/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.text;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.sonar.squid.measures.Metric;
import org.sonar.squid.recognizer.CodeRecognizer;

import com.puppycrawl.tools.checkstyle.api.TextBlock;

public class Source {

  private Line[]         lines;
  private CodeRecognizer codeRecognizer;
  private Set<Integer>   noSonarTagLines = new HashSet<Integer>();

  public Source(String[] stringLines, CodeRecognizer codeRecognizer, TextBlock... comments) {
    this(stringLines, codeRecognizer, Arrays.asList(comments));
  }

  public Source(String[] stringLines, CodeRecognizer codeRecognizer, Collection<TextBlock> comments) {
    lines = new Line[stringLines.length];
    this.codeRecognizer = codeRecognizer;
    int index = 0;
    for (String stringLine : stringLines) {
      lines[index++] = new Line(stringLine);
    }
    attachTextBlockToLine(comments);
    for (int lineIndex = 0; lineIndex < lines.length; lineIndex++) {
      computeBlankLine(lines[lineIndex]);
      computeHeaderCommentLine(lines[lineIndex]);
      computeCommentLine(lines[lineIndex]);
      computeCommentBlankLine(lines[lineIndex]);
      computeLineOfCode(lines[lineIndex]);
      computeNoSonarTag(lines[lineIndex], lineIndex + 1);
    }
  }

  private void computeNoSonarTag(Line line, int lineIndex) {
    if (line.isThereNoSonarTag()) {
      noSonarTagLines.add(lineIndex);
    }
  }

  private void computeLineOfCode(Line line) {
    if (line.isThereCode()) {
      line.setMeasure(Metric.LINES_OF_CODE, 1);
    }
  }

  private void computeHeaderCommentLine(Line line) {
    if (line.isThereComment() && !line.isThereBlankComment() && line.isThereLicenseHeaderComment()) {
      line.setMeasure(Metric.HEADER_COMMENT_LINES, 1);
    }
  }

  private void computeCommentLine(Line line) {
    if (line.isThereComment() && !line.isThereBlankComment()) {
      if (line.isThereJavadoc() || line.isThereLicenseHeaderComment()) {
        line.setMeasure(Metric.COMMENT_LINES, 1);
        return;
      }

      boolean isCommentedOutCode = codeRecognizer.isLineOfCode(line.getComment());
      if (!isCommentedOutCode) {
        line.setMeasure(Metric.COMMENT_LINES, 1);
      } else {
        line.setMeasure(Metric.COMMENTED_OUT_CODE_LINES, 1);
      }
    }
  }

  private void computeBlankLine(Line line) {
    if (line.isBlank()) {
      line.setMeasure(Metric.BLANK_LINES, 1);
    }
  }

  private void computeCommentBlankLine(Line line) {
    if (line.isThereBlankComment()) {
      line.setMeasure(Metric.COMMENT_BLANK_LINES, 1);
    }
  }

  private void attachTextBlockToLine(Collection<TextBlock> comments) {
    for (TextBlock comment : comments) {
      boolean isJavadoc = comment.getText()[0].startsWith("/**");
      boolean isLicenseHeader = (comment.getStartLineNo() == 1);
      for (int index = comment.getStartLineNo(); index <= comment.getEndLineNo(); index++) {
        lines[index - 1].setComment(comment.getText()[index - comment.getStartLineNo()], isJavadoc, isLicenseHeader);
      }
    }
  }

  public int getMeasure(Metric metric) {
    return getMeasure(metric, 1, lines.length);
  }

  public int getMeasure(Metric metric, int fromLine, int toLine) {
    if (toLine > lines.length) {
      throw new IllegalStateException("There are only " + lines.length + " lines in the file and you're trying to reach line " + toLine);
    }
    if (fromLine < 1) {
      throw new IllegalStateException("Line index starts from 1 and not from " + fromLine);
    }

    int measure = 0;
    for (int index = fromLine; index < toLine + 1; index++) {
      measure += lines[index - 1].getInt(metric);
    }
    return measure;
  }

  public Set<Integer> getNoSonarTagLines() {
    return noSonarTagLines;
  }
}
