/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.text;

import java.io.BufferedReader;
import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;

import org.sonar.squid.api.AnalysisException;

public class CodeReader {

  private final Reader reader;
  private int HISTORY_SIZE = 3;
  private int[] lastChars = new int[HISTORY_SIZE];

  public CodeReader(Reader reader) {
    this.reader = new BufferedReader(reader);
    for (int i = 0; i < lastChars.length; i++) {
      lastChars[i] = -1;
    }
  }

  public CodeReader(String code) {
    this(new StringReader(code));
  }

  private int read() throws IOException {
    if (!reader.ready()) {
      return -1;
    }
    return reader.read();
  }

  public final void mark(int readAheadLimit) throws IOException {
    if (reader.markSupported()) {
      throw new AnalysisException("Mark are not supported on provided Reader.");
    }
    reader.mark(readAheadLimit);
  }

  public final int lastChar() {
    return lastChars[0];
  }

  public final int peek() {
    try {
      reader.mark(1);
      int nextChar = read();
      reader.reset();
      return nextChar;
    } catch (IOException e) {
      throw new AnalysisException("Unable to read on input stream.", e);
    }
  }

  public final int pop() {
    try {
      setLastChar(read());
      return lastChar();
    } catch (IOException e) {
      throw new AnalysisException("Unable to read on input stream.", e);
    }
  }

  private void setLastChar(int read) {
    for (int i = lastChars.length - 1; i > 0; i--) {
      lastChars[i] = lastChars[i - 1];
    }
    lastChars[0] = read;

  }

  public String readTo(EndTokenMatcher matcher) {
    CharArrayWriter result = new CharArrayWriter();
    result.append((char) pop());
    while (!matcher.match(peek()) && peek() != -1) {
      result.append((char) pop());
    }
    return result.toString();
  }

  public void close() {
    try {
      reader.close();
    } catch (IOException e) {
      throw new AnalysisException("Unable to close the reader on source code.", e);
    }
  }

  public char[] peek(int index) {
    try {
      char[] result = new char[index];
      reader.mark(index);
      reader.read(result, 0, index);
      reader.reset();
      return result;
    } catch (IOException e) {
      throw new AnalysisException("Unable to read on input stream.", e);
    }
  }

  public int lastChar(int index) {
    return lastChars[index];
  }
}
