/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.ast.visitor;

import org.sonar.squid.api.SourcePackage;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.indexer.SquidIndex;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.FullIdent;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

public class PackageVisitor extends AstVisitor {

  private SquidIndex indexer;

  public PackageVisitor(SquidIndex indexer) {
    this.indexer = indexer;
  }

  @Override
  public void visitFile(DetailAST ast) {
    if (ast == null) {
      // ast can be null for empty classes
      return;
    }
    SourceCode packageRes = extractPackage(ast);
    if (peekResource().hasChild(packageRes)) {
      packageRes = indexer.search(packageRes.getKey());
    }
    addResource(packageRes);
  }

  @Override
  public void leaveFile(DetailAST ast) {
    if (ast == null) {
      // ast can be null for empty classes
      return;
    }
    popResource();
  }

  private SourcePackage extractPackage(DetailAST ast) {
    SourcePackage packageRes;
    if (ast.getType() != TokenTypes.PACKAGE_DEF) {
      packageRes = new SourcePackage("[default]");
    } else {
      String packageName = FullIdent.createFullIdent(ast.getLastChild().getPreviousSibling()).getText();
      packageRes = new SourcePackage(packageName);
    }
    return packageRes;
  }
}
