/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.ast.visitor;

import java.util.Arrays;
import java.util.List;

import org.sonar.squid.api.SourceCode;
import org.sonar.squid.measures.Metric;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.Scope;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

public class PublicApiVisitor extends AstVisitor {

  private static final List<Integer> wantedTokens = Arrays.asList(TokenTypes.CLASS_DEF, TokenTypes.INTERFACE_DEF, TokenTypes.METHOD_DEF,
                                                      TokenTypes.CTOR_DEF, TokenTypes.ANNOTATION_DEF, TokenTypes.ANNOTATION_FIELD_DEF,
                                                      TokenTypes.VARIABLE_DEF);

  public PublicApiVisitor() {
  }

  @Override
  public List<Integer> getWantedTokens() {
    return wantedTokens;
  }

  @Override
  public void visitToken(DetailAST ast) {
    SourceCode currentResource = peekResource();
    if (AstUtils.isScope(AstUtils.getScope(ast), Scope.PUBLIC) || AstUtils.isType(ast, TokenTypes.ANNOTATION_FIELD_DEF)) {
      // final static vars are not part of public API
      // to avoid noise on public constants class or interfaces
      if ((AstUtils.isClassVariable(ast) || AstUtils.isInterfaceVariable(ast)) && AstUtils.isFinal(ast) && AstUtils.isStatic(ast)) {
        return;
      }
      currentResource.add(Metric.PUBLIC_API, 1);
      boolean isDocumented = getFileContents().getJavadocBefore(ast.getLineNo()) != null;
      if (isDocumented) {
        currentResource.add(Metric.PUBLIC_DOC_API, 1);
      }
    }
  }
}
