/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.bytecode.asm;

import org.objectweb.asm.Type;
import org.objectweb.asm.signature.SignatureReader;

public class AsmMethod extends AsmUnit {

  String    name;
  String    key;
  String[]  exceptions;
  AsmType   returnType;
  AsmType[] arguments;
  String[]  signatureClasses;

  public AsmMethod(int access, String name, String descriptor, String descriptorWithGenerics, String[] exceptions) {
    super(access);
    this.name = name;
    key = name + descriptor;
    this.exceptions = exceptions;
    initReturnType(descriptor);
    initArguments(descriptor);
    if (descriptorWithGenerics == null) {
      extractClasses(descriptor);
    } else {
      extractClasses(descriptorWithGenerics);
    }
  }

  private void initArguments(String descriptor) {
    Type[] types = Type.getArgumentTypes(descriptor);
    arguments = new AsmType[types.length];
    for (int i = 0; i < arguments.length; i++) {
      arguments[i] = new AsmType(types[i]);
    }
  }

  private void extractClasses(String signature) {
    AsmSignatureVisitor signatureVisitor = new AsmSignatureVisitor();
    new SignatureReader(signature).accept(signatureVisitor);
    signatureClasses = signatureVisitor.getInternalNames().toArray(new String[0]);
  }

  private void initReturnType(String descriptor) {
    returnType = new AsmType(Type.getReturnType(descriptor));
  }

  public String getName() {
    return name;
  }

  public String getKey() {
    return key;
  }

  public boolean hasReturnType() {
    return !returnType.isVoid();
  }

  public AsmType getReturnType() {
    return returnType;
  }

  public boolean hasArguments() {
    return arguments.length != 0;
  }

  public AsmType[] getArguments() {
    return arguments;
  }

  public String[] getExceptions() {
    return exceptions;
  }
  
  public String[] getSignatureClasses(){
    return signatureClasses;
  }

  public String getReturnTypePath() {
    return returnType.getPath();
  }
}
