/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.graph;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

public class CycleDetector {

  private final Collection<? extends Node> resourcesToAnalyze;
  private final List<Node> alreadyAnalyzed;
  private final List<Cycle>                cycles = new ArrayList<Cycle>(); ;

  public CycleDetector(Collection<? extends Node> resourcesToAnalyze) {
    this.resourcesToAnalyze = resourcesToAnalyze;
    this.alreadyAnalyzed = new ArrayList<Node>();
    analyze();
  }

  public CycleDetector(Node... resources) {
    this(Arrays.asList(resources));
  }

  private void analyze() {
    for (Node resource : resourcesToAnalyze) {
      if (! alreadyAnalyzed.contains(resource)) {
        analyze(resource, new ArrayList<Node>());
      }
    }
  }

  private void analyze(Node node, List<Node> path) {
    path.add(node);
    alreadyAnalyzed.add(node);
    for (Node dependency : node.dependsUpon()) {
      if (path.contains(dependency)) {
        Cycle cycle = new Cycle(new ArrayList<Node>(path.subList(path.indexOf(dependency), path.indexOf(node) + 1)));
        if (!cycles.contains(cycle)) {
          cycles.add(cycle);
        }
      } else {
        analyze(dependency, new ArrayList<Node>(path));
      }
    }
  }

  public List<Cycle> getCycles() {
    return cycles;
  }

  public Cycle getCycles(int cycleNumber) {
    return cycles.get(cycleNumber);
  }

  public int getCyclesNumber() {
    return cycles.size();
  }

  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append("There are " + getCyclesNumber() + " cycles.\n");
    for (Cycle cycle : cycles) {
      builder.append(cycle);
    }
    return builder.toString();
  }

  public boolean contains(Cycle cycle) {
    return cycles.contains(cycle);
  }
}
