/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.text.colorizer;

import java.util.Arrays;

import org.sonar.squid.text.CodeReader;
import org.sonar.squid.text.EndTokenMatcher;

public abstract class MultilinesDocTokenizer extends Tokenizer {

  private boolean      commentStartedOnPreviousLine = false;
  private final char[] startToken;

  public MultilinesDocTokenizer(String startToken, String tagBefore, String tagAfter) {
    super(tagBefore, tagAfter);
    this.startToken = startToken.toCharArray();
  }

  public MultilinesDocTokenizer(String startToken) {
    this("", "", startToken);
  }

  @Override
  public boolean hasNextToken(CodeReader code) {
    return code.peek() != '\n' && code.peek() != '\r'
        && (commentStartedOnPreviousLine || (code.peek() == startToken[0] && Arrays.equals(code.peek(startToken.length), startToken)));
  }

  @Override
  public String nextToken(CodeReader code) {
    String comment = code.readTo(new JavadocEndTokenMatcher(code));
    return tagBefore + comment + tagAfter;
  }

  private class JavadocEndTokenMatcher implements EndTokenMatcher {

    CodeReader code;

    public JavadocEndTokenMatcher(CodeReader code) {
      this.code = code;
    }

    public boolean match(int endFlag) {
      if (code.lastChar() == '/' && code.lastChar(1) == '*') {
        commentStartedOnPreviousLine = false;
        return true;
      }
      if (endFlag == '\r' || endFlag == '\n') {
        commentStartedOnPreviousLine = true;
        return true;
      }
      return false;
    }
  }

}
