/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.graph;

import java.util.Arrays;
import java.util.List;

public class Cycle {

  private List<Node> nodes;
  private int        hashCode;

  public Cycle(List<Node> nodes) {
    this.nodes = nodes;
    hashCode = 0;
    for (Node node : nodes) {
      hashCode += node.hashCode();
    }
  }

  public Cycle(Node... nodes) {
    this(Arrays.asList(nodes));
  }

  public int getCycleLength() {
    return nodes.size();
  }

  public boolean contains(Node resource) {
    return nodes.contains(resource);
  }

  public List<Node> getNodes() {
    return nodes;
  }

  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    Edge weaker = getWeakerEdge();
    builder.append("Cycle between " + getCycleLength() + " resources : \n");
    for (Node from : nodes) {
      Edge relation = getEdgeFrom(from);
      String weakerFlag = "   ";
      if (weaker.equals(relation)) {
        weakerFlag = " * ";
      }
      builder.append(weakerFlag + relation.getFrom().getKey() + " --" + relation.getWeight() + "--> " + "\n");
    }
    return builder.toString();
  }

  @Override
  public boolean equals(Object object) {
    if (!(object instanceof Cycle)) {
      return false;
    }
    return hashCode == object.hashCode();
  }
  
  @Override
  public int hashCode(){
    return hashCode;
  }

  public Edge getWeakerEdge() {
    Edge weaker = null;
    for (Node from : nodes) {
      Edge relation = getEdgeFrom(from);
      if (weaker == null || weaker.getWeight() > relation.getWeight()) {
        weaker = relation;
      }
    }
    return weaker;
  }

  private Edge getEdgeFrom(Node from) {
    int index = nodes.indexOf(from);
    Node to = nodes.get((index + 1 < nodes.size() ? index + 1 : 0));
    return from.getEdgeTo(to);
  }
}
