/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.ast.visitor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.sonar.java.language.ArgumentAndReturnType;
import org.sonar.java.language.ArgumentType;
import org.sonar.java.language.JavaMethodSignature;
import org.sonar.java.language.JavaType;
import org.sonar.java.language.ReturnType;
import org.sonar.squid.api.SourceMethod;
import org.sonar.squid.measures.Metric;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

public class MethodVisitor extends AstVisitor {

  private static final String CONSTRUCTOR = "<init>";

  private static final List<Integer> wantedTokens = Arrays.asList(TokenTypes.CTOR_DEF, TokenTypes.METHOD_DEF);
  private static final Map<Integer, JavaType> tokenJavaTypeMapping = new HashMap<Integer, JavaType>();

  static {
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_BYTE, JavaType.BYTE);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_CHAR, JavaType.CHAR);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_SHORT, JavaType.SHORT);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_INT, JavaType.INT);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_LONG, JavaType.LONG);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_BOOLEAN, JavaType.BOOLEAN);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_FLOAT, JavaType.FLOAT);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_DOUBLE, JavaType.DOUBLE);
    tokenJavaTypeMapping.put(TokenTypes.LITERAL_VOID, JavaType.VOID);
  }

  @Override
  public List<Integer> getWantedTokens() {
    return wantedTokens;
  }

  @Override
  public void visitToken(DetailAST ast) {
    boolean isConstructor = isConstructor(ast);
    String methodName = buildMethodSignature(ast, isConstructor);
    SourceMethod sourceMethod = new SourceMethod(peekParentClass(), methodName, ast.getLineNo());
    sourceMethod.setMeasure(Metric.METHODS, 1);
    if (isConstructor) {
      sourceMethod.setMeasure(Metric.CONSTRUCTORS, 1);
    }
    addResource(sourceMethod);
  }

  private boolean isConstructor(DetailAST ast) {
    return ast.getType() == TokenTypes.CTOR_DEF;
  }

  @Override
  public void leaveToken(DetailAST ast) {
    popResource();
  }

  private String buildMethodSignature(DetailAST ast, boolean isConstructor) {
    String methodName = extractMethodName(ast, isConstructor);
    ReturnType returnType = extractMethodReturnType(ast, isConstructor);
    List<ArgumentType> argumentTypes = extractMethodArgumentTypes(ast);
    JavaMethodSignature signature = new JavaMethodSignature(methodName, returnType, argumentTypes);
    return signature.dumpMethodSignature();
  }

  private List<ArgumentType> extractMethodArgumentTypes(DetailAST ast) {
    List<ArgumentType> argumentTypes = new ArrayList<ArgumentType>();
    DetailAST child = ast.findFirstToken(TokenTypes.PARAMETERS).findFirstToken(TokenTypes.PARAMETER_DEF);
    while (child != null) {
      if (child.getType() == TokenTypes.PARAMETER_DEF) {
        ArgumentAndReturnType argumentType = extractArgumentAndRreturnType(child.findFirstToken(TokenTypes.TYPE));
        argumentTypes.add(new ArgumentType(argumentType));
      }
      child = (DetailAST) child.getNextSibling();
    }
    return argumentTypes;
  }

  private String extractMethodName(DetailAST ast, boolean isConstructor) {
    if (isConstructor) {
      return CONSTRUCTOR;
    }
    return ast.findFirstToken(TokenTypes.IDENT).getText();
  }

  private ReturnType extractMethodReturnType(DetailAST ast, boolean isConstructor) {
    if (isConstructor) {
      return null;
    }
    ArgumentAndReturnType returnType = extractArgumentAndRreturnType(ast.findFirstToken(TokenTypes.TYPE));
    return new ReturnType(returnType);
  }

  private ArgumentAndReturnType extractArgumentAndRreturnType(DetailAST ast) {
    boolean isArray = isArrayType(ast);
    for (Integer tokenType : tokenJavaTypeMapping.keySet()) {
      if (ast.branchContains(tokenType)) {
        return new ArgumentAndReturnType(tokenJavaTypeMapping.get(tokenType), isArray);
      }
    }
    if (isObjectType(ast)) {
      String className;
      if (isArray) {
        className = extractClassName(ast.findFirstToken(TokenTypes.ARRAY_DECLARATOR));
      } else {
        className = extractClassName(ast);
      }
      return new ArgumentAndReturnType(className, isArray);
    }
    throw new IllegalStateException("No Known TokenType has been found at line " + ast.getLineNo() + " of file "
        + getFileContents().getFilename());
  }

  private String extractClassName(DetailAST ast) {
    if (ast.findFirstToken(TokenTypes.DOT) != null) {
      return ast.findFirstToken(TokenTypes.DOT).getLastChild().getText();
    } else {
      return ast.findFirstToken(TokenTypes.IDENT).getText();
    }
  }

  private boolean isObjectType(DetailAST ast) {
    return ast.branchContains(TokenTypes.IDENT);
  }

  private boolean isArrayType(DetailAST ast) {
    if(ast.findFirstToken(TokenTypes.IDENT) != null){
      return false;
    }
    return ast.branchContains(TokenTypes.ARRAY_DECLARATOR);
  }
}
