/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.ast;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Stack;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.java.ast.visitor.AccessorVisitor;
import org.sonar.java.ast.visitor.AnonymousInnerClassVisitor;
import org.sonar.java.ast.visitor.AstVisitor;
import org.sonar.java.ast.visitor.BlankLinesVisitor;
import org.sonar.java.ast.visitor.BranchVisitor;
import org.sonar.java.ast.visitor.ClassVisitor;
import org.sonar.java.ast.visitor.CommentVisitor;
import org.sonar.java.ast.visitor.ComplexityVisitor;
import org.sonar.java.ast.visitor.EndAtLineVisitor;
import org.sonar.java.ast.visitor.FileVisitor;
import org.sonar.java.ast.visitor.LinesOfCodeVisitor;
import org.sonar.java.ast.visitor.LinesVisitor;
import org.sonar.java.ast.visitor.MethodVisitor;
import org.sonar.java.ast.visitor.PackageVisitor;
import org.sonar.java.ast.visitor.PublicApiVisitor;
import org.sonar.java.ast.visitor.StatementVisitor;
import org.sonar.squid.Squid;
import org.sonar.squid.api.AnalysisException;
import org.sonar.squid.api.CodeScanner;
import org.sonar.squid.api.CodeVisitor;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.api.SquidConfiguration;

import com.puppycrawl.tools.checkstyle.Checker;
import com.puppycrawl.tools.checkstyle.ConfigurationLoader;
import com.puppycrawl.tools.checkstyle.DefaultLogger;
import com.puppycrawl.tools.checkstyle.PropertiesExpander;
import com.puppycrawl.tools.checkstyle.api.AuditEvent;
import com.puppycrawl.tools.checkstyle.api.AuditListener;
import com.puppycrawl.tools.checkstyle.api.Configuration;

/**
 * Squid uses Checkstyle to get an out-of-the-box java parser with AST generation and visitor pattern support.
 */
public class JavaAstScanner implements CodeScanner {

  private SquidConfiguration conf;
  private List<AstVisitor> visitors = new ArrayList<AstVisitor>();

  public JavaAstScanner(SquidConfiguration conf) {
    this.conf = conf;
  }

  /**
   * Create and execute the Checkstyle engine.
   * 
   * @param files
   *          collection of files to analyse. This list shouldn't contain and directory.
   * @param charset
   *          the default charset to use to read files
   */
  private void launchCheckstyleEngine(Collection<File> files, Charset charset) {
    ErrorsListener listener = new ErrorsListener();
    Checker c = createChecker(charset, listener);
    File[] processedFiles = new File[files.size()];
    files.toArray(processedFiles);
    c.process(processedFiles);
    c.destroy();
    if (!listener.getErrors().isEmpty()) {
      throw listener.getErrors().get(0);
    }
  }

  /**
   * Creates the Checkstyle Checker object.
   * 
   * @return a nice new fresh Checkstyle Checker
   */
  private Checker createChecker(Charset charset, ErrorsListener errorsListener) {
    try {
      InputStream checkstyleConfig = JavaAstScanner.class.getClassLoader().getResourceAsStream("checkstyle-configuration.xml");
      String readenConfig = IOUtils.toString(checkstyleConfig);
      readenConfig = readenConfig.replace("${charset}", charset.toString());
      checkstyleConfig = new ByteArrayInputStream(readenConfig.getBytes());
      Configuration config = ConfigurationLoader.loadConfiguration(checkstyleConfig, new PropertiesExpander(System.getProperties()), false);
      Checker c = new Checker();
      c.configure(config);
      final Logger logger = LoggerFactory.getLogger(Squid.class);
      StreamLogger infoLogger = new StreamLogger() {

        @Override
        public void log(String log) {
          logger.info(log);
        }
      };
      StreamLogger errorLogger = new StreamLogger() {

        @Override
        public void log(String log) {
          logger.error(log);
        }
      };
      c.addListener(new DefaultLogger(infoLogger, true, errorLogger, true));
      c.addListener(errorsListener);
      return c;
    } catch (final Exception e) {
      throw new AnalysisException(
          "Unable to create Checkstyle Checker object with 'checkstyle-configuration.xml' as Checkstyle configuration file name", e);
    }
  }

  private static class ErrorsListener implements AuditListener {

    private List<AnalysisException> errors = new ArrayList<AnalysisException>();

    /**
     * {@inheritDoc}
     */
    public void addError(AuditEvent evt) {
      // some projects can have file parsing errors (tapestry for example)
      // currently do not throw an error.
      // see
      // http://sourceforge.net/tracker/?func=detail&atid=397078&aid=1667137&group_id=29721
      if (evt.getMessage().contains("expecting EOF, found")) {
        return;
      }
      errors.add(new AnalysisException(evt.getMessage()));
    }

    /**
     * {@inheritDoc}
     */
    public void addException(AuditEvent evt, Throwable throwable) {
      errors.add(new AnalysisException(evt.getMessage(), throwable));
    }

    /**
     * {@inheritDoc}
     */
    public void auditFinished(AuditEvent evt) {
    }

    /**
     * {@inheritDoc}
     */
    public void auditStarted(AuditEvent evt) {
    }

    /**
     * {@inheritDoc}
     */
    public void fileFinished(AuditEvent evt) {
    }

    /**
     * {@inheritDoc}
     */
    public void fileStarted(AuditEvent evt) {
    }

    /**
     * {@inheritDoc}
     */
    public List<AnalysisException> getErrors() {
      return errors;
    }
  }

  private abstract static class StreamLogger extends OutputStream {

    private StringBuilder builder = new StringBuilder(256);

    @Override
    public void write(int byteToWrite) throws IOException {
      char character = (char) byteToWrite;
      if (character == '\n') {
        logAndResetBuffer();
      } else {
        builder.append(character);
      }
    }

    private void logAndResetBuffer() {
      log(builder.toString().trim());
      builder.setLength(0);
    }

    public abstract void log(String log);

    @Override
    public void close() throws IOException {
      if (builder.length() > 0) {
        logAndResetBuffer();
      }
      super.close();
    }
  }

  public void scanCode(SourceCode project, Collection<File> filesToAnalyse) {
    Stack<SourceCode> resourcesStack = new Stack<SourceCode>();
    resourcesStack.add(project);
    for (AstVisitor visitor : visitors) {
      visitor.setResourcesStack(resourcesStack);
    }
    CheckstyleSquidBridge.setASTVisitors(visitors);
    CheckstyleSquidBridge.setSquidConfiguration(conf);
    launchCheckstyleEngine(filesToAnalyse, conf.getCharset());
  }

  public void accept(CodeVisitor visitor) {
    visitors.add((AstVisitor) visitor);
  }

  public Collection<Class<? extends CodeVisitor>> getVisitors() {
    List<Class<? extends CodeVisitor>> visitorClasses = new ArrayList<Class<? extends CodeVisitor>>();
    visitorClasses.add(PackageVisitor.class);
    visitorClasses.add(FileVisitor.class);
    visitorClasses.add(ClassVisitor.class);
    visitorClasses.add(AnonymousInnerClassVisitor.class);
    visitorClasses.add(MethodVisitor.class);
    visitorClasses.add(EndAtLineVisitor.class);
    visitorClasses.add(LinesVisitor.class);
    visitorClasses.add(BlankLinesVisitor.class);
    visitorClasses.add(CommentVisitor.class);
    visitorClasses.add(PublicApiVisitor.class);
    visitorClasses.add(BranchVisitor.class);
    visitorClasses.add(StatementVisitor.class);
    if (conf.isAnalysePropertyAccessors()) {
      visitorClasses.add(AccessorVisitor.class);
    }
    visitorClasses.add(ComplexityVisitor.class);
    visitorClasses.add(LinesOfCodeVisitor.class);
    return visitorClasses;
  }
}
