/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.ast;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Stack;

import org.apache.commons.io.IOUtils;
import org.sonar.java.ast.visitor.AccessorVisitor;
import org.sonar.java.ast.visitor.AnonymousInnerClassVisitor;
import org.sonar.java.ast.visitor.AstVisitor;
import org.sonar.java.ast.visitor.BlankLinesVisitor;
import org.sonar.java.ast.visitor.BranchVisitor;
import org.sonar.java.ast.visitor.ClassVisitor;
import org.sonar.java.ast.visitor.CommentVisitor;
import org.sonar.java.ast.visitor.ComplexityVisitor;
import org.sonar.java.ast.visitor.EndAtLineVisitor;
import org.sonar.java.ast.visitor.FileVisitor;
import org.sonar.java.ast.visitor.LinesOfCodeVisitor;
import org.sonar.java.ast.visitor.LinesVisitor;
import org.sonar.java.ast.visitor.MethodVisitor;
import org.sonar.java.ast.visitor.PackageVisitor;
import org.sonar.java.ast.visitor.PublicApiVisitor;
import org.sonar.java.ast.visitor.StatementVisitor;
import org.sonar.squid.api.AnalysisException;
import org.sonar.squid.api.CodeScanner;
import org.sonar.squid.api.CodeVisitor;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.api.SquidConfiguration;

import com.puppycrawl.tools.checkstyle.Checker;
import com.puppycrawl.tools.checkstyle.ConfigurationLoader;
import com.puppycrawl.tools.checkstyle.PropertiesExpander;
import com.puppycrawl.tools.checkstyle.api.Configuration;

/**
 * Squid uses Checkstyle to get an out-of-the-box java parser with AST generation and visitor pattern support.
 */
public class JavaAstScanner implements CodeScanner {

  private SquidConfiguration conf;
  private List<AstVisitor> visitors = new ArrayList<AstVisitor>();

  public JavaAstScanner(SquidConfiguration conf) {
    this.conf = conf;
  }

  /**
   * Create and execute the Checkstyle engine.
   * 
   * @param files
   *          collection of files to analyse. This list shouldn't contain and directory.
   * @param charset
   *          the default charset to use to read files
   */
  private void launchCheckstyleEngine(Collection<File> files, Charset charset) {
    Checker c = createChecker(charset);
    File[] processedFiles = new File[files.size()];
    files.toArray(processedFiles);
    c.process(processedFiles);
    c.destroy();
  }

  /**
   * Creates the Checkstyle Checker object.
   * 
   * @return a nice new fresh Checkstyle Checker
   */
  private Checker createChecker(Charset charset) {
    try {
      InputStream checkstyleConfig = JavaAstScanner.class.getClassLoader().getResourceAsStream("checkstyle-configuration.xml");
      String readenConfig = IOUtils.toString(checkstyleConfig);
      readenConfig = readenConfig.replace("${charset}", charset.toString());
      checkstyleConfig = new ByteArrayInputStream(readenConfig.getBytes());
      Configuration config = ConfigurationLoader.loadConfiguration(checkstyleConfig, new PropertiesExpander(System.getProperties()), false);
      Checker c = new Checker();
      c.configure(config);
      c.addListener(new CheckstyleAuditListener());
      return c;
    } catch (final Exception e) {
      throw new AnalysisException(
          "Unable to create Checkstyle Checker object with 'checkstyle-configuration.xml' as Checkstyle configuration file name", e);
    }
  }

  public void scanCode(SourceCode project, Collection<File> filesToAnalyse) {
    Stack<SourceCode> resourcesStack = new Stack<SourceCode>();
    resourcesStack.add(project);
    for (AstVisitor visitor : visitors) {
      visitor.setResourcesStack(resourcesStack);
    }
    CheckstyleSquidBridge.setASTVisitors(visitors);
    CheckstyleSquidBridge.setSquidConfiguration(conf);
    launchCheckstyleEngine(filesToAnalyse, conf.getCharset());
  }

  public void accept(CodeVisitor visitor) {
    visitors.add((AstVisitor) visitor);
  }

  public Collection<Class<? extends CodeVisitor>> getVisitors() {
    List<Class<? extends CodeVisitor>> visitorClasses = new ArrayList<Class<? extends CodeVisitor>>();
    visitorClasses.add(PackageVisitor.class);
    visitorClasses.add(FileVisitor.class);
    visitorClasses.add(ClassVisitor.class);
    visitorClasses.add(AnonymousInnerClassVisitor.class);
    visitorClasses.add(MethodVisitor.class);
    visitorClasses.add(EndAtLineVisitor.class);
    visitorClasses.add(LinesVisitor.class);
    visitorClasses.add(BlankLinesVisitor.class);
    visitorClasses.add(CommentVisitor.class);
    visitorClasses.add(PublicApiVisitor.class);
    visitorClasses.add(BranchVisitor.class);
    visitorClasses.add(StatementVisitor.class);
    if (conf.isAnalysePropertyAccessors()) {
      visitorClasses.add(AccessorVisitor.class);
    }
    visitorClasses.add(ComplexityVisitor.class);
    visitorClasses.add(LinesOfCodeVisitor.class);
    return visitorClasses;
  }
}
