/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.bytecode;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.objectweb.asm.FieldVisitor;
import org.objectweb.asm.Label;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.EmptyVisitor;
import org.objectweb.asm.signature.SignatureReader;
import org.sonar.java.bytecode.asm.AsmSignatureVisitor;
import org.sonar.squid.api.SourceClass;
import org.sonar.squid.indexer.SquidIndex;

public class AsmSquidBridge extends EmptyVisitor {

  private List<AsmVisitor> visitors = new ArrayList<AsmVisitor>();
  private SourceClass currentClass;
  private final SquidIndex indexer;

  public AsmSquidBridge(SquidIndex indexer, List<AsmVisitor> asmVisitors) {
    this.indexer = indexer;
    this.visitors = asmVisitors;
  }

  @Override
  public void visit(final int version, final int access, final String className, final String signature, final String superClass,
      final String[] interfaces) {
    currentClass = searchSquidClass(className);
    for (AsmVisitor visitor : visitors) {
      visitor.visitClass(new AccessFlags(access), currentClass, searchSquidClass(superClass), searchSquidClasses(interfaces),
          searchSquidClasses(analyzeSignature(signature)));
    }
  }

  @Override
  public FieldVisitor visitField(final int access, final String fieldName, final String fieldType, final String signature,
      final Object value) {
    for (AsmVisitor visitor : visitors) {
      visitor.visitField(new AccessFlags(access), currentClass, fieldName, getSquidClassFromDescriptor(fieldType),
          searchSquidClasses(analyzeSignature(signature)), analyzeFieldValue(value));
    }
    return this;
  }

  @Override
  public MethodVisitor visitMethod(final int access, final String methodName, final String methodDesc, final String signature,
      final String[] exceptions) {
    for (AsmVisitor visitor : visitors) {
      visitor.visitMethod(new AccessFlags(access), currentClass, methodName, getReturnSquidClass(methodDesc),
          getArgumentSquidClasses(methodDesc), searchSquidClasses(analyzeSignature(signature)), searchSquidClasses(exceptions));
    }
    return this;
  }

  @Override
  public void visitFieldInsn(final int opcode, final String ownerClass, final String name, final String fieldDescription) {
    for (AsmVisitor visitor : visitors) {
      visitor.visitOutsideFieldAccess(currentClass, searchSquidClass(ownerClass), getSquidClassFromDescriptor(fieldDescription));
    }
  }

  @Override
  public void visitTypeInsn(final int opcode, final String type) {
    for (AsmVisitor visitor : visitors) {
      visitor.visitTypeInsn(currentClass, searchSquidClass(type));
    }
  }

  @Override
  public void visitMethodInsn(final int opcode, final String ownerClass, final String methodName, final String methodDesc) {
    for (AsmVisitor visitor : visitors) {
      visitor.visitOutsideMethodAccess(currentClass, searchSquidClass(ownerClass), getArgumentSquidClasses(methodDesc));
    }
  }

  @Override
  public void visitTryCatchBlock(final Label start, final Label end, final Label handler, final String exception) {
    for (AsmVisitor visitor : visitors) {
      visitor.visitTryCatchBlock(currentClass, searchSquidClass(exception));
    }
  }

  private Set<SourceClass> getArgumentSquidClasses(String desc) {
    Set<SourceClass> resources = new HashSet<SourceClass>();
    Type[] types = Type.getArgumentTypes(desc);
    for (int i = 0; i < types.length; i++) {
      resources.add(getSquidClassFromType(types[i]));
    }
    return resources;
  }

  private SourceClass getReturnSquidClass(String desc) {
    return getSquidClassFromType(Type.getReturnType(desc));
  }

  private SourceClass analyzeFieldValue(Object value) {
    if (value instanceof Type) {
      return getSquidClassFromType((Type) value);
    }
    return null;
  }

  private String[] analyzeSignature(String signature) {
    AsmSignatureVisitor signatureVisitor = new AsmSignatureVisitor();
    if (signature != null) {
      new SignatureReader(signature).accept(signatureVisitor);
    }
    Set<String> internalNames = signatureVisitor.getInternalNames();
    return internalNames.toArray(new String[internalNames.size()]);
  }

  private SourceClass getSquidClassFromType(Type type) {
    String className;
    switch (type.getSort()) {
      case Type.OBJECT:
        className = type.getInternalName();
        break;
      case Type.ARRAY:
        if (type.getElementType().getSort() == Type.OBJECT) {
          className = type.getElementType().getInternalName();
        } else {
          return null;
        }
        break;
      default:
        return null;
    }
    return (SourceClass) indexer.search(className);
  }

  private SourceClass getSquidClassFromDescriptor(String typeDescription) {
    Type type = Type.getType(typeDescription);
    return getSquidClassFromType(type);
  }

  private SourceClass searchSquidClass(String internalName) {
    if (internalName == null) {
      return null;
    }
    internalName = attachAnoInnerClassToParentClass(internalName);
    return (SourceClass) indexer.search(internalName);
  }

  private String attachAnoInnerClassToParentClass(String internalName) {
    if (indexer.search(internalName) == null && internalName.indexOf('$') != -1 && internalName.matches(".*\\d")) {
      return internalName.substring(0, internalName.indexOf('$'));
    }
    return internalName;
  }

  private Set<SourceClass> searchSquidClasses(String[] internalNames) {
    Set<SourceClass> resources = new HashSet<SourceClass>();
    if (internalNames == null) {
      return resources;
    }
    for (int i = 0; i < internalNames.length; i++) {
      resources.add(searchSquidClass(internalNames[i]));
    }
    return resources;
  }
}
