/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.bytecode;

import java.util.Set;

import org.sonar.squid.api.SourceClass;
import org.sonar.squid.api.SourceFile;
import org.sonar.squid.api.SourcePackage;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.graph.Edge;
import org.sonar.squid.graph.EdgeUsage;
import org.sonar.squid.measures.Metric;

public class DependencyVisitor extends AsmVisitor {

  @Override
  protected void visitClass(AccessFlags accessFlags, SourceClass currentClass, SourceClass superClass, Set<SourceClass> interfaces,
      Set<SourceClass> generics) {
    link(currentClass, superClass, EdgeUsage.EXTENDS);
    link(currentClass, interfaces, EdgeUsage.IMPLEMENTS);
    link(currentClass, generics, EdgeUsage.CONTAINS);
  }

  @Override
  public void visitField(AccessFlags accessFlags, SourceClass currentClass, String fieldName, SourceClass fieldResource,
      Set<SourceClass> generics, SourceClass value) {
    link(currentClass, fieldResource, EdgeUsage.CONTAINS);
    link(currentClass, generics, EdgeUsage.CONTAINS);
    link(currentClass, value, EdgeUsage.USES);
  }

  @Override
  public void visitMethod(AccessFlags accessFlags, SourceClass currentClass, String methodName, SourceClass returnResource,
      Set<SourceClass> argResources, Set<SourceClass> generics, Set<SourceClass> exceptions) {
    link(currentClass, returnResource, EdgeUsage.USES);
    link(currentClass, argResources, EdgeUsage.USES);
    link(currentClass, generics, EdgeUsage.USES);
    link(currentClass, exceptions, EdgeUsage.USES);
  }

  @Override
  public void visitOutsideFieldAccess(SourceClass currentClass, SourceClass fieldOwner, SourceClass fieldType) {
    link(currentClass, fieldOwner, EdgeUsage.USES);
    link(currentClass, fieldType, EdgeUsage.USES);
  }

  @Override
  public void visitTypeInsn(SourceClass currentClass, SourceClass usedClass) {
    link(currentClass, usedClass, EdgeUsage.USES);
  }

  @Override
  public void visitOutsideMethodAccess(SourceClass currentClass, SourceClass outsideClass, Set<SourceClass> methodArgs) {
    link(currentClass, outsideClass, EdgeUsage.USES);
    link(currentClass, methodArgs, EdgeUsage.USES);
  }

  @Override
  public void visitTryCatchBlock(SourceClass currentClass, SourceClass exceptionClass) {
    link(currentClass, exceptionClass, EdgeUsage.USES);
  }

  private void link(SourceClass from, Set<SourceClass> tos, EdgeUsage link) {
    for (SourceClass to : tos) {
      link(from, to, link);
    }
  }

  private void link(SourceClass from, SourceClass to, EdgeUsage link) {
    if (canWeLinkNodes(from, to) && from.getEdgeTo(to) == null) {
      from.createEdgeWith(to, link);
      from.add(Metric.CE, 1);
      to.add(Metric.CA, 1);
      createEdgeBetweenParents(SourceFile.class, from, to, from.getEdgeTo(to));
      createEdgeBetweenParents(SourcePackage.class, from, to, from.getEdgeTo(to));
    }
  }

  private void createEdgeBetweenParents(Class<? extends SourceCode> type, SourceClass from, SourceClass to, Edge rootEdge) {
    SourceCode fromParent = from.getParent(type);
    SourceCode toParent = to.getParent(type);

    if (canWeLinkNodes(fromParent, toParent)) {
      if (fromParent.getEdgeTo(toParent) == null) {
        fromParent.createEdgeWith(toParent, EdgeUsage.USES, rootEdge);
        fromParent.add(Metric.CE, 1);
        toParent.add(Metric.CA, 1);
      } else {
        if (!fromParent.getEdgeTo(toParent).hasAnEdgeFromRootNode(from)) {
          toParent.add(Metric.CA, 1);
        }
        if (!fromParent.getEdgeTo(toParent).hasAnEdgeToRootNode(to)) {
          fromParent.add(Metric.CE, 1);
        }
        fromParent.getEdgeTo(toParent).addRootEdge(rootEdge);
      }
    }
  }

  private boolean canWeLinkNodes(SourceCode from, SourceCode to) {
    boolean result = from != null && to != null && !from.equals(to);
    return result;
  }

}
