/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.bytecode.asm;

import java.util.Set;

import org.objectweb.asm.Type;
import org.objectweb.asm.signature.SignatureReader;

public class AsmMethod extends AsmUnit {

  String name;
  String key;
  String[] exceptions;
  AsmType returnType;
  AsmType[] arguments;
  String[] signatureClasses;

  public AsmMethod(int access, String name, String descriptor, String descriptorWithGenerics, String[] exceptions) {
    super(access);
    this.name = name;
    key = name + descriptor;
    this.exceptions = exceptions;
    initReturnType(descriptor);
    initArguments(descriptor);
    if (descriptorWithGenerics == null) {
      extractClasses(descriptor);
    } else {
      extractClasses(descriptorWithGenerics);
    }
  }

  private void initArguments(String descriptor) {
    Type[] types = Type.getArgumentTypes(descriptor);
    arguments = new AsmType[types.length];
    for (int i = 0; i < arguments.length; i++) {
      arguments[i] = new AsmType(types[i]);
    }
  }

  private void extractClasses(String signature) {
    AsmSignatureVisitor signatureVisitor = new AsmSignatureVisitor();
    new SignatureReader(signature).accept(signatureVisitor);
    Set<String> internalNames = signatureVisitor.getInternalNames();
    signatureClasses = internalNames.toArray(new String[internalNames.size()]);
  }

  private void initReturnType(String descriptor) {
    returnType = new AsmType(Type.getReturnType(descriptor));
  }

  String getName() {
    return name;
  }

  String getKey() {
    return key;
  }

  boolean hasReturnType() {
    return !returnType.isVoid();
  }

  AsmType getReturnType() {
    return returnType;
  }

  boolean hasArguments() {
    return arguments.length != 0;
  }

  AsmType[] getArguments() {
    return arguments;
  }

  String[] getExceptions() {
    return exceptions;
  }

  String[] getSignatureClasses() {
    return signatureClasses;
  }

  String getReturnTypePath() {
    return returnType.getPath();
  }
}
