/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.language;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

public class JavaMethodSignature {

  private final String methodName;
  private final ReturnType returnType;
  private final List<ArgumentType> argumentTypes;
  private final boolean isConstructor;

  private static Map<String, JavaType> javaTypes = new Hashtable<String, JavaType>();
  private static final char ARRAY = '[';

  static {
    for (JavaType javaType : Arrays.asList(JavaType.values())) {
      javaTypes.put(javaType.JVMTypeIdentifier, javaType);
    }
  }

  public JavaMethodSignature(String methodName, ReturnType returnType, ArgumentType... argumentTypes) {
    this(methodName, returnType, Arrays.asList(argumentTypes));
  }

  public JavaMethodSignature(String methodName, ReturnType returnType, List<ArgumentType> argumentTypes) {
    this.methodName = methodName;
    this.returnType = returnType;
    this.argumentTypes = argumentTypes;
    if (returnType == null) {
      isConstructor = true;
    } else {
      isConstructor = false;
    }

  }

  public static JavaMethodSignature create(String methodSignature) {
    int leftBracketIndex = methodSignature.indexOf('(');
    int rightBracketIndex = methodSignature.indexOf(')');
    String methodName = methodSignature.substring(0, leftBracketIndex);
    ReturnType returnType = extractReturnType(methodSignature.substring(rightBracketIndex + 1));
    List<ArgumentType> argumentTypes = extractArgumentTypes(methodSignature.substring(leftBracketIndex + 1, rightBracketIndex));
    return new JavaMethodSignature(methodName, returnType, argumentTypes);
  }

  static List<ArgumentType> extractArgumentTypes(String argumentTypesSignature) {
    List<ArgumentType> argumentTypes = new ArrayList<ArgumentType>();
    int argumentTypeStartIndex = 0;
    int argumentTypeEndIndex = 0;
    while ((argumentTypeEndIndex = argumentTypesSignature.indexOf(';', argumentTypeStartIndex)) != -1) {
      ArgumentType argumentAndReturnType = extractArgumentType(argumentTypesSignature.substring(argumentTypeStartIndex,
          argumentTypeEndIndex));
      argumentTypes.add(argumentAndReturnType);
      argumentTypeStartIndex = argumentTypeEndIndex + 1;
    }
    return argumentTypes;
  }

  private static ArgumentType extractArgumentType(String argumentTypeSignature) {
    return new ArgumentType(extractType(argumentTypeSignature));
  }

  private static ReturnType extractReturnType(String returnTypeSignature) {
    if (StringUtils.isBlank(returnTypeSignature)) {
      return null;
    }
    return new ReturnType(extractType(returnTypeSignature));
  }

  static ArgumentAndReturnType extractType(String stringTypeSignature) {
    boolean isArray = false;
    int index = 0;
    String classCanonicalName = null;
    if (stringTypeSignature.charAt(0) == ARRAY) {
      isArray = true;
      index++;
    }
    JavaType javaType = javaTypes.get(stringTypeSignature.substring(index, index + 1));
    index++;
    if (javaType == JavaType.OBJECT) {
      classCanonicalName = stringTypeSignature.substring(index);
      classCanonicalName = StringUtils.remove(classCanonicalName, ';');
    }
    return new ArgumentType(javaType, classCanonicalName, isArray);
  }

  public String dumpMethodSignature() {
    StringBuilder builder = new StringBuilder();
    builder.append(methodName);
    builder.append("(");
    for (ArgumentAndReturnType argumentType : argumentTypes) {
      builder.append(dumpTypeSignature(argumentType));
      builder.append(';');
    }
    builder.append(")");
    if (!isConstructor) {
      builder.append(dumpTypeSignature(returnType));
      if (!returnType.isVoid()) {
        builder.append(";");
      }
    }
    return builder.toString();
  }

  static String dumpTypeSignature(ArgumentAndReturnType argumentType) {
    String result = "";
    if (argumentType.isArray()) {
      result += ARRAY;
    }
    result += argumentType.getJavaType().JVMTypeIdentifier;
    if (argumentType.isOject()) {
      result += argumentType.getClassCanonicalName();
    }
    return result;
  }
}
