/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.picocontainer.DefaultPicoContainer;
import org.picocontainer.MutablePicoContainer;
import org.sonar.squid.api.CodeScanner;
import org.sonar.squid.api.CodeVisitor;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.api.SourceProject;
import org.sonar.squid.api.SquidConfiguration;
import org.sonar.squid.indexer.Query;
import org.sonar.squid.indexer.SquidIndex;

public class Squid {

  private MutablePicoContainer pico;
  private SourceProject        project;
  private SquidIndex           squidIndex;

  public Squid(SquidConfiguration conf) {
    pico = new DefaultPicoContainer();
    pico.addComponent(conf);
    project = new SourceProject("Project");
    squidIndex = new SquidIndex();
    squidIndex.index(project);
    pico.addComponent(squidIndex);
  }

  public Squid() {
    this(new SquidConfiguration());
  }

  public void scanDir(Class<? extends CodeScanner> codeScanner, File dirToAnalyse) {
    if (dirToAnalyse == null) {
      throw new IllegalStateException("There is no directory to analyze as the File object is null.");
    }
    scanFiles(codeScanner, traverse(dirToAnalyse));
  }

  public void scanFile(Class<? extends CodeScanner> codeScanner, File fileToAnalyse) {
    if (fileToAnalyse == null) {
      throw new IllegalStateException("There is no file to analyze as the File object is null.");
    }
    Set<File> files = new HashSet<File>();
    files.add(fileToAnalyse);
    scanFiles(codeScanner, files);
  }

  public void scanFiles(Class<? extends CodeScanner> codeScanner, Collection<File> filesToAnalyse) {
    addToPicocontainer(codeScanner);
    CodeScanner scanner = (CodeScanner) pico.getComponent(codeScanner);
    for (Class<? extends CodeVisitor> visitorClasses : scanner.getVisitors()) {
      addToPicocontainer(visitorClasses);
      scanner.accept(pico.getComponent(visitorClasses));
    }
    scanner.scanCode(project, filesToAnalyse);
  }

  public SourceCode computeMeasures() {
    project.computeMeasures();
    return project;
  }

  private void addToPicocontainer(Class classToExpose) {
    if (pico.getComponent(classToExpose) == null) {
      pico.addComponent(classToExpose);
    }
  }

  public SourceCode search(String key) {
    return squidIndex.search(key);
  }

  public Collection<SourceCode> search(Query... query) {
    return squidIndex.search(query);
  }

  private static List<File> traverse(File aNode) {
    List<File> files = new ArrayList<File>();
    if (aNode.canRead()) {
      if (aNode.isDirectory()) {
        final File[] nodes = aNode.listFiles();
        for (File node : nodes) {
          files.addAll(traverse(node));
        }
      } else if (aNode.isFile()) {
        files.add(aNode);
      }
    }
    return files;
  }
}
