/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.graph;

import java.util.HashSet;
import java.util.Set;

public class Edge {

  private final Node      from;
  private final Node      to;
  private final Set<Edge> rootEdges = new HashSet<Edge>();
  private final Set<Node> rootFromNodes = new HashSet<Node>();
  private final Set<Node> rootToNodes = new HashSet<Node>();
  private final EdgeUsage usage;

  public Edge(Node from, Node to, EdgeUsage link) {
    this(from, to, link, null);
  }

  public Edge(Node from, Node to, EdgeUsage usage, Edge rootEdge) {
    this.from = from;
    this.to = to;
    this.usage = usage;
    addRootEdge(rootEdge);
  }

  public Node getFrom() {
    return from;
  }

  public Node getTo() {
    return to;
  }

  public boolean hasAnEdgeFromRootNode(Node rootFromNode) {
    return rootFromNodes.contains(rootFromNode);
  }

  public boolean hasAnEdgeToRootNode(Node rootToNode) {
    return rootToNodes.contains(rootToNode);
  }
  public EdgeUsage getUsage() {
    return usage;
  }

  public Set<Edge> getRootEdges() {
    return rootEdges;
  }

  public int getNumberOfRootFromNodes() {
    return rootFromNodes.size();
  }

  public final void addRootEdge(Edge rootRelationShip) {
    if (rootRelationShip != null) {
      rootEdges.add(rootRelationShip);
      rootFromNodes.add(rootRelationShip.from);
      rootToNodes.add(rootRelationShip.to);
    }
  }

  public int getWeight() {
    return rootEdges.size();
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Edge)) {
      return false;
    }
    Edge edge = (Edge) obj;
    return from.equals(edge.from) && to.equals(edge.to);
  }

  @Override
  public int hashCode() {
    return from.hashCode() + to.hashCode() + usage.hashCode();
  }
}
