/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.text;

import java.io.IOException;
import java.io.Reader;

public class StringArrayReader extends Reader {

  private String[] array;
  private int rowIndex = 0;
  private int rowIndexMark = 0;
  private int colIndex = 0;
  private int colIndexMark = 0;
  private static final char END_OF_LINE_CHAR = '\n';
  private boolean ready = true;

  public StringArrayReader(String array[]) {
    this.array = array;
  }

  @Override
  public void close() throws IOException {
    array = null;
    ready = false;
  }

  @Override
  public boolean ready() throws IOException {
    return ready;
  }

  @Override
  public boolean markSupported() {
    return true;
  }

  @Override
  public void mark(int readAheadLimit) throws IOException {
    rowIndexMark = rowIndex;
    colIndexMark = colIndex;
  }

  @Override
  public void reset() throws IOException {
    rowIndex = rowIndexMark;
    colIndex = colIndexMark;
  }

  @Override
  public int read(char[] cbuf, int off, int len) throws IOException {
    if (isEndOfArray()) {
      return -1;
    }
    if (isEndOfCurrentLine()) {
      switchToNewLine(cbuf, off);
      if (len > 1) {
        int numberOfReadChars = 1 + read(cbuf, off + 1, len - 1);
        return (numberOfReadChars == 0 ? 1 : numberOfReadChars);
      } else {
        return 1;
      }
    }
    if (isThereEnoughCharsOnCurrentLine(len)) {
      getCurrentLine().getChars(colIndex, colIndex + len, cbuf, off);
      colIndex += len;
      return len;
    } else {
      getCurrentLine().getChars(colIndex, getCurrentLine().length(), cbuf, off);
      int readLength = getCurrentLine().length() - colIndex;
      colIndex = getCurrentLine().length();
      int nextReadLength = read(cbuf, off + readLength, len - readLength);
      return readLength + (nextReadLength > -1 ? nextReadLength : 0);
    }
  }

  private boolean isThereEnoughCharsOnCurrentLine(int len) {
    return colIndex + len <= getCurrentLine().length();
  }

  private boolean isEndOfArray() {
    return array == null || array.length == 0 || (isEndOfCurrentLine() && rowIndex == array.length - 1);
  }

  private boolean isEndOfCurrentLine() {
    return colIndex >= getCurrentLine().length();
  }

  private String getCurrentLine() {
    return array[rowIndex];
  }

  private void switchToNewLine(char[] cbuf, int off) {
    cbuf[off] = END_OF_LINE_CHAR;
    colIndex = 0;
    rowIndex++;
  }

}
