/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.entities;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;
import java.util.SortedSet;
import java.util.StringTokenizer;
import java.util.TreeSet;

import org.apache.commons.io.IOUtils;

public class Resource implements Comparable<Resource> {

  public enum Type {
    PROJECT, PACKAGE, FILE, CLASS, METHOD
  }

  private final Type type;

  private final String name;

  private Resource parent;

  private final Measures measures;

  private SortedSet<Resource> children = new TreeSet<Resource>();

  public Resource(String name, Type type) {
    this.name = name;
    this.type = type;
    this.measures = new Measures(this);
  }

  public void addChild(Resource resource) {
    resource.setParent(this);
    if (!children.contains(resource)) {
      children.add(resource);
    }
  }

  public Measures getMeasures() {
    return measures;
  }

  public Resource getFirstChild() {
    return !children.isEmpty() ? children.first() : null;
  }

  public Resource getLastChild() {
    return !children.isEmpty() ? children.last() : null;
  }

  private void setParent(Resource parent) {
    this.parent = parent;
  }

  public int compareTo(Resource resource) {
    return this.name.compareTo(resource.getName());
  }

  public String getName() {
    return name;
  }

  public String getFullName() {
    if (parent != null && !parent.getType().equals(Resource.Type.PROJECT)) {
      return new StringBuilder(32).append(parent.getFullName()).append(".").append(getName()).toString();
    }
    return getName();
  }

  public Type getType() {
    return type;
  }

  public Resource getParent() {
    return parent;
  }

  public Set<Resource> getChildren() {
    return children;
  }

  public boolean equals(Object obj) {
    if (!(obj instanceof Resource)) {
      return false;
    }
    Resource resource = (Resource) obj;
    return name.equals(resource.getName()) && type.equals(resource.getType());
  }

  public int hashCode() {
    return name.hashCode() + type.hashCode();
  }

  public String toString() {
    StringBuilder sb = new StringBuilder(512);
    sb.append(getType()).append(" : ").append(getName()).append(":(").append(measures).append(")").append(IOUtils.LINE_SEPARATOR);
    for (Resource child : children) {
      String childTree = child.toString();
      StringTokenizer tokenizer = new StringTokenizer(childTree, IOUtils.LINE_SEPARATOR);
      while (tokenizer.hasMoreTokens()) {
        sb.append("-").append(tokenizer.nextToken()).append(IOUtils.LINE_SEPARATOR);
      }

    }
    return sb.toString();
  }

  public boolean contains(Resource wantedRes, boolean recursive) {
    if (children.contains(wantedRes)) {
      return true;
    }
    if (recursive) {
      for (Resource child : children) {
        if (child.contains(wantedRes, recursive)) {
          return true;
        }
      }
    }
    return false;
  }

  public Resource find(Resource wantedRes) {
    if (wantedRes.equals(this)) {
      return this;
    }
    for (Resource child : children) {
      Resource res = child.find(wantedRes);
      if (res != null) {
        return res;
      }
    }
    return null;
  }

  public Resource find(String resourceName, Resource.Type resourceType) {
    return find(new Resource(resourceName, resourceType));
  }

  public Collection<Resource> find(Resource.Type resourceType) {
    Collection<Resource> resources = new ArrayList<Resource>();
    find(resources, resourceType);
    return resources;
  }

  private void find(Collection<Resource> resources, Resource.Type resourceType) {
    if (this.getType().equals(resourceType)) {
      resources.add(this);
    }
    for (Resource child : children) {
      child.find(resources, resourceType);
    }
  }

  public final void compute() {
    for (Resource child : getChildren()) {
      child.compute();
    }
    getMeasures().consolidate();
  }
}
