/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.entities;

import java.util.Map;

import org.sonar.squid.entities.Resource.Type;
import org.sonar.squid.utils.CountDistribution;

public class Measures {

  private Resource resource;

  private int methods = 0;
  private int attrAccessorMethods = 0;
  private int classes = 0;
  private int files = 0;
  private int packages = 0;

  private int loc = 0;
  private int ncloc = 0;
  private int blankLines = 0;

  private int statements = 0;
  private int branches = 0;
  private int complexity = 0;

  private int commentLines = 0;
  private int commentLinesOnLineOfCode = 0;
  private int blankCommentLines = 0;
  private int headerCommentLines = 0;
  private int publicApi = 0;
  private int publicDocumentedApi = 0;

  private int[] methodsComplexity = new int[0];
  private int[] classesComplexity = new int[0];

  public Measures(Resource resource) {
    this.resource = resource;
    if (resource.getType() == Resource.Type.PACKAGE) {
      packages++;
    } else if (resource.getType() == Resource.Type.FILE) {
      files++;
    } else if (resource.getType() == Resource.Type.CLASS) {
      classes++;
    } else if (resource.getType() == Resource.Type.METHOD) {
      methods++;
    }
  }

  public int[] getMethodsComplexity() {
    return methodsComplexity;
  }

  public Map<Integer, Integer> getMethodsComplexityDistribution(int... distributionLimits) {
    return CountDistribution.count(distributionLimits, methodsComplexity);
  }

  public int[] getClassesComplexity() {
    return classesComplexity;
  }

  public Map<Integer, Integer> getClassesComplexityDistribution(int... distributionLimits) {
    return CountDistribution.count(distributionLimits, classesComplexity);
  }

  public int getLoc() {
    return loc;
  }

  public int getNcloc() {
    if (ncloc == 0) {
      ncloc = getLoc() - getCommentLines() - getBlankLines() - getBlankCommentLines() + getCommentLinesOnLineOfCode();
    }
    return ncloc;
  }

  public void setNcloc(int ncloc) {
    this.ncloc = ncloc;
  }

  public int getBlankLines() {
    return blankLines;
  }

  public int getStatements() {
    return statements;
  }

  public void addStatement() {
    statements++;
  }

  public int getCommentLines() {
    return commentLines;
  }

  public void setCommentLines(int commentLines) {
    this.commentLines = commentLines;
  }

  public int getCommentLinesOnLineOfCode() {
    return commentLinesOnLineOfCode;
  }

  public void setCommentLinesOnLineOfCode(int commentLinesOnLineOfCode) {
    this.commentLinesOnLineOfCode = commentLinesOnLineOfCode;
  }

  public int getRealCommentLines() {
    // commentLines = header + javadoc + cpp comments + c style comments
    return commentLines - headerCommentLines;
  }

  public int getComplexity() {
    return complexity;
  }

  public void setComplexity(int complexity) {
    this.complexity = complexity;
    if (resource.getType() == Resource.Type.METHOD) {
      methodsComplexity = new int[] { complexity };
    } else if (resource.getType() == Resource.Type.CLASS) {
      classesComplexity = new int[] { complexity };
    }
  }

  public int getBranches() {
    return branches;
  }

  public void addBranch() {
    branches++;
  }
  
  public void addPublicApi() {
    this.publicApi++;
  }

  public void addPublicDocumentedApi() {
    this.publicDocumentedApi++;
  }

  public int getMethods() {
    return methods;
  }

  public void setMethods(int methods) {
    this.methods = methods;
  }

  public int getClasses() {
    return classes;
  }

  public void setClasses(int classes) {
    this.classes = classes;
  }

  public int getFiles() {
    return files;
  }

  public int getPackages() {
    return packages;
  }

  public int getBlankCommentLines() {
    return blankCommentLines;
  }

  public void setBlankCommentLines(int blankCommentLines) {
    this.blankCommentLines = blankCommentLines;
  }

  private int[] addComplexity(int[] currentComplexity, int[] complexityToAdd) {
    if (complexityToAdd.length == 0)
      return currentComplexity;
    int[] added = new int[currentComplexity.length + complexityToAdd.length];
    System.arraycopy(currentComplexity, 0, added, 0, currentComplexity.length);
    System.arraycopy(complexityToAdd, 0, added, currentComplexity.length, complexityToAdd.length);
    return added;
  }

  protected void consolidate() {
    for (Resource child : resource.getChildren()) {
      Measures childMeasures = child.getMeasures();

      methods += childMeasures.getMethods();
      attrAccessorMethods += childMeasures.getAttrAccessorMethods();

      classes += childMeasures.getClasses();
      files += childMeasures.getFiles();
      packages += childMeasures.getPackages();

      blankLines += childMeasures.getBlankLines();
      loc += childMeasures.getLoc();
      ncloc += childMeasures.getNcloc();

      branches += childMeasures.getBranches();
      complexity += childMeasures.getComplexity();
      statements += childMeasures.getStatements();

      commentLines += childMeasures.getCommentLines();
      commentLinesOnLineOfCode += childMeasures.getCommentLinesOnLineOfCode();
      blankCommentLines += childMeasures.getBlankCommentLines();
      headerCommentLines += childMeasures.getHeaderCommentLines();
      
      publicApi += childMeasures.getPublicApi();
      publicDocumentedApi += childMeasures.getPublicDocumentedApi();
    }
    
    setComplexity(complexity);
    if (resource.getType() == Type.METHOD && isAttrAccessorMethod()) {
      publicApi = 0;
      publicDocumentedApi = 0;
      complexity = 0;
      methodsComplexity = new int[0];
      methods = 0;
    }
    for (Resource child : resource.getChildren()) {
      Measures childMeasures = child.getMeasures();
      classesComplexity = addComplexity(classesComplexity, childMeasures.getClassesComplexity());
      methodsComplexity = addComplexity(methodsComplexity, childMeasures.getMethodsComplexity());
    }
  }

  public int getAttrAccessorMethods() {
    return attrAccessorMethods;
  }

  public void setAttrAccessorMethod() {
    attrAccessorMethods = 1;
  }

  public boolean isAttrAccessorMethod() {
    return attrAccessorMethods != 0;
  }

  public Measures setStatements(int statements) {
    this.statements = statements;
    return this;
  }

  public void setCommentsBlock(int commentLines, int blankCommentLines, int commentLinesOnLineOfCode) {
    this.commentLines = commentLines;
    this.blankCommentLines += blankCommentLines;
    this.commentLinesOnLineOfCode = commentLinesOnLineOfCode;
  }

  public int getHeaderCommentLines() {
    return headerCommentLines;
  }

  public void setHeaderCommentLines(int headerCommentLines) {
    this.headerCommentLines = headerCommentLines;
  }

  public void setFiles(int files) {
    this.files = files;
  }

  public void setPackages(int packages) {
    this.packages = packages;
  }

  public void setLoc(int loc) {
    this.loc = loc;
  }

  public void setBlankLines(int blankLines) {
    this.blankLines = blankLines;
  }

  public void setBranches(int branches) {
    this.branches = branches;
  }

  public int getPublicApi() {
    return publicApi;
  }

  public int getPublicDocumentedApi() {
    return publicDocumentedApi;
  }

  public double getPercentOfApiDoc() {
    return getPublicApi() != 0 ? ((double) getPublicDocumentedApi() / (double) getPublicApi()) : 0;
  }

  public double getPercentOfCommentLines() {
    if (loc != 0) {
      int comments = getRealCommentLines();
      return (double) comments / ((double) getNcloc() + comments);
    } else {
      throw new IllegalStateException("Unable to compute the percentage of comment lines as 'loc' == 0");
    }
  }

  public String toString() {
    return "loc=" + loc + ",ncloc=" + getNcloc() + ",cmp=" + complexity + ",stmts=" + statements + ",meth="
    + methods + ",cla=" + classes;
  }
}