/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.handlers;

import com.puppycrawl.tools.checkstyle.Checker;
import com.puppycrawl.tools.checkstyle.ConfigurationLoader;
import com.puppycrawl.tools.checkstyle.DefaultLogger;
import com.puppycrawl.tools.checkstyle.PropertiesExpander;
import com.puppycrawl.tools.checkstyle.api.AuditEvent;
import com.puppycrawl.tools.checkstyle.api.AuditListener;
import com.puppycrawl.tools.checkstyle.api.Configuration;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.squid.AnalysisException;
import org.sonar.squid.Squid;
import org.sonar.squid.entities.Resource;
import org.sonar.squid.sensors.*;

import java.io.*;
import java.nio.charset.Charset;
import java.util.*;

/**
 * Squid uses Checkstyle to get an out-of-the-box java parser with AST
 * generation and visitor pattern support.
 */
public class JavaCheckstyleHandler implements Handler {

  private boolean anaylseAccessors = false;

  public JavaCheckstyleHandler(boolean anaylseAccessors) {
    this.anaylseAccessors = anaylseAccessors;
  }

  /**
   * Create and execute the Checkstyle engine.
   *
   * @param files        collection of files to analyse. This list shouldn't contain
   *                     and directory.
   * @param charset      the default charset to use to read files
   * @param confFileName the Checkstyle configuration file name to load.
   */
  private void launchCheckstyleEngine(Collection<File> files, Charset charset) throws AnalysisException {
    ErrorsListener listener = new ErrorsListener();
    Checker c = createChecker(charset, listener);
    File[] processedFiles = new File[files.size()];
    files.toArray(processedFiles);
    c.process(processedFiles);
    c.destroy();
    if (!listener.getErrors().isEmpty()) {
      throw listener.getErrors().get(0);
    }
  }

  /**
   * Creates the Checkstyle Checker object.
   *
   * @return a nice new fresh Checkstyle Checker
   */
  private Checker createChecker(Charset charset, ErrorsListener errorsListener) {
    try {
      InputStream checkstyleConfig = JavaCheckstyleHandler.class.getClassLoader().getResourceAsStream(
          "checkstyle-configuration.xml");
      String readenConfig = IOUtils.toString(checkstyleConfig);
      readenConfig = readenConfig.replace("${charset}", charset.toString());
      checkstyleConfig = new ByteArrayInputStream(readenConfig.getBytes());
      Configuration config = ConfigurationLoader.loadConfiguration(checkstyleConfig, new PropertiesExpander(
          System.getProperties()), false);
      Checker c = new Checker();
      c.configure(config);

      final Logger logger = LoggerFactory.getLogger(Squid.class);

      StreamLogger infoLogger = new StreamLogger() {
        @Override
        public void log(String log) {
          logger.info(log);
        }
      };
      StreamLogger errorLogger = new StreamLogger() {
        @Override
        public void log(String log) {
          logger.error(log);
        }
      };

      c.addListener(new DefaultLogger(infoLogger, true, errorLogger, true));
      c.addListener(errorsListener);
      return c;
    } catch (final Exception e) {
      throw new RuntimeException(
          "Unable to create Checkstyle Checker object with 'checkstyle-configuration.xml' as Checkstyle configuration file name",
          e);
    }
  }

  private static class ErrorsListener implements AuditListener {

    private List<AnalysisException> errors = new ArrayList<AnalysisException>();

    public void addError(AuditEvent evt) {
      // some projects can have file parsing errors (tapestry for example)
      // currently do not throw an error.
      // see http://sourceforge.net/tracker/?func=detail&atid=397078&aid=1667137&group_id=29721
      if (evt.getMessage().contains("Got an exception - expecting EOF, found")) return;
      errors.add(new AnalysisException(evt.getMessage()));
    }

    public void addException(AuditEvent evt, Throwable throwable) {
      errors.add(new AnalysisException(evt.getMessage(), throwable));
    }

    public void auditFinished(AuditEvent evt) {
    }

    public void auditStarted(AuditEvent evt) {
    }

    public void fileFinished(AuditEvent evt) {
    }

    public void fileStarted(AuditEvent evt) {
    }

    public List<AnalysisException> getErrors() {
      return errors;
    }

  }

  private abstract static class StreamLogger extends OutputStream {

    private StringBuilder builder = new StringBuilder(256);

    @Override
    public void write(int byteToWrite) throws IOException {
      char character = (char) byteToWrite;
      if (character == '\n') {
        logAndResetBuffer();
      } else {
        builder.append(character);
      }
    }

    private void logAndResetBuffer() {
      log(builder.toString().trim());
      builder.setLength(0);
    }

    public abstract void log(String log);

    @Override
    public void close() throws IOException {
      if (builder.length() > 0) {
        logAndResetBuffer();
      }
      super.close();
    }
  }

  public void analyzeSources(Resource project, Collection<File> filesToAnalyse, Charset charset) {
    List<ASTSensor> visitors = getVisitors();
    Stack<Resource> resourcesStack = new Stack<Resource>();
    resourcesStack.add(project);
    for (ASTSensor visitor : visitors) {
      visitor.setResourcesStack(resourcesStack);
    }
    JavaCheckstyleSquidBridge.setASTVisitors(visitors);
    launchCheckstyleEngine(filesToAnalyse, charset);
    project.compute();
  }

  private List<ASTSensor> getVisitors() {
    AccessorSensor accessorSensor = new AccessorSensor();
    List<ASTSensor> visitors = new ArrayList<ASTSensor>(Arrays.asList(new PackageSensor(), new FileSensor(),
        new HeaderCommentSensor(), new ClassSensor(), new MethodSensor(),
        accessorSensor, new LocSensor(), new BlankLineSensor(), new CommentSensors(), new PublicApiSensor(),
        new StatementSensor(), new BranchSensor(), new ComplexitySensor()));
    if (!anaylseAccessors) {
      visitors.remove(accessorSensor);
    }
    return visitors;
	}
}
