/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.squid.sensors;

import antlr.collections.AST;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.FileContents;
import com.puppycrawl.tools.checkstyle.api.Scope;
import com.puppycrawl.tools.checkstyle.api.ScopeUtils;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

import org.sonar.squid.AnalysisException;
import org.sonar.squid.entities.Resource;
import org.sonar.squid.entities.Resource.Type;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

public abstract class ASTSensor {

  private Stack<Resource> resourcesStack;
  private FileContents fileContents;

  public final void setFileContents(FileContents fileContents) {
    this.fileContents = fileContents;
  }

  public final FileContents getFileContents() {
    return fileContents;
  }

  public List<Integer> getWantedTokens() {
    return new ArrayList<Integer>();
  }

  public final void setResourcesStack(Stack<Resource> resourcesStack) {
    this.resourcesStack = resourcesStack;
  }

  public final void addResource(Resource child) {
    peekResource().addChild(child);
    resourcesStack.add(child);
  }

  public final void popResource() {
    resourcesStack.pop();
  }

  public final Resource peekResource() {
    return resourcesStack.peek();
  }

  public void visitFile(DetailAST ast) {
  }

  public void visitToken(DetailAST ast) {
  }

  public void leaveToken(DetailAST ast) {
  }

  public void leaveFile(DetailAST ast) {
  }
  
  protected AST findType(DetailAST ast) {
    DetailAST typeAst = ast.findFirstToken(TokenTypes.TYPE);
    if (typeAst != null) {
      return typeAst.getFirstChild();
    }
    return null;
  }
  
  protected boolean isClassVariable(DetailAST ast) {
    return ast.getType() == TokenTypes.VARIABLE_DEF && 
      ast.getParent().getType() == TokenTypes.OBJBLOCK && 
      isClass(ast.getParent().getParent());
  }
  
  protected boolean isClass(DetailAST ast) {
    return ast.getType() == TokenTypes.CLASS_DEF || 
      ast.getType() == TokenTypes.ENUM_DEF || 
      ast.getType() == TokenTypes.ANNOTATION_DEF;
  }
  
  protected boolean isInterfaceVariable(DetailAST ast) {
    return ast.getType() == TokenTypes.VARIABLE_DEF && 
      ast.getParent().getType() == TokenTypes.OBJBLOCK && 
      isInterface(ast.getParent().getParent());
  }
  
  protected boolean isInterface(DetailAST ast) {
    return ast.getType() == TokenTypes.INTERFACE_DEF;
  }
  
  protected boolean isFinal(DetailAST detailAst) {
    return isModifier(detailAst, TokenTypes.FINAL);
  }
  
  protected boolean isStatic(DetailAST detailAst) {
    return isModifier(detailAst, TokenTypes.LITERAL_STATIC);
  }

  private boolean isModifier(DetailAST detailAst, int modifierType) {
    DetailAST modifiers = detailAst.findFirstToken(TokenTypes.MODIFIERS);
    if (modifiers != null) {
      boolean isModifierMatching = modifiers.branchContains(modifierType);
      if (!isModifierMatching && isInterfaceVariable(detailAst)) {
        // by default if not specified, a var def in an interface is public final static
        return (modifierType == TokenTypes.LITERAL_STATIC || modifierType == TokenTypes.FINAL);
      }
      return isModifierMatching;
    }
    return false;
  }
  
  protected Scope getScope(DetailAST ast) {
    DetailAST modifierAst = ast.findFirstToken(TokenTypes.MODIFIERS);
    Scope found =  modifierAst != null ? ScopeUtils.getScopeFromMods(modifierAst) : Scope.NOTHING;
    if (found.compareTo(Scope.PACKAGE) == 0 && 
        ( ast.getType() == TokenTypes.METHOD_DEF || ast.getType() == TokenTypes.VARIABLE_DEF)) {
      // check if we found a parent interface declaration
      // interface methods or var defs are by default public when not specified
      found = (isScope(Scope.PACKAGE, found) && findParent(ast, TokenTypes.INTERFACE_DEF) != null) ? Scope.PUBLIC : found;
    }
    return found;
  }
  
  protected boolean isScope(Scope toCompare, Scope scope) {
    return scope.compareTo(toCompare) == 0;
  }
  
  protected boolean isType(DetailAST ast, int type) {
    return ast.getType() == type;
  }
  
  protected DetailAST findParent(DetailAST ast, int tokenType) {
    DetailAST parent = ast.getParent();
    if (parent != null) {
      if (parent.getType() == tokenType) {
        return parent;
      } else {
        return findParent(parent, tokenType);
      }
    }
    return null;
  }
  
  protected void ensureResourceType(Resource resource, Type type) {
    if (resource.getType() != type) {
      throw new AnalysisException("Resource " + resource.getFullName() + " must be of type " + type);
    }
  }

  @Override
  public int hashCode() {
    return this.getClass().hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().equals(obj.getClass());
  }
  
  
}
