/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.java.ast;

import com.puppycrawl.tools.checkstyle.Checker;
import com.puppycrawl.tools.checkstyle.ConfigurationLoader;
import com.puppycrawl.tools.checkstyle.PropertiesExpander;
import com.puppycrawl.tools.checkstyle.api.Configuration;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.java.ast.visitor.*;
import org.sonar.squid.api.AnalysisException;
import org.sonar.squid.api.CodeScanner;
import org.sonar.squid.api.SourceCode;
import org.sonar.squid.api.SquidConfiguration;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.*;

/**
 * Squid uses Checkstyle to get an out-of-the-box java parser with AST generation and visitor pattern support.
 */
public class JavaAstScanner extends CodeScanner<AstVisitor> {

  private static final Logger LOG = LoggerFactory.getLogger(JavaAstScanner.class);
  private SquidConfiguration conf;
  private SourceCode project;

  public JavaAstScanner(SquidConfiguration conf, SourceCode project) {
    this.conf = conf;
    this.project = project;
  }

  /**
   * Create and execute the Checkstyle engine.
   *
   * @param files   collection of files to analyse. This list shouldn't contain and directory.
   * @param charset the default charset to use to read files
   */
  private void launchCheckstyleEngine(Collection<File> files, Charset charset) {
    Checker c = createChecker(charset);
    File[] processedFiles = new File[files.size()];
    files.toArray(processedFiles);
    c.process(processedFiles);
    c.destroy();
  }

  /**
   * Creates the Checkstyle Checker object.
   *
   * @return a nice new fresh Checkstyle Checker
   */
  private Checker createChecker(Charset charset) {
    InputStream checkstyleConfig = null;
    try {
      checkstyleConfig = JavaAstScanner.class.getClassLoader().getResourceAsStream("checkstyle-configuration.xml");
      String readenConfig = IOUtils.toString(checkstyleConfig);
      readenConfig = readenConfig.replace("${charset}", charset.toString());
      checkstyleConfig = new ByteArrayInputStream(readenConfig.getBytes());
      Configuration config = ConfigurationLoader.loadConfiguration(checkstyleConfig, new PropertiesExpander(System.getProperties()), false);
      Checker c = new Checker();
      c.configure(config);
      c.addListener(new CheckstyleAuditListener());
      return c;

    } catch (Exception e) {
      throw new AnalysisException(
          "Unable to create Checkstyle Checker object with 'checkstyle-configuration.xml' as Checkstyle configuration file name", e);
    } finally {
      IOUtils.closeQuietly(checkstyleConfig);
    }
  }

  public JavaAstScanner scanDirectory(File javaSourceDirectory) {
    Collection<File> files = (Collection<File>) FileUtils.listFiles(javaSourceDirectory, FileFilterUtils.fileFileFilter(), FileFilterUtils.directoryFileFilter());
    return scanFiles(files);
  }

  public JavaAstScanner scanFile(File javaFile) {
    return scanFiles(Arrays.asList(javaFile));
  }

  public JavaAstScanner scanFiles(Collection<File> javaFiles) {
    if (LOG.isDebugEnabled()) {
      LOG.debug("----- Java sources analyzed by Squid:");
      for (File javaFile : javaFiles) {
        LOG.debug(javaFile.getAbsolutePath());
      }
      LOG.debug("-----");
    }

    Stack<SourceCode> resourcesStack = new Stack<SourceCode>();
    resourcesStack.add(project);
    for (AstVisitor visitor : getVisitors()) {
      visitor.setResourcesStack(resourcesStack);
    }
    CheckstyleSquidBridge.setASTVisitors(getVisitors());
    CheckstyleSquidBridge.setSquidConfiguration(conf);
    launchCheckstyleEngine(javaFiles, conf.getCharset());
    return this;
  }

  @Override
  public Collection<Class<? extends AstVisitor>> getVisitorClasses() {
    List<Class<? extends AstVisitor>> visitorClasses = new ArrayList<Class<? extends AstVisitor>>();
    visitorClasses.add(PackageVisitor.class);
    visitorClasses.add(FileVisitor.class);
    visitorClasses.add(ClassVisitor.class);
    visitorClasses.add(AnonymousInnerClassVisitor.class);
    visitorClasses.add(MethodVisitor.class);
    visitorClasses.add(EndAtLineVisitor.class);
    visitorClasses.add(LinesVisitor.class);
    visitorClasses.add(BlankLinesVisitor.class);
    visitorClasses.add(CommentVisitor.class);
    visitorClasses.add(PublicApiVisitor.class);
    visitorClasses.add(BranchVisitor.class);
    visitorClasses.add(StatementVisitor.class);
    if (conf.isAnalysePropertyAccessors()) {
      visitorClasses.add(AccessorVisitor.class);
    }
    visitorClasses.add(ComplexityVisitor.class);
    visitorClasses.add(LinesOfCodeVisitor.class);
    return visitorClasses;
  }
}
