/*
 * Sonar, open source software quality management tool.
 * Copyright (C) 2009 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * Sonar is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * Sonar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sonar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02
 */
package org.sonar.updatecenter;

import org.apache.commons.lang.StringUtils;

public class Version implements Comparable<Version> {

  private String major = normalizePart("0");
  private String minor = normalizePart("0");
  private String patch = normalizePart("0");
  private String patch2 = normalizePart("0");
  private String name;

  public Version(String version) {
    this.name = StringUtils.trimToEmpty(version);
    String[] split = StringUtils.split(name, '.');
    if (split.length >= 1) {
      major = normalizePart(split[0]);
    }
    if (split.length >= 2) {
      minor = normalizePart(split[1]);
    }
    if (split.length >= 3) {
      patch = normalizePart(split[2]);
    }
    if (split.length >= 4) {
      patch2 = normalizePart(split[3]);
    }
  }

  private static String normalizePart(String part) {
    return StringUtils.leftPad(part, 4, '0');
  }

  public String getMajor() {
    return major;
  }

  public String getMinor() {
    return minor;
  }

  public String getPatch() {
    return patch;
  }

  public String getPatch2() {
    return patch2;
  }

  public String getVersion() {
    return name;
  }

  public boolean isRelease() {
    return StringUtils.isNumeric(major) &&
        (StringUtils.isNumeric(minor) || StringUtils.contains(minor, "GA")) &&
        StringUtils.isNumeric(patch) &&
        StringUtils.isNumeric(patch2);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    Version version = (Version) o;
    if (!major.equals(version.major)) {
      return false;
    }
    if (!minor.equals(version.minor)) {
      return false;
    }
    if (!patch.equals(version.patch)) {
      return false;
    }
    if (!patch2.equals(version.patch2)) {
      return false;
    }
    return true;
  }

  @Override
  public int hashCode() {
    int result = major.hashCode();
    result = 31 * result + minor.hashCode();
    result = 31 * result + patch.hashCode();
    result = 31 * result + patch2.hashCode();
    return result;
  }

  public int compareTo(Version other) {
    // TODO : manage RC, alpha, ...
    int c = major.compareTo(other.major);
    if (c == 0) {
      c = minor.compareTo(other.minor);
      if (c == 0) {
        c = patch.compareTo(other.patch);
        if (c == 0) {
          c = patch2.compareTo(other.patch2);
        }
      }
    }
    return c;
  }

  @Override
  public String toString() {
    return name;
  }

  public static Version create(String version) {
    return new Version(version);
  }
}
