/*
 * Decompiled with CFR 0.152.
 */
package org.codelibs.fess.ds.atlassian.api.util;

import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.codelibs.fess.ds.atlassian.AtlassianDataStoreException;
import org.codelibs.fess.ds.atlassian.api.util.UrlUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OAuthUtil {
    private static Logger logger = LoggerFactory.getLogger(OAuthUtil.class);
    private static final SecureRandom RANDOM = new SecureRandom();
    private static final String SIGNATURE_METHOD = "RSA-SHA1";

    private OAuthUtil() {
    }

    public static PrivateKey getPrivateKey(String privateKey) {
        try {
            String key = privateKey.replaceAll("\\\\n|-----[A-Z ]+-----", "");
            byte[] privateBytes = Base64.getDecoder().decode(key);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateBytes);
            KeyFactory kf = KeyFactory.getInstance("RSA");
            return kf.generatePrivate(keySpec);
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new AtlassianDataStoreException("Failed to generate private key.", e);
        }
    }

    public static String getAuthorizationHeader(String consumerKey, PrivateKey privateKey, String token, String verifier, String requestMethod, URL url) {
        String nonce = OAuthUtil.generateNonce();
        String timestamp = OAuthUtil.generateTimestamp();
        String signature = OAuthUtil.generateSignature(consumerKey, privateKey, token, verifier, nonce, timestamp, requestMethod, url);
        StringBuilder buf = new StringBuilder("OAuth");
        OAuthUtil.appendParameter(buf, "oauth_consumer_key", consumerKey);
        OAuthUtil.appendParameter(buf, "oauth_nonce", nonce);
        OAuthUtil.appendParameter(buf, "oauth_signature", signature);
        OAuthUtil.appendParameter(buf, "oauth_signature_method", SIGNATURE_METHOD);
        OAuthUtil.appendParameter(buf, "oauth_timestamp", timestamp);
        OAuthUtil.appendParameter(buf, "oauth_token", token);
        OAuthUtil.appendParameter(buf, "oauth_verifier", verifier);
        return buf.substring(0, buf.length() - 1);
    }

    public static void appendParameter(StringBuilder buf, String name, String value) {
        buf.append(' ').append(UrlUtil.encode(name)).append("=\"").append(UrlUtil.encode(value)).append("\",");
    }

    public static Map<String, String> getQueryMapFromUrl(URL url) {
        String query = url.getQuery();
        if (query == null) {
            return Collections.emptyMap();
        }
        return Arrays.stream(query.split("&")).collect(Collectors.toMap(p -> p.split("=")[0], p -> UrlUtil.decode(p.split("=")[1])));
    }

    public static String generateSignature(String consumerKey, PrivateKey privateKey, String token, String verifier, String nonce, String timestamp, String requestMethod, URL url) {
        TreeMap<String, String> parameters = new TreeMap<String, String>();
        parameters.put("oauth_consumer_key", consumerKey);
        parameters.put("oauth_nonce", nonce);
        parameters.put("oauth_signature_method", SIGNATURE_METHOD);
        parameters.put("oauth_timestamp", timestamp);
        parameters.put("oauth_token", token);
        parameters.put("oauth_verifier", verifier);
        parameters.putAll(OAuthUtil.getQueryMapFromUrl(url));
        String normalizedParameters = UrlUtil.buildQueryParameters(parameters);
        String normalizedPath = url.getProtocol() + "://" + url.getAuthority() + url.getPath();
        StringBuilder signatureBaseString = new StringBuilder();
        signatureBaseString.append(UrlUtil.encode(requestMethod)).append('&');
        signatureBaseString.append(UrlUtil.encode(normalizedPath)).append('&');
        signatureBaseString.append(UrlUtil.encode(normalizedParameters));
        return OAuthUtil.computeSignature(privateKey, signatureBaseString.toString());
    }

    public static String generateNonce() {
        if (logger.isDebugEnabled()) {
            logger.debug("Using '{}:{}' as PRNG for generating nonce.", (Object)RANDOM.getProvider().getName(), (Object)RANDOM.getAlgorithm());
        }
        return Long.toHexString(Math.abs(RANDOM.nextLong()));
    }

    public static String generateTimestamp() {
        return Long.toString(System.currentTimeMillis() / 1000L);
    }

    public static String computeSignature(PrivateKey privateKey, String signatureBaseString) {
        try {
            Signature signature = Signature.getInstance("SHA1withRSA");
            signature.initSign(privateKey);
            signature.update(signatureBaseString.getBytes(StandardCharsets.UTF_8));
            return Base64.getEncoder().encodeToString(signature.sign());
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | SignatureException e) {
            logger.warn("Failed to compute OAuth Signature.", (Throwable)e);
            return "";
        }
    }
}

