/*
 * Decompiled with CFR 0.152.
 */
package org.cognitor.cassandra.migration;

import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SimpleStatement;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.exceptions.DriverException;
import java.io.Closeable;
import java.util.Date;
import org.cognitor.cassandra.migration.DbMigration;
import org.cognitor.cassandra.migration.MigrationException;
import org.cognitor.cassandra.migration.cql.SimpleCQLLexer;
import org.cognitor.cassandra.migration.keyspace.Keyspace;
import org.cognitor.cassandra.migration.util.Ensure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Database
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(Database.class);
    private static final String SCHEMA_CF = "schema_migration";
    private static final String INSERT_MIGRATION = "insert into %s(applied_successful, version, script_name, script, executed_at) values(?, ?, ?, ?, ?)";
    private static final String CREATE_MIGRATION_CF = "CREATE TABLE %s (applied_successful boolean, version int, script_name varchar, script text, executed_at timestamp, PRIMARY KEY (applied_successful, version))";
    private static final String VERSION_QUERY = "select version from %s where applied_successful = True order by version desc limit 1";
    private static final String MIGRATION_ERROR_MSG = "Error during migration of script %s while executing '%s'";
    private final String keyspaceName;
    private final Keyspace keyspace;
    private final Cluster cluster;
    private final Session session;
    private final PreparedStatement logMigrationStatement;

    public Database(Cluster cluster, Keyspace keyspace) {
        this.cluster = Ensure.notNull(cluster, "cluster");
        this.keyspace = Ensure.notNull(keyspace, "keyspace");
        this.keyspaceName = keyspace.getKeyspaceName();
        this.createKeyspaceIfRequired();
        this.session = cluster.connect(this.keyspaceName);
        this.ensureSchemaTable();
        this.logMigrationStatement = this.session.prepare(String.format(INSERT_MIGRATION, SCHEMA_CF));
    }

    private void createKeyspaceIfRequired() {
        if (this.keyspaceExists()) {
            return;
        }
        try (Session session = this.cluster.connect();){
            session.execute(this.keyspace.getCqlStatement());
        }
        catch (DriverException exception) {
            throw new MigrationException(String.format("Unable to create keyspace %s.", this.keyspaceName), exception);
        }
    }

    private boolean keyspaceExists() {
        return this.cluster.getMetadata().getKeyspace(this.keyspace.getKeyspaceName()) != null;
    }

    public Database(Cluster cluster, String keyspaceName) {
        this.cluster = Ensure.notNull(cluster, "cluster");
        this.keyspaceName = Ensure.notNullOrEmpty(keyspaceName, "keyspaceName");
        this.keyspace = null;
        this.session = cluster.connect(keyspaceName);
        this.ensureSchemaTable();
        this.logMigrationStatement = this.session.prepare(String.format(INSERT_MIGRATION, SCHEMA_CF));
    }

    @Override
    public void close() {
        this.session.close();
    }

    public int getVersion() {
        ResultSet resultSet = this.session.execute(String.format(VERSION_QUERY, SCHEMA_CF));
        Row result = resultSet.one();
        if (result == null) {
            return 0;
        }
        return result.getInt(0);
    }

    public String getKeyspaceName() {
        return this.keyspaceName;
    }

    private void ensureSchemaTable() {
        if (this.schemaTablesIsNotExisting()) {
            this.createSchemaTable();
        }
    }

    private boolean schemaTablesIsNotExisting() {
        return this.cluster.getMetadata().getKeyspace(this.keyspaceName).getTable(SCHEMA_CF) == null;
    }

    private void createSchemaTable() {
        this.session.execute(String.format(CREATE_MIGRATION_CF, SCHEMA_CF));
    }

    public void execute(DbMigration migration) {
        Ensure.notNull(migration, "migration");
        LOGGER.debug(String.format("About to execute migration %s to version %d", migration.getScriptName(), migration.getVersion()));
        String lastStatement = null;
        try {
            SimpleCQLLexer lexer = new SimpleCQLLexer(migration.getMigrationScript());
            for (String statement : lexer.getCqlQueries()) {
                lastStatement = statement = statement.trim();
                this.executeStatement(statement);
            }
            this.logMigration(migration, true);
            LOGGER.debug(String.format("Successfully applied migration %s to version %d", migration.getScriptName(), migration.getVersion()));
        }
        catch (Exception exception) {
            this.logMigration(migration, false);
            String errorMessage = String.format(MIGRATION_ERROR_MSG, migration.getScriptName(), lastStatement);
            throw new MigrationException(errorMessage, (Throwable)exception, migration.getScriptName(), lastStatement);
        }
    }

    private void executeStatement(String statement) {
        if (!statement.isEmpty()) {
            SimpleStatement simpleStatement = new SimpleStatement(statement);
            simpleStatement.setConsistencyLevel(ConsistencyLevel.QUORUM);
            this.session.execute((Statement)simpleStatement);
        }
    }

    private void logMigration(DbMigration migration, boolean wasSuccessful) {
        BoundStatement boundStatement = this.logMigrationStatement.bind(new Object[]{wasSuccessful, migration.getVersion(), migration.getScriptName(), migration.getMigrationScript(), new Date()});
        this.session.execute((Statement)boundStatement);
    }
}

