/*
 * Decompiled with CFR 0.152.
 */
package org.cognitor.cassandra.migration;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Pattern;
import org.cognitor.cassandra.migration.DbMigration;
import org.cognitor.cassandra.migration.MigrationException;
import org.cognitor.cassandra.migration.collector.FailOnDuplicatesCollector;
import org.cognitor.cassandra.migration.collector.ScriptCollector;
import org.cognitor.cassandra.migration.collector.ScriptFile;
import org.cognitor.cassandra.migration.scanner.LocationScanner;
import org.cognitor.cassandra.migration.scanner.ScannerRegistry;
import org.cognitor.cassandra.migration.util.Ensure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationRepository {
    public static final String DEFAULT_SCRIPT_PATH = "cassandra/migration";
    public static final String SCRIPT_EXTENSION = ".cql";
    public static final String SCRIPT_ENCODING = "UTF-8";
    public static final String VERSION_NAME_DELIMITER = "_";
    public static final String SINGLE_LINE_COMMENT_PATTERN = "(^--.*)|(^//.*)";
    private static final Logger LOGGER = LoggerFactory.getLogger(MigrationRepository.class);
    private static final String EXTRACT_VERSION_ERROR_MSG = "Error for script %s. Unable to extract version.";
    private static final String SCANNING_SCRIPT_FOLDER_ERROR_MSG = "Error while scanning script folder for new scripts.";
    private static final String READING_SCRIPT_ERROR_MSG = "Error while reading script %s";
    private static final String PATH_SEPARATOR_CHAR = "/";
    private final Pattern commentPattern;
    private final ScannerRegistry scannerRegistry;
    private List<ScriptFile> migrationScripts;
    private final ScriptCollector scriptCollector;

    public MigrationRepository() {
        this(DEFAULT_SCRIPT_PATH);
    }

    public MigrationRepository(String scriptPath) {
        this(scriptPath, new FailOnDuplicatesCollector());
    }

    public MigrationRepository(String scriptPath, ScriptCollector scriptCollector) {
        this(scriptPath, scriptCollector, new ScannerRegistry());
    }

    public MigrationRepository(String scriptPath, ScriptCollector scriptCollector, ScannerRegistry scannerRegistry) {
        this.scriptCollector = Ensure.notNull(scriptCollector, "scriptCollector");
        this.scannerRegistry = Ensure.notNull(scannerRegistry, "scannerRegistry");
        this.commentPattern = Pattern.compile(SINGLE_LINE_COMMENT_PATTERN);
        try {
            this.migrationScripts = this.scanForScripts(this.normalizePath(Ensure.notNullOrEmpty(scriptPath, "scriptPath")));
        }
        catch (IOException | URISyntaxException exception) {
            throw new MigrationException(SCANNING_SCRIPT_FOLDER_ERROR_MSG, exception);
        }
    }

    private String normalizePath(String scriptPath) {
        StringBuilder builder = new StringBuilder(scriptPath.length() + 1);
        if (scriptPath.startsWith(PATH_SEPARATOR_CHAR)) {
            builder.append(scriptPath.substring(1));
        } else {
            builder.append(scriptPath);
        }
        if (!scriptPath.endsWith(PATH_SEPARATOR_CHAR)) {
            builder.append(PATH_SEPARATOR_CHAR);
        }
        return builder.toString();
    }

    public int getLatestVersion() {
        if (this.migrationScripts.isEmpty()) {
            return 0;
        }
        return this.migrationScripts.get(this.migrationScripts.size() - 1).getVersion();
    }

    private List<ScriptFile> scanForScripts(String scriptPath) throws IOException, URISyntaxException {
        LOGGER.debug("Scanning for cql migration scripts in " + scriptPath);
        Enumeration<URL> scriptResources = this.getClass().getClassLoader().getResources(scriptPath);
        while (scriptResources.hasMoreElements()) {
            URI script = scriptResources.nextElement().toURI();
            LOGGER.debug("Potential script folder: {}", (Object)script.toString());
            if (!this.scannerRegistry.supports(script.getScheme())) {
                LOGGER.debug("No LocationScanner available for scheme '{}'. Skipping it.");
                continue;
            }
            LocationScanner scanner = this.scannerRegistry.getScanner(script.getScheme());
            for (String resource : scanner.findResourceNames(scriptPath, script)) {
                if (MigrationRepository.isMigrationScript(resource)) {
                    String scriptName = this.extractScriptName(resource);
                    int version = MigrationRepository.extractScriptVersion(scriptName);
                    this.scriptCollector.collect(new ScriptFile(version, resource, scriptName));
                    continue;
                }
                LOGGER.warn(String.format("Ignoring file %s because it is not a cql file.", resource));
            }
        }
        ArrayList<ScriptFile> scripts = new ArrayList<ScriptFile>(this.scriptCollector.getScriptFiles());
        LOGGER.info(String.format("Found %d migration scripts", scripts.size()));
        Collections.sort(scripts);
        return scripts;
    }

    private static int extractScriptVersion(String scriptName) {
        String[] splittedName = scriptName.split(VERSION_NAME_DELIMITER);
        int folderSeperatorPos = splittedName[0].lastIndexOf(PATH_SEPARATOR_CHAR);
        String versionString = folderSeperatorPos >= 0 ? splittedName[0].substring(folderSeperatorPos + 1) : splittedName[0];
        try {
            return Integer.parseInt(versionString);
        }
        catch (NumberFormatException exception) {
            throw new MigrationException(String.format(EXTRACT_VERSION_ERROR_MSG, scriptName), exception, scriptName);
        }
    }

    private static boolean isMigrationScript(String resource) {
        return resource.endsWith(SCRIPT_EXTENSION);
    }

    private String extractScriptName(String resourceName) {
        int slashIndex = resourceName.lastIndexOf(PATH_SEPARATOR_CHAR);
        if (slashIndex > -1) {
            return resourceName.substring(slashIndex + 1);
        }
        return resourceName;
    }

    public List<DbMigration> getMigrationsSinceVersion(int version) {
        ArrayList<DbMigration> dbMigrations = new ArrayList<DbMigration>();
        this.migrationScripts.stream().filter(script -> script.getVersion() > version).forEach(script -> {
            String content = this.loadScriptContent((ScriptFile)script);
            dbMigrations.add(new DbMigration(script.getScriptName(), script.getVersion(), content));
        });
        return dbMigrations;
    }

    private String loadScriptContent(ScriptFile script) {
        try {
            return this.readResourceFileAsString(script.getResourceName(), this.getClass().getClassLoader());
        }
        catch (IOException exception) {
            throw new MigrationException(String.format(READING_SCRIPT_ERROR_MSG, script.getResourceName()), exception, script.getScriptName());
        }
    }

    private String readResourceFileAsString(String resourceName, ClassLoader classLoader) throws IOException {
        StringBuilder fileContent = new StringBuilder(256);
        new BufferedReader(new InputStreamReader(classLoader.getResourceAsStream(resourceName), SCRIPT_ENCODING)).lines().filter(line -> !this.isLineComment((String)line)).forEach(fileContent::append);
        return fileContent.toString();
    }

    private boolean isLineComment(String line) {
        return this.commentPattern.matcher(line).matches();
    }
}

