/*
 *  Copyright (C) 2011-2017 Cojen.org
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as
 *  published by the Free Software Foundation, either version 3 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.cojen.tupl.io;

import java.io.File;
import java.io.IOException;

import java.nio.ByteBuffer;

import java.util.EnumSet;
import java.util.Objects;

/**
 * Basic {@link PageArray} implementation which accesses a file.
 *
 * @author Brian S O'Neill
 */
public class FilePageArray extends PageArray {
    final FileIO mFio;

    public FilePageArray(int pageSize, File file, EnumSet<OpenOption> options) throws IOException {
        this(pageSize, FileIO.open(file, options));
    }

    public FilePageArray(int pageSize, FileIO fio) {
        super(pageSize);
        mFio = Objects.requireNonNull(fio);
    }

    @Override
    public int directPageSize() {
        int size = pageSize();
        if (mFio.isDirectIO()) {
            size = -size;
        }
        return size;
    }

    @Override
    public boolean isReadOnly() {
        return mFio.isReadOnly();
    }

    @Override
    public boolean isEmpty() throws IOException {
        return mFio.length() == 0;
    }

    @Override
    public long pageCount() throws IOException {
        // Always round page count down. A partial last page effectively doesn't exist.
        return mFio.length() / mPageSize;
    }

    @Override
    public void truncatePageCount(long count) throws IOException {
        if (allowPageCountAdjust(count)) {
            mFio.truncateLength(count * mPageSize);
        }
    }

    @Override
    public void expandPageCount(long count) throws IOException {
        if (allowPageCountAdjust(count)) {
            mFio.expandLength(count * mPageSize, LengthOption.PREALLOCATE_OPTIONAL);
        }
    }

    private boolean allowPageCountAdjust(long count) {
        if (count < 0) {
            throw new IllegalArgumentException(String.valueOf(count));
        }
        return !isReadOnly();
    }

    @Override
    public void readPage(long index, byte[] dst, int offset, int length) throws IOException {
        if (index < 0) {
            throw new IndexOutOfBoundsException(String.valueOf(index));
        }
        mFio.read(index * mPageSize, dst, offset, length);
    }

    @Override
    public void readPage(long index, byte[] dst, int offset, int length, ByteBuffer tail)
        throws IOException
    {
        if (index < 0) {
            throw new IndexOutOfBoundsException(String.valueOf(index));
        }
        mFio.read(index * mPageSize, dst, offset, length, tail);
    }

    @Override
    public void readPage(long index, long dstPtr, int offset, int length) throws IOException {
        if (index < 0) {
            throw new IndexOutOfBoundsException(String.valueOf(index));
        }
        mFio.read(index * mPageSize, dstPtr, offset, length);
    }

    @Override
    public void readPage(long index, long dstPtr, int offset, int length, ByteBuffer tail)
        throws IOException
    {
        if (index < 0) {
            throw new IndexOutOfBoundsException(String.valueOf(index));
        }
        mFio.read(index * mPageSize, dstPtr, offset, length, tail);
    }

    @Override
    public void writePage(long index, byte[] src, int offset) throws IOException {
        int pageSize = mPageSize;
        mFio.write(index * pageSize, src, offset, pageSize);
    }

    @Override
    public void writePage(long index, byte[] src, int offset, ByteBuffer tail)
        throws IOException
    {
        int pageSize = mPageSize;
        mFio.write(index * pageSize, src, offset, pageSize - tail.remaining(), tail);
    }

    @Override
    public void writePage(long index, long srcPtr, int offset) throws IOException {
        int pageSize = mPageSize;
        mFio.write(index * pageSize, srcPtr, offset, pageSize);
    }

    @Override
    public void writePage(long index, long srcPtr, int offset, ByteBuffer tail)
        throws IOException
    {
        int pageSize = mPageSize;
        mFio.write(index * pageSize, srcPtr, offset, pageSize - tail.remaining(), tail);
    }

    @Override
    public void sync(boolean metadata) throws IOException {
        mFio.sync(metadata);
        // If mapped, now is a good time to remap if length has changed.
        mFio.remap();
    }

    @Override
    public void close(Throwable cause) throws IOException {
        Utils.close(mFio, cause);
    }
}
