/*
 *  Copyright (C) 2022 Cojen.org
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as
 *  published by the Free Software Foundation, either version 3 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.cojen.tupl.rows;

import java.lang.invoke.CallSite;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;

import java.lang.ref.WeakReference;

import java.util.Map;

import org.cojen.maker.Label;
import org.cojen.maker.MethodMaker;
import org.cojen.maker.Variable;

import org.cojen.tupl.DatabaseException;
import org.cojen.tupl.Index;

import org.cojen.tupl.core.RowPredicateLock;

/**
 * Makes Table classes which dynamically generates code which depends on RowStore and tableId
 * constants. The generated classes extend those generated by StaticTableMaker.
 *
 * @author Brian S O'Neill
 */
public class DynamicTableMaker extends TableMaker {
    private final RowStore mStore;
    private final long mTableId;

    private final Class mBaseClass;

    /**
     * Constructor for making a primary table.
     *
     * @param rowGen describes row encoding
     * @param store generated class is pinned to this specific instance
     * @param tableId primary table index id
     */
    DynamicTableMaker(Class<?> type, RowGen rowGen, RowStore store, long tableId) {
        super(type, rowGen, rowGen, null);

        mStore = store;
        mTableId = tableId;

        mBaseClass = StaticTableMaker.obtain(type, rowGen);
    }

    /**
     * Constructor for making an unjoined secondary index table.
     *
     * @param rowGen describes row encoding
     * @param codecGen describes key and value codecs (different than rowGen)
     * @param secondaryDesc secondary index descriptor
     * @param store generated class is pinned to this specific instance
     * @param tableId primary table index id
     */
    DynamicTableMaker(Class<?> type, RowGen rowGen, RowGen codecGen, byte[] secondaryDesc,
                      RowStore store, long tableId)
    {
        super(type, rowGen, codecGen, secondaryDesc);

        mStore = store;
        mTableId = tableId;

        mBaseClass = StaticTableMaker.obtain(type, rowGen, secondaryDesc);
    }

    /**
     * Return a constructor which accepts a (TableManager, Index, RowPredicateLock) and returns
     * a BaseTable or BaseTableIndex implementation.
     */
    MethodHandle finish() {
        String suffix = isPrimaryTable() ? "table" : "unjoined";
        mClassMaker = mCodecGen.beginClassMaker(getClass(), mRowType, suffix).public_()
            .extend(mBaseClass);

        MethodType mt = MethodType.methodType
            (void.class, TableManager.class, Index.class, RowPredicateLock.class);

        MethodMaker ctor = mClassMaker.addConstructor(mt);
        ctor.invokeSuperConstructor(ctor.param(0), ctor.param(1), ctor.param(2));

        if (isPrimaryTable()) {
            addDynamicEncodeValueColumns();
            addDynamicDecodeValueColumns();

            // Override the inherited abstract delegate methods to call the static methods
            // defined above.

            MethodMaker mm = mClassMaker.addMethod
                (byte[].class, "doEncodeValue", mRowClass).protected_();
            mm.return_(mm.invoke("encodeValue", mm.param(0)));

            mm = mClassMaker.addMethod
                (null, "doDecodeValue", mRowClass, byte[].class).protected_();

            addDynamicUpdateValueColumns();
            addDoUpdateMethod();

            mm.invoke("decodeValue", mm.param(0), mm.param(1));

            addDynamicWriteRowMethod();
        }

        addUnfilteredMethods(mTableId);

        return doFinish(mt);
    }

    /**
     * Implements: static byte[] encodeValue(RowClass row)
     *
     * Method isn't implemented until needed, delaying acquisition/creation of the current
     * schema version. This allows replicas to decode existing rows even when the class
     * definition has changed, but encoding will still fail.
     */
    private void addDynamicEncodeValueColumns() {
        MethodMaker mm = mClassMaker.addMethod(byte[].class, "encodeValue", mRowClass).static_();
        var indy = mm.var(DynamicTableMaker.class).indy
            ("indyEncodeValueColumns", mStore.ref(), mRowType, mTableId);
        mm.return_(indy.invoke(byte[].class, "encodeValue", null, mm.param(0)));
    }

    public static CallSite indyEncodeValueColumns
        (MethodHandles.Lookup lookup, String name, MethodType mt,
         WeakReference<RowStore> storeRef, Class<?> rowType, long tableId)
    {
        return doIndyEncode
            (lookup, name, mt, storeRef, rowType, tableId, (mm, info, schemaVersion) -> {
                ColumnCodec[] codecs = info.rowGen().valueCodecs();
                addEncodeColumns(mm, ColumnCodec.bind(schemaVersion, codecs, mm));
            });
    }

    /**
     * Implements: static byte[] updateValue(RowClass row, byte[] original)
     *
     * Method isn't fully implemented until needed, delaying acquisition/creation of the
     * current schema version. This allows replicas to decode existing rows even when the class
     * definition has changed, but encoding will still fail.
     */
    private void addDynamicUpdateValueColumns() {
        MethodMaker mm = mClassMaker
            .addMethod(byte[].class, "updateValue", mRowClass, byte[].class).static_();

        if (mCodecGen.info.valueColumns.isEmpty()) {
            // If the checkValueAllDirty method was defined, it would always return true.
            mm.return_(mm.var(RowUtils.class).field("EMPTY_BYTES"));
            return;
        }

        Variable rowVar = mm.param(0);

        Label partiallyDirty = mm.label();
        mm.invoke("checkValueAllDirty", rowVar).ifFalse(partiallyDirty);
        mm.return_(mm.invoke("encodeValue", rowVar));
        partiallyDirty.here();

        var indy = mm.var(DynamicTableMaker.class).indy
            ("indyUpdateValueColumns", mStore.ref(), mRowType, mTableId);
        mm.return_(indy.invoke(byte[].class, "updateValue", null, rowVar, mm.param(1)));
    }

    public static CallSite indyUpdateValueColumns
        (MethodHandles.Lookup lookup, String name, MethodType mt,
         WeakReference<RowStore> storeRef, Class<?> rowType, long tableId)
    {
        return doIndyEncode
            (lookup, name, mt, storeRef, rowType, tableId, (mm, info, schemaVersion) -> {
                // These variables were provided by the indy call in addDynamicUpdateValueColumns.
                Variable rowVar = mm.param(0);
                Variable originalVar = mm.param(1); // byte[]

                var tableVar = mm.var(lookup.lookupClass());
                var ue = encodeUpdateValue(mm, info, schemaVersion, tableVar, rowVar, originalVar);

                mm.return_(ue.newEntryVar);
            });
    }

    @FunctionalInterface
    static interface EncodeFinisher {
        void finish(MethodMaker mm, RowInfo info, int schemaVersion);
    }

    /**
     * Does the work to obtain the current schema version, handling any exceptions. The given
     * finisher completes the definition of the encode method when no exception was thrown when
     * trying to obtain the schema version. If an exception was thrown, the finisher might be
     * called at a later time.
     */
    private static CallSite doIndyEncode(MethodHandles.Lookup lookup, String name, MethodType mt,
                                         WeakReference<RowStore> storeRef,
                                         Class<?> rowType, long tableId,
                                         EncodeFinisher finisher)
    {
        return ExceptionCallSite.make(() -> {
            MethodMaker mm = MethodMaker.begin(lookup, name, mt);
            RowStore store = storeRef.get();
            if (store == null) {
                mm.new_(DatabaseException.class, "Closed").throw_();
            } else {
                RowInfo info = RowInfo.find(rowType);
                int schemaVersion;
                try {
                    schemaVersion = store.schemaVersion(info, false, tableId, true);
                } catch (Exception e) {
                    return new ExceptionCallSite.Failed(mt, mm, e);
                }
                finisher.finish(mm, info, schemaVersion);
            }
            return mm.finish();
        });
    }

    private void addDynamicDecodeValueColumns() {
        // First define a method which generates the SwitchCallSite.
        {
            MethodMaker mm = mClassMaker.addMethod
                (SwitchCallSite.class, "decodeValueSwitchCallSite").static_();
            var condy = mm.var(DynamicTableMaker.class).condy
                ("condyDecodeValueColumns",  mStore.ref(), mRowType, mRowClass, mTableId);
            mm.return_(condy.invoke(SwitchCallSite.class, "_"));
        }

        // Also define a method to obtain a MethodHandle which decodes for a given schema
        // version. This must be defined here to ensure that the correct lookup is used. It
        // must always refer to this table class.
        {
            MethodMaker mm = mClassMaker.addMethod
                (MethodHandle.class, "decodeValueHandle", int.class).static_();
            var lookup = mm.var(MethodHandles.class).invoke("lookup");
            var mh = mm.invoke("decodeValueSwitchCallSite").invoke("getCase", lookup, mm.param(0));
            mm.return_(mh);
        }

        MethodMaker mm = mClassMaker.addMethod
            (null, "decodeValue", mRowClass, byte[].class).static_().public_();

        var data = mm.param(1);
        var schemaVersion = mm.var(RowUtils.class).invoke("decodeSchemaVersion", data);

        var indy = mm.var(DynamicTableMaker.class).indy("indyDecodeValueColumns");
        indy.invoke(null, "decodeValue", null, schemaVersion, mm.param(0), data);
    }

    /**
     * Returns a SwitchCallSite instance suitable for decoding all value columns. By defining
     * it via a "condy" method, the SwitchCallSite instance can be shared by other methods. In
     * particular, filter subclasses are generated against specific schema versions, and so
     * they need direct access to just one of the cases. This avoids a redundant version check.
     *
     * MethodType is: void (int schemaVersion, RowClass row, byte[] data)
     */
    public static SwitchCallSite condyDecodeValueColumns
        (MethodHandles.Lookup lookup, String name, Class<?> type,
         WeakReference<RowStore> storeRef, Class<?> rowType, Class<?> rowClass, long tableId)
    {
        MethodType mt = MethodType.methodType(void.class, int.class, rowClass, byte[].class);

        return new SwitchCallSite(lookup, mt, schemaVersion -> {
            MethodMaker mm = MethodMaker.begin(lookup, null, "decode", rowClass, byte[].class);

            RowStore store = storeRef.get();
            if (store == null) {
                mm.new_(DatabaseException.class, "Closed").throw_();
            } else {
                RowInfo dstRowInfo = RowInfo.find(rowType);

                if (schemaVersion == 0) {
                    // No columns to decode, so assign defaults.
                    for (Map.Entry<String, ColumnInfo> e : dstRowInfo.valueColumns.entrySet()) {
                        Converter.setDefault(mm, e.getValue(), mm.param(0).field(e.getKey()));
                    }
                } else {
                    RowInfo srcRowInfo;
                    try {
                        srcRowInfo = store.rowInfo(rowType, tableId, schemaVersion);
                    } catch (Exception e) {
                        return new ExceptionCallSite.Failed
                            (MethodType.methodType(void.class, rowClass, byte[].class), mm, e);
                    }

                    ColumnCodec[] srcCodecs = srcRowInfo.rowGen().valueCodecs();
                    int fixedOffset = schemaVersion < 128 ? 1 : 4;

                    addDecodeColumns(mm, dstRowInfo, srcCodecs, fixedOffset);

                    if (dstRowInfo != srcRowInfo) {
                        // Assign defaults for any missing columns.
                        for (Map.Entry<String, ColumnInfo> e : dstRowInfo.valueColumns.entrySet()) {
                            String fieldName = e.getKey();
                            if (!srcRowInfo.valueColumns.containsKey(fieldName)) {
                                Converter.setDefault
                                    (mm, e.getValue(), mm.param(0).field(fieldName));
                            }
                        }
                    }
                }
            }

            return mm.finish();
        });
    }

    /**
     * This just returns the SwitchCallSite generated by condyDecodeValueColumns.
     */
    public static SwitchCallSite indyDecodeValueColumns(MethodHandles.Lookup lookup,
                                                        String name, MethodType mt)
        throws Throwable
    {
        MethodHandle mh = lookup.findStatic(lookup.lookupClass(), "decodeValueSwitchCallSite",
                                            MethodType.methodType(SwitchCallSite.class));
        return (SwitchCallSite) mh.invokeExact();
    }

    /**
     * Add a method for remotely serializing rows.
     *
     * @see RowEvaluator#writeRow
     */
    private void addDynamicWriteRowMethod() {
        MethodMaker mm = mClassMaker.addMethod
            (null, "writeRow", RowWriter.class, byte[].class, byte[].class).static_();

        var writerVar = mm.param(0);
        var keyVar = mm.param(1);
        var valueVar = mm.param(2);

        var schemaVersion = mm.var(RowUtils.class).invoke("decodeSchemaVersion", valueVar);

        var indy = mm.var(WriteRowMaker.class).indy
            ("indyWriteRow", mStore.ref(), mRowType, mTableId, null);
        indy.invoke(null, "writeRow", null, schemaVersion, writerVar, keyVar, valueVar);
    }

    /**
     * Override in order for addDoUpdateMethod to work.
     */
    @Override
    protected void finishDoUpdate(MethodMaker mm,
                                  Variable rowVar, Variable mergeVar, Variable cursorVar)
    {
        var indy = mm.var(DynamicTableMaker.class).indy
            ("indyDoUpdate", mStore.ref(), mRowType, mTableId, supportsTriggers() ? 1 : 0);
        indy.invoke(null, "doUpdate", null, mm.this_(), rowVar, mergeVar, cursorVar);
    }

    /**
     * @param triggers 0 for false, 1 for true
     */
    public static CallSite indyDoUpdate(MethodHandles.Lookup lookup, String name, MethodType mt,
                                        WeakReference<RowStore> storeRef,
                                        Class<?> rowType, long tableId, int triggers)
    {
        return doIndyEncode
            (lookup, name, mt, storeRef, rowType, tableId, (mm, info, schemaVersion) -> {
                // All these variables were provided by the indy call in addDoUpdateMethod.
                Variable tableVar = mm.param(0);
                Variable rowVar = mm.param(1);
                Variable mergeVar = mm.param(2);
                Variable cursorVar = mm.param(3);

                finishDoUpdate(mm, info, schemaVersion, triggers, false,
                               tableVar, rowVar, mergeVar, cursorVar);
            });
    }
}
