/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.conductor.redis.dao;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.netflix.conductor.common.metadata.events.EventExecution;
import com.netflix.conductor.common.metadata.tasks.TaskDef;
import com.netflix.conductor.core.config.ConductorProperties;
import com.netflix.conductor.core.exception.TransientException;
import com.netflix.conductor.dao.ConcurrentExecutionLimitDAO;
import com.netflix.conductor.dao.ExecutionDAO;
import com.netflix.conductor.metrics.Monitors;
import com.netflix.conductor.model.TaskModel;
import com.netflix.conductor.model.WorkflowModel;
import com.netflix.conductor.redis.config.AnyRedisCondition;
import com.netflix.conductor.redis.config.RedisProperties;
import com.netflix.conductor.redis.dao.BaseDynoDAO;
import com.netflix.conductor.redis.jedis.JedisProxy;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Component;

@Component
@Conditional(value={AnyRedisCondition.class})
public class RedisExecutionDAO
extends BaseDynoDAO
implements ExecutionDAO,
ConcurrentExecutionLimitDAO {
    public static final Logger LOGGER = LoggerFactory.getLogger(RedisExecutionDAO.class);
    private static final String TASK_LIMIT_BUCKET = "TASK_LIMIT_BUCKET";
    private static final String IN_PROGRESS_TASKS = "IN_PROGRESS_TASKS";
    private static final String TASKS_IN_PROGRESS_STATUS = "TASKS_IN_PROGRESS_STATUS";
    private static final String WORKFLOW_TO_TASKS = "WORKFLOW_TO_TASKS";
    private static final String SCHEDULED_TASKS = "SCHEDULED_TASKS";
    private static final String TASK = "TASK";
    private static final String WORKFLOW = "WORKFLOW";
    private static final String PENDING_WORKFLOWS = "PENDING_WORKFLOWS";
    private static final String WORKFLOW_DEF_TO_WORKFLOWS = "WORKFLOW_DEF_TO_WORKFLOWS";
    private static final String CORR_ID_TO_WORKFLOWS = "CORR_ID_TO_WORKFLOWS";
    private static final String EVENT_EXECUTION = "EVENT_EXECUTION";
    private final int ttlEventExecutionSeconds;

    public RedisExecutionDAO(JedisProxy jedisProxy, ObjectMapper objectMapper, ConductorProperties conductorProperties, RedisProperties properties) {
        super(jedisProxy, objectMapper, conductorProperties, properties);
        this.ttlEventExecutionSeconds = (int)properties.getEventExecutionPersistenceTTL().getSeconds();
    }

    private static String dateStr(Long timeInMs) {
        Date date = new Date(timeInMs);
        return RedisExecutionDAO.dateStr(date);
    }

    private static String dateStr(Date date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return format.format(date);
    }

    private static List<String> dateStrBetweenDates(Long startdatems, Long enddatems) {
        ArrayList<String> dates = new ArrayList<String>();
        GregorianCalendar calendar = new GregorianCalendar();
        Date startdate = new Date(startdatems);
        Date enddate = new Date(enddatems);
        calendar.setTime(startdate);
        while (calendar.getTime().before(enddate) || calendar.getTime().equals(enddate)) {
            Date result = calendar.getTime();
            dates.add(RedisExecutionDAO.dateStr(result));
            ((Calendar)calendar).add(5, 1);
        }
        return dates;
    }

    public List<TaskModel> getPendingTasksByWorkflow(String taskName, String workflowId) {
        LinkedList<TaskModel> tasks = new LinkedList<TaskModel>();
        List<TaskModel> pendingTasks = this.getPendingTasksForTaskType(taskName);
        pendingTasks.forEach(pendingTask -> {
            if (pendingTask.getWorkflowInstanceId().equals(workflowId)) {
                tasks.add((TaskModel)pendingTask);
            }
        });
        return tasks;
    }

    public List<TaskModel> getTasks(String taskDefName, String startKey, int count) {
        LinkedList<TaskModel> tasks = new LinkedList<TaskModel>();
        List<TaskModel> pendingTasks = this.getPendingTasksForTaskType(taskDefName);
        boolean startKeyFound = startKey == null;
        int foundcount = 0;
        for (TaskModel pendingTask : pendingTasks) {
            if (!startKeyFound && pendingTask.getTaskId().equals(startKey)) {
                startKeyFound = true;
                if (startKey != null) continue;
            }
            if (!startKeyFound || foundcount >= count) continue;
            tasks.add(pendingTask);
            ++foundcount;
        }
        return tasks;
    }

    public List<TaskModel> createTasks(List<TaskModel> tasks) {
        LinkedList<TaskModel> tasksCreated = new LinkedList<TaskModel>();
        for (TaskModel task : tasks) {
            this.validate(task);
            this.recordRedisDaoRequests("createTask", task.getTaskType(), task.getWorkflowType());
            String taskKey = task.getReferenceTaskName() + task.getRetryCount();
            Long added = this.jedisProxy.hset(this.nsKey(SCHEDULED_TASKS, task.getWorkflowInstanceId()), taskKey, task.getTaskId());
            if (added < 1L) {
                LOGGER.debug("Task already scheduled, skipping the run " + task.getTaskId() + ", ref=" + task.getReferenceTaskName() + ", key=" + taskKey);
                continue;
            }
            if (task.getStatus() != null && !task.getStatus().isTerminal() && task.getScheduledTime() == 0L) {
                task.setScheduledTime(System.currentTimeMillis());
            }
            this.correlateTaskToWorkflowInDS(task.getTaskId(), task.getWorkflowInstanceId());
            LOGGER.debug("Scheduled task added to WORKFLOW_TO_TASKS workflowId: {}, taskId: {}, taskType: {} during createTasks", new Object[]{task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType()});
            String inProgressTaskKey = this.nsKey(IN_PROGRESS_TASKS, task.getTaskDefName());
            this.jedisProxy.sadd(inProgressTaskKey, task.getTaskId());
            LOGGER.debug("Scheduled task added to IN_PROGRESS_TASKS with inProgressTaskKey: {}, workflowId: {}, taskId: {}, taskType: {} during createTasks", new Object[]{inProgressTaskKey, task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType()});
            this.updateTask(task);
            tasksCreated.add(task);
        }
        return tasksCreated;
    }

    public void updateTask(TaskModel task) {
        Set<String> taskIds;
        Optional taskDefinition = task.getTaskDefinition();
        if (taskDefinition.isPresent() && ((TaskDef)taskDefinition.get()).concurrencyLimit() > 0) {
            if (task.getStatus() != null && task.getStatus().equals((Object)TaskModel.Status.IN_PROGRESS)) {
                this.jedisProxy.sadd(this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName()), task.getTaskId());
                LOGGER.debug("Workflow Task added to TASKS_IN_PROGRESS_STATUS with tasksInProgressKey: {}, workflowId: {}, taskId: {}, taskType: {}, taskStatus: {} during updateTask", new Object[]{this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName(), task.getTaskId()), task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType(), task.getStatus().name()});
            } else {
                this.jedisProxy.srem(this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName()), task.getTaskId());
                LOGGER.debug("Workflow Task removed from TASKS_IN_PROGRESS_STATUS with tasksInProgressKey: {}, workflowId: {}, taskId: {}, taskType: {}, taskStatus: {} during updateTask", new Object[]{this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName(), task.getTaskId()), task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType(), task.getStatus().name()});
                String key = this.nsKey(TASK_LIMIT_BUCKET, task.getTaskDefName());
                this.jedisProxy.zrem(key, task.getTaskId());
                LOGGER.debug("Workflow Task removed from TASK_LIMIT_BUCKET with taskLimitBucketKey: {}, workflowId: {}, taskId: {}, taskType: {}, taskStatus: {} during updateTask", new Object[]{key, task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType(), task.getStatus().name()});
            }
        }
        String payload = this.toJson(task);
        this.recordRedisDaoPayloadSize("updateTask", payload.length(), taskDefinition.map(TaskDef::getName).orElse("n/a"), task.getWorkflowType());
        this.recordRedisDaoRequests("updateTask", task.getTaskType(), task.getWorkflowType());
        this.jedisProxy.set(this.nsKey(TASK, task.getTaskId()), payload);
        LOGGER.debug("Workflow task payload saved to TASK with taskKey: {}, workflowId: {}, taskId: {}, taskType: {} during updateTask", new Object[]{this.nsKey(TASK, task.getTaskId()), task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType()});
        if (task.getStatus() != null && task.getStatus().isTerminal()) {
            this.jedisProxy.srem(this.nsKey(IN_PROGRESS_TASKS, task.getTaskDefName()), task.getTaskId());
            LOGGER.debug("Workflow Task removed from TASKS_IN_PROGRESS_STATUS with tasksInProgressKey: {}, workflowId: {}, taskId: {}, taskType: {}, taskStatus: {} during updateTask", new Object[]{this.nsKey(IN_PROGRESS_TASKS, task.getTaskDefName()), task.getWorkflowInstanceId(), task.getTaskId(), task.getTaskType(), task.getStatus().name()});
        }
        if (!(taskIds = this.jedisProxy.smembers(this.nsKey(WORKFLOW_TO_TASKS, task.getWorkflowInstanceId()))).contains(task.getTaskId())) {
            this.correlateTaskToWorkflowInDS(task.getTaskId(), task.getWorkflowInstanceId());
        }
    }

    public boolean exceedsLimit(TaskModel task) {
        boolean rateLimited;
        Optional taskDefinition = task.getTaskDefinition();
        if (taskDefinition.isEmpty()) {
            return false;
        }
        int limit = ((TaskDef)taskDefinition.get()).concurrencyLimit();
        if (limit <= 0) {
            return false;
        }
        long current = this.getInProgressTaskCount(task.getTaskDefName());
        if (current >= (long)limit) {
            LOGGER.info("Task execution count limited. task - {}:{}, limit: {}, current: {}", new Object[]{task.getTaskId(), task.getTaskDefName(), limit, current});
            Monitors.recordTaskConcurrentExecutionLimited((String)task.getTaskDefName(), (int)limit);
            return true;
        }
        String rateLimitKey = this.nsKey(TASK_LIMIT_BUCKET, task.getTaskDefName());
        double score = System.currentTimeMillis();
        String taskId = task.getTaskId();
        this.jedisProxy.zaddnx(rateLimitKey, score, taskId);
        this.recordRedisDaoRequests("checkTaskRateLimiting", task.getTaskType(), task.getWorkflowType());
        Set<String> ids = this.jedisProxy.zrangeByScore(rateLimitKey, 0.0, score + 1.0, limit);
        boolean bl = rateLimited = !ids.contains(taskId);
        if (rateLimited) {
            LOGGER.info("Task execution count limited. task - {}:{}, limit: {}, current: {}", new Object[]{task.getTaskId(), task.getTaskDefName(), limit, current});
            String inProgressKey = this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName());
            ids.stream().filter(id -> !this.jedisProxy.sismember(inProgressKey, (String)id)).forEach(id2 -> this.jedisProxy.zrem(rateLimitKey, (String)id2));
            Monitors.recordTaskRateLimited((String)task.getTaskDefName(), (int)limit);
        }
        return rateLimited;
    }

    private void removeTaskMappings(TaskModel task) {
        String taskKey = task.getReferenceTaskName() + task.getRetryCount();
        this.jedisProxy.hdel(this.nsKey(SCHEDULED_TASKS, task.getWorkflowInstanceId()), taskKey);
        this.jedisProxy.srem(this.nsKey(IN_PROGRESS_TASKS, task.getTaskDefName()), task.getTaskId());
        this.jedisProxy.srem(this.nsKey(WORKFLOW_TO_TASKS, task.getWorkflowInstanceId()), task.getTaskId());
        this.jedisProxy.srem(this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName()), task.getTaskId());
        this.jedisProxy.zrem(this.nsKey(TASK_LIMIT_BUCKET, task.getTaskDefName()), task.getTaskId());
    }

    private void removeTaskMappingsWithExpiry(TaskModel task) {
        String taskKey = task.getReferenceTaskName() + task.getRetryCount();
        this.jedisProxy.hdel(this.nsKey(SCHEDULED_TASKS, task.getWorkflowInstanceId()), taskKey);
        this.jedisProxy.srem(this.nsKey(IN_PROGRESS_TASKS, task.getTaskDefName()), task.getTaskId());
        this.jedisProxy.srem(this.nsKey(TASKS_IN_PROGRESS_STATUS, task.getTaskDefName()), task.getTaskId());
        this.jedisProxy.zrem(this.nsKey(TASK_LIMIT_BUCKET, task.getTaskDefName()), task.getTaskId());
    }

    public boolean removeTask(String taskId) {
        TaskModel task = this.getTask(taskId);
        if (task == null) {
            LOGGER.warn("No such task found by id {}", (Object)taskId);
            return false;
        }
        this.removeTaskMappings(task);
        this.jedisProxy.del(this.nsKey(TASK, task.getTaskId()));
        this.recordRedisDaoRequests("removeTask", task.getTaskType(), task.getWorkflowType());
        return true;
    }

    private boolean removeTaskWithExpiry(String taskId, int ttlSeconds) {
        TaskModel task = this.getTask(taskId);
        if (task == null) {
            LOGGER.warn("No such task found by id {}", (Object)taskId);
            return false;
        }
        this.removeTaskMappingsWithExpiry(task);
        this.jedisProxy.expire(this.nsKey(TASK, task.getTaskId()), ttlSeconds);
        this.recordRedisDaoRequests("removeTask", task.getTaskType(), task.getWorkflowType());
        return true;
    }

    public TaskModel getTask(String taskId) {
        Preconditions.checkNotNull((Object)taskId, (Object)"taskId cannot be null");
        return Optional.ofNullable(this.jedisProxy.get(this.nsKey(TASK, taskId))).map(json -> {
            TaskModel task = this.readValue((String)json, TaskModel.class);
            this.recordRedisDaoRequests("getTask", task.getTaskType(), task.getWorkflowType());
            this.recordRedisDaoPayloadSize("getTask", this.toJson(task).length(), task.getTaskType(), task.getWorkflowType());
            return task;
        }).orElse(null);
    }

    public List<TaskModel> getTasks(List<String> taskIds) {
        return taskIds.stream().map(taskId -> this.nsKey(TASK, (String)taskId)).map(this.jedisProxy::get).filter(Objects::nonNull).map(jsonString -> {
            TaskModel task = this.readValue((String)jsonString, TaskModel.class);
            this.recordRedisDaoRequests("getTask", task.getTaskType(), task.getWorkflowType());
            this.recordRedisDaoPayloadSize("getTask", jsonString.length(), task.getTaskType(), task.getWorkflowType());
            return task;
        }).collect(Collectors.toList());
    }

    public List<TaskModel> getTasksForWorkflow(String workflowId) {
        Preconditions.checkNotNull((Object)workflowId, (Object)"workflowId cannot be null");
        Set<String> taskIds = this.jedisProxy.smembers(this.nsKey(WORKFLOW_TO_TASKS, workflowId));
        this.recordRedisDaoRequests("getTasksForWorkflow");
        return this.getTasks(new ArrayList<String>(taskIds));
    }

    public List<TaskModel> getPendingTasksForTaskType(String taskName) {
        Preconditions.checkNotNull((Object)taskName, (Object)"task name cannot be null");
        Set<String> taskIds = this.jedisProxy.smembers(this.nsKey(IN_PROGRESS_TASKS, taskName));
        this.recordRedisDaoRequests("getPendingTasksForTaskType");
        return this.getTasks(new ArrayList<String>(taskIds));
    }

    public String createWorkflow(WorkflowModel workflow) {
        return this.insertOrUpdateWorkflow(workflow, false);
    }

    public String updateWorkflow(WorkflowModel workflow) {
        return this.insertOrUpdateWorkflow(workflow, true);
    }

    public boolean removeWorkflow(String workflowId) {
        WorkflowModel workflow = this.getWorkflow(workflowId, true);
        if (workflow != null) {
            this.recordRedisDaoRequests("removeWorkflow");
            String key = this.nsKey(WORKFLOW_DEF_TO_WORKFLOWS, workflow.getWorkflowName(), RedisExecutionDAO.dateStr(workflow.getCreateTime()));
            this.jedisProxy.srem(key, workflowId);
            this.jedisProxy.srem(this.nsKey(CORR_ID_TO_WORKFLOWS, workflow.getCorrelationId()), workflowId);
            this.jedisProxy.srem(this.nsKey(PENDING_WORKFLOWS, workflow.getWorkflowName()), workflowId);
            this.jedisProxy.del(this.nsKey(WORKFLOW, workflowId));
            for (TaskModel task : workflow.getTasks()) {
                this.removeTask(task.getTaskId());
            }
            return true;
        }
        return false;
    }

    public boolean removeWorkflowWithExpiry(String workflowId, int ttlSeconds) {
        WorkflowModel workflow = this.getWorkflow(workflowId, true);
        if (workflow != null) {
            this.recordRedisDaoRequests("removeWorkflow");
            String key = this.nsKey(WORKFLOW_DEF_TO_WORKFLOWS, workflow.getWorkflowName(), RedisExecutionDAO.dateStr(workflow.getCreateTime()));
            this.jedisProxy.srem(key, workflowId);
            this.jedisProxy.srem(this.nsKey(CORR_ID_TO_WORKFLOWS, workflow.getCorrelationId()), workflowId);
            this.jedisProxy.srem(this.nsKey(PENDING_WORKFLOWS, workflow.getWorkflowName()), workflowId);
            this.jedisProxy.expire(this.nsKey(WORKFLOW, workflowId), ttlSeconds);
            for (TaskModel task : workflow.getTasks()) {
                this.removeTaskWithExpiry(task.getTaskId(), ttlSeconds);
            }
            this.jedisProxy.expire(this.nsKey(WORKFLOW_TO_TASKS, workflowId), ttlSeconds);
            return true;
        }
        return false;
    }

    public void removeFromPendingWorkflow(String workflowType, String workflowId) {
        this.recordRedisDaoRequests("removePendingWorkflow");
        this.jedisProxy.del(this.nsKey(SCHEDULED_TASKS, workflowId));
        this.jedisProxy.srem(this.nsKey(PENDING_WORKFLOWS, workflowType), workflowId);
    }

    public WorkflowModel getWorkflow(String workflowId) {
        return this.getWorkflow(workflowId, true);
    }

    public WorkflowModel getWorkflow(String workflowId, boolean includeTasks) {
        String json = this.jedisProxy.get(this.nsKey(WORKFLOW, workflowId));
        WorkflowModel workflow = null;
        if (json != null) {
            workflow = this.readValue(json, WorkflowModel.class);
            this.recordRedisDaoRequests("getWorkflow", "n/a", workflow.getWorkflowName());
            this.recordRedisDaoPayloadSize("getWorkflow", json.length(), "n/a", workflow.getWorkflowName());
            if (includeTasks) {
                List<TaskModel> tasks = this.getTasksForWorkflow(workflowId);
                tasks.sort(Comparator.comparingInt(TaskModel::getSeq));
                workflow.setTasks(tasks);
            }
        }
        return workflow;
    }

    public List<String> getRunningWorkflowIds(String workflowName, int version) {
        Preconditions.checkNotNull((Object)workflowName, (Object)"workflowName cannot be null");
        this.recordRedisDaoRequests("getRunningWorkflowsByName");
        Set<String> pendingWorkflows = this.jedisProxy.smembers(this.nsKey(PENDING_WORKFLOWS, workflowName));
        LinkedList<String> workflowIds = new LinkedList<String>(pendingWorkflows);
        return workflowIds;
    }

    public List<WorkflowModel> getPendingWorkflowsByType(String workflowName, int version) {
        Preconditions.checkNotNull((Object)workflowName, (Object)"workflowName cannot be null");
        List<String> workflowIds = this.getRunningWorkflowIds(workflowName, version);
        return workflowIds.stream().map(this::getWorkflow).filter(workflow -> workflow.getWorkflowVersion() == version).collect(Collectors.toList());
    }

    public List<WorkflowModel> getWorkflowsByType(String workflowName, Long startTime, Long endTime) {
        Preconditions.checkNotNull((Object)workflowName, (Object)"workflowName cannot be null");
        Preconditions.checkNotNull((Object)startTime, (Object)"startTime cannot be null");
        Preconditions.checkNotNull((Object)endTime, (Object)"endTime cannot be null");
        LinkedList<WorkflowModel> workflows = new LinkedList<WorkflowModel>();
        List<String> dateStrs = RedisExecutionDAO.dateStrBetweenDates(startTime, endTime);
        dateStrs.forEach(dateStr -> {
            String key = this.nsKey(WORKFLOW_DEF_TO_WORKFLOWS, workflowName, (String)dateStr);
            this.jedisProxy.smembers(key).forEach(workflowId -> {
                try {
                    WorkflowModel workflow = this.getWorkflow((String)workflowId);
                    if (workflow.getCreateTime() >= startTime && workflow.getCreateTime() <= endTime) {
                        workflows.add(workflow);
                    }
                }
                catch (Exception e) {
                    LOGGER.error("Failed to get workflow: {}", workflowId, (Object)e);
                }
            });
        });
        return workflows;
    }

    public List<WorkflowModel> getWorkflowsByCorrelationId(String workflowName, String correlationId, boolean includeTasks) {
        throw new UnsupportedOperationException("This method is not implemented in RedisExecutionDAO. Please use ExecutionDAOFacade instead.");
    }

    public boolean canSearchAcrossWorkflows() {
        return false;
    }

    private String insertOrUpdateWorkflow(WorkflowModel workflow, boolean update) {
        Preconditions.checkNotNull((Object)workflow, (Object)"workflow object cannot be null");
        List tasks = workflow.getTasks();
        workflow.setTasks(new LinkedList());
        String payload = this.toJson(workflow);
        this.jedisProxy.set(this.nsKey(WORKFLOW, workflow.getWorkflowId()), payload);
        this.recordRedisDaoRequests("storeWorkflow", "n/a", workflow.getWorkflowName());
        this.recordRedisDaoPayloadSize("storeWorkflow", payload.length(), "n/a", workflow.getWorkflowName());
        if (!update) {
            String key = this.nsKey(WORKFLOW_DEF_TO_WORKFLOWS, workflow.getWorkflowName(), RedisExecutionDAO.dateStr(workflow.getCreateTime()));
            this.jedisProxy.sadd(key, workflow.getWorkflowId());
            if (workflow.getCorrelationId() != null) {
                this.jedisProxy.sadd(this.nsKey(CORR_ID_TO_WORKFLOWS, workflow.getCorrelationId()), workflow.getWorkflowId());
            }
        }
        if (workflow.getStatus().isTerminal()) {
            this.jedisProxy.srem(this.nsKey(PENDING_WORKFLOWS, workflow.getWorkflowName()), workflow.getWorkflowId());
        } else {
            this.jedisProxy.sadd(this.nsKey(PENDING_WORKFLOWS, workflow.getWorkflowName()), workflow.getWorkflowId());
        }
        workflow.setTasks(tasks);
        return workflow.getWorkflowId();
    }

    @VisibleForTesting
    void correlateTaskToWorkflowInDS(String taskId, String workflowInstanceId) {
        String workflowToTaskKey = this.nsKey(WORKFLOW_TO_TASKS, workflowInstanceId);
        this.jedisProxy.sadd(workflowToTaskKey, taskId);
        LOGGER.debug("Task mapped in WORKFLOW_TO_TASKS with workflowToTaskKey: {}, workflowId: {}, taskId: {}", new Object[]{workflowToTaskKey, workflowInstanceId, taskId});
    }

    public long getPendingWorkflowCount(String workflowName) {
        String key = this.nsKey(PENDING_WORKFLOWS, workflowName);
        this.recordRedisDaoRequests("getPendingWorkflowCount");
        return this.jedisProxy.scard(key);
    }

    public long getInProgressTaskCount(String taskDefName) {
        String inProgressKey = this.nsKey(TASKS_IN_PROGRESS_STATUS, taskDefName);
        this.recordRedisDaoRequests("getInProgressTaskCount");
        return this.jedisProxy.scard(inProgressKey);
    }

    public boolean addEventExecution(EventExecution eventExecution) {
        try {
            boolean added;
            String key = this.nsKey(EVENT_EXECUTION, eventExecution.getName(), eventExecution.getEvent(), eventExecution.getMessageId());
            String json = this.objectMapper.writeValueAsString((Object)eventExecution);
            this.recordRedisDaoEventRequests("addEventExecution", eventExecution.getEvent());
            this.recordRedisDaoPayloadSize("addEventExecution", json.length(), eventExecution.getEvent(), "n/a");
            boolean bl = added = this.jedisProxy.hsetnx(key, eventExecution.getId(), json) == 1L;
            if (this.ttlEventExecutionSeconds > 0) {
                this.jedisProxy.expire(key, this.ttlEventExecutionSeconds);
            }
            return added;
        }
        catch (Exception e) {
            throw new TransientException("Unable to add event execution for " + eventExecution.getId(), (Throwable)e);
        }
    }

    public void updateEventExecution(EventExecution eventExecution) {
        try {
            String key = this.nsKey(EVENT_EXECUTION, eventExecution.getName(), eventExecution.getEvent(), eventExecution.getMessageId());
            String json = this.objectMapper.writeValueAsString((Object)eventExecution);
            LOGGER.info("updating event execution {}", (Object)key);
            this.jedisProxy.hset(key, eventExecution.getId(), json);
            this.recordRedisDaoEventRequests("updateEventExecution", eventExecution.getEvent());
            this.recordRedisDaoPayloadSize("updateEventExecution", json.length(), eventExecution.getEvent(), "n/a");
        }
        catch (Exception e) {
            throw new TransientException("Unable to update event execution for " + eventExecution.getId(), (Throwable)e);
        }
    }

    public void removeEventExecution(EventExecution eventExecution) {
        try {
            String key = this.nsKey(EVENT_EXECUTION, eventExecution.getName(), eventExecution.getEvent(), eventExecution.getMessageId());
            LOGGER.info("removing event execution {}", (Object)key);
            this.jedisProxy.hdel(key, eventExecution.getId());
            this.recordRedisDaoEventRequests("removeEventExecution", eventExecution.getEvent());
        }
        catch (Exception e) {
            throw new TransientException("Unable to remove event execution for " + eventExecution.getId(), (Throwable)e);
        }
    }

    public List<EventExecution> getEventExecutions(String eventHandlerName, String eventName, String messageId, int max) {
        try {
            String field;
            String value;
            String key = this.nsKey(EVENT_EXECUTION, eventHandlerName, eventName, messageId);
            LOGGER.info("getting event execution {}", (Object)key);
            LinkedList<EventExecution> executions = new LinkedList<EventExecution>();
            for (int i = 0; i < max && (value = this.jedisProxy.hget(key, field = messageId + "_" + i)) != null; ++i) {
                this.recordRedisDaoEventRequests("getEventExecution", eventHandlerName);
                this.recordRedisDaoPayloadSize("getEventExecution", value.length(), eventHandlerName, "n/a");
                EventExecution eventExecution = (EventExecution)this.objectMapper.readValue(value, EventExecution.class);
                executions.add(eventExecution);
            }
            return executions;
        }
        catch (Exception e) {
            throw new TransientException("Unable to get event executions for " + eventHandlerName, (Throwable)e);
        }
    }

    private void validate(TaskModel task) {
        try {
            Preconditions.checkNotNull((Object)task, (Object)"task object cannot be null");
            Preconditions.checkNotNull((Object)task.getTaskId(), (Object)"Task id cannot be null");
            Preconditions.checkNotNull((Object)task.getWorkflowInstanceId(), (Object)"Workflow instance id cannot be null");
            Preconditions.checkNotNull((Object)task.getReferenceTaskName(), (Object)"Task reference name cannot be null");
        }
        catch (NullPointerException npe) {
            throw new IllegalArgumentException(npe.getMessage(), npe);
        }
    }
}

