/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.conductor.redis.dao;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.netflix.conductor.common.metadata.tasks.TaskDef;
import com.netflix.conductor.common.metadata.workflow.WorkflowDef;
import com.netflix.conductor.core.config.ConductorProperties;
import com.netflix.conductor.core.exception.ConflictException;
import com.netflix.conductor.core.exception.NotFoundException;
import com.netflix.conductor.dao.MetadataDAO;
import com.netflix.conductor.metrics.Monitors;
import com.netflix.conductor.redis.config.AnyRedisCondition;
import com.netflix.conductor.redis.config.RedisProperties;
import com.netflix.conductor.redis.dao.BaseDynoDAO;
import com.netflix.conductor.redis.jedis.JedisProxy;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Component;

@Component
@Conditional(value={AnyRedisCondition.class})
public class RedisMetadataDAO
extends BaseDynoDAO
implements MetadataDAO {
    private static final Logger LOGGER = LoggerFactory.getLogger(RedisMetadataDAO.class);
    private static final String ALL_TASK_DEFS = "TASK_DEFS";
    private static final String WORKFLOW_DEF_NAMES = "WORKFLOW_DEF_NAMES";
    private static final String WORKFLOW_DEF = "WORKFLOW_DEF";
    private static final String LATEST = "latest";
    private static final String className = RedisMetadataDAO.class.getSimpleName();
    private Map<String, TaskDef> taskDefCache = new HashMap<String, TaskDef>();

    public RedisMetadataDAO(JedisProxy jedisProxy, ObjectMapper objectMapper, ConductorProperties conductorProperties, RedisProperties properties) {
        super(jedisProxy, objectMapper, conductorProperties, properties);
        this.refreshTaskDefs();
        long cacheRefreshTime = properties.getTaskDefCacheRefreshInterval().getSeconds();
        Executors.newSingleThreadScheduledExecutor().scheduleWithFixedDelay(this::refreshTaskDefs, cacheRefreshTime, cacheRefreshTime, TimeUnit.SECONDS);
    }

    public TaskDef createTaskDef(TaskDef taskDef) {
        return this.insertOrUpdateTaskDef(taskDef);
    }

    public TaskDef updateTaskDef(TaskDef taskDef) {
        return this.insertOrUpdateTaskDef(taskDef);
    }

    private TaskDef insertOrUpdateTaskDef(TaskDef taskDef) {
        String payload = this.toJson(taskDef);
        this.jedisProxy.hset(this.nsKey(ALL_TASK_DEFS), taskDef.getName(), payload);
        this.recordRedisDaoRequests("storeTaskDef");
        this.recordRedisDaoPayloadSize("storeTaskDef", payload.length(), taskDef.getName(), "n/a");
        this.refreshTaskDefs();
        return taskDef;
    }

    private void refreshTaskDefs() {
        try {
            HashMap<String, TaskDef> map = new HashMap<String, TaskDef>();
            this.getAllTaskDefs().forEach(taskDef -> map.put(taskDef.getName(), (TaskDef)taskDef));
            this.taskDefCache = map;
            LOGGER.debug("Refreshed task defs " + this.taskDefCache.size());
        }
        catch (Exception e) {
            Monitors.error((String)className, (String)"refreshTaskDefs");
            LOGGER.error("refresh TaskDefs failed ", (Throwable)e);
        }
    }

    public TaskDef getTaskDef(String name) {
        return Optional.ofNullable(this.taskDefCache.get(name)).orElseGet(() -> this.getTaskDefFromDB(name));
    }

    private TaskDef getTaskDefFromDB(String name) {
        Preconditions.checkNotNull((Object)name, (Object)"TaskDef name cannot be null");
        TaskDef taskDef = null;
        String taskDefJsonStr = this.jedisProxy.hget(this.nsKey(ALL_TASK_DEFS), name);
        if (taskDefJsonStr != null) {
            taskDef = this.readValue(taskDefJsonStr, TaskDef.class);
            this.recordRedisDaoRequests("getTaskDef");
            this.recordRedisDaoPayloadSize("getTaskDef", taskDefJsonStr.length(), taskDef.getName(), "n/a");
        }
        this.setDefaults(taskDef);
        return taskDef;
    }

    private void setDefaults(TaskDef taskDef) {
        if (taskDef != null && taskDef.getResponseTimeoutSeconds() == 0L) {
            taskDef.setResponseTimeoutSeconds(taskDef.getTimeoutSeconds() == 0L ? 3600L : taskDef.getTimeoutSeconds() - 1L);
        }
    }

    public List<TaskDef> getAllTaskDefs() {
        LinkedList<TaskDef> allTaskDefs = new LinkedList<TaskDef>();
        this.recordRedisDaoRequests("getAllTaskDefs");
        Map<String, String> taskDefs = this.jedisProxy.hgetAll(this.nsKey(ALL_TASK_DEFS));
        int size = 0;
        if (taskDefs.size() > 0) {
            for (String taskDefJsonStr : taskDefs.values()) {
                if (taskDefJsonStr == null) continue;
                TaskDef taskDef = this.readValue(taskDefJsonStr, TaskDef.class);
                this.setDefaults(taskDef);
                allTaskDefs.add(taskDef);
                size += taskDefJsonStr.length();
            }
            this.recordRedisDaoPayloadSize("getAllTaskDefs", size, "n/a", "n/a");
        }
        return allTaskDefs;
    }

    public void removeTaskDef(String name) {
        Preconditions.checkNotNull((Object)name, (Object)"TaskDef name cannot be null");
        Long result = this.jedisProxy.hdel(this.nsKey(ALL_TASK_DEFS), name);
        if (!result.equals(1L)) {
            throw new NotFoundException("Cannot remove the task - no such task definition");
        }
        this.recordRedisDaoRequests("removeTaskDef");
        this.refreshTaskDefs();
    }

    public void createWorkflowDef(WorkflowDef def) {
        if (this.jedisProxy.hexists(this.nsKey(WORKFLOW_DEF, def.getName()), String.valueOf(def.getVersion())).booleanValue()) {
            throw new ConflictException("Workflow with %s already exists!", new Object[]{def.key()});
        }
        this._createOrUpdate(def);
    }

    public void updateWorkflowDef(WorkflowDef def) {
        this._createOrUpdate(def);
    }

    public Optional<WorkflowDef> getLatestWorkflowDef(String name) {
        String latestdata;
        Preconditions.checkNotNull((Object)name, (Object)"WorkflowDef name cannot be null");
        WorkflowDef workflowDef = null;
        Optional<Integer> optionalMaxVersion = this.getWorkflowMaxVersion(name);
        if (optionalMaxVersion.isPresent() && (latestdata = this.jedisProxy.hget(this.nsKey(WORKFLOW_DEF, name), optionalMaxVersion.get().toString())) != null) {
            workflowDef = this.readValue(latestdata, WorkflowDef.class);
        }
        return Optional.ofNullable(workflowDef);
    }

    private Optional<Integer> getWorkflowMaxVersion(String workflowName) {
        return this.jedisProxy.hkeys(this.nsKey(WORKFLOW_DEF, workflowName)).stream().filter(key -> !key.equals(LATEST)).map(Integer::valueOf).max(Comparator.naturalOrder());
    }

    public List<WorkflowDef> getAllVersions(String name) {
        Preconditions.checkNotNull((Object)name, (Object)"WorkflowDef name cannot be null");
        LinkedList<WorkflowDef> workflows = new LinkedList<WorkflowDef>();
        this.recordRedisDaoRequests("getAllWorkflowDefsByName");
        Map<String, String> workflowDefs = this.jedisProxy.hgetAll(this.nsKey(WORKFLOW_DEF, name));
        int size = 0;
        for (String key : workflowDefs.keySet()) {
            if (key.equals(LATEST)) continue;
            String workflowDef = workflowDefs.get(key);
            workflows.add(this.readValue(workflowDef, WorkflowDef.class));
            size += workflowDef.length();
        }
        this.recordRedisDaoPayloadSize("getAllWorkflowDefsByName", size, "n/a", name);
        return workflows;
    }

    public Optional<WorkflowDef> getWorkflowDef(String name, int version) {
        Preconditions.checkNotNull((Object)name, (Object)"WorkflowDef name cannot be null");
        WorkflowDef def = null;
        this.recordRedisDaoRequests("getWorkflowDef");
        String workflowDefJsonString = this.jedisProxy.hget(this.nsKey(WORKFLOW_DEF, name), String.valueOf(version));
        if (workflowDefJsonString != null) {
            def = this.readValue(workflowDefJsonString, WorkflowDef.class);
            this.recordRedisDaoPayloadSize("getWorkflowDef", workflowDefJsonString.length(), "n/a", name);
        }
        return Optional.ofNullable(def);
    }

    public void removeWorkflowDef(String name, Integer version) {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)name), (Object)"WorkflowDef name cannot be null");
        Preconditions.checkNotNull((Object)version, (Object)"Input version cannot be null");
        Long result = this.jedisProxy.hdel(this.nsKey(WORKFLOW_DEF, name), String.valueOf(version));
        if (!result.equals(1L)) {
            throw new NotFoundException("Cannot remove the workflow - no such workflow definition: %s version: %d", new Object[]{name, version});
        }
        Optional<Integer> optionMaxVersion = this.getWorkflowMaxVersion(name);
        if (optionMaxVersion.isEmpty()) {
            this.jedisProxy.srem(this.nsKey(WORKFLOW_DEF_NAMES), name);
        }
        this.recordRedisDaoRequests("removeWorkflowDef");
    }

    public List<String> findAll() {
        Set<String> wfNames = this.jedisProxy.smembers(this.nsKey(WORKFLOW_DEF_NAMES));
        return new ArrayList<String>(wfNames);
    }

    public List<WorkflowDef> getAllWorkflowDefs() {
        LinkedList<WorkflowDef> workflows = new LinkedList<WorkflowDef>();
        this.recordRedisDaoRequests("getAllWorkflowDefs");
        Set<String> wfNames = this.jedisProxy.smembers(this.nsKey(WORKFLOW_DEF_NAMES));
        int size = 0;
        for (String wfName : wfNames) {
            Map<String, String> workflowDefs = this.jedisProxy.hgetAll(this.nsKey(WORKFLOW_DEF, wfName));
            for (String key : workflowDefs.keySet()) {
                if (key.equals(LATEST)) continue;
                String workflowDef = workflowDefs.get(key);
                workflows.add(this.readValue(workflowDef, WorkflowDef.class));
                size += workflowDef.length();
            }
        }
        this.recordRedisDaoPayloadSize("getAllWorkflowDefs", size, "n/a", "n/a");
        return workflows;
    }

    public List<WorkflowDef> getAllWorkflowDefsLatestVersions() {
        LinkedList<WorkflowDef> workflows = new LinkedList<WorkflowDef>();
        this.recordRedisDaoRequests("getAllWorkflowLatestVersionsDefs");
        Set<String> wfNames = this.jedisProxy.smembers(this.nsKey(WORKFLOW_DEF_NAMES));
        int size = 0;
        for (String wfName : wfNames) {
            WorkflowDef def = this.getLatestWorkflowDef(wfName).orElse(null);
            if (def == null) continue;
            workflows.add(def);
            size += def.toString().length();
        }
        this.recordRedisDaoPayloadSize("getAllWorkflowLatestVersionsDefs", size, "n/a", "n/a");
        return workflows;
    }

    private void _createOrUpdate(WorkflowDef workflowDef) {
        this.jedisProxy.hset(this.nsKey(WORKFLOW_DEF, workflowDef.getName()), String.valueOf(workflowDef.getVersion()), this.toJson(workflowDef));
        this.jedisProxy.sadd(this.nsKey(WORKFLOW_DEF_NAMES), workflowDef.getName());
        this.recordRedisDaoRequests("storeWorkflowDef", "n/a", workflowDef.getName());
    }
}

