/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.cryptofs.ch;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.NonReadableChannelException;
import java.nio.channels.NonWritableChannelException;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.function.Supplier;
import javax.inject.Inject;
import org.cryptomator.cryptofs.CryptoFileSystemStats;
import org.cryptomator.cryptofs.EffectiveOpenOptions;
import org.cryptomator.cryptofs.ch.AbstractFileChannel;
import org.cryptomator.cryptofs.ch.ChannelCloseListener;
import org.cryptomator.cryptofs.ch.ChannelScoped;
import org.cryptomator.cryptofs.ch.CleartextFileLock;
import org.cryptomator.cryptofs.ch.MustWriteHeader;
import org.cryptomator.cryptofs.fh.BufferPool;
import org.cryptomator.cryptofs.fh.ByteSource;
import org.cryptomator.cryptofs.fh.Chunk;
import org.cryptomator.cryptofs.fh.ChunkCache;
import org.cryptomator.cryptofs.fh.ExceptionsDuringWrite;
import org.cryptomator.cryptofs.fh.OpenFileModifiedDate;
import org.cryptomator.cryptofs.fh.OpenFileSize;
import org.cryptomator.cryptolib.api.Cryptor;
import org.cryptomator.cryptolib.api.FileHeader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ChannelScoped
public class CleartextFileChannel
extends AbstractFileChannel {
    private static final Logger LOG = LoggerFactory.getLogger(CleartextFileChannel.class);
    private final FileChannel ciphertextFileChannel;
    private final FileHeader fileHeader;
    private final Cryptor cryptor;
    private final ChunkCache chunkCache;
    private final BufferPool bufferPool;
    private final EffectiveOpenOptions options;
    private final AtomicLong fileSize;
    private final AtomicReference<Instant> lastModified;
    private final Supplier<BasicFileAttributeView> attrViewProvider;
    private final ExceptionsDuringWrite exceptionsDuringWrite;
    private final ChannelCloseListener closeListener;
    private final CryptoFileSystemStats stats;
    private boolean mustWriteHeader;

    @Inject
    public CleartextFileChannel(FileChannel ciphertextFileChannel, FileHeader fileHeader, @MustWriteHeader boolean mustWriteHeader, ReadWriteLock readWriteLock, Cryptor cryptor, ChunkCache chunkCache, BufferPool bufferPool, EffectiveOpenOptions options, @OpenFileSize AtomicLong fileSize, @OpenFileModifiedDate AtomicReference<Instant> lastModified, Supplier<BasicFileAttributeView> attrViewProvider, ExceptionsDuringWrite exceptionsDuringWrite, ChannelCloseListener closeListener, CryptoFileSystemStats stats) {
        super(readWriteLock);
        this.ciphertextFileChannel = ciphertextFileChannel;
        this.fileHeader = fileHeader;
        this.cryptor = cryptor;
        this.chunkCache = chunkCache;
        this.bufferPool = bufferPool;
        this.options = options;
        this.fileSize = fileSize;
        this.lastModified = lastModified;
        this.attrViewProvider = attrViewProvider;
        this.exceptionsDuringWrite = exceptionsDuringWrite;
        this.closeListener = closeListener;
        this.stats = stats;
        if (options.append()) {
            this.position = fileSize.get();
        }
        this.mustWriteHeader = mustWriteHeader;
        if (options.createNew() || options.create()) {
            lastModified.compareAndSet(Instant.EPOCH, Instant.now());
        }
    }

    @Override
    public long size() throws IOException {
        this.assertOpen();
        return this.fileSize.get();
    }

    @Override
    protected boolean isWritable() {
        return this.options.writable();
    }

    @Override
    protected boolean isReadable() {
        return this.options.readable();
    }

    @Override
    protected int readLocked(ByteBuffer dst, long position) throws IOException {
        int origLimit = dst.limit();
        long limitConsideringEof = this.fileSize.get() - position;
        if (limitConsideringEof < 1L) {
            return -1;
        }
        dst.limit((int)Math.min((long)origLimit, limitConsideringEof));
        int read = 0;
        int payloadSize = this.cryptor.fileContentCryptor().cleartextChunkSize();
        while (dst.hasRemaining()) {
            long pos = position + (long)read;
            long chunkIndex = pos / (long)payloadSize;
            int offsetInChunk = (int)(pos % (long)payloadSize);
            ByteBuffer data = this.chunkCache.get(chunkIndex).data().duplicate().position(offsetInChunk);
            int len = Math.min(dst.remaining(), data.remaining());
            dst.put(data.limit(data.position() + len));
            read += len;
        }
        dst.limit(origLimit);
        this.stats.addBytesRead(read);
        return read;
    }

    @Override
    protected int writeLocked(ByteBuffer src, long position) throws IOException {
        long written;
        long oldFileSize = this.fileSize.get();
        if (position > oldFileSize) {
            long gapLen = position - oldFileSize;
            ByteSource byteSource = ByteSource.undefinedNoise(gapLen).followedBy(src);
            written = this.writeLockedInternal(byteSource, oldFileSize) - gapLen;
        } else {
            ByteSource byteSource = ByteSource.from(src);
            written = this.writeLockedInternal(byteSource, position);
        }
        assert (written <= (long)src.capacity());
        return (int)written;
    }

    private long writeLockedInternal(ByteSource src, long position) throws IOException {
        Preconditions.checkArgument((position <= this.fileSize.get() ? 1 : 0) != 0);
        this.writeHeaderIfNeeded();
        int cleartextChunkSize = this.cryptor.fileContentCryptor().cleartextChunkSize();
        long written = 0L;
        while (src.hasRemaining()) {
            long currentPosition = position + written;
            long chunkIndex = currentPosition / (long)cleartextChunkSize;
            assert (chunkIndex >= 0L);
            int offsetInChunk = (int)(currentPosition % (long)cleartextChunkSize);
            assert (offsetInChunk < cleartextChunkSize);
            int len = (int)Math.min(src.remaining(), (long)(cleartextChunkSize - offsetInChunk));
            assert (len <= cleartextChunkSize);
            if (offsetInChunk == 0 && len == cleartextChunkSize) {
                ByteBuffer cleartextChunkData = this.bufferPool.getCleartextBuffer();
                src.copyTo(cleartextChunkData);
                cleartextChunkData.flip();
                Chunk chunk = new Chunk(cleartextChunkData, true);
                this.chunkCache.set(chunkIndex, chunk);
            } else {
                Chunk chunk = this.chunkCache.get(chunkIndex);
                chunk.data().limit(Math.max(chunk.data().limit(), offsetInChunk + len));
                src.copyTo(chunk.data().duplicate().position(offsetInChunk));
                chunk.dirty().set(true);
            }
            written += (long)len;
        }
        long minSize = position + written;
        long newSize = this.fileSize.updateAndGet(size -> Math.max(minSize, size));
        assert (newSize >= minSize);
        this.lastModified.set(Instant.now());
        if (this.options.syncData()) {
            this.forceInternal(this.options.syncDataAndMetadata());
        }
        this.stats.addBytesWritten(written);
        return written;
    }

    private void writeHeaderIfNeeded() throws IOException {
        if (this.mustWriteHeader) {
            LOG.trace("{} - Writing file header.", (Object)this);
            ByteBuffer encryptedHeader = this.cryptor.fileHeaderCryptor().encryptHeader(this.fileHeader);
            this.ciphertextFileChannel.write(encryptedHeader, 0L);
            this.mustWriteHeader = false;
        }
    }

    @Override
    protected void truncateLocked(long newSize) throws IOException {
        Preconditions.checkArgument((newSize >= 0L ? 1 : 0) != 0);
        if (newSize < this.fileSize.get()) {
            int cleartextChunkSize = this.cryptor.fileContentCryptor().cleartextChunkSize();
            long indexOfLastChunk = (newSize + (long)cleartextChunkSize - 1L) / (long)cleartextChunkSize - 1L;
            int sizeOfIncompleteChunk = (int)(newSize % (long)cleartextChunkSize);
            if (sizeOfIncompleteChunk > 0) {
                Chunk chunk = this.chunkCache.get(indexOfLastChunk);
                chunk.data().limit(sizeOfIncompleteChunk);
                chunk.dirty().set(true);
            }
            long ciphertextFileSize = (long)this.cryptor.fileHeaderCryptor().headerSize() + this.cryptor.fileContentCryptor().ciphertextSize(newSize);
            this.chunkCache.invalidateAll();
            this.ciphertextFileChannel.truncate(ciphertextFileSize);
            this.position = Math.min(newSize, this.position);
            this.fileSize.set(newSize);
            this.lastModified.set(Instant.now());
        }
    }

    @Override
    public void force(boolean metaData) throws IOException {
        this.assertOpen();
        this.forceInternal(metaData);
    }

    private void forceInternal(boolean metaData) throws IOException {
        this.flush();
        this.ciphertextFileChannel.force(metaData);
        if (metaData) {
            this.persistLastModified();
        }
    }

    private void flush() throws IOException {
        if (this.isWritable()) {
            this.writeHeaderIfNeeded();
            this.chunkCache.invalidateAll();
            this.exceptionsDuringWrite.throwIfPresent();
        }
    }

    private void persistLastModified() throws IOException {
        FileTime lastModifiedTime = this.isWritable() ? FileTime.from(this.lastModified.get()) : null;
        FileTime lastAccessTime = FileTime.from(Instant.now());
        this.attrViewProvider.get().setTimes(lastModifiedTime, lastAccessTime, null);
    }

    @Override
    public MappedByteBuffer map(FileChannel.MapMode mode, long position, long size) {
        throw new UnsupportedOperationException();
    }

    @Override
    public FileLock lock(long pos, long size, boolean shared) throws IOException {
        FileLock ciphertextLock;
        this.assertOpen();
        if (shared && !this.options.readable()) {
            throw new NonReadableChannelException();
        }
        if (!shared && !this.options.writable()) {
            throw new NonWritableChannelException();
        }
        long beginOfFirstChunk = this.beginOfChunk(pos);
        long beginOfLastChunk = this.beginOfChunk(pos + size);
        if (beginOfFirstChunk == Long.MAX_VALUE || beginOfLastChunk == Long.MAX_VALUE) {
            ciphertextLock = this.ciphertextFileChannel.lock(0L, Long.MAX_VALUE, shared);
        } else {
            long endOfLastChunk = beginOfLastChunk + (long)this.cryptor.fileContentCryptor().ciphertextChunkSize();
            ciphertextLock = this.ciphertextFileChannel.lock(beginOfFirstChunk, endOfLastChunk - beginOfFirstChunk, shared);
        }
        return new CleartextFileLock((FileChannel)this, ciphertextLock, pos, size);
    }

    @Override
    public FileLock tryLock(long pos, long size, boolean shared) throws IOException {
        FileLock ciphertextLock;
        this.assertOpen();
        if (shared && !this.options.readable()) {
            throw new NonReadableChannelException();
        }
        if (!shared && !this.options.writable()) {
            throw new NonWritableChannelException();
        }
        long beginOfFirstChunk = this.beginOfChunk(pos);
        long beginOfLastChunk = this.beginOfChunk(pos + size);
        if (beginOfFirstChunk == Long.MAX_VALUE || beginOfLastChunk == Long.MAX_VALUE) {
            ciphertextLock = this.ciphertextFileChannel.tryLock(0L, Long.MAX_VALUE, shared);
        } else {
            long endOfLastChunk = beginOfLastChunk + (long)this.cryptor.fileContentCryptor().ciphertextChunkSize();
            ciphertextLock = this.ciphertextFileChannel.tryLock(beginOfFirstChunk, endOfLastChunk - beginOfFirstChunk, shared);
        }
        if (ciphertextLock == null) {
            return null;
        }
        return new CleartextFileLock((FileChannel)this, ciphertextLock, pos, size);
    }

    long beginOfChunk(long cleartextPos) {
        long maxCiphertextPayloadSize = Long.MAX_VALUE - (long)this.cryptor.fileHeaderCryptor().headerSize();
        long maxChunks = maxCiphertextPayloadSize / (long)this.cryptor.fileContentCryptor().ciphertextChunkSize();
        long chunk = cleartextPos / (long)this.cryptor.fileContentCryptor().cleartextChunkSize();
        if (chunk > maxChunks) {
            return Long.MAX_VALUE;
        }
        return chunk * (long)this.cryptor.fileContentCryptor().ciphertextChunkSize() + (long)this.cryptor.fileHeaderCryptor().headerSize();
    }

    @Override
    protected void implCloseChannel() throws IOException {
        try {
            this.flush();
            this.persistLastModified();
        }
        finally {
            super.implCloseChannel();
            this.closeListener.closed(this);
        }
    }
}

