/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.cryptofs.dir;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Named;
import org.cryptomator.cryptofs.VaultConfig;
import org.cryptomator.cryptofs.dir.DirectoryStreamScoped;
import org.cryptomator.cryptofs.dir.Node;
import org.cryptomator.cryptolib.api.Cryptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@DirectoryStreamScoped
class C9rConflictResolver {
    private static final Logger LOG = LoggerFactory.getLogger(C9rConflictResolver.class);
    private final Cryptor cryptor;
    private final byte[] dirId;
    private final int maxC9rFileNameLength;
    private final int maxCleartextFileNameLength;

    @Inject
    public C9rConflictResolver(Cryptor cryptor, @Named(value="dirId") String dirId, VaultConfig vaultConfig) {
        this.cryptor = cryptor;
        this.dirId = dirId.getBytes(StandardCharsets.US_ASCII);
        this.maxC9rFileNameLength = vaultConfig.getShorteningThreshold();
        this.maxCleartextFileNameLength = (this.maxC9rFileNameLength - 4) / 4 * 3 - 16;
    }

    public Stream<Node> process(Node node) {
        Preconditions.checkArgument((node.extractedCiphertext != null ? 1 : 0) != 0, (Object)"Can only resolve conflicts if extractedCiphertext is set");
        Preconditions.checkArgument((node.cleartextName != null ? 1 : 0) != 0, (Object)"Can only resolve conflicts if cleartextName is set");
        String canonicalCiphertextFileName = node.extractedCiphertext + ".c9r";
        if (node.fullCiphertextFileName.equals(canonicalCiphertextFileName)) {
            return Stream.of(node);
        }
        if (node.fullCiphertextFileName.startsWith(".")) {
            LOG.debug("Ignoring hidden file {}", (Object)node.ciphertextPath);
            return Stream.empty();
        }
        try {
            Path canonicalPath = node.ciphertextPath.resolveSibling(canonicalCiphertextFileName);
            return this.resolveConflict(node, canonicalPath);
        }
        catch (IOException e) {
            LOG.error("Failed to resolve conflict for " + node.ciphertextPath, (Throwable)e);
            return Stream.empty();
        }
    }

    private Stream<Node> resolveConflict(Node conflicting, Path canonicalPath) throws IOException {
        Path conflictingPath = conflicting.ciphertextPath;
        if (this.resolveConflictTrivially(canonicalPath, conflictingPath)) {
            Node resolved = new Node(canonicalPath);
            resolved.cleartextName = conflicting.cleartextName;
            resolved.extractedCiphertext = conflicting.extractedCiphertext;
            return Stream.of(resolved);
        }
        return Stream.of(this.renameConflictingFile(canonicalPath, conflictingPath, conflicting.cleartextName));
    }

    private Node renameConflictingFile(Path canonicalPath, Path conflictingPath, String cleartext) throws IOException {
        String alternativeCleartext;
        String alternativeCiphertext;
        String alternativeCiphertextName;
        Path alternativePath;
        assert (Files.exists(canonicalPath, new LinkOption[0]));
        int beginOfFileExtension = cleartext.lastIndexOf(46);
        String fileExtension = beginOfFileExtension > 0 ? cleartext.substring(beginOfFileExtension) : "";
        String basename = beginOfFileExtension > 0 ? cleartext.substring(0, beginOfFileExtension) : cleartext;
        String lengthRestrictedBasename = basename.substring(0, Math.min(basename.length(), this.maxCleartextFileNameLength - fileExtension.length() - 5));
        int i = 1;
        do {
            alternativeCleartext = lengthRestrictedBasename + " (" + i++ + ")" + fileExtension;
        } while (Files.exists(alternativePath = canonicalPath.resolveSibling(alternativeCiphertextName = (alternativeCiphertext = this.cryptor.fileNameCryptor().encryptFilename(BaseEncoding.base64Url(), alternativeCleartext, (byte[][])new byte[][]{this.dirId})) + ".c9r"), new LinkOption[0]));
        assert (alternativeCiphertextName.length() <= this.maxC9rFileNameLength);
        LOG.info("Moving conflicting file {} to {}", (Object)conflictingPath, (Object)alternativePath);
        Files.move(conflictingPath, alternativePath, StandardCopyOption.ATOMIC_MOVE);
        Node node = new Node(alternativePath);
        node.cleartextName = alternativeCleartext;
        node.extractedCiphertext = alternativeCiphertext;
        return node;
    }

    private boolean resolveConflictTrivially(Path canonicalPath, Path conflictingPath) throws IOException {
        if (!Files.exists(canonicalPath, new LinkOption[0])) {
            Files.move(conflictingPath, canonicalPath, new CopyOption[0]);
            return true;
        }
        if (this.hasSameFileContent(conflictingPath.resolve("dir.c9r"), canonicalPath.resolve("dir.c9r"), 36)) {
            LOG.info("Removing conflicting directory {} (identical to {})", (Object)conflictingPath, (Object)canonicalPath);
            MoreFiles.deleteRecursively((Path)conflictingPath, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            return true;
        }
        if (this.hasSameFileContent(conflictingPath.resolve("symlink.c9r"), canonicalPath.resolve("symlink.c9r"), Short.MAX_VALUE)) {
            LOG.info("Removing conflicting symlink {} (identical to {})", (Object)conflictingPath, (Object)canonicalPath);
            MoreFiles.deleteRecursively((Path)conflictingPath, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            return true;
        }
        return false;
    }

    private boolean hasSameFileContent(Path conflictingPath, Path canonicalPath, int numBytesToCompare) throws IOException {
        if (!Files.isDirectory(conflictingPath.getParent(), new LinkOption[0]) || !Files.isDirectory(canonicalPath.getParent(), new LinkOption[0])) {
            return false;
        }
        try {
            return -1L == Files.mismatch(conflictingPath, canonicalPath);
        }
        catch (NoSuchFileException e) {
            return false;
        }
    }
}

