/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.cryptofs.health.dirid;

import com.google.common.io.BaseEncoding;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ByteChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import org.cryptomator.cryptofs.CryptoPathMapper;
import org.cryptomator.cryptofs.DirectoryIdBackup;
import org.cryptomator.cryptofs.VaultConfig;
import org.cryptomator.cryptofs.common.CiphertextFileType;
import org.cryptomator.cryptofs.health.api.DiagnosticResult;
import org.cryptomator.cryptolib.api.AuthenticationFailedException;
import org.cryptomator.cryptolib.api.Cryptor;
import org.cryptomator.cryptolib.api.FileNameCryptor;
import org.cryptomator.cryptolib.api.Masterkey;
import org.cryptomator.cryptolib.common.ByteBuffers;
import org.cryptomator.cryptolib.common.DecryptingReadableByteChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OrphanContentDir
implements DiagnosticResult {
    private static final Logger LOG = LoggerFactory.getLogger(OrphanContentDir.class);
    private static final String FILE_PREFIX = "file";
    private static final String DIR_PREFIX = "directory";
    private static final String SYMLINK_PREFIX = "symlink";
    private static final String LONG_NAME_SUFFIX_BASE = "_withVeryLongName";
    final Path contentDir;

    OrphanContentDir(Path contentDir) {
        this.contentDir = contentDir;
    }

    @Override
    public DiagnosticResult.Severity getSeverity() {
        return DiagnosticResult.Severity.WARN;
    }

    @Override
    public String toString() {
        return String.format("Orphan directory: %s", this.contentDir);
    }

    @Override
    public Map<String, String> details() {
        return Map.of("Encrypted Path", this.contentDir.toString());
    }

    @Override
    public Optional<DiagnosticResult.Fix> getFix(Path pathToVault, VaultConfig config, Masterkey masterkey, Cryptor cryptor) {
        return Optional.of(() -> this.fix(pathToVault, config, cryptor));
    }

    private void fix(Path pathToVault, VaultConfig config, Cryptor cryptor) throws IOException {
        MessageDigest sha1 = OrphanContentDir.getSha1MessageDigest();
        String runId = Integer.toString((short)UUID.randomUUID().getMostSignificantBits(), 32);
        Path dataDir = pathToVault.resolve("d");
        Path orphanedDir = dataDir.resolve(this.contentDir);
        String orphanDirIdHash = this.contentDir.getParent().getFileName().toString() + this.contentDir.getFileName().toString();
        Path recoveryDir = this.prepareRecoveryDir(pathToVault, cryptor.fileNameCryptor());
        if (recoveryDir.toAbsolutePath().equals(orphanedDir.toAbsolutePath())) {
            return;
        }
        CryptoPathMapper.CiphertextDirectory stepParentDir = this.prepareStepParent(dataDir, recoveryDir, cryptor, orphanDirIdHash);
        AtomicInteger fileCounter = new AtomicInteger(1);
        AtomicInteger dirCounter = new AtomicInteger(1);
        AtomicInteger symlinkCounter = new AtomicInteger(1);
        String longNameSuffix = OrphanContentDir.createClearnameToBeShortened(config.getShorteningThreshold());
        Optional<String> dirId = this.retrieveDirId(orphanedDir, cryptor);
        try (DirectoryStream<Path> orphanedContentStream = Files.newDirectoryStream(orphanedDir, this::matchesEncryptedContentPattern);){
            for (Path orphanedResource : orphanedContentStream) {
                boolean isShortened = orphanedResource.toString().endsWith(".c9s");
                String newClearName = dirId.map(id -> {
                    try {
                        return this.decryptFileName(orphanedResource, isShortened, (String)id, cryptor.fileNameCryptor());
                    }
                    catch (IOException | AuthenticationFailedException e) {
                        LOG.warn("Unable to read and decrypt file name of {}:", (Object)orphanedResource, (Object)e);
                        return null;
                    }
                }).orElseGet(() -> (switch (OrphanContentDir.determineCiphertextFileType(orphanedResource)) {
                    default -> throw new IncompatibleClassChangeError();
                    case CiphertextFileType.FILE -> FILE_PREFIX + fileCounter.getAndIncrement();
                    case CiphertextFileType.DIRECTORY -> DIR_PREFIX + dirCounter.getAndIncrement();
                    case CiphertextFileType.SYMLINK -> SYMLINK_PREFIX + symlinkCounter.getAndIncrement();
                }) + "_" + runId + (isShortened ? longNameSuffix : ""));
                this.adoptOrphanedResource(orphanedResource, newClearName, isShortened, stepParentDir, cryptor.fileNameCryptor(), sha1);
            }
        }
        Files.deleteIfExists(orphanedDir.resolve("dirid.c9r"));
        try (DirectoryStream<Path> nonCryptomatorFiles = Files.newDirectoryStream(orphanedDir);){
            for (Path p : nonCryptomatorFiles) {
                Files.move(p, stepParentDir.path.resolve(p.getFileName()), LinkOption.NOFOLLOW_LINKS);
            }
        }
        Files.delete(orphanedDir);
    }

    private boolean matchesEncryptedContentPattern(Path path) {
        String tmp = path.getFileName().toString();
        return tmp.length() >= 26 && (tmp.endsWith(".c9r") || tmp.endsWith(".c9s"));
    }

    Path prepareRecoveryDir(Path pathToVault, FileNameCryptor cryptor) throws IOException {
        Path dataDir = pathToVault.resolve("d");
        String rootDirHash = cryptor.hashDirectoryId("");
        Path vaultCipherRootPath = dataDir.resolve(rootDirHash.substring(0, 2)).resolve(rootDirHash.substring(2)).toAbsolutePath();
        String cipherRecoveryDirName = OrphanContentDir.encrypt(cryptor, "LOST+FOUND", "");
        Path cipherRecoveryDirFile = vaultCipherRootPath.resolve(cipherRecoveryDirName + "/dir.c9r");
        if (Files.notExists(cipherRecoveryDirFile, LinkOption.NOFOLLOW_LINKS)) {
            Files.createDirectories(cipherRecoveryDirFile.getParent(), new FileAttribute[0]);
            Files.writeString(cipherRecoveryDirFile, (CharSequence)"recovery", StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
        } else {
            String uuid = Files.readString(cipherRecoveryDirFile, StandardCharsets.UTF_8);
            if (!"recovery".equals(uuid)) {
                throw new FileAlreadyExistsException("Directory /LOST+FOUND already exists, but with wrong directory id.");
            }
        }
        String recoveryDirHash = cryptor.hashDirectoryId("recovery");
        Path cipherRecoveryDir = dataDir.resolve(recoveryDirHash.substring(0, 2)).resolve(recoveryDirHash.substring(2)).toAbsolutePath();
        Files.createDirectories(cipherRecoveryDir, new FileAttribute[0]);
        return cipherRecoveryDir;
    }

    CryptoPathMapper.CiphertextDirectory prepareStepParent(Path dataDir, Path cipherRecoveryDir, Cryptor cryptor, String clearStepParentDirName) throws IOException {
        String stepParentUUID;
        String cipherStepParentDirName = OrphanContentDir.encrypt(cryptor.fileNameCryptor(), clearStepParentDirName, "recovery");
        Path cipherStepParentDirFile = cipherRecoveryDir.resolve(cipherStepParentDirName + "/dir.c9r");
        if (Files.exists(cipherStepParentDirFile, LinkOption.NOFOLLOW_LINKS)) {
            stepParentUUID = Files.readString(cipherStepParentDirFile, StandardCharsets.UTF_8);
        } else {
            Files.createDirectories(cipherStepParentDirFile.getParent(), new FileAttribute[0]);
            stepParentUUID = UUID.randomUUID().toString();
            Files.writeString(cipherStepParentDirFile, (CharSequence)stepParentUUID, StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
        }
        String stepParentDirHash = cryptor.fileNameCryptor().hashDirectoryId(stepParentUUID);
        Path stepParentDir = dataDir.resolve(stepParentDirHash.substring(0, 2)).resolve(stepParentDirHash.substring(2)).toAbsolutePath();
        Files.createDirectories(stepParentDir, new FileAttribute[0]);
        CryptoPathMapper.CiphertextDirectory stepParentCipherDir = new CryptoPathMapper.CiphertextDirectory(stepParentUUID, stepParentDir);
        try {
            DirectoryIdBackup.backupManually(cryptor, stepParentCipherDir);
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
            // empty catch block
        }
        return stepParentCipherDir;
    }

    Optional<String> retrieveDirId(Path orphanedDir, Cryptor cryptor) {
        Path dirIdFile = orphanedDir.resolve("dirid.c9r");
        ByteBuffer dirIdBuffer = ByteBuffer.allocate(36);
        try (SeekableByteChannel channel = Files.newByteChannel(dirIdFile, StandardOpenOption.READ);
             DecryptingReadableByteChannel decryptingChannel = this.createDecryptingReadableByteChannel(channel, cryptor);){
            ByteBuffers.fill((ReadableByteChannel)decryptingChannel, (ByteBuffer)dirIdBuffer);
            dirIdBuffer.flip();
        }
        catch (IOException e) {
            LOG.info("Unable to read {}.", (Object)dirIdFile, (Object)e);
            return Optional.empty();
        }
        return Optional.of(StandardCharsets.US_ASCII.decode(dirIdBuffer).toString());
    }

    DecryptingReadableByteChannel createDecryptingReadableByteChannel(ByteChannel channel, Cryptor cryptor) {
        return new DecryptingReadableByteChannel((ReadableByteChannel)channel, cryptor, true);
    }

    String decryptFileName(Path orphanedResource, boolean isShortened, String dirId, FileNameCryptor cryptor) throws IOException, AuthenticationFailedException {
        String filenameWithExtension = isShortened ? Files.readString(orphanedResource.resolve("name.c9s")) : orphanedResource.getFileName().toString();
        String filename = filenameWithExtension.substring(0, filenameWithExtension.length() - ".c9r".length());
        return cryptor.decryptFilename(BaseEncoding.base64Url(), filename, (byte[][])new byte[][]{dirId.getBytes(StandardCharsets.UTF_8)});
    }

    void adoptOrphanedResource(Path oldCipherPath, String newClearName, boolean isShortened, CryptoPathMapper.CiphertextDirectory stepParentDir, FileNameCryptor cryptor, MessageDigest sha1) throws IOException {
        String newCipherName = OrphanContentDir.encrypt(cryptor, newClearName, stepParentDir.dirId);
        if (isShortened) {
            String deflatedName = BaseEncoding.base64Url().encode(sha1.digest(newCipherName.getBytes(StandardCharsets.UTF_8))) + ".c9s";
            Path targetPath = stepParentDir.path.resolve(deflatedName);
            Files.move(oldCipherPath, targetPath, new CopyOption[0]);
            try (SeekableByteChannel fc = Files.newByteChannel(targetPath.resolve("name.c9s"), StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
                fc.write(ByteBuffer.wrap(newCipherName.getBytes(StandardCharsets.UTF_8)));
            }
        } else {
            Path targetPath = stepParentDir.path.resolve(newCipherName);
            Files.move(oldCipherPath, targetPath, new CopyOption[0]);
        }
    }

    private static String createClearnameToBeShortened(int threshold) {
        int neededLength = (threshold - 4) / 4 * 3 - 16;
        return LONG_NAME_SUFFIX_BASE.repeat(neededLength % LONG_NAME_SUFFIX_BASE.length() + 1);
    }

    private static String encrypt(FileNameCryptor cryptor, String clearTextName, String dirId) {
        return cryptor.encryptFilename(BaseEncoding.base64Url(), clearTextName, (byte[][])new byte[][]{dirId.getBytes(StandardCharsets.UTF_8)}) + ".c9r";
    }

    private static CiphertextFileType determineCiphertextFileType(Path ciphertextPath) {
        if (Files.exists(ciphertextPath.resolve("dir.c9r"), LinkOption.NOFOLLOW_LINKS)) {
            return CiphertextFileType.DIRECTORY;
        }
        if (Files.exists(ciphertextPath.resolve("symlink.c9r"), LinkOption.NOFOLLOW_LINKS)) {
            return CiphertextFileType.SYMLINK;
        }
        return CiphertextFileType.FILE;
    }

    private static MessageDigest getSha1MessageDigest() {
        try {
            return MessageDigest.getInstance("SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Every JVM needs to provide a SHA1 implementation.");
        }
    }
}

