/*
 * Decompiled with CFR 0.152.
 */
package org.cuberact.storage;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.spi.FileSystemProvider;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.cuberact.storage.GlobMatcher;
import org.cuberact.storage.Resource;
import org.cuberact.storage.StorageException;
import org.cuberact.storage.StorageFileSystem;
import org.cuberact.storage.StorageType;
import org.cuberact.storage.deferred.DeferredExecutor;
import org.cuberact.storage.deferred.DeferredTask;

public class Storage {
    private final Path path;
    private final URI uri;
    private final StorageType type;
    private final Charset charset;
    private final Predicate<DeferredTask> matcher = task -> task instanceof Resource.WriteTask && ((Resource.WriteTask)task).resource.getStorage().equals(this);
    private static final FileSystemProvider ZIP_FILE_SYSTEM_PROVIDER = FileSystemProvider.installedProviders().stream().filter(fsp -> "jar".equalsIgnoreCase(fsp.getScheme())).findFirst().orElseThrow(() -> new StorageException("Can't find ZIP FILE SYSTEM PROVIDER"));
    private static final Map<String, String> ZIP_ENV = Stream.of(new AbstractMap.SimpleEntry<String, String>("create", "true")).collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));
    private static final byte[] ZIP_HEADER = new byte[]{80, 75, 3, 4};
    private static final byte[] ZIP_HEADER_EMPTY = new byte[]{80, 75, 5, 6};

    public Storage(String path) {
        this(path, null);
    }

    public Storage(String path, StorageType type) {
        this(path, type, StandardCharsets.UTF_8);
    }

    public Storage(String path, StorageType type, Charset charset) {
        Path storagePath = Paths.get(Storage.normalizePath(path), new String[0]).toAbsolutePath();
        if (type == null) {
            type = Storage.resolveType(storagePath);
        }
        if (Files.exists(storagePath, new LinkOption[0])) {
            switch (type) {
                case DIRECTORY: {
                    if (Files.isDirectory(storagePath, new LinkOption[0])) break;
                    throw new StorageException("Storage '" + storagePath + "' is not directory");
                }
                case ZIP: {
                    if (Storage.isZipFile(storagePath)) break;
                    throw new StorageException("Storage '" + storagePath + "' is not zip file");
                }
            }
        }
        this.path = Objects.requireNonNull(storagePath, "Path");
        this.uri = Storage.createURI("file:/" + Storage.normalizePath(this.path.toString()));
        this.type = Objects.requireNonNull(type, "Type");
        this.charset = Objects.requireNonNull(charset);
    }

    public Path getPath() {
        return this.path;
    }

    public Charset getCharset() {
        return this.charset;
    }

    public StorageType getType() {
        return this.type;
    }

    public boolean exists() {
        DeferredExecutor.runImmediately(this.matcher);
        return Files.exists(this.path, new LinkOption[0]);
    }

    public void delete() {
        DeferredExecutor.runImmediately(this.matcher);
        if (this.exists()) {
            Storage.delete(this.path);
        }
    }

    public Resource getResource(String path) {
        return new Resource(this, Storage.normalizePath(path));
    }

    public List<Resource> findResources(String glob) {
        DeferredExecutor.runImmediately(this.matcher);
        if (this.exists()) {
            return this.runInStorage(fs -> Storage.collectChildren(this, fs.getPath("/", new String[0]), new GlobMatcher(glob)));
        }
        return Collections.emptyList();
    }

    public Resource findResource(String glob) {
        List<Resource> resources = this.findResources(glob);
        if (resources.size() > 1) {
            StringBuilder sb = new StringBuilder();
            for (Resource resource : resources) {
                sb.append("  ").append(resource.toString()).append("\n");
            }
            throw new StorageException("Found more then one resources in Storage '" + this.path.toString() + "' with glob '" + glob + "'\n\n" + sb.toString());
        }
        return resources.isEmpty() ? null : resources.get(0);
    }

    public String toString() {
        return "Storage '" + Storage.normalizePath(this.path.toString()) + "' [exists: " + this.exists() + ", type: " + (Object)((Object)this.getType()) + "]";
    }

    public int hashCode() {
        return this.uri.hashCode() * 31;
    }

    public boolean equals(Object obj) {
        if (obj instanceof Storage) {
            Storage other = (Storage)obj;
            return Objects.equals(this.uri, other.uri);
        }
        return false;
    }

    static URI createURI(String line) {
        return URI.create(line.replaceAll(" ", "%20"));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    <E> E runInStorage(StorageRunner<E> storageRunner) {
        try {
            if (!Files.exists(this.path, new LinkOption[0])) {
                try {
                    switch (this.type) {
                        case DIRECTORY: {
                            Files.createDirectories(this.path, new FileAttribute[0]);
                            break;
                        }
                        case ZIP: {
                            Files.createDirectories(this.path.getParent(), new FileAttribute[0]);
                            break;
                        }
                    }
                }
                catch (Throwable t) {
                    throw new StorageException(t);
                }
            }
            if (this.type != StorageType.ZIP) return storageRunner.run(new StorageFileSystem(this.path));
            try (FileSystem zipFileSystem = ZIP_FILE_SYSTEM_PROVIDER.newFileSystem(this.path, ZIP_ENV);){
                E e = storageRunner.run(zipFileSystem);
                return e;
            }
            catch (Throwable e) {
                throw new StorageException(e);
            }
        }
        catch (Throwable e) {
            throw new StorageException(e);
        }
    }

    private static StorageType resolveType(Path path) {
        if (Files.exists(path, new LinkOption[0])) {
            if (Files.isDirectory(path, new LinkOption[0])) {
                return StorageType.DIRECTORY;
            }
            if (Storage.isZipFile(path)) {
                return StorageType.ZIP;
            }
            throw new RuntimeException("Can't init Storage '" + path + "'. Only directory or zip file are allowed.");
        }
        String strPath = path.toString();
        return strPath.endsWith(".zip") || strPath.endsWith(".jar") ? StorageType.ZIP : StorageType.DIRECTORY;
    }

    static String normalizePath(String path) {
        String normalized = path.replace('\\', '/');
        if (normalized.endsWith("/")) {
            normalized = normalized.substring(0, normalized.length() - 1);
        }
        return normalized;
    }

    private static List<Resource> collectChildren(final Storage storage, final Path path, final GlobMatcher globMatcher) {
        try {
            final ArrayList<Resource> children = new ArrayList<Resource>();
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path child, BasicFileAttributes basicFileAttributes) {
                    String subPath = Storage.normalizePath(path.relativize(child).toString());
                    if (!"".equals(subPath) && globMatcher.matches(subPath)) {
                        children.add(new Resource(storage, subPath));
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
            return children;
        }
        catch (Throwable t) {
            throw new StorageException(t);
        }
    }

    static void delete(Path path) {
        try {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes basicFileAttributes) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (Throwable e) {
            throw new StorageException(e);
        }
    }

    static void closeQuietly(Closeable closeable) {
        try {
            closeable.close();
        }
        catch (IOException e) {
            throw new StorageException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean isZipFile(Path path) {
        try (InputStream is = Files.newInputStream(path, new OpenOption[0]);){
            byte[] header = new byte[4];
            int count = is.read(header);
            boolean bl = count == 4 && (Arrays.equals(header, ZIP_HEADER) || Arrays.equals(header, ZIP_HEADER_EMPTY));
            return bl;
        }
        catch (IOException e) {
            return false;
        }
    }

    static interface StorageRunner<E> {
        public E run(FileSystem var1) throws Throwable;
    }
}

