/*
 * Copyright (C) 2011-2015 Volker Bergmann (volker.bergmann@bergmann-it.de).
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.databene.formats.text;

import java.text.MessageFormat;

import org.databene.commons.ConversionException;
import org.databene.commons.Converter;
import org.databene.commons.StringUtil;
import org.databene.commons.converter.ThreadSafeConverter;

/**
 * {@link Converter} that transforms an object into its hexadecimal representation.
 * It works with integral numbers, characters and strings.
 * Created: 29.10.2009 08:44:53
 * @since 0.5.0
 * @author Volker Bergmann
 */
public class ToHexConverter extends ThreadSafeConverter<Object, String> {
	
	private boolean upperCase;
	private String pattern;
	private int length;

	public ToHexConverter() {
	    this(false);
    }

	public ToHexConverter(boolean upperCase) {
	    this(upperCase, null);
    }

	public ToHexConverter(boolean upperCase, String pattern) {
	    this(upperCase, pattern, -1);
    }

	public ToHexConverter(boolean upperCase, String pattern, int length) {
	    super(Object.class, String.class);
	    this.upperCase = upperCase;
	    this.pattern = pattern;
	    this.length = length;
    }

	@Override
	public String convert(Object sourceValue) throws ConversionException {
		if (sourceValue == null)
			return null;
		Class<?> sourceType = sourceValue.getClass();
		if (sourceType == Long.class)
			return convertLong((Long) sourceValue, upperCase, pattern, length);
		else if (sourceType == Integer.class)
			return convertInt((Integer) sourceValue, upperCase, pattern, length);
		else if (sourceType == Short.class)
			return convertShort((Short) sourceValue, upperCase, pattern, length);
		else if (sourceType == Byte.class)
			return convertByte((Byte) sourceValue, upperCase, pattern, length);
		else if (sourceType == Character.class)
			return convertChar((Character) sourceValue, upperCase, pattern, length);
		else if (sourceType == String.class)
			return convertString((String) sourceValue, upperCase, pattern, length);
		else
			throw new IllegalArgumentException("Can't render '" + sourceType + "' in hex format.");
    }

	public static String convertLong(Long sourceValue, boolean upperCase, String pattern, int length) {
	    String base = Long.toHexString(sourceValue);
	    return postProcess(base, upperCase, pattern, length);
    }

	public static String convertInt(int sourceValue, boolean upperCase, String pattern, int length) {
	    String base = Integer.toHexString(sourceValue);
	    return postProcess(base, upperCase, pattern, length);
    }

	public static String convertShort(short sourceValue, boolean upperCase, String pattern, int length) {
	    String base = Integer.toHexString(sourceValue);
	    if (base.length() == 8)
	    	base = base.substring(4);
	    return postProcess(base, upperCase, pattern, length);
    }

	public static String convertByte(byte sourceValue, boolean upperCase, String pattern, int length) {
	    String base = Integer.toHexString(sourceValue);
	    if (base.length() == 8)
	    	base = base.substring(6);
	    return postProcess(base, upperCase, pattern, length);
    }

	public static String convertChar(Character sourceValue, boolean upperCase, String pattern, int length) {
		String base = convertInt(sourceValue, upperCase, null, 2);
		return postProcess(base, upperCase, pattern, length);
    }

	public static String convertString(String sourceValue, boolean upperCase, String pattern, int length) {
		StringBuilder builder = new StringBuilder(sourceValue.length() * 2);
		for (int i = 0; i < sourceValue.length() ; i++)
			builder.append(convertChar(sourceValue.charAt(i), upperCase, null, 2));
		return postProcess(builder.toString(), upperCase, pattern, length);
    }

	private static String postProcess(String base, boolean upperCase, String pattern, int length) {
	    if (upperCase)
	    	base = base.toUpperCase();
	    if (length > 0)
	    	base = StringUtil.padLeft(base, length, '0');
	    if (pattern != null)
	    	base = MessageFormat.format(pattern, base);
	    return base;
    }

}
