/**********************************************************************
Copyright (c) 2010 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.api.jdo.query;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.Date;
import java.sql.Time;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.jdo.FetchPlan;
import javax.jdo.JDOException;
import javax.jdo.JDOFatalUserException;
import javax.jdo.JDOQLTypedQuery;
import javax.jdo.JDOQLTypedSubquery;
import javax.jdo.JDOUnsupportedOptionException;
import javax.jdo.JDOUserException;
import javax.jdo.PersistenceManager;
import javax.jdo.query.BooleanExpression;
import javax.jdo.query.CharacterExpression;
import javax.jdo.query.CollectionExpression;
import javax.jdo.query.DateExpression;
import javax.jdo.query.DateTimeExpression;
import javax.jdo.query.Expression;
import javax.jdo.query.IfThenElseExpression;
import javax.jdo.query.ListExpression;
import javax.jdo.query.MapExpression;
import javax.jdo.query.NumericExpression;
import javax.jdo.query.OrderExpression;
import javax.jdo.query.PersistableExpression;
import javax.jdo.query.StringExpression;
import javax.jdo.query.TimeExpression;
import javax.jdo.spi.JDOPermission;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.api.jdo.JDOFetchPlan;
import org.datanucleus.api.jdo.JDOPersistenceManagerFactory;
import org.datanucleus.api.jdo.JDOQuery;
import org.datanucleus.api.jdo.JDOAdapter;
import org.datanucleus.api.jdo.query.geospatial.GeospatialHelperImpl;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.flush.FlushMode;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.QueryLanguage;
import org.datanucleus.metadata.QueryMetaData;
import org.datanucleus.store.query.NoQueryResultsException;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.query.Query.QueryType;
import org.datanucleus.store.query.compiler.QueryCompilation;
import org.datanucleus.store.query.expression.Literal;
import org.datanucleus.store.query.expression.ParameterExpression;
import org.datanucleus.store.query.expression.PrimaryExpression;
import org.datanucleus.store.query.expression.VariableExpression;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.StringUtils;

/**
 * Implementation of a JDOQLTypedQuery.
 * Note that a JDOQLTypedQuery only supports named parameters.
 */
public class JDOQLTypedQueryImpl<T> extends AbstractJDOQLTypedQuery<T> implements JDOQLTypedQuery<T>
{
    private static final long serialVersionUID = -8359479260893321900L;

    private boolean closed = false;

    protected JDOFetchPlan fetchPlan;
    protected boolean ignoreCache = false;
    protected Boolean serializeRead = null;
    protected Integer datastoreReadTimeout = null;
    protected Integer datastoreWriteTimeout = null;
    protected Map<String, Object> extensions = null;

    protected Collection<T> candidates = null;

    boolean unmodifiable = false;

    /** Map of parameter expression keyed by the name. */
    protected Map<String, ExpressionImpl> parameterExprByName = null;

    /** Map of parameters keyed by their name/expression. */
    protected Map<String, Object> parameterValuesByName = null;

    /** Set of any subqueries used by this query. */
    protected transient Set<JDOQLTypedSubqueryImpl> subqueries = null;

    /** Internal queries generated by this typesafe query. Managed so that they can be closed. TODO Use just one? */
    protected transient Set<Query> internalQueries = null;

    protected transient Object geospatialHelper = null;

    /**
     * Constructor for a typesafe query.
     * @param pm Persistence Manager
     * @param candidateClass The candidate class
     */
    public JDOQLTypedQueryImpl(PersistenceManager pm, Class<T> candidateClass)
    {
        super(pm, candidateClass, "this", null);
    }

    /* (non-Javadoc)
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() throws IOException
    {
        if (closed)
        {
            return;
        }

        closeAll();

        Boolean closeableQuery = ec.getBooleanProperty(JDOQuery.PROPERTY_CLOSEABLE_QUERY);
        if (closeableQuery == Boolean.TRUE)
        {
            // User has requested a closeable Query, so release connection to PM and underlying query etc
            if (this.fetchPlan != null)
            {
                this.fetchPlan.clearGroups();
                this.fetchPlan = null;
            }
            this.parameterExprByName = null;
            this.parameterValuesByName = null;
            this.ec = null;
            this.pm = null;
            this.internalQueries = null;
            this.subqueries = null;

            this.closed = true;
        }
    }

    /**
     * Accessor for whether this Query is closed.
     * @return Whether this Query is closed.
     */
    public boolean isClosed()
    {
        return closed;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#close(java.lang.Object)
     */
    public void close(Object result)
    {
        assertIsOpen();
        if (internalQueries != null)
        {
            Iterator<Query> iter = internalQueries.iterator();
            while (iter.hasNext())
            {
                Query query = iter.next();
                query.close(result);
            }
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#closeAll()
     */
    public void closeAll()
    {
        assertIsOpen();
        if (internalQueries != null)
        {
            Iterator<Query> iter = internalQueries.iterator();
            while (iter.hasNext())
            {
                Query query = iter.next();
                query.closeAll();
            }
            internalQueries.clear();
            internalQueries = null;
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#candidate()
     */
    public PersistableExpression candidate()
    {
        assertIsOpen();
        String candName = candidateCls.getName();
        int pos = candName.lastIndexOf('.');
        String qName = candName.substring(0, pos+1) + getQueryClassNameForClassName(candName.substring(pos+1));
        try
        {
            // Use the candidate() static method for access
            Class qClass = ec.getClassLoaderResolver().classForName(qName);
            Method method = qClass.getMethod("candidate", new Class[] {});
            Object candObj = method.invoke(null, (Object[])null);
            if (candObj == null || !(candObj instanceof PersistableExpression))
            {
                throw new JDOException("Class " + candidateCls.getName() + " has a Query class but the candidate is invalid");
            }
            return (PersistableExpression)candObj;
        }
        catch (NoSuchMethodException nsfe)
        {
            throw new JDOException("Class " + candidateCls.getName() + " has a Query class but the candidate is invalid");
        }
        catch (InvocationTargetException ite)
        {
            throw new JDOException("Class " + candidateCls.getName() + " has a Query class but the candidate is invalid");
        }
        catch (IllegalAccessException iae)
        {
            throw new JDOException("Class " + candidateCls.getName() + " has a Query class but the candidate is invalid");
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#parameter(java.lang.String, java.lang.Class)
     */
    public <P> Expression<P> parameter(String name, Class<P> type)
    {
        assertIsOpen();
        discardCompiled();

        ExpressionImpl paramExpr = null;
        if (type == Boolean.class || type == boolean.class)
        {
            paramExpr = new BooleanExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Byte.class || type == byte.class)
        {
            paramExpr = new ByteExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Character.class || type == char.class)
        {
            paramExpr = new CharacterExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Double.class || type == double.class)
        {
            paramExpr = new NumericExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Float.class || type == float.class)
        {
            paramExpr = new NumericExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Integer.class || type == int.class)
        {
            paramExpr = new NumericExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Long.class || type == long.class)
        {
            paramExpr = new NumericExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == Short.class || type == short.class)
        {
            paramExpr = new NumericExpressionImpl(type, name, ExpressionType.PARAMETER);
        }
        else if (type == String.class)
        {
            paramExpr = new StringExpressionImpl((Class<String>) type, name, ExpressionType.PARAMETER);
        }
        else if (Time.class.isAssignableFrom(type))
        {
            paramExpr = new TimeExpressionImpl((Class<Time>) type, name, ExpressionType.PARAMETER);
        }
        else if (Date.class.isAssignableFrom(type))
        {
            paramExpr = new DateExpressionImpl((Class<Date>) type, name, ExpressionType.PARAMETER);
        }
        else if (java.util.Date.class.isAssignableFrom(type))
        {
            paramExpr = new DateTimeExpressionImpl((Class<java.util.Date>) type, name, ExpressionType.PARAMETER);
        }
        else if (ec.getApiAdapter().isPersistable(type))
        {
            // Persistable class
            String typeName = type.getName();
            int pos = typeName.lastIndexOf('.');
            String qName = typeName.substring(0, pos+1) + getQueryClassNameForClassName(typeName.substring(pos+1));
            try
            {
                Class qClass = ec.getClassLoaderResolver().classForName(qName);
                Constructor ctr = qClass.getConstructor(new Class[] {Class.class, String.class, ExpressionType.class});
                Object candObj = ctr.newInstance(new Object[] {type, name, ExpressionType.PARAMETER});
                paramExpr = (ExpressionImpl)candObj;
            }
            catch (NoSuchMethodException nsme)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for parameters");
            }
            catch (IllegalAccessException iae)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for parameters");
            }
            catch (InvocationTargetException ite)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for parameters");
            }
            catch (InstantiationException ie)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for parameters");
            }
        }
        else
        {
            paramExpr = new ObjectExpressionImpl(type, name, ExpressionType.PARAMETER);
        }

        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);

        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#stringParameter(java.lang.String)
     */
    public StringExpression stringParameter(String name)
    {
        assertIsOpen();
        StringExpressionImpl paramExpr = new StringExpressionImpl(String.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#characterParameter(java.lang.String)
     */
    public CharacterExpression characterParameter(String name)
    {
        assertIsOpen();
        CharacterExpressionImpl paramExpr = new CharacterExpressionImpl(Character.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#numericParameter(java.lang.String)
     */
    @Override
    public NumericExpression<? extends Number> numericParameter(String name)
    {
        assertIsOpen();
        NumericExpressionImpl<Float> paramExpr = new NumericExpressionImpl(Number.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#numericParameter(java.lang.String, java.lang.Class)
     */
    @Override
    public <N extends Number> NumericExpression<N> numericParameter(String name, Class<N> type)
    {
        assertIsOpen();
        NumericExpressionImpl paramExpr = new NumericExpressionImpl(type, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#dateParameter(java.lang.String)
     */
    public DateExpression dateParameter(String name)
    {
        assertIsOpen();
        DateExpressionImpl paramExpr = new DateExpressionImpl(Date.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#timeParameter(java.lang.String)
     */
    public TimeExpression timeParameter(String name)
    {
        assertIsOpen();
        TimeExpressionImpl paramExpr = new TimeExpressionImpl(Time.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#datetimeParameter(java.lang.String)
     */
    public DateTimeExpression datetimeParameter(String name)
    {
        assertIsOpen();
        DateTimeExpressionImpl paramExpr = new DateTimeExpressionImpl(java.util.Date.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#collectionParameter(java.lang.String)
     */
    public CollectionExpression collectionParameter(String name)
    {
        assertIsOpen();
        CollectionExpressionImpl paramExpr = new CollectionExpressionImpl(java.util.Collection.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#collectionParameter(java.lang.String, java.lang.Class)
     */
    @Override
    public <E> CollectionExpression<Collection<E>, E> collectionParameter(String name, Class<E> elementType)
    {
        assertIsOpen();
        CollectionExpressionImpl paramExpr = new CollectionExpressionImpl(java.util.Collection.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#mapParameter(java.lang.String)
     */
    public MapExpression mapParameter(String name)
    {
        assertIsOpen();
        MapExpressionImpl paramExpr = new MapExpressionImpl(java.util.Map.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#mapParameter(java.lang.String, java.lang.Class, java.lang.Class)
     */
    @Override
    public <K, V> MapExpression<Map<K, V>, K, V> mapParameter(String name, Class<K> keyType, Class<V> valueType)
    {
        assertIsOpen();
        MapExpressionImpl paramExpr = new MapExpressionImpl(java.util.Map.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#listParameter(java.lang.String)
     */
    public ListExpression listParameter(String name)
    {
        assertIsOpen();
        ListExpressionImpl paramExpr = new ListExpressionImpl(java.util.List.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#listParameter(java.lang.String, java.lang.Class)
     */
    @Override
    public <E> ListExpression<List<E>, E> listParameter(String name, Class<E> elementType)
    {
        assertIsOpen();
        ListExpressionImpl paramExpr = new ListExpressionImpl(java.util.List.class, name, ExpressionType.PARAMETER);
        if (parameterExprByName == null)
        {
            parameterExprByName = new ConcurrentHashMap<>();
        }
        parameterExprByName.put(name, paramExpr);
        return paramExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#variable(java.lang.String, java.lang.Class)
     */
    public <V> Expression<V> variable(String name, Class<V> type)
    {
        assertIsOpen();
        discardCompiled();

        Expression varExpr = null;
        if (ec.getApiAdapter().isPersistable(type))
        {
            // Persistable class
            String typeName = type.getName();
            int pos = typeName.lastIndexOf('.');
            String qName = typeName.substring(0, pos+1) + getQueryClassNameForClassName(typeName.substring(pos+1));
            try
            {
                Class qClass = ec.getClassLoaderResolver().classForName(qName);
                Constructor ctr = qClass.getConstructor(new Class[] {Class.class, String.class, ExpressionType.class});
                Object candObj = ctr.newInstance(new Object[] {type, name, ExpressionType.VARIABLE});
                varExpr = (Expression)candObj;
            }
            catch (NoSuchMethodException nsme)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for variables");
            }
            catch (IllegalAccessException iae)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for variables");
            }
            catch (InvocationTargetException ite)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for variables");
            }
            catch (InstantiationException ie)
            {
                throw new JDOException("Class " + typeName + " has a Query class but has no constructor for variables");
            }
        }
        else if (type == Boolean.class || type == boolean.class)
        {
            varExpr = new BooleanExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Byte.class || type == byte.class)
        {
            varExpr = new ByteExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Character.class || type == char.class)
        {
            varExpr = new CharacterExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Double.class || type == double.class)
        {
            varExpr = new NumericExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Float.class || type == float.class)
        {
            varExpr = new NumericExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Integer.class || type == int.class)
        {
            varExpr = new NumericExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Long.class || type == long.class)
        {
            varExpr = new NumericExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == Short.class || type == short.class)
        {
            varExpr = new NumericExpressionImpl(type, name, ExpressionType.VARIABLE);
        }
        else if (type == String.class)
        {
            varExpr = new StringExpressionImpl((Class<String>) type, name, ExpressionType.VARIABLE);
        }
        else if (Time.class.isAssignableFrom(type))
        {
            varExpr = new TimeExpressionImpl((Class<Time>) type, name, ExpressionType.VARIABLE);
        }
        else if (Date.class.isAssignableFrom(type))
        {
            varExpr = new DateExpressionImpl((Class<Date>) type, name, ExpressionType.VARIABLE);
        }
        else if (java.util.Date.class.isAssignableFrom(type))
        {
            varExpr = new DateTimeExpressionImpl((Class<java.util.Date>) type, name, ExpressionType.VARIABLE);
        }
        else
        {
            varExpr = new ObjectExpressionImpl(type, name, ExpressionType.VARIABLE);
        }

        return varExpr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ifThenElse(java.lang.Class, javax.jdo.query.BooleanExpression, javax.jdo.query.Expression, javax.jdo.query.Expression)
     */
    @Override
    public <V> IfThenElseExpression<V> ifThenElse(Class<V> type, BooleanExpression ifExpr, Expression<V> thenValueExpr, Expression<V> elseValueExpr)
    {
        IfThenElseExpression expr = new IfThenElseExpressionImpl();
        expr.ifThen(ifExpr, thenValueExpr);
        expr.elseEnd(elseValueExpr);
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ifThenElse(javax.jdo.query.BooleanExpression, javax.jdo.query.Expression, java.lang.Object)
     */
    @Override
    public <V> IfThenElseExpression<V> ifThenElse(BooleanExpression cond, Expression<V> thenValueExpr, V elseValue)
    {
        IfThenElseExpression expr = new IfThenElseExpressionImpl();
        expr.ifThen(cond, thenValueExpr);
        expr.elseEnd(elseValue);
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ifThenElse(javax.jdo.query.BooleanExpression, java.lang.Object, javax.jdo.query.Expression)
     */
    @Override
    public <V> IfThenElseExpression<V> ifThenElse(BooleanExpression cond, V thenValue, Expression<V> elseValueExpr)
    {
        IfThenElseExpression expr = new IfThenElseExpressionImpl();
        expr.ifThen(cond, thenValue);
        expr.elseEnd(elseValueExpr);
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ifThenElse(javax.jdo.query.BooleanExpression, java.lang.Object, java.lang.Object)
     */
    @Override
    public <V> IfThenElseExpression<V> ifThenElse(BooleanExpression cond, V thenValue, V elseValue)
    {
        IfThenElseExpression expr = new IfThenElseExpressionImpl();
        expr.ifThen(cond, thenValue);
        expr.elseEnd(elseValue);
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ifThen(java.lang.Class, javax.jdo.query.BooleanExpression, javax.jdo.query.Expression)
     */
    @Override
    public <V> IfThenElseExpression<V> ifThen(Class<V> type, BooleanExpression cond, Expression<V> thenValueExpr)
    {
        IfThenElseExpression expr = new IfThenElseExpressionImpl();
        expr.ifThen(cond, thenValueExpr);
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ifThen(javax.jdo.query.BooleanExpression, java.lang.Object)
     */
    @Override
    public <V> IfThenElseExpression<V> ifThen(BooleanExpression cond, V thenValue)
    {
        IfThenElseExpression expr = new IfThenElseExpressionImpl();
        expr.ifThen(cond, thenValue);
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#excludeSubclasses()
     */
    public JDOQLTypedQuery<T> excludeSubclasses()
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();
        this.subclasses = false;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#includeSubclasses()
     */
    public JDOQLTypedQuery<T> includeSubclasses()
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();
        this.subclasses = true;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#filter(org.datanucleus.query.typesafe.BooleanExpression)
     */
    public JDOQLTypedQuery<T> filter(BooleanExpression expr)
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();
        this.filter = (BooleanExpressionImpl)expr;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#groupBy(org.datanucleus.query.typesafe.Expression[])
     */
    public JDOQLTypedQuery<T> groupBy(Expression... exprs)
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();
        if (exprs != null && exprs.length > 0)
        {
            grouping = new ArrayList<ExpressionImpl>();
            for (int i=0;i<exprs.length;i++)
            {
                grouping.add((ExpressionImpl)exprs[i]);
            }
        }
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#having(org.datanucleus.query.typesafe.Expression)
     */
    public JDOQLTypedQuery<T> having(Expression expr)
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();
        this.having = (ExpressionImpl)expr;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#orderBy(org.datanucleus.query.typesafe.OrderExpression[])
     */
    public JDOQLTypedQuery<T> orderBy(OrderExpression... exprs)
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();
        if (exprs != null && exprs.length > 0)
        {
            ordering = new ArrayList<OrderExpressionImpl>();
            for (int i=0;i<exprs.length;i++)
            {
                ordering.add((OrderExpressionImpl)exprs[i]);
            }
        }
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#range(long, long)
     */
    public JDOQLTypedQuery<T> range(long lowerIncl, long upperExcl)
    {
        assertIsOpen();
        discardCompiled();
        this.rangeLowerExpr = new NumericExpressionImpl(new Literal(lowerIncl));
        this.rangeUpperExpr = new NumericExpressionImpl(new Literal(upperExcl));
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#range(org.datanucleus.query.typesafe.NumericExpression, org.datanucleus.query.typesafe.NumericExpression)
     */
    public JDOQLTypedQuery<T> range(NumericExpression lowerInclExpr, NumericExpression upperExclExpr)
    {
        assertIsOpen();
        discardCompiled();
        this.rangeLowerExpr = (ExpressionImpl)lowerInclExpr;
        this.rangeUpperExpr = (ExpressionImpl)upperExclExpr;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#range(java.lang.String, java.lang.String)
     */
    public JDOQLTypedQuery<T> range(Expression paramLowerInclExpr, Expression paramUpperExclExpr)
    {
        assertIsOpen();
        discardCompiled();
        if (!((ExpressionImpl)paramLowerInclExpr).isParameter())
        {
            throw new JDOUserException("lower inclusive expression should be a parameter");
        }
        else if (!((ExpressionImpl)paramUpperExclExpr).isParameter())
        {
            throw new JDOUserException("upper exclusive expression should be a parameter");
        }
        this.rangeLowerExpr = (ExpressionImpl)paramLowerInclExpr;
        this.rangeUpperExpr = (ExpressionImpl)paramUpperExclExpr;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#subquery(Class, String)
     */
    public <S> JDOQLTypedSubquery<S> subquery(Class<S> candidateClass, String candidateAlias)
    {
        assertIsOpen();
        discardCompiled();
        JDOQLTypedSubqueryImpl<S> subquery = new JDOQLTypedSubqueryImpl<S>(pm, candidateClass, candidateAlias, this);
        if (subqueries == null)
        {
            subqueries = ConcurrentHashMap.newKeySet();
        }
        subqueries.add(subquery);
        return subquery;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#subquery(java.lang.String)
     */
    public JDOQLTypedSubquery<T> subquery(String candidateAlias)
    {
        assertIsOpen();
        discardCompiled();
        JDOQLTypedSubqueryImpl<T> subquery = new JDOQLTypedSubqueryImpl<T>(pm, this.candidateCls, candidateAlias, this);
        if (subqueries == null)
        {
            subqueries = ConcurrentHashMap.newKeySet();
        }
        subqueries.add(subquery);
        return subquery;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#subquery(CollectionExpression, Class, String)
     */
    @Override
    public <E> JDOQLTypedSubquery<E> subquery(CollectionExpression<Collection<E>, E> candidateCollection, Class<E> candidateClass, String candidateAlias)
    {
        assertIsOpen();
        discardCompiled();
        JDOQLTypedSubqueryImpl<E> subquery = new JDOQLTypedSubqueryImpl<E>(pm, candidateClass, candidateAlias, (CollectionExpressionImpl)candidateCollection, this);
        if (subqueries == null)
        {
            subqueries = ConcurrentHashMap.newKeySet();
        }
        subqueries.add(subquery);
        return subquery;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#setParameters(java.util.Map)
     */
    @Override
    public JDOQLTypedQuery<T> setParameters(Map namedParamMap)
    {
        assertIsOpen();
        discardCompiled();
        if (namedParamMap == null || namedParamMap.isEmpty())
        {
            parameterValuesByName = null;
            return this;
        }

        if (parameterValuesByName == null)
        {
            parameterValuesByName = new ConcurrentHashMap<>();
        }

        Iterator<Map.Entry> entryIter = namedParamMap.entrySet().iterator();
        while (entryIter.hasNext())
        {
            Map.Entry entry = entryIter.next();
            Object key = entry.getKey();
            Object val = entry.getValue();

            if (key instanceof String)
            {
                if (parameterExprByName == null || !parameterExprByName.containsKey(key))
                {
                    throw new JDOUserException("Parameter with name " + key + " doesnt exist for this query");
                }
                parameterValuesByName.put((String)key, val);
            }
            else if (key instanceof Expression)
            {
                ParameterExpression internalParamExpr = (ParameterExpression) ((ExpressionImpl)key).getQueryExpression();
                if (parameterExprByName == null || !parameterExprByName.containsKey(internalParamExpr.getAlias()))
                {
                    throw new JDOUserException("Parameter with name " + internalParamExpr.getAlias() + " doesnt exist for this query");
                }
                parameterValuesByName.put(internalParamExpr.getAlias(), val);
            }
        }

        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#setParameter(org.datanucleus.query.typesafe.Expression, java.lang.Object)
     */
    public JDOQLTypedQuery<T> setParameter(Expression paramExpr, Object value)
    {
        assertIsOpen();
        discardCompiled();

        ParameterExpression internalParamExpr = (ParameterExpression) ((ExpressionImpl)paramExpr).getQueryExpression();
        if (parameterExprByName == null || !parameterExprByName.containsKey(internalParamExpr.getAlias()))
        {
            throw new JDOUserException("Parameter with name " + internalParamExpr.getAlias() + " doesnt exist for this query");
        }

        if (parameterValuesByName == null)
        {
            parameterValuesByName = new ConcurrentHashMap<>();
        }
        parameterValuesByName.put(internalParamExpr.getAlias(), value);
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#setParameter(java.lang.String, java.lang.Object)
     */
    public JDOQLTypedQuery<T> setParameter(String paramName, Object value)
    {
        assertIsOpen();
        discardCompiled();

        if (parameterExprByName == null || !parameterExprByName.containsKey(paramName))
        {
            throw new JDOUserException("Parameter with name " + paramName + " doesnt exist for this query");
        }

        if (parameterValuesByName == null)
        {
            parameterValuesByName = new ConcurrentHashMap<String, Object>();
        }
        parameterValuesByName.put(paramName, value);
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#setCandidates(java.util.Collection)
     */
    public JDOQLTypedQuery<T> setCandidates(Collection<T> candidates)
    {
        assertIsOpen();
        if (candidates != null)
        {
            this.candidates = new ArrayList<T>(candidates);
        }
        else
        {
            this.candidates = null;
        }
        return null;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#result(boolean, javax.jdo.query.Expression[])
     */
    @Override
    public JDOQLTypedQuery<T> result(boolean distinct, Expression<?>... exprs)
    {
        assertIsOpen();
        assertIsModifiable();
        discardCompiled();

        result = null;
        if (exprs != null && exprs.length > 0)
        {
            result = new ArrayList<ExpressionImpl>();
            for (int i=0;i<exprs.length;i++)
            {
                result.add((ExpressionImpl)exprs[i]);
            }
        }
        this.resultDistinct = distinct;

        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#executeResultList(java.lang.Class)
     */
    @Override
    public <R> List<R> executeResultList(Class<R> resultCls)
    {
        assertIsOpen();
        checkCandidateResult();
        if (result == null && resultCls == null)
        {
            throw new JDOUserException("Cannot call executeResultList method when query has result AND resultClass unset. Call executeList instead.");
        }
        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        this.unique = false;
        this.resultClass = resultCls;

        return (List<R>)executeInternalQuery(getInternalQuery());
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#executeResultUnique(java.lang.Class)
     */
    @Override
    public <R> R executeResultUnique(Class<R> resultCls)
    {
        assertIsOpen();
        checkCandidateResult();
        if (result == null && resultCls == null)
        {
            throw new JDOUserException("Cannot call executeResultUnique method when query has result AND resultClass unset. Call executeUnique instead.");
        }
        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        this.unique = true;
        this.resultClass = resultCls;

        return (R)executeInternalQuery(getInternalQuery());
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#executeResultList()
     */
    @Override
    public List executeResultList()
    {
        assertIsOpen();
        checkCandidateResult();
        if (result == null)
        {
            throw new JDOUserException("Cannot call executeResultList method when query has result unset. Call executeList instead.");
        }
        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        this.unique = false;
        this.resultClass = null;

        return (List) executeInternalQuery(getInternalQuery());
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#executeResultUnique()
     */
    @Override
    public Object executeResultUnique()
    {
        assertIsOpen();
        checkCandidateResult();
        if (result == null)
        {
            throw new JDOUserException("Cannot call executeResultUnique method when query has result unset. Call executeUnique instead.");
        }
        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        this.unique = true;
        this.resultClass = null;

        return executeInternalQuery(getInternalQuery());
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#executeList()
     */
    public List<T> executeList()
    {
        assertIsOpen();
        checkCandidateResult();
        if (result != null)
        {
            throw new JDOUserException("Cannot call executeList method when query has result set to " + StringUtils.collectionToString(result) + ". Call executeResultList instead.");
        }
        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        unique = false;

        return (List<T>)executeInternalQuery(getInternalQuery());
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#executeUnique()
     */
    public T executeUnique()
    {
        assertIsOpen();
        checkCandidateResult();
        if (result != null)
        {
            throw new JDOUserException("Cannot call executeUnique method when query has result set to " + StringUtils.collectionToString(result) + ". Call executeResultUnique instead.");
        }
        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        unique = true;

        return (T)executeInternalQuery(getInternalQuery());
    }

    /**
     * Helper method to check the result expression(s).
     * If the user has set the result to "this", then just ignore since default is "distinct this"
     */
    private void checkCandidateResult()
    {
        if (result != null && result.size() == 1 && isCandidateThisExpression(result.get(0)))
        {
            result = null;
        }
    }

    /**
     * Checks whether the specified result expression is a result of a candidate("this") call
     * @param resultExpr the result expression to be checked
     * @return Whether the candidate is "this"
     */
    private boolean isCandidateThisExpression(ExpressionImpl resultExpr)
    {
        if ((resultExpr != null)  && (resultExpr instanceof PersistableExpressionImpl))
        {
            // result expression is a PersistableExpressionImpl
            PersistableExpressionImpl persExpr = (PersistableExpressionImpl)resultExpr;
            if (!persExpr.isParameter() && !persExpr.isVariable() && persExpr.getQueryExpression() instanceof PrimaryExpression) 
            {
                // it is not a variable or parameter and its query expression is a PrimaryExpression
                PrimaryExpression primaryExpr = (PrimaryExpression)persExpr.getQueryExpression();
                List<String> tuples = primaryExpr.getTuples();
                // the primary Expression does not have a parent and its tuple list consists of the string "this"
                return (primaryExpr.getLeft() == null && tuples != null && tuples.size() == 1 && "this".equalsIgnoreCase(tuples.get(0)));
            }
        }
        return false;
    }

    /**
     * Convenience method to generate an internal DataNucleus Query and apply the generic compilation to it.
     * @return The internal DataNucleus query
     */
    protected Query getInternalQuery()
    {
        // Create a DataNucleus query and set the generic compilation
        Query internalQuery = ec.getStoreManager().newQuery(QueryLanguage.JDOQL.name(), ec, toString());

        if (ec.getFlushMode() == FlushMode.QUERY)
        {
            // Flush mode implies flush all before executing the query so set the necessary property
            internalQuery.addExtension(Query.EXTENSION_FLUSH_BEFORE_EXECUTION, Boolean.TRUE);
        }
        internalQuery.setIgnoreCache(ignoreCache);
        if (extensions != null)
        {
            internalQuery.setExtensions(extensions);
        }
        if (fetchPlan != null)
        {
            internalQuery.setFetchPlan(fetchPlan.getInternalFetchPlan());
        }
        if (serializeRead != null)
        {
            internalQuery.setSerializeRead(serializeRead);
        }
        if (datastoreReadTimeout != null)
        {
            internalQuery.setDatastoreReadTimeoutMillis(datastoreReadTimeout);
        }
        if (datastoreWriteTimeout != null)
        {
            internalQuery.setDatastoreWriteTimeoutMillis(datastoreWriteTimeout);
        }

        if (!subclasses)
        {
            internalQuery.setSubclasses(false);
        }
        if (type == QueryType.SELECT)
        {
            internalQuery.setType(Query.QueryType.SELECT);
            if (resultDistinct != null)
            {
                internalQuery.setResultDistinct(resultDistinct.booleanValue());
            }
            internalQuery.setResultClass(resultClass);
            internalQuery.setUnique(unique);
            if (candidates != null)
            {
                internalQuery.setCandidates(candidates);
            }
        }
        else if (type == QueryType.BULK_UPDATE)
        {
            internalQuery.setType(Query.QueryType.BULK_UPDATE);
        }
        else if (type == QueryType.BULK_DELETE)
        {
            internalQuery.setType(Query.QueryType.BULK_DELETE);
        }

        QueryCompilation compilation = getCompilation();
        internalQuery.setCompilation(compilation);

        return internalQuery;
    }

    protected Object executeInternalQuery(Query internalQuery)
    {
        // Cache the internal query
        if (internalQueries == null)
        {
            internalQueries = ConcurrentHashMap.newKeySet();
        }
        internalQueries.add(internalQuery);

        try
        {
            if (parameterValuesByName != null || parameterExprByName != null)
            {
                validateParameters();

                return internalQuery.executeWithMap(parameterValuesByName);
            }
            return internalQuery.execute();
        }
        catch (NoQueryResultsException nqre)
        {
            return null;
        }
        catch (NucleusException jpe)
        {
            // Convert any exceptions into what JDO expects
            throw JDOAdapter.getJDOExceptionForNucleusException(jpe);
        }
        finally
        {
            // Parameter values are not retained beyond the subsequent execute/deletePersistentAll call
            parameterValuesByName = null;
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#deletePersistentAll()
     */
    public long deletePersistentAll()
    {
        assertIsOpen();
        if (result != null || resultClass != null)
        {
            throw new JDOUserException("Cannot call deletePersistentAll method when query has result or resultClass set. Remove the result setting.");
        }

        type = QueryType.SELECT;
        updateExprs = null;
        updateVals = null;
        unique = false;

        try
        {
            Query internalQuery = getInternalQuery();
            if (parameterValuesByName != null || parameterExprByName != null)
            {
                validateParameters();

                return internalQuery.deletePersistentAll(parameterValuesByName);
            }
            return internalQuery.deletePersistentAll();
        }
        catch (NucleusException jpe)
        {
            // Convert any exceptions into what JDO expects
            throw JDOAdapter.getJDOExceptionForNucleusException(jpe);
        }
        finally
        {
            // Parameter values are not retained beyond the subsequent execute/deletePersistentAll call
            parameterValuesByName = null;
        }
    }

    /**
     * Convenience method to validate the defined parameters, and the values provided for these parameters.
     * @throws JDOUserException if they are inconsistent
     */
    private void validateParameters()
    {
        int numParams = (parameterExprByName != null ? parameterExprByName.size() : 0);
        int numValues = (parameterValuesByName != null ? parameterValuesByName.size() : 0);

        if (numParams == 0 && numValues == 0)
        {
            return;
        }

        // Validate the defined parameters and the provided values
        if (numParams != numValues)
        {
            throw new JDOUserException("Query has " + numParams + " but " + numValues + " values have been provided");
        }

        if (parameterExprByName != null && !parameterExprByName.isEmpty())
        {
            for (String paramName : parameterExprByName.keySet())
            {
                if (parameterValuesByName == null || !parameterValuesByName.containsKey(paramName))
                {
                    throw new JDOUserException("Query has a parameter " + paramName + " defined but no value supplied");
                }
            }
        }
    }

    /**
     * Method to specify the update of a field of the candidate.
     * @param expr Expression for field of the candidate
     * @param val The new value
     * @return The query
     */
    public JDOQLTypedQuery<T> set(Expression expr, Object val)
    {
        assertIsOpen();
        type = QueryType.BULK_UPDATE;

        // TODO Check that expr relates to the candidate
        if (updateExprs == null)
        {
            updateExprs = new ArrayList<ExpressionImpl>();
            updateVals = new ArrayList<ExpressionImpl>();
        }

        ExpressionImpl valExpr = null;
        org.datanucleus.store.query.expression.Expression literalExpr = new Literal(val);
        if (val instanceof String)
        {
            valExpr = new StringExpressionImpl(literalExpr);
        }
        else if (val instanceof java.sql.Time)
        {
            valExpr = new TimeExpressionImpl(literalExpr);
        }
        else if (val instanceof java.sql.Date)
        {
            valExpr = new DateExpressionImpl(literalExpr);
        }
        else if (val instanceof java.util.Date)
        {
            valExpr = new DateTimeExpressionImpl(literalExpr);
        }
        else if (val instanceof Boolean)
        {
            valExpr = new BooleanExpressionImpl(literalExpr);
        }
        else if (val instanceof Byte)
        {
            valExpr = new ByteExpressionImpl(literalExpr);
        }
        else if (val instanceof Number)
        {
            valExpr = new NumericExpressionImpl(literalExpr);
        }
        else if (val instanceof Enum)
        {
            valExpr = new EnumExpressionImpl(literalExpr);
        }

        updateExprs.add((ExpressionImpl) expr);
        updateVals.add(valExpr);

        return this;
    }

    /**
     * Extension method to provide bulk update capabilities (not part of JDO).
     * @return Number of instances that were updated
     */
    public long update()
    {
        assertIsOpen();
        type = QueryType.BULK_UPDATE;
        if (updateExprs == null || updateExprs.isEmpty())
        {
            throw new JDOUserException("No update expressions defined. Use set() method");
        }

        return (Long)executeInternalQuery(getInternalQuery());
    }

    /**
     * Extension method to provide bulk delete capabilities (not part of JDO).
     * This differs from deletePersistentAll() in that it doesn't cascade to related objects (unless the
     * datastore does that automatically), and that it doesn't attempt to update cached objects state to
     * reflect the deletion.
     * @return Number of instances that were deleted
     */
    public long delete()
    {
        assertIsOpen();
        type = QueryType.BULK_DELETE;
        updateExprs = null;
        updateVals = null;
        return (Long)executeInternalQuery(getInternalQuery());
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#getFetchPlan()
     */
    public FetchPlan getFetchPlan()
    {
        assertIsOpen();
        if (fetchPlan == null)
        {
            fetchPlan = new JDOFetchPlan(ec.getFetchPlan().getCopy());
        }
        return fetchPlan;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#getPersistenceManager()
     */
    public PersistenceManager getPersistenceManager()
    {
        assertIsOpen();
        return pm;
    }

    /**
     * Method to compile the typesafe query.
     * @return The generic compilation
     */
    public QueryCompilation compile(MetaDataManager mmgr, ClassLoaderResolver clr)
    {
        assertIsOpen();
        QueryCompilation compilation = super.compile(mmgr, clr);

        // Add compilation of any subqueries
        if (subqueries != null && !subqueries.isEmpty())
        {
            Iterator<JDOQLTypedSubqueryImpl> iter = subqueries.iterator();
            while (iter.hasNext())
            {
                JDOQLTypedSubqueryImpl subquery = iter.next();
                QueryCompilation subqueryCompilation = subquery.getCompilation();
                compilation.addSubqueryCompilation(subquery.getAlias(), subqueryCompilation);
            }
        }

        return compilation;
    }

    /**
     * Method to return the (simple) name of the query class for a specified class name.
     * Currently just returns "Q{className}"
     * @param name Simple name of the class (without package)
     * @return Simple name of the query class
     */
    public static String getQueryClassNameForClassName(String name)
    {
        return QUERY_CLASS_PREFIX + name;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#getDatastoreReadTimeoutMillis()
     */
    @Override
    public Integer getDatastoreReadTimeoutMillis()
    {
        assertIsOpen();
        return datastoreReadTimeout;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#datastoreReadTimeoutMillis(java.lang.Integer)
     */
    @Override
    public JDOQLTypedQuery<T> datastoreReadTimeoutMillis(Integer interval)
    {
        assertIsOpen();
        this.datastoreReadTimeout = interval;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#getDatastoreWriteTimeoutMillis()
     */
    @Override
    public Integer getDatastoreWriteTimeoutMillis()
    {
        assertIsOpen();
        return datastoreWriteTimeout;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#datastoreWriteTimeoutMillis(java.lang.Integer)
     */
    @Override
    public JDOQLTypedQuery<T> datastoreWriteTimeoutMillis(Integer interval)
    {
        assertIsOpen();
        this.datastoreWriteTimeout = interval;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#getSerializeRead()
     */
    @Override
    public Boolean getSerializeRead()
    {
        assertIsOpen();
        return serializeRead;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#serializeRead(java.lang.Boolean)
     */
    @Override
    public JDOQLTypedQuery<T> serializeRead(Boolean serialize)
    {
        assertIsOpen();
        this.serializeRead = serialize;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#isUnmodifiable()
     */
    @Override
    public boolean isUnmodifiable()
    {
        assertIsOpen();
        return unmodifiable;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#unmodifiable()
     */
    @Override
    public JDOQLTypedQuery<T> unmodifiable()
    {
        assertIsOpen();
        this.unmodifiable = true;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#getIgnoreCache()
     */
    @Override
    public boolean getIgnoreCache()
    {
        assertIsOpen();
        return ignoreCache;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#ignoreCache(boolean)
     */
    @Override
    public JDOQLTypedQuery<T> ignoreCache(boolean flag)
    {
        assertIsOpen();
        this.ignoreCache = flag;
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#extension(java.lang.String, java.lang.Object)
     */
    @Override
    public JDOQLTypedQuery<T> extension(String key, Object value)
    {
        assertIsOpen();
        if (extensions == null)
        {
            extensions = new ConcurrentHashMap<>();
        }
        extensions.put(key, value);
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#extensions(java.util.Map)
     */
    @Override
    public JDOQLTypedQuery<T> extensions(Map values)
    {
        assertIsOpen();
        this.extensions = new ConcurrentHashMap(extensions);
        return this;
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#cancelAll()
     */
    @Override
    public void cancelAll()
    {
        assertIsOpen();
        if (internalQueries == null || internalQueries.isEmpty())
        {
            return;
        }
        try
        {
            Iterator<Query> iter = internalQueries.iterator();
            while (iter.hasNext())
            {
                Query query = iter.next();
                query.cancel();
            }
        }
        catch (NucleusException ne)
        {
            throw new JDOException("Error in calling Query.cancelAll. See the nested exception", ne);
        }
        catch (UnsupportedOperationException uoe)
        {
            throw new JDOUnsupportedOptionException();
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#cancel(java.lang.Thread)
     */
    @Override
    public void cancel(Thread thread)
    {
        assertIsOpen();
        if (internalQueries == null || internalQueries.isEmpty())
        {
            return;
        }
        try
        {
            Iterator<Query> iter = internalQueries.iterator();
            while (iter.hasNext())
            {
                Query query = iter.next();
                query.cancel(thread);
            }
        }
        catch (NucleusException ne)
        {
            throw new JDOException("Error in calling Query.cancelAll. See the nested exception", ne);
        }
        catch (UnsupportedOperationException uoe)
        {
            throw new JDOUnsupportedOptionException();
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#saveAsNamedQuery(java.lang.String)
     */
    @Override
    public JDOQLTypedQuery<T> saveAsNamedQuery(String name)
    {
        assertIsOpen();
        JDOPersistenceManagerFactory.checkJDOPermission(JDOPermission.GET_METADATA);

        QueryMetaData qmd = new QueryMetaData(name);
        qmd.setLanguage(QueryLanguage.JDOQL.name());
        Query query = getInternalQuery();
        qmd.setQuery(query.toString());
        qmd.setResultClass(query.getResultClassName());
        qmd.setUnique(query.isUnique());
        Map<String, Object> queryExts = query.getExtensions();
        if (queryExts != null && !queryExts.isEmpty())
        {
            Iterator<Map.Entry<String, Object>> queryExtsIter = queryExts.entrySet().iterator();
            while (queryExtsIter.hasNext())
            {
                Map.Entry<String, Object> queryExtEntry = queryExtsIter.next();
                qmd.addExtension(queryExtEntry.getKey(), "" + queryExtEntry.getValue());
            }
        }
        query.getExecutionContext().getMetaDataManager().registerNamedQuery(qmd);

        return this;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.api.jdo.query.AbstractJDOQLTypedQuery#getJDOQLForExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    public String getJDOQLForExpression(org.datanucleus.store.query.expression.Expression expr)
    {
        if (expr instanceof VariableExpression)
        {
            VariableExpression varExpr = (VariableExpression)expr;
            if (subqueries != null)
            {
                for (JDOQLTypedSubqueryImpl subq : subqueries)
                {
                    if (varExpr.getId().equals(subq.getAlias()))
                    {
                        // This variable represents a subquery so return the subquery text, so we form a single-string including subqueries
                        return "(" + subq.toString() + ")";
                    }
                }
            }
        }

        return super.getJDOQLForExpression(expr);
    }

    /**
     * Method to throw an exception if the query is currently not modifiable.
     * @throws NucleusUserException Thrown when it is unmodifiable
     */
    protected void assertIsModifiable()
    {
        if (unmodifiable)
        {
            throw new NucleusUserException(Localiser.msg("021014"));
        }
    }

    /**
     * Method to assert if this Query is open.
     * @throws JDOFatalUserException if the Query is closed.
     */
    protected void assertIsOpen()
    {
        if (closed)
        {
            throw new JDOFatalUserException(Localiser.msg("011100"));
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.JDOQLTypedQuery#geospatialHelper()
     */
    @Override
    public javax.jdo.query.geospatial.GeospatialHelper geospatialHelper()
    {
        if (geospatialHelper == null)
        {
            geospatialHelper = new GeospatialHelperImpl();
        }
        return (javax.jdo.query.geospatial.GeospatialHelper) geospatialHelper;
    }
}
