/**********************************************************************
Copyright (c) 2006 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 

Contributors:
2006 Andy Jefferson - javadocs
    ...
**********************************************************************/
package org.datanucleus.api.jpa;

import java.util.Map;

import javax.persistence.EntityManagerFactory;
import javax.persistence.spi.LoadState;
import javax.persistence.spi.PersistenceProvider;
import javax.persistence.spi.PersistenceUnitInfo;
import javax.persistence.spi.ProviderUtil;

import org.datanucleus.ExecutionContext;
import org.datanucleus.api.jpa.exceptions.NoPersistenceUnitException;
import org.datanucleus.api.jpa.exceptions.NoPersistenceXmlException;
import org.datanucleus.api.jpa.exceptions.NotProviderException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.state.ObjectProvider;

/**
 * PersistenceProvider implementation.
 * Provides a means of creating EntityManagerFactory objects.
 */
public class PersistenceProviderImpl implements PersistenceProvider, ProviderUtil
{
    /**
     * Constructor.
     */
    public PersistenceProviderImpl()
    {
    }

    /**
     * Method to create an EntityManagerFactory when running in J2EE.
     * The container will have parsed the persistence.xml files to provide this PersistenceUnitInfo.
     * @param unitInfo The "persistence-unit"
     * @param properties EntityManagerFactory properties to override those in the persistence unit
     * @return The EntityManagerFactory
     */
    public EntityManagerFactory createContainerEntityManagerFactory(PersistenceUnitInfo unitInfo, Map properties)
    {
        try
        {
            return new JPAEntityManagerFactory(unitInfo, properties);
        }
        catch (SingletonEMFException se)
        {
            return se.getSingleton();
        }
        catch (NotProviderException npe)
        {
            return null;
        }
        catch (NoPersistenceUnitException npue)
        {
            return null;
        }
        catch (NoPersistenceXmlException npxe)
        {
            return null;
        }
    }

    /**
     * Method to create an EntityManagerFactory when running in J2SE.
     * @param unitName Name of the "persistence-unit"
     * @param properties EntityManagerFactory properties to override those in the persistence unit
     * @return The EntityManagerFactory
     */
    public EntityManagerFactory createEntityManagerFactory(String unitName, Map properties)
    {
        try
        {
            return new JPAEntityManagerFactory(unitName, properties);
        }
        catch (SingletonEMFException se)
        {
            return se.getSingleton();
        }
        catch (NotProviderException npe)
        {
            return null;
        }
        catch (NoPersistenceUnitException npue)
        {
            return null;
        }
        catch (NoPersistenceXmlException npxe)
        {
            return null;
        }
    }

    /**
     * If the provider determines that the entity has been provided
     * by itself and that the state of the specified attribute has
     * been loaded, this method returns LoadState.LOADED.
     * If the provider determines that the entity has been provided
     * by itself and that either entity attributes with FetchType
     * EAGER have not been loaded or that the state of the specified
     * attribute has not been loaded, this methods returns
     * LoadState.NOT_LOADED.
     * If a provider cannot determine the load state, this method
     * returns LoadState.UNKNOWN.
     * The provider's implementation of this method must not obtain
     * a reference to an attribute value, as this could trigger the
     * loading of entity state if the entity has been provided by a
     * different provider.
     * @param entity The entity
     * @param memberName Name of member whose load status is to be determined
     * @return load status of the attribute
     */
    public LoadState isLoadedWithoutReference(Object entity, String memberName)
    {
        if (entity == null)
        {
            return LoadState.UNKNOWN;
        }

        JPAAdapter adapter = new JPAAdapter();
        ExecutionContext ec = adapter.getExecutionContext(entity);
        if (ec == null)
        {
            // TODO Handle detached entities
            return LoadState.UNKNOWN;
        }
        ObjectProvider sm = ec.findObjectProvider(entity);
        if (sm == null)
        {
            return LoadState.UNKNOWN;
        }
        else
        {
            String[] loadedFields = sm.getLoadedFieldNames();
            if (loadedFields != null)
            {
                for (int i=0;i<loadedFields.length;i++)
                {
                    if (loadedFields[i].equals(memberName))
                    {
                        return LoadState.LOADED;
                    }
                }
            }
            return LoadState.NOT_LOADED;
        }
    }

    /**
     * If the provider determines that the entity has been provided
     * by itself and that the state of the specified attribute has
     * been loaded, this method returns LoadState.LOADED.
     * If a provider determines that the entity has been provided
     * by itself and that either the entity attributes with FetchType
     * EAGER have not been loaded or that the state of the specified
     * attribute has not been loaded, this method returns
     * return LoadState.NOT_LOADED.
     * If the provider cannot determine the load state, this method
     * returns LoadState.UNKNOWN.
     * The provider's implementation of this method is permitted to
     * obtain a reference to the attribute value. (This access is
     * safe because providers which might trigger the loading of the
     * attribute state will have already been determined by
     * isLoadedWithoutReference.)
     * @param entity The entity
     * @param memberName name of member whose load status is to be determined
     * @return load status of the member
     */
    public LoadState isLoadedWithReference(Object entity, String memberName)
    {
        // Just make use of other implementation
        return isLoadedWithoutReference(entity, memberName);
    }

    /**
     * If the provider determines that the entity has been provided
     * by itself and that the state of all attributes for which
     * FetchType EAGER has been specified have been loaded, this
     * method returns LoadState.LOADED.
     * If the provider determines that the entity has been provided
     * by itself and that not all attributes with FetchType EAGER
     * have been loaded, this method returns LoadState.NOT_LOADED.
     * If the provider cannot determine if the entity has been
     * provided by itself, this method returns LoadState.UNKNOWN.
     * The provider's implementation of this method must not obtain
     * a reference to any attribute value, as this could trigger the
     * loading of entity state if the entity has been provided by a different provider.
     * @param entity whose loaded status is to be determined
     * @return load status of the entity
     */
    public LoadState isLoaded(Object entity)
    {
        if (entity == null)
        {
            return LoadState.UNKNOWN;
        }

        JPAAdapter adapter = new JPAAdapter();
        ExecutionContext ec = adapter.getExecutionContext(entity);
        if (ec == null)
        {
            // TODO Handle detached entities
            return LoadState.UNKNOWN;
        }

        ObjectProvider sm = ec.findObjectProvider(entity);
        if (sm == null)
        {
            return LoadState.UNKNOWN;
        }
        else
        {
            boolean allLoaded = true;
            AbstractClassMetaData cmd =
                ec.getMetaDataManager().getMetaDataForClass(entity.getClass(), ec.getClassLoaderResolver());
            int[] dfgFieldNumbers = cmd.getDFGMemberPositions();
            for (int i=0;i<dfgFieldNumbers.length;i++)
            {
                AbstractMemberMetaData mmd= cmd.getMetaDataForManagedMemberAtAbsolutePosition(dfgFieldNumbers[i]);
                String[] loadedFields = sm.getLoadedFieldNames();

                boolean memberLoaded = false;
                if (loadedFields != null)
                {
                    for (int j=0;j<loadedFields.length;j++)
                    {
                        if (loadedFields[j].equals(mmd.getName()))
                        {
                            memberLoaded = true;
                            break;
                        }
                    }
                }
                if (!memberLoaded)
                {
                    allLoaded = false;
                    break;
                }
            }
            return (allLoaded ? LoadState.LOADED : LoadState.NOT_LOADED);
        }
    }

    /* (non-Javadoc)
     * @see javax.persistence.spi.PersistenceProvider#getProviderUtil()
     */
    public ProviderUtil getProviderUtil()
    {
        return this;
    }
}