/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.geometry.S1Angle;
import com.google.common.geometry.S1ChordAngle;
import com.google.common.geometry.S2BuilderGraph;
import com.google.common.geometry.S2BuilderLayer;
import com.google.common.geometry.S2BuilderSnapFunctions;
import com.google.common.geometry.S2BuilderUtil;
import com.google.common.geometry.S2CellId;
import com.google.common.geometry.S2ClosestEdgeQuery;
import com.google.common.geometry.S2ClosestPointQuery;
import com.google.common.geometry.S2CrossingEdgesQuery;
import com.google.common.geometry.S2EdgeUtil;
import com.google.common.geometry.S2Error;
import com.google.common.geometry.S2Loop;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2PointIndex;
import com.google.common.geometry.S2Polygon;
import com.google.common.geometry.S2Polyline;
import com.google.common.geometry.S2PolylineSimplifier;
import com.google.common.geometry.S2Predicates;
import com.google.common.geometry.S2RobustCrossProd;
import com.google.common.geometry.S2Shape;
import com.google.common.geometry.S2ShapeIndex;
import com.google.common.geometry.primitives.IdSetLexicon;
import com.google.common.geometry.primitives.IntVector;
import com.google.common.geometry.primitives.Ints;
import com.google.common.geometry.primitives.Sorter;
import com.google.common.primitives.UnsignedLongs;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

public class S2Builder {
    private final Builder options;
    private final S1ChordAngle siteSnapRadiusChordAngle;
    private final S1ChordAngle edgeSnapRadiusChordAngle;
    private final boolean checkAllSiteCrossings;
    private final S1Angle maxEdgeDeviation;
    private final S1ChordAngle edgeSiteQueryRadiusChordAngle;
    private final S1ChordAngle minEdgeLengthToSplitChordAngle;
    private final S1Angle minSiteSeparation;
    private final S1ChordAngle minSiteSeparationChordAngle;
    private final S1ChordAngle minEdgeSiteSeparationChordAngle;
    private final S1ChordAngle minEdgeSiteSeparationChordAngleLimit;
    private final S1ChordAngle maxAdjacentSiteSeparationChordAngle;
    private final double edgeSnapRadiusSin2;
    private S2Error error;
    private final boolean snappingRequested;
    private boolean snappingNeeded;
    private final List<S2BuilderLayer> layers = new ArrayList<S2BuilderLayer>();
    private final List<GraphOptions> layerOptions = new ArrayList<GraphOptions>();
    private final IntVector layerBegins = new IntVector();
    private final List<IsFullPolygonPredicate> layerIsFullPolygonPredicates = new ArrayList<IsFullPolygonPredicate>();
    private boolean labelSetModified;
    private final List<S2Point> inputVertices = new ArrayList<S2Point>();
    private final S2BuilderGraph.EdgeList inputEdges = new S2BuilderGraph.EdgeList();
    private final IntVector labelSetIds = new IntVector();
    private final IdSetLexicon labelSetLexicon = new IdSetLexicon();
    private final IntVector labelSet = new IntVector();
    private int labelSetId;
    private int numForcedSites;
    private final List<S2Point> sites = new ArrayList<S2Point>();
    private S2CrossingEdgesQuery crossingEdgesQuery;
    private final EdgeSites edgeSites = new EdgeSites();

    private S2Builder(Builder builder) {
        this.options = builder;
        SnapFunction snapFunction = builder.snapFunction();
        S1Angle snapRadius = snapFunction.snapRadius();
        Preconditions.checkArgument((boolean)snapRadius.lessOrEquals(S2BuilderSnapFunctions.maxSnapRadius()));
        this.siteSnapRadiusChordAngle = S1ChordAngle.fromS1Angle(snapRadius);
        S1Angle edgeSnapRadius = this.options.edgeSnapRadius();
        this.edgeSnapRadiusChordAngle = S2Builder.roundUp(edgeSnapRadius);
        this.snappingRequested = edgeSnapRadius.greaterThan(S1Angle.ZERO);
        this.maxEdgeDeviation = this.options.maxEdgeDeviation();
        this.edgeSiteQueryRadiusChordAngle = S1ChordAngle.fromS1Angle(this.maxEdgeDeviation.add(snapFunction.minEdgeVertexSeparation()));
        this.minEdgeLengthToSplitChordAngle = !this.snappingRequested ? S1ChordAngle.INFINITY : S1ChordAngle.fromRadians(2.0 * Math.acos(Math.sin(edgeSnapRadius.radians()) / Math.sin(this.maxEdgeDeviation.radians())));
        this.checkAllSiteCrossings = this.options.maxEdgeDeviation().greaterThan(this.options.edgeSnapRadius().add(snapFunction.minEdgeVertexSeparation()));
        if (this.options.intersectionTolerance().lessOrEquals(S1Angle.ZERO)) {
            Preconditions.checkState((!this.checkAllSiteCrossings ? 1 : 0) != 0);
        }
        this.minSiteSeparation = snapFunction.minVertexSeparation();
        this.minSiteSeparationChordAngle = S1ChordAngle.fromS1Angle(this.minSiteSeparation);
        this.minEdgeSiteSeparationChordAngle = S1ChordAngle.fromS1Angle(snapFunction.minEdgeVertexSeparation());
        this.minEdgeSiteSeparationChordAngleLimit = S2Builder.addPointToEdgeError(this.minEdgeSiteSeparationChordAngle);
        this.maxAdjacentSiteSeparationChordAngle = S2Builder.addPointToPointError(S2Builder.roundUp(edgeSnapRadius.mul(2.0)));
        double d = Math.sin(edgeSnapRadius.radians());
        this.edgeSnapRadiusSin2 = d * d + ((9.5 * d + 2.5 + 2.0 * Math.sqrt(3.0)) * d + 1.9984014443252818E-15) * 2.220446049250313E-16;
        this.labelSetId = Integer.MIN_VALUE;
        this.labelSetModified = false;
        this.snappingNeeded = false;
    }

    public Builder toBuilder() {
        return new Builder(this.options);
    }

    public static IsFullPolygonPredicate isFullPolygon(boolean isFull) {
        return graph -> isFull;
    }

    private static S1ChordAngle roundUp(S1Angle a) {
        S1ChordAngle ca = S1ChordAngle.fromS1Angle(a);
        return ca.plusError(ca.getS1AngleConstructorMaxError());
    }

    private static S1ChordAngle addPointToPointError(S1ChordAngle ca) {
        return ca.plusError(ca.getS2PointConstructorMaxError());
    }

    private static S1ChordAngle addPointToEdgeError(S1ChordAngle ca) {
        return ca.plusError(S2EdgeUtil.getMinDistanceMaxError(ca));
    }

    public S2BuilderLayer currentLayer() {
        return this.layers.get(this.layers.size() - 1);
    }

    public GraphOptions currentLayerOptions() {
        return this.layerOptions.get(this.layers.size() - 1);
    }

    public void startLayer(S2BuilderLayer layer) {
        this.layerOptions.add(layer.graphOptions());
        this.layerBegins.add(this.inputEdges.size());
        this.layerIsFullPolygonPredicates.add(S2Builder.isFullPolygon(false));
        this.layers.add(layer);
    }

    @CanIgnoreReturnValue
    private int addVertex(S2Point v) {
        if (this.inputVertices.isEmpty() || !v.equalsPoint(this.inputVertices.get(this.inputVertices.size() - 1))) {
            this.inputVertices.add(v);
        }
        return this.inputVertices.size() - 1;
    }

    boolean isForced(int siteId) {
        return siteId < this.numForcedSites;
    }

    public void addPoint(S2Point v) {
        this.addEdge(v, v);
    }

    public void addEdge(S2Point v0, S2Point v1) {
        Preconditions.checkState((!this.layers.isEmpty() ? 1 : 0) != 0, (Object)"Call startLayer before adding any edges.");
        if (v0.equalsPoint(v1) && this.currentLayerOptions().degenerateEdges() == GraphOptions.DegenerateEdges.DISCARD) {
            return;
        }
        int j0 = this.addVertex(v0);
        int j1 = this.addVertex(v1);
        this.inputEdges.add(j0, j1);
        if (this.labelSetModified) {
            if (this.labelSetIds.isEmpty()) {
                this.labelSetIds.enlarge(this.inputEdges.size() - 1);
                this.labelSetIds.fill(this.labelSetId);
            }
            this.labelSetId = this.labelSetLexicon.add(this.labelSet);
            this.labelSetIds.push(this.labelSetId);
            this.labelSetModified = false;
        } else if (!this.labelSetIds.isEmpty()) {
            this.labelSetIds.push(this.labelSetId);
        }
    }

    public void addPolyline(S2Polyline polyline) {
        int n = polyline.numVertices();
        if (n == 1) {
            this.addPoint(polyline.vertex(0));
            return;
        }
        for (int i = 1; i < n; ++i) {
            this.addEdge(polyline.vertex(i - 1), polyline.vertex(i));
        }
    }

    public void addLoop(S2Loop loop) {
        if (loop.isEmptyOrFull()) {
            return;
        }
        for (int i = 0; i < loop.numVertices(); ++i) {
            this.addEdge(loop.orientedVertex(i), loop.orientedVertex(i + 1));
        }
    }

    public void addPolygon(S2Polygon polygon) {
        for (int i = 0; i < polygon.numLoops(); ++i) {
            this.addLoop(polygon.loop(i));
        }
    }

    public void addShape(S2Shape shape) {
        S2Shape.MutableEdge edge = new S2Shape.MutableEdge();
        int n = shape.numEdges();
        for (int e = 0; e < n; ++e) {
            shape.getEdge(e, edge);
            this.addEdge(edge.a, edge.b);
        }
    }

    public void addIntersection(S2Point vertex) {
        Preconditions.checkState((boolean)this.options.intersectionTolerance().greaterOrEquals(S1Angle.ZERO));
        this.snappingNeeded = true;
        this.addVertex(vertex);
    }

    public void addIsFullPolygonPredicate(IsFullPolygonPredicate predicate) {
        this.layerIsFullPolygonPredicates.set(this.layerIsFullPolygonPredicates.size() - 1, predicate);
    }

    public void forceVertex(S2Point vertex) {
        this.sites.add(vertex);
    }

    public void setLabel(int label) {
        Preconditions.checkArgument((label >= 0 ? 1 : 0) != 0);
        this.labelSet.clear();
        this.labelSet.push(label);
        this.labelSetModified = true;
    }

    public void clearLabels() {
        this.labelSet.clear();
        this.labelSetModified = true;
    }

    public void pushLabel(int label) {
        Preconditions.checkArgument((label >= 0 ? 1 : 0) != 0);
        this.labelSet.push(label);
        this.labelSetModified = true;
    }

    public void popLabel() {
        this.labelSet.pop();
        this.labelSetModified = true;
    }

    public boolean build(S2Error error) {
        Preconditions.checkNotNull((Object)error);
        this.error = error;
        this.error.clear();
        this.layerBegins.push(this.inputEdges.size());
        if (this.snappingRequested && !this.options.idempotent()) {
            this.snappingNeeded = true;
        }
        try {
            this.chooseSites();
        }
        catch (IllegalArgumentException e) {
            if (e.getMessage() != null && e.getMessage().contains("NaN")) {
                error.init(S2Error.Code.BUILDER_SNAP_RADIUS_TOO_SMALL, "NaN in input geometry?", new Object[0]);
                this.clear();
                return false;
            }
            throw e;
        }
        this.buildLayers();
        this.clear();
        return error.ok();
    }

    public void clear() {
        this.inputVertices.clear();
        this.inputEdges.clear();
        this.layers.clear();
        this.layerOptions.clear();
        this.layerBegins.clear();
        this.layerIsFullPolygonPredicates.clear();
        this.labelSetIds.clear();
        this.labelSetLexicon.clear();
        this.labelSet.clear();
        this.labelSetModified = false;
        this.sites.clear();
        this.edgeSites.clear();
        this.snappingNeeded = false;
    }

    private void chooseSites() {
        if (this.inputVertices.isEmpty()) {
            return;
        }
        S2ShapeIndex inputEdgeIndex = new S2ShapeIndex();
        inputEdgeIndex.add(new S2BuilderUtil.GraphShape(this.inputEdges, this.inputVertices));
        if (this.options.splitCrossingEdges()) {
            this.addEdgeCrossings(inputEdgeIndex);
        }
        if (this.snappingRequested) {
            S2PointIndex<Integer> siteIndex = new S2PointIndex<Integer>();
            this.addForcedSites(siteIndex);
            this.chooseInitialSites(siteIndex);
            this.collectSiteEdges(siteIndex);
        }
        if (this.snappingNeeded) {
            this.addExtraSites(inputEdgeIndex);
        } else {
            this.chooseAllVerticesAsSites();
        }
    }

    private void chooseAllVerticesAsSites() {
        this.sites.clear();
        IntVector sortedInputVertexIds = this.sortInputVertices();
        IntVector vmap = IntVector.ofSize(this.inputVertices.size());
        S2Point previousSite = S2Point.ORIGIN;
        for (int in = 0; in < sortedInputVertexIds.size(); ++in) {
            int vertexId = sortedInputVertexIds.get(in);
            S2Point candidateSite = this.inputVertices.get(vertexId);
            if (!previousSite.equalsPoint(candidateSite)) {
                this.sites.add(candidateSite);
                previousSite = candidateSite;
            }
            vmap.set(vertexId, this.sites.size() - 1);
        }
        this.inputVertices.clear();
        this.inputVertices.addAll(this.sites);
        for (int edgeId = 0; edgeId < this.inputEdges.size(); ++edgeId) {
            int newSrcId = vmap.get(this.inputEdges.getSrcId(edgeId));
            int newDstId = vmap.get(this.inputEdges.getDstId(edgeId));
            this.inputEdges.set(edgeId, newSrcId, newDstId);
        }
    }

    private IntVector sortInputVertices() {
        final IntVector sortedVertexIds = new IntVector();
        final long[] cellIds = new long[this.inputVertices.size()];
        sortedVertexIds.ensureCapacity(this.inputVertices.size());
        for (int inputVertexId = 0; inputVertexId < this.inputVertices.size(); ++inputVertexId) {
            cellIds[inputVertexId] = S2CellId.fromPoint(this.inputVertices.get(inputVertexId)).id();
            sortedVertexIds.add(inputVertexId);
        }
        Sorter.sort(new Sorter.SortableCollection(){

            @Override
            public int size() {
                return sortedVertexIds.size();
            }

            @Override
            public void truncate(int start) {
                sortedVertexIds.truncate(start);
            }

            @Override
            public void swap(int a, int b) {
                sortedVertexIds.swap(a, b);
                long t = cellIds[a];
                cellIds[a] = cellIds[b];
                cellIds[b] = t;
            }

            @Override
            public boolean less(int a, int b) {
                S2Point vertexB;
                int c = UnsignedLongs.compare((long)cellIds[a], (long)cellIds[b]);
                if (c != 0) {
                    return c < 0;
                }
                S2Point vertexA = S2Builder.this.inputVertices.get(sortedVertexIds.get(a));
                return vertexA.compareTo(vertexB = S2Builder.this.inputVertices.get(sortedVertexIds.get(b))) < 0;
            }
        });
        return sortedVertexIds;
    }

    private void addEdgeCrossings(S2ShapeIndex inputEdgeIndex) {
        inputEdgeIndex.applyUpdates();
        ArrayList newVertices = new ArrayList();
        if (this.crossingEdgesQuery == null) {
            this.crossingEdgesQuery = new S2CrossingEdgesQuery(S2CrossingEdgesQuery.CrossingType.INTERIOR);
        }
        this.crossingEdgesQuery.visitCrossingEdgePairs(inputEdgeIndex, (shapeA, edgeIdA, aEdgeSrc, aEdgeDst, shapeB, edgeIdB, bEdgeSrc, bEdgeDst, isInterior) -> {
            newVertices.add(S2EdgeUtil.getIntersection(aEdgeSrc, aEdgeDst, bEdgeSrc, bEdgeDst));
            return true;
        });
        if (newVertices.isEmpty()) {
            return;
        }
        this.snappingNeeded = true;
        this.inputVertices.addAll(newVertices);
    }

    private void addForcedSites(S2PointIndex<Integer> siteIndex) {
        Collections.sort(this.sites);
        S2BuilderUtil.deduplicateSortedList(this.sites);
        for (int siteId = 0; siteId < this.sites.size(); ++siteId) {
            siteIndex.add(this.sites.get(siteId), siteId);
        }
        this.numForcedSites = this.sites.size();
    }

    private void chooseInitialSites(S2PointIndex<Integer> siteIndex) {
        S2ClosestPointQuery<Integer> siteQuery = new S2ClosestPointQuery<Integer>(siteIndex);
        siteQuery.setConservativeMaxDistance(this.minSiteSeparationChordAngle);
        ArrayList nearbyExistingSites = new ArrayList();
        IntVector inputVertexIds = this.sortInputVertices();
        for (int i = 0; i < inputVertexIds.size(); ++i) {
            int inputVertexId = inputVertexIds.get(i);
            S2Point vertex = this.inputVertices.get(inputVertexId);
            S2Point site = this.snapSite(vertex);
            this.snappingNeeded = this.snappingNeeded || !site.equalsPoint(vertex);
            boolean addSite = true;
            if (this.siteSnapRadiusChordAngle.isZero()) {
                addSite = this.sites.isEmpty() || !site.equalsPoint(this.sites.get(this.sites.size() - 1));
            } else {
                nearbyExistingSites.clear();
                siteQuery.findClosestPoints(nearbyExistingSites, site);
                for (S2ClosestPointQuery.Result result : nearbyExistingSites) {
                    if (S2Predicates.compareDistance(site, result.entry().point(), this.minSiteSeparationChordAngle.getLength2()) > 0) continue;
                    addSite = false;
                    this.snappingNeeded = this.snappingNeeded || !site.equalsPoint(result.entry().point());
                }
            }
            if (!addSite) continue;
            siteIndex.add(site, this.sites.size());
            this.sites.add(site);
            siteQuery.reset();
        }
    }

    private S2Point snapSite(S2Point point) {
        if (!this.snappingRequested) {
            return point;
        }
        S2Point site = this.options.snapFunction().snapPoint(point);
        S1ChordAngle distMoved = new S1ChordAngle(site, point);
        if (distMoved.greaterThan(this.siteSnapRadiusChordAngle)) {
            this.error.init(S2Error.Code.BUILDER_SNAP_RADIUS_TOO_SMALL, "Snap function moved vertex (%s, %s, %s) by %s, which is more than the specified snap radius of %s", point.x, point.y, point.z, distMoved.radians(), this.siteSnapRadiusChordAngle.radians());
        }
        return site;
    }

    private void collectSiteEdges(S2PointIndex<Integer> siteIndex) {
        S2ClosestPointQuery<Integer> siteQuery = new S2ClosestPointQuery<Integer>(siteIndex);
        siteQuery.setConservativeMaxDistance(this.edgeSiteQueryRadiusChordAngle);
        ArrayList nearbySites = new ArrayList();
        for (int inputEdgeId = 0; inputEdgeId < this.inputEdges.size(); ++inputEdgeId) {
            S2Point v0 = this.inputVertices.get(this.inputEdges.getSrcId(inputEdgeId));
            S2Point v1 = this.inputVertices.get(this.inputEdges.getDstId(inputEdgeId));
            nearbySites.clear();
            siteQuery.findClosestPointsToEdge(nearbySites, v0, v1);
            IntVector sitesNearEdge = new IntVector();
            sitesNearEdge.ensureCapacity(nearbySites.size());
            for (S2ClosestPointQuery.Result result : nearbySites) {
                sitesNearEdge.add((Integer)result.entry().data());
                if (this.snappingNeeded || !result.distance().lessThan(this.minEdgeSiteSeparationChordAngleLimit) || result.entry().point().equalsPoint(v0) || result.entry().point().equalsPoint(v1)) continue;
                this.snappingNeeded = S2Predicates.compareEdgeDistance(result.entry().point(), v0, v1, this.minEdgeSiteSeparationChordAngle.getLength2()) < 0;
            }
            sitesNearEdge.sort(new SiteIdDistanceComparator(v0));
            this.edgeSites.add(inputEdgeId, sitesNearEdge);
        }
    }

    private void addExtraSites(S2ShapeIndex inputEdgeIndex) {
        IntOpenHashSet edgesToResnap = new IntOpenHashSet();
        IntVector siteIdChain = new IntVector();
        for (int inputEdgeId = 0; inputEdgeId < this.inputEdges.size(); ++inputEdgeId) {
            this.snapEdge(inputEdgeId, siteIdChain);
            edgesToResnap.remove(inputEdgeId);
            this.maybeAddExtraSites(inputEdgeId, siteIdChain, inputEdgeIndex, (IntSet)edgesToResnap);
        }
        while (!edgesToResnap.isEmpty()) {
            HashSet edgesToSnap = new HashSet(edgesToResnap);
            edgesToResnap.clear();
            Iterator iterator = edgesToSnap.iterator();
            while (iterator.hasNext()) {
                int inputEdgeId = (Integer)iterator.next();
                this.snapEdge(inputEdgeId, siteIdChain);
                edgesToResnap.remove(inputEdgeId);
                this.maybeAddExtraSites(inputEdgeId, siteIdChain, inputEdgeIndex, (IntSet)edgesToResnap);
            }
        }
    }

    private void maybeAddExtraSites(int inputEdgeId, IntVector siteIdsChain, S2ShapeIndex inputEdgeIndex, IntSet edgesToResnap) {
        if (siteIdsChain.isEmpty()) {
            return;
        }
        S2Point a0 = this.inputVertices.get(this.inputEdges.getSrcId(inputEdgeId));
        S2Point a1 = this.inputVertices.get(this.inputEdges.getDstId(inputEdgeId));
        Ints.IntList nearbySites = this.edgeSites.get(inputEdgeId);
        int i = 0;
        for (int j = 0; j < nearbySites.size(); ++j) {
            int siteId = nearbySites.get(j);
            if (siteId == siteIdsChain.get(i)) {
                S2Point v1;
                if (++i == siteIdsChain.size()) break;
                S2Point v0 = this.sites.get(siteIdsChain.get(i - 1));
                if (new S1ChordAngle(v0, v1 = this.sites.get(siteIdsChain.get(i))).lessThan(this.minEdgeLengthToSplitChordAngle) || S2EdgeUtil.isEdgeBNearEdgeA(a0, a1, v0, v1, this.maxEdgeDeviation)) continue;
                S2Point mid = S2Point.add(S2EdgeUtil.project(v0, a0, a1), S2EdgeUtil.project(v1, a0, a1)).normalize();
                S2Point newSite = this.getSeparationSite(mid, v0, v1, inputEdgeId);
                this.addExtraSite(newSite, inputEdgeIndex, edgesToResnap);
                return;
            }
            if (i == 0) continue;
            S2Point siteToAvoid = this.sites.get(siteId);
            S2Point v0 = this.sites.get(siteIdsChain.get(i - 1));
            S2Point v1 = this.sites.get(siteIdsChain.get(i));
            boolean addSeparationSite = false;
            if (!this.isForced(siteId) && this.minEdgeSiteSeparationChordAngle.greaterThan(S1ChordAngle.ZERO) && S2Predicates.compareEdgeDistance(siteToAvoid, v0, v1, this.minEdgeSiteSeparationChordAngle.getLength2()) < 0) {
                addSeparationSite = true;
            }
            if (!addSeparationSite && (this.isForced(siteId) || this.checkAllSiteCrossings) && S2Predicates.sign(a0, a1, siteToAvoid) != S2Predicates.sign(v0, v1, siteToAvoid) && S2Predicates.compareEdgeDirections(a0, a1, a0, siteToAvoid) > 0 && S2Predicates.compareEdgeDirections(a0, a1, siteToAvoid, a1) > 0 && S2Predicates.compareEdgeDirections(a0, a1, v0, siteToAvoid) > 0 && S2Predicates.compareEdgeDirections(a0, a1, siteToAvoid, v1) > 0) {
                addSeparationSite = true;
            }
            if (!addSeparationSite) continue;
            S2Point newSite = this.getSeparationSite(siteToAvoid, v0, v1, inputEdgeId);
            assert (!siteToAvoid.equalsPoint(newSite));
            this.addExtraSite(newSite, inputEdgeIndex, edgesToResnap);
            while (nearbySites.get(j + 1) != siteIdsChain.get(i)) {
                ++j;
            }
        }
    }

    private void addExtraSite(S2Point newSite, S2ShapeIndex inputEdgeIndex, IntSet edgesToResnap) {
        assert (this.sites.isEmpty() || !newSite.equalsPoint(this.sites.get(this.sites.size() - 1)));
        int newSiteId = this.sites.size();
        this.sites.add(newSite);
        S2ClosestEdgeQuery.Builder builder = new S2ClosestEdgeQuery.Builder();
        builder.setConservativeMaxDistance(this.edgeSiteQueryRadiusChordAngle);
        builder.setIncludeInteriors(false);
        inputEdgeIndex.applyUpdates();
        S2ClosestEdgeQuery.Query query = builder.build(inputEdgeIndex);
        S2ClosestEdgeQuery.PointTarget newSiteTarget = new S2ClosestEdgeQuery.PointTarget(newSite);
        query.findClosestEdges(newSiteTarget, (distance, shapeId, edgeId) -> {
            S2Point v0 = this.inputVertices.get(this.inputEdges.getSrcId(edgeId));
            this.edgeSites.insertSiteIdByDistance(edgeId, newSiteId, v0);
            edgesToResnap.add(edgeId);
            return true;
        });
    }

    private S2Point getSeparationSite(S2Point siteToAvoid, S2Point v0, S2Point v1, int inputEdgeId) {
        S2Point x = this.inputVertices.get(this.inputEdges.getSrcId(inputEdgeId));
        S2Point y = this.inputVertices.get(this.inputEdges.getDstId(inputEdgeId));
        S2Point xyDir = y.sub(x);
        S2Point n = S2RobustCrossProd.robustCrossProd(x, y);
        S2Point newSite = S2EdgeUtil.project(siteToAvoid, x, y, n);
        S2Point gapMin = this.getCoverageEndpoint(v0, n);
        S2Point gapMax = this.getCoverageEndpoint(v1, n.neg());
        if (newSite.sub(gapMin).dotProd(xyDir) < 0.0) {
            newSite = gapMin;
        } else if (gapMax.sub(newSite).dotProd(xyDir) < 0.0) {
            newSite = gapMax;
        }
        newSite = this.snapSite(newSite);
        assert (!v0.equalsPoint(newSite));
        assert (!v1.equalsPoint(newSite));
        return newSite;
    }

    private S2Point getCoverageEndpoint(S2Point p, S2Point n) {
        double n2 = n.norm2();
        double nDp = n.dotProd(p);
        S2Point nXp = n.crossProd(p);
        S2Point nXpXn = p.mul(n2).sub(n.mul(nDp));
        S2Point om = nXpXn.mul(Math.sqrt(1.0 - this.edgeSnapRadiusSin2));
        double mr2 = this.edgeSnapRadiusSin2 * n2 - nDp * nDp;
        S2Point mr = nXp.mul(Math.sqrt(Math.max(0.0, mr2)));
        return om.add(mr).normalize();
    }

    private void snapEdge(int inputEdgeId, IntVector siteIdChain) {
        siteIdChain.clear();
        int srcSiteId = this.inputEdges.getSrcId(inputEdgeId);
        int dstSiteId = this.inputEdges.getDstId(inputEdgeId);
        if (!this.snappingNeeded) {
            siteIdChain.push(srcSiteId);
            siteIdChain.push(dstSiteId);
            return;
        }
        S2Point x = this.inputVertices.get(srcSiteId);
        S2Point y = this.inputVertices.get(dstSiteId);
        Ints.IntList candidates = this.edgeSites.get(inputEdgeId);
        Ints.OfInt siteIter = candidates.intIterator();
        while (siteIter.hasNext()) {
            S2Point b;
            S1ChordAngle bc;
            int siteId = siteIter.nextInt();
            S2Point c = this.sites.get(siteId);
            if (S2Predicates.compareEdgeDistance(c, x, y, this.edgeSnapRadiusChordAngle.getLength2()) > 0) continue;
            boolean addSiteC = true;
            while (!siteIdChain.isEmpty() && !(bc = new S1ChordAngle(b = this.sites.get(siteIdChain.peek()), c)).greaterOrEquals(this.maxAdjacentSiteSeparationChordAngle)) {
                S2Predicates.Excluded result = S2Predicates.getVoronoiSiteExclusion(b, c, x, y, this.edgeSnapRadiusChordAngle.getLength2());
                if (result != S2Predicates.Excluded.FIRST) {
                    S2Point a;
                    S1ChordAngle ac;
                    if (result == S2Predicates.Excluded.SECOND) {
                        addSiteC = false;
                        break;
                    }
                    assert (result == S2Predicates.Excluded.NEITHER);
                    if (siteIdChain.size() < 2 || (ac = new S1ChordAngle(a = this.sites.get(siteIdChain.get(siteIdChain.size() - 2)), c)).greaterOrEquals(this.maxAdjacentSiteSeparationChordAngle)) break;
                    int xyb = S2Predicates.sign(x, y, b);
                    if (S2Predicates.sign(a, b, c) == xyb || S2Predicates.edgeCircumcenterSign(x, y, a, b, c) != xyb) break;
                }
                siteIdChain.pop();
            }
            if (!addSiteC) continue;
            siteIdChain.push(siteId);
        }
        assert (!siteIdChain.isEmpty());
        S2Point lastSite = this.sites.get(siteIdChain.peek());
        assert (this.checkSnappingInvariant(y, lastSite, candidates));
    }

    private boolean checkSnappingInvariant(S2Point endpoint, S2Point lastSite, Ints.IntList candidates) {
        Ints.OfInt i = candidates.intIterator();
        while (i.hasNext()) {
            if (S2Predicates.compareDistances(endpoint, lastSite, this.sites.get(i.nextInt())) <= 0) continue;
            return false;
        }
        return true;
    }

    private void buildLayers() {
        ArrayList<S2BuilderGraph.EdgeList> layerEdges = new ArrayList<S2BuilderGraph.EdgeList>();
        ArrayList<IntVector> layerInputEdgeIdSetIds = new ArrayList<IntVector>();
        IdSetLexicon inputEdgeIdSetLexicon = new IdSetLexicon();
        ArrayList<ArrayList<S2Point>> layerVertices = null;
        this.buildLayerEdges(layerEdges, layerInputEdgeIdSetIds, inputEdgeIdSetLexicon);
        int kMinLayersForVertexFiltering = 10;
        if (this.layers.size() >= 10) {
            boolean allowVertexFiltering = true;
            for (GraphOptions options : this.layerOptions) {
                allowVertexFiltering &= options.allowVertexFiltering();
            }
            if (allowVertexFiltering) {
                layerVertices = new ArrayList<ArrayList<S2Point>>();
                layerVertices.ensureCapacity(this.layers.size());
                IntVector tmp1 = new IntVector();
                IntVector tmp2 = new IntVector();
                for (int i = 0; i < this.layers.size(); ++i) {
                    layerVertices.add(S2BuilderGraph.filterVertices(this.sites, layerEdges.get(i), tmp1, tmp2));
                }
            }
        }
        for (int i = 0; i < this.layers.size(); ++i) {
            List vertices = layerVertices == null ? this.sites : (List)layerVertices.get(i);
            S2BuilderGraph graph = new S2BuilderGraph(this.layerOptions.get(i), vertices, layerEdges.get(i), layerInputEdgeIdSetIds.get(i), inputEdgeIdSetLexicon, this.labelSetIds, this.labelSetLexicon, this.layerIsFullPolygonPredicates.get(i));
            boolean bl = this.layers.get(i).build(graph, this.error);
        }
    }

    private void buildLayerEdges(ArrayList<S2BuilderGraph.EdgeList> layerEdges, ArrayList<IntVector> layerInputEdgeIdSetIds, IdSetLexicon inputEdgeIdSetLexicon) {
        int i;
        boolean simplify = this.snappingNeeded && this.options.simplifyEdgeChains();
        ArrayList<IntVector> siteVertices = new ArrayList<IntVector>();
        if (simplify) {
            for (i = 0; i < this.sites.size(); ++i) {
                siteVertices.add(new IntVector());
            }
        }
        for (i = 0; i < this.layers.size(); ++i) {
            layerEdges.add(new S2BuilderGraph.EdgeList());
            layerInputEdgeIdSetIds.add(new IntVector());
        }
        for (int layer = 0; layer < this.layers.size(); ++layer) {
            this.addSnappedEdges(this.layerBegins.get(layer), this.layerBegins.get(layer + 1), this.layerOptions.get(layer), layerEdges.get(layer), layerInputEdgeIdSetIds.get(layer), inputEdgeIdSetLexicon, siteVertices);
        }
        if (simplify) {
            this.simplifyEdgeChains(siteVertices, layerEdges, layerInputEdgeIdSetIds, inputEdgeIdSetLexicon);
            siteVertices.clear();
        }
        this.edgeSites.clear();
        for (i = 0; i < this.layers.size(); ++i) {
            S2BuilderGraph.processEdges(this.layerOptions.get(i), layerEdges.get(i), layerInputEdgeIdSetIds.get(i), inputEdgeIdSetLexicon, this.error);
        }
    }

    private void simplifyEdgeChains(ArrayList<IntVector> siteVertices, ArrayList<S2BuilderGraph.EdgeList> layerEdges, ArrayList<IntVector> layerInputEdgeIdSetIds, IdSetLexicon inputEdgeIdSetLexicon) {
        if (this.layers.isEmpty()) {
            return;
        }
        S2BuilderGraph.EdgeList mergedEdges = new S2BuilderGraph.EdgeList();
        IntVector mergedInputEdgeIdSetIds = new IntVector();
        IntVector mergedEdgeLayers = new IntVector();
        this.mergeLayerEdges(layerEdges, layerInputEdgeIdSetIds, mergedEdges, mergedInputEdgeIdSetIds, mergedEdgeLayers);
        for (S2BuilderGraph.EdgeList edges : layerEdges) {
            edges.clear();
        }
        for (IntVector inputEdgeIds : layerInputEdgeIdSetIds) {
            inputEdgeIds.clear();
        }
        GraphOptions graphOptions = new GraphOptions(EdgeType.DIRECTED, GraphOptions.DegenerateEdges.KEEP, GraphOptions.DuplicateEdges.KEEP, GraphOptions.SiblingPairs.KEEP);
        S2BuilderGraph mergedGraph = new S2BuilderGraph(graphOptions, this.sites, mergedEdges, mergedInputEdgeIdSetIds, inputEdgeIdSetLexicon, null, null, S2Builder.isFullPolygon(false));
        EdgeChainSimplifier simplifier = new EdgeChainSimplifier(this, mergedGraph, mergedEdgeLayers, siteVertices, layerEdges, layerInputEdgeIdSetIds, inputEdgeIdSetLexicon);
        simplifier.run();
    }

    private void mergeLayerEdges(ArrayList<S2BuilderGraph.EdgeList> layerEdges, ArrayList<IntVector> layerInputEdgeIdSetIds, S2BuilderGraph.EdgeList mergedEdges, IntVector mergedInputEdgeIdSetIds, IntVector mergedEdgeLayers) {
        LayerEdgeIdList order = new LayerEdgeIdList(layerEdges);
        for (int layer = 0; layer < layerEdges.size(); ++layer) {
            for (int edgeIndex = 0; edgeIndex < layerEdges.get(layer).size(); ++edgeIndex) {
                order.add(layer, edgeIndex);
            }
        }
        order.sort();
        mergedEdges.ensureCapacity(order.size());
        mergedInputEdgeIdSetIds.ensureCapacity(order.size());
        mergedEdgeLayers.ensureCapacity(order.size());
        for (int i = 0; i < order.size(); ++i) {
            mergedEdges.add(order.getSrcId(i), order.getDstId(i));
            mergedInputEdgeIdSetIds.add(layerInputEdgeIdSetIds.get(order.getLayerId(i)).get(order.getEdgeId(i)));
            mergedEdgeLayers.add(order.getLayerId(i));
        }
    }

    private void addSnappedEdges(int beginInputEdgeId, int endInputEdgeId, GraphOptions options, S2BuilderGraph.EdgeList edges, IntVector inputEdgeIdSetIds, IdSetLexicon inputEdgeIdSetLexicon, ArrayList<IntVector> siteVertices) {
        boolean discardDegenerateEdges = options.degenerateEdges() == GraphOptions.DegenerateEdges.DISCARD;
        IntVector siteIdChain = new IntVector();
        for (int inputEdgeId = beginInputEdgeId; inputEdgeId < endInputEdgeId; ++inputEdgeId) {
            int inputEdgeIdSetId = inputEdgeIdSetLexicon.addSingleton(inputEdgeId);
            this.snapEdge(inputEdgeId, siteIdChain);
            if (siteIdChain.isEmpty()) continue;
            this.maybeAddInputVertex(this.inputEdges.getSrcId(inputEdgeId), siteIdChain.get(0), siteVertices);
            if (siteIdChain.size() == 1) {
                if (discardDegenerateEdges) continue;
                this.addSnappedEdge(siteIdChain.get(0), siteIdChain.get(0), inputEdgeIdSetId, options.edgeType(), edges, inputEdgeIdSetIds);
                continue;
            }
            this.maybeAddInputVertex(this.inputEdges.getDstId(inputEdgeId), siteIdChain.peek(), siteVertices);
            for (int i = 1; i < siteIdChain.size(); ++i) {
                this.addSnappedEdge(siteIdChain.get(i - 1), siteIdChain.get(i), inputEdgeIdSetId, options.edgeType(), edges, inputEdgeIdSetIds);
            }
        }
    }

    private void maybeAddInputVertex(int inputVertexId, int siteId, ArrayList<IntVector> siteVertices) {
        if (siteVertices.isEmpty()) {
            return;
        }
        IntVector vertices = siteVertices.get(siteId);
        if (vertices.isEmpty() || vertices.peek() != inputVertexId) {
            vertices.push(inputVertexId);
        }
    }

    private void addSnappedEdge(int srcSiteId, int dstSiteId, int inputEdgeIdSetId, EdgeType edgeType, S2BuilderGraph.EdgeList edges, IntVector inputEdgeIdSetIds) {
        edges.add(srcSiteId, dstSiteId);
        inputEdgeIdSetIds.add(inputEdgeIdSetId);
        if (edgeType == EdgeType.UNDIRECTED) {
            edges.add(dstSiteId, srcSiteId);
            inputEdgeIdSetIds.add(Integer.MIN_VALUE);
        }
    }

    public static interface IsFullPolygonPredicate {
        public boolean test(S2BuilderGraph var1);
    }

    public static class GraphOptions {
        private EdgeType edgeType;
        private DegenerateEdges degenerateEdges;
        private DuplicateEdges duplicateEdges;
        private SiblingPairs siblingPairs;
        private boolean allowVertexFiltering;

        public GraphOptions(EdgeType edgeType, DegenerateEdges degenerateEdges, DuplicateEdges duplicateEdges, SiblingPairs siblingPairs) {
            this.edgeType = edgeType;
            this.degenerateEdges = degenerateEdges;
            this.duplicateEdges = duplicateEdges;
            this.siblingPairs = siblingPairs;
            this.allowVertexFiltering = true;
        }

        public GraphOptions() {
            this.edgeType = EdgeType.DIRECTED;
            this.degenerateEdges = DegenerateEdges.KEEP;
            this.duplicateEdges = DuplicateEdges.KEEP;
            this.siblingPairs = SiblingPairs.KEEP;
            this.allowVertexFiltering = true;
        }

        public GraphOptions(GraphOptions other) {
            this.edgeType = other.edgeType();
            this.degenerateEdges = other.degenerateEdges();
            this.duplicateEdges = other.duplicateEdges();
            this.siblingPairs = other.siblingPairs();
            this.allowVertexFiltering = other.allowVertexFiltering();
        }

        public String toString() {
            return Strings.lenientFormat((String)"EdgeType %s, DegenerateEdges %s, DuplicateEdges %s, SiblingPairs %s, AllowVertexFiltering %s", (Object[])new Object[]{this.edgeType, this.degenerateEdges, this.duplicateEdges, this.siblingPairs, this.allowVertexFiltering});
        }

        public EdgeType edgeType() {
            return this.edgeType;
        }

        public void setEdgeType(EdgeType edgeType) {
            this.edgeType = edgeType;
        }

        public DegenerateEdges degenerateEdges() {
            return this.degenerateEdges;
        }

        public void setDegenerateEdges(DegenerateEdges degenerateEdges) {
            this.degenerateEdges = degenerateEdges;
        }

        public DuplicateEdges duplicateEdges() {
            return this.duplicateEdges;
        }

        public void setDuplicateEdges(DuplicateEdges duplicateEdges) {
            this.duplicateEdges = duplicateEdges;
        }

        public SiblingPairs siblingPairs() {
            return this.siblingPairs;
        }

        public void setSiblingPairs(SiblingPairs siblingPairs) {
            this.siblingPairs = siblingPairs;
        }

        public boolean allowVertexFiltering() {
            return this.allowVertexFiltering;
        }

        public void setAllowVertexFiltering(boolean allowVertexFiltering) {
            this.allowVertexFiltering = allowVertexFiltering;
        }

        public boolean equals(Object other) {
            if (!(other instanceof GraphOptions)) {
                return false;
            }
            GraphOptions otherGraphOptions = (GraphOptions)other;
            return this.edgeType == otherGraphOptions.edgeType && this.degenerateEdges == otherGraphOptions.degenerateEdges && this.duplicateEdges == otherGraphOptions.duplicateEdges && this.siblingPairs == otherGraphOptions.siblingPairs && this.allowVertexFiltering == otherGraphOptions.allowVertexFiltering;
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.edgeType, this.degenerateEdges, this.duplicateEdges, this.siblingPairs, this.allowVertexFiltering});
        }

        public static enum SiblingPairs {
            DISCARD,
            DISCARD_EXCESS,
            KEEP,
            REQUIRE,
            CREATE;

        }

        public static enum DuplicateEdges {
            MERGE,
            KEEP;

        }

        public static enum DegenerateEdges {
            DISCARD,
            DISCARD_EXCESS,
            KEEP;

        }
    }

    public static class Builder {
        private SnapFunction snapFunction = new S2BuilderSnapFunctions.IdentitySnapFunction(S1Angle.ZERO);
        private boolean splitCrossingEdges = false;
        private S1Angle intersectionTolerance = S1Angle.ZERO;
        private boolean simplifyEdgeChains = false;
        private boolean idempotent = true;

        public Builder() {
        }

        public Builder(Builder options) {
            this.setSnapFunction(options.snapFunction);
            this.setIdempotent(options.idempotent);
            this.setSimplifyEdgeChains(options.simplifyEdgeChains);
            this.setSplitCrossingEdges(options.splitCrossingEdges);
            this.setIntersectionTolerance(options.intersectionTolerance);
        }

        public Builder(SnapFunction snapFunction) {
            this.setSnapFunction(snapFunction);
        }

        public S2Builder build() {
            return new S2Builder(this);
        }

        @CanIgnoreReturnValue
        public Builder setSnapFunction(SnapFunction snapFunction) {
            this.snapFunction = snapFunction;
            return this;
        }

        public SnapFunction snapFunction() {
            return this.snapFunction;
        }

        public S1Angle edgeSnapRadius() {
            return this.snapFunction().snapRadius().add(this.intersectionTolerance());
        }

        public S1Angle maxEdgeDeviation() {
            Preconditions.checkState((boolean)this.snapFunction().snapRadius().lessOrEquals(S2BuilderSnapFunctions.maxSnapRadius()));
            double kMaxEdgeDeviationRatio = 1.1;
            return this.edgeSnapRadius().mul(1.1);
        }

        public boolean splitCrossingEdges() {
            return this.splitCrossingEdges;
        }

        @CanIgnoreReturnValue
        public Builder setSplitCrossingEdges(boolean splitCrossingEdges) {
            this.splitCrossingEdges = splitCrossingEdges;
            return this;
        }

        public S1Angle intersectionTolerance() {
            if (!this.splitCrossingEdges()) {
                return this.intersectionTolerance;
            }
            return S1Angle.max(this.intersectionTolerance, S1Angle.radians(1.7763568394002505E-15));
        }

        @CanIgnoreReturnValue
        public Builder setIntersectionTolerance(S1Angle intersectionTolerance) {
            Preconditions.checkArgument((boolean)intersectionTolerance.greaterThan(S1Angle.ZERO));
            this.intersectionTolerance = intersectionTolerance;
            return this;
        }

        public boolean simplifyEdgeChains() {
            return this.simplifyEdgeChains;
        }

        @CanIgnoreReturnValue
        public Builder setSimplifyEdgeChains(boolean simplifyEdgeChains) {
            this.simplifyEdgeChains = simplifyEdgeChains;
            this.setIdempotent(false);
            return this;
        }

        public boolean idempotent() {
            return this.idempotent;
        }

        @CanIgnoreReturnValue
        public Builder setIdempotent(boolean idempotent) {
            this.idempotent = idempotent;
            return this;
        }
    }

    public static interface SnapFunction {
        public S1Angle snapRadius();

        public S1Angle minVertexSeparation();

        public S1Angle minEdgeVertexSeparation();

        public S2Point snapPoint(S2Point var1);
    }

    public static enum EdgeType {
        DIRECTED,
        UNDIRECTED;

    }

    private static class LayerEdgeIdList
    implements Sorter.SortableCollection {
        private final ArrayList<S2BuilderGraph.EdgeList> layerEdges;
        private final IntVector layerIds = new IntVector();
        private final IntVector edgeIds = new IntVector();

        public LayerEdgeIdList(ArrayList<S2BuilderGraph.EdgeList> layerEdges) {
            this.layerEdges = layerEdges;
        }

        public void add(int layerId, int edgeId) {
            this.layerIds.add(layerId);
            this.edgeIds.add(edgeId);
        }

        @Override
        public int size() {
            return this.layerIds.size();
        }

        @Override
        public void truncate(int start) {
            this.layerIds.truncate(start);
            this.edgeIds.truncate(start);
        }

        public int getLayerId(int index) {
            return this.layerIds.get(index);
        }

        public int getEdgeId(int index) {
            return this.edgeIds.get(index);
        }

        public int getSrcId(int index) {
            return this.layerEdges.get(this.layerIds.get(index)).getSrcId(this.edgeIds.get(index));
        }

        public int getDstId(int index) {
            return this.layerEdges.get(this.layerIds.get(index)).getDstId(this.edgeIds.get(index));
        }

        @Override
        public boolean less(int indexA, int indexB) {
            int dstB;
            int srcB;
            int layerA = this.layerIds.get(indexA);
            int layerB = this.layerIds.get(indexB);
            int edgeIdA = this.edgeIds.get(indexA);
            int edgeIdB = this.edgeIds.get(indexB);
            S2BuilderGraph.EdgeList layerEdgesA = this.layerEdges.get(layerA);
            S2BuilderGraph.EdgeList layerEdgesB = this.layerEdges.get(layerB);
            int srcA = layerEdgesA.getSrcId(edgeIdA);
            if (srcA < (srcB = layerEdgesB.getSrcId(edgeIdB))) {
                return true;
            }
            if (srcB < srcA) {
                return false;
            }
            int dstA = layerEdgesA.getDstId(edgeIdA);
            if (dstA < (dstB = layerEdgesB.getDstId(edgeIdB))) {
                return true;
            }
            if (dstB < dstA) {
                return false;
            }
            if (layerA < layerB) {
                return true;
            }
            if (layerB < layerA) {
                return false;
            }
            return edgeIdA < edgeIdB;
        }

        @Override
        public void swap(int indexA, int indexB) {
            this.layerIds.swap(indexA, indexB);
            this.edgeIds.swap(indexA, indexB);
        }
    }

    private static class EdgeChainSimplifier {
        private final S2Builder builder;
        private final S2BuilderGraph graph;
        private final S2BuilderGraph.VertexInMap vertexInMap;
        private final S2BuilderGraph.VertexOutMap vertexOutMap;
        private final IntVector edgeLayers;
        private final List<IntVector> siteVertices;
        private final List<S2BuilderGraph.EdgeList> layerEdges;
        private final List<IntVector> layerInputEdgeIdSetIds;
        private final IdSetLexicon inputEdgeIdSetLexicon;
        private final IntVector layerBegins;
        private static final int TRUE = 1;
        private static final int FALSE = 0;
        private final IntVector isInterior;
        private final IntVector used;
        private final IntVector tmpVertices = new IntVector();
        private final IntVector tmpEdges = new IntVector();
        private final IntSet usedVertices = new IntOpenHashSet();
        private final S2BuilderGraph.EdgeList newEdges = new S2BuilderGraph.EdgeList();
        private final IntVector newInputEdgeIdSetIds = new IntVector();
        private final IntVector newEdgeLayers = new IntVector();

        public EdgeChainSimplifier(S2Builder builder, S2BuilderGraph graph, IntVector edgeLayers, List<IntVector> siteVertices, List<S2BuilderGraph.EdgeList> layerEdges, List<IntVector> layerInputEdgeIdSetIds, IdSetLexicon inputEdgeIdSetLexicon) {
            this.builder = builder;
            this.graph = graph;
            this.edgeLayers = edgeLayers;
            this.siteVertices = siteVertices;
            this.vertexInMap = new S2BuilderGraph.VertexInMap(graph);
            this.vertexOutMap = new S2BuilderGraph.VertexOutMap(graph);
            this.layerEdges = layerEdges;
            this.layerInputEdgeIdSetIds = layerInputEdgeIdSetIds;
            this.inputEdgeIdSetLexicon = inputEdgeIdSetLexicon;
            this.layerBegins = builder.layerBegins;
            this.isInterior = IntVector.ofSize(graph.numVertices());
            this.used = IntVector.ofSize(graph.numEdges());
            this.newEdges.ensureCapacity(graph.numEdges());
            this.newInputEdgeIdSetIds.ensureCapacity(graph.numEdges());
            this.newEdgeLayers.ensureCapacity(graph.numEdges());
        }

        public void run() {
            int dstVertex;
            int srcVertex;
            int edgeId;
            for (int vertexId = 0; vertexId < this.graph.numVertices(); ++vertexId) {
                this.isInterior.set(vertexId, this.isInterior(vertexId));
            }
            for (edgeId = 0; edgeId < this.graph.numEdges(); ++edgeId) {
                srcVertex = this.graph.edgeSrcId(edgeId);
                dstVertex = this.graph.edgeDstId(edgeId);
                if (this.used.get(edgeId) == 1 || this.isInterior.get(srcVertex) == 1) continue;
                if (this.isInterior.get(dstVertex) == 0) {
                    this.outputEdge(edgeId);
                    continue;
                }
                this.simplifyChain(srcVertex, dstVertex);
            }
            for (edgeId = 0; edgeId < this.graph.numEdges(); ++edgeId) {
                if (this.used.get(edgeId) == 1) continue;
                srcVertex = this.graph.edgeSrcId(edgeId);
                if (srcVertex == (dstVertex = this.graph.edgeDstId(edgeId))) {
                    this.outputEdge(edgeId);
                    continue;
                }
                this.simplifyChain(srcVertex, dstVertex);
            }
            for (int index = 0; index < this.newEdges.size(); ++index) {
                int layer = this.newEdgeLayers.get(index);
                int src = this.newEdges.getSrcId(index);
                int dst = this.newEdges.getDstId(index);
                int newInputEdgeIdSetId = this.newInputEdgeIdSetIds.get(index);
                this.layerEdges.get(layer).add(src, dst);
                this.layerInputEdgeIdSetIds.get(layer).add(newInputEdgeIdSetId);
            }
        }

        private void outputEdge(int edgeId) {
            this.newEdges.add(this.graph.edges().getSrcId(edgeId), this.graph.edges().getDstId(edgeId));
            this.newInputEdgeIdSetIds.push(this.graph.inputEdgeIdSetId(edgeId));
            this.newEdgeLayers.push(this.edgeLayers.get(edgeId));
            this.used.set(edgeId, 1);
        }

        private int graphEdgeLayer(int edgeId) {
            return this.edgeLayers.get(edgeId);
        }

        private int inputEdgeLayer(int inputEdgeId) {
            assert (inputEdgeId >= 0);
            return this.layerBegins.upperBound(inputEdgeId) - (this.layerBegins.get(0) + 1);
        }

        private int isInterior(int vertexId) {
            if (this.vertexOutMap.outDegree(vertexId) == 0) {
                return 0;
            }
            if (this.vertexOutMap.outDegree(vertexId) != this.vertexInMap.inDegree(vertexId)) {
                return 0;
            }
            if (this.builder.isForced(vertexId)) {
                return 0;
            }
            IntVector edgeIds = this.tmpEdges;
            edgeIds.clear();
            this.vertexOutMap.edgeIds(vertexId).forEach(edgeIds::push);
            this.vertexInMap.edgeIds(vertexId).forEach(edgeIds::push);
            edgeIds.sort((x, y) -> Integer.compare(this.graphEdgeLayer(x), this.graphEdgeLayer(y)));
            InteriorVertexMatcher matcher = new InteriorVertexMatcher(vertexId);
            int e = 0;
            while (e < edgeIds.size()) {
                int edgeId = edgeIds.get(e);
                int layer = this.graphEdgeLayer(edgeId);
                matcher.startLayer();
                while (e < edgeIds.size() && this.graphEdgeLayer(edgeIds.get(e)) == layer) {
                    edgeId = edgeIds.get(e);
                    int srcVertex = this.graph.edgeSrcId(edgeId);
                    int dstVertex = this.graph.edgeDstId(edgeId);
                    if (srcVertex == vertexId) {
                        matcher.tally(dstVertex, true);
                    }
                    if (dstVertex == vertexId) {
                        matcher.tally(srcVertex, false);
                    }
                    ++e;
                }
                if (matcher.matches()) continue;
                return 0;
            }
            return 1;
        }

        private void simplifyChain(int v0, int v1) {
            IntVector chain = this.tmpVertices;
            S2PolylineSimplifier simplifier = new S2PolylineSimplifier();
            int vStart = v0;
            boolean done = false;
            do {
                this.usedVertices.clear();
                chain.push(v0);
                this.usedVertices.add(v0);
                simplifier.init(this.graph.vertex(v0));
                boolean simplify = this.avoidSites(v0, v0, v1, this.usedVertices, simplifier);
                do {
                    chain.push(v1);
                    this.usedVertices.add(v1);
                    boolean bl = done = this.isInterior.get(v1) == 0 || v1 == vStart;
                    if (done) break;
                    int vPrev = v0;
                    v0 = v1;
                    v1 = this.followChain(vPrev, v0);
                } while (simplify && this.targetInputVertices(v0, simplifier) && this.avoidSites(chain.get(0), v0, v1, this.usedVertices, simplifier) && simplifier.extend(this.graph.vertex(v1)));
                if (chain.size() == 2) {
                    this.outputAllEdges(chain.get(0), chain.get(1));
                } else {
                    this.mergeChain(chain);
                }
                chain.clear();
            } while (!done);
        }

        private int followChain(int v0, int v1) {
            assert (this.isInterior.get(v1) == 1);
            Ints.OfInt edgeIter = this.vertexOutMap.edgeIds(v1).intIterator();
            while (edgeIter.hasNext()) {
                int edgeId = edgeIter.nextInt();
                int edgeDstVertex = this.graph.edgeDstId(edgeId);
                if (edgeDstVertex == v0 || edgeDstVertex == v1) continue;
                return edgeDstVertex;
            }
            throw new IllegalStateException("Could not find next edge in edge chain");
        }

        private void outputAllEdges(int v0, int v1) {
            this.vertexOutMap.edgeIds(v0, v1).forEach(this::outputEdge);
            this.vertexOutMap.edgeIds(v1, v0).forEach(this::outputEdge);
        }

        private boolean targetInputVertices(int vertexId, S2PolylineSimplifier simplifier) {
            Ints.OfInt iter = this.siteVertices.get(vertexId).intIterator();
            while (iter.hasNext()) {
                int inputVertexId = iter.nextInt();
                if (simplifier.targetDisc(this.builder.inputVertices.get(inputVertexId), this.builder.edgeSnapRadiusChordAngle)) continue;
                return false;
            }
            return true;
        }

        private boolean avoidSites(int v0, int v1, int v2, IntSet usedVertices, S2PolylineSimplifier simplifier) {
            int inputEdgeId;
            Ints.OfInt inputEdgeIter;
            int edgeId;
            S1ChordAngle r1;
            S2Point p0 = this.graph.vertex(v0);
            S2Point p1 = this.graph.vertex(v1);
            S2Point p2 = this.graph.vertex(v2);
            S1ChordAngle r2 = new S1ChordAngle(p0, p2);
            if (r2.lessThan(r1 = new S1ChordAngle(p0, p1))) {
                return false;
            }
            if (r2.greaterOrEquals(this.builder.minEdgeLengthToSplitChordAngle)) {
                return false;
            }
            int bestInputEdgeId = -1;
            EdgeSites edgeSites = this.builder.edgeSites;
            Ints.OfInt edgeIter = this.vertexOutMap.edgeIds(v1, v2).intIterator();
            while (edgeIter.hasNext()) {
                edgeId = edgeIter.nextInt();
                inputEdgeIter = this.graph.inputEdgeIds(edgeId).intIterator();
                while (inputEdgeIter.hasNext()) {
                    inputEdgeId = inputEdgeIter.nextInt();
                    if (bestInputEdgeId >= 0 && edgeSites.get(inputEdgeId).size() >= edgeSites.get(bestInputEdgeId).size()) continue;
                    bestInputEdgeId = inputEdgeId;
                }
            }
            edgeIter = this.vertexOutMap.edgeIds(v2, v1).intIterator();
            while (edgeIter.hasNext()) {
                edgeId = edgeIter.nextInt();
                inputEdgeIter = this.graph.inputEdgeIds(edgeId).intIterator();
                while (inputEdgeIter.hasNext()) {
                    inputEdgeId = inputEdgeIter.nextInt();
                    if (bestInputEdgeId >= 0 && edgeSites.get(inputEdgeId).size() >= edgeSites.get(bestInputEdgeId).size()) continue;
                    bestInputEdgeId = inputEdgeId;
                }
            }
            assert (bestInputEdgeId >= 0);
            Ints.OfInt iter = edgeSites.get(bestInputEdgeId).intIterator();
            while (iter.hasNext()) {
                int vertexId = iter.nextInt();
                S2Point p = this.graph.vertex(vertexId);
                S1ChordAngle r = new S1ChordAngle(p0, p);
                if (r.greaterOrEquals(r2) || !usedVertices.add(vertexId)) continue;
                boolean discOnLeft = v1 == v0 ? S2Predicates.sign(p1, p2, p) > 0 : S2Predicates.orderedCCW(p0, p2, p, p1);
                if (simplifier.avoidDisc(p, this.builder.minEdgeSiteSeparationChordAngle, discOnLeft)) continue;
                return false;
            }
            return true;
        }

        private void clearMergedInputIds(List<IntVector> mergedInputIds, int entries, int capacity) {
            for (IntVector vec : mergedInputIds) {
                vec.ensureCapacity(capacity);
                vec.clear();
            }
            for (int j = mergedInputIds.size(); j < entries; ++j) {
                mergedInputIds.add(IntVector.ofCapacity(capacity));
            }
        }

        private void mergeChain(IntVector vertices) {
            ArrayList<IntVector> mergedInputIds = new ArrayList<IntVector>();
            IntVector degenerateIds = new IntVector();
            for (int i = 1; i < vertices.size(); ++i) {
                Ints.OfInt idIter;
                IntVector merged;
                int edgeId2;
                int v0 = vertices.get(i - 1);
                int v1 = vertices.get(i);
                S2BuilderGraph.VertexOutEdgeIds v0v1Edges = this.vertexOutMap.edgeIds(v0, v1);
                S2BuilderGraph.VertexOutEdgeIds v1v0Edges = this.vertexOutMap.edgeIds(v1, v0);
                if (i == 1) {
                    int numEdges = v0v1Edges.size() + v1v0Edges.size();
                    this.clearMergedInputIds(mergedInputIds, numEdges, vertices.size() - 1);
                } else {
                    assert (this.isInterior.get(v0) == 1);
                    this.vertexOutMap.edgeIds(v0, v0).forEach(edgeId -> {
                        this.graph.inputEdgeIds(edgeId).forEach(degenerateIds::add);
                        this.used.set(edgeId, 1);
                    });
                }
                int j = 0;
                Ints.OfInt edgeIter = v0v1Edges.intIterator();
                while (edgeIter.hasNext()) {
                    edgeId2 = edgeIter.nextInt();
                    merged = (IntVector)mergedInputIds.get(j);
                    idIter = this.graph.inputEdgeIds(edgeId2).intIterator();
                    while (idIter.hasNext()) {
                        merged.add(idIter.nextInt());
                    }
                    this.used.set(edgeId2, 1);
                    ++j;
                }
                edgeIter = v1v0Edges.intIterator();
                while (edgeIter.hasNext()) {
                    edgeId2 = edgeIter.nextInt();
                    merged = (IntVector)mergedInputIds.get(j);
                    idIter = this.graph.inputEdgeIds(edgeId2).intIterator();
                    while (idIter.hasNext()) {
                        merged.add(idIter.nextInt());
                    }
                    this.used.set(edgeId2, 1);
                    ++j;
                }
                assert (mergedInputIds.size() == j);
            }
            if (!degenerateIds.isEmpty()) {
                degenerateIds.sort();
                this.assignDegenerateEdges(degenerateIds, mergedInputIds);
            }
            int v0 = vertices.get(0);
            int v1 = vertices.get(1);
            int vb = vertices.peek();
            this.vertexOutMap.edgeIds(v0, v1).forEach(edgeId -> {
                this.newEdges.add(v0, vb);
                this.newEdgeLayers.add(this.graphEdgeLayer(edgeId));
            });
            this.vertexOutMap.edgeIds(v1, v0).forEach(edgeId -> {
                this.newEdges.add(vb, v0);
                this.newEdgeLayers.add(this.graphEdgeLayer(edgeId));
            });
            for (IntVector ids : mergedInputIds) {
                this.newInputEdgeIdSetIds.add(this.inputEdgeIdSetLexicon.add(ids));
            }
        }

        private void assignDegenerateEdges(Ints.IntList degenerateIds, List<IntVector> mergedIds) {
            for (IntVector ids : mergedIds) {
                ids.sort();
            }
            IntVector order = IntVector.ofCapacity(mergedIds.size());
            for (int i2 = 0; i2 < mergedIds.size(); ++i2) {
                if (mergedIds.get(i2).isEmpty()) continue;
                order.add(i2);
            }
            order.sort((i, j) -> Integer.compare(((IntVector)mergedIds.get(i)).get(0), ((IntVector)mergedIds.get(j)).get(0)));
            Ints.OfInt degenerateEdgeIter = degenerateIds.intIterator();
            while (degenerateEdgeIter.hasNext()) {
                int degenerateEdgeId = degenerateEdgeIter.nextInt();
                int layer = this.inputEdgeLayer(degenerateEdgeId);
                int index = order.upperBound(degenerateEdgeId, (x, y) -> Integer.compare(x, ((IntVector)mergedIds.get(y)).get(0)));
                if (index != 0 && mergedIds.get(order.get(index - 1)).get(0) >= this.layerBegins.get(layer)) {
                    --index;
                }
                int edgeId = order.get(index);
                assert (layer == this.inputEdgeLayer(mergedIds.get(edgeId).get(0)));
                mergedIds.get(edgeId).add(degenerateEdgeId);
            }
        }

        private static class InteriorVertexMatcher {
            private final int v0;
            private int v1;
            private int v2;
            private int n0;
            private int n1;
            private int n2;
            private int excessOut;
            private boolean tooManyEndpoints;

            public InteriorVertexMatcher(int v0) {
                this.v0 = v0;
                this.v1 = -1;
                this.v2 = -2;
                this.n0 = 0;
                this.n1 = 0;
                this.n2 = 0;
                this.excessOut = 0;
                this.tooManyEndpoints = false;
            }

            public void startLayer() {
                this.excessOut = 0;
                this.n0 = 0;
                this.n1 = 0;
                this.n2 = 0;
            }

            public void tally(int v, boolean outgoing) {
                this.excessOut += outgoing ? 1 : -1;
                if (v == this.v0) {
                    ++this.n0;
                } else {
                    if (this.v1 < 0) {
                        this.v1 = v;
                    }
                    if (this.v1 == v) {
                        ++this.n1;
                    } else {
                        if (this.v2 < 0) {
                            this.v2 = v;
                        }
                        if (this.v2 == v) {
                            ++this.n2;
                        } else {
                            this.tooManyEndpoints = true;
                        }
                    }
                }
            }

            public boolean matches() {
                return !this.tooManyEndpoints && this.excessOut == 0 && this.n1 == this.n2 && (this.n0 == 0 || this.n1 > 0);
            }
        }
    }

    private class EdgeSites {
        private final HashMap<Integer, IntVector> map = new HashMap();

        public void clear() {
            this.map.clear();
        }

        public void add(int inputEdgeId, IntVector sitesNearEdge) {
            this.map.put(inputEdgeId, sitesNearEdge);
        }

        public void insertSiteIdByDistance(int edgeId, int newSiteId, S2Point x) {
            IntVector sitesNearEdge = this.map.get(edgeId);
            int pos = sitesNearEdge.lowerBound(newSiteId, new SiteIdDistanceComparator(x));
            Preconditions.checkState((pos == sitesNearEdge.size() || sitesNearEdge.get(pos) != newSiteId ? 1 : 0) != 0);
            sitesNearEdge.insert(pos, newSiteId);
        }

        public Ints.IntList get(int edgeId) {
            if (!this.map.containsKey(edgeId)) {
                this.map.put(edgeId, new IntVector());
            }
            return this.map.get(edgeId);
        }
    }

    private class SiteIdDistanceComparator
    implements Ints.IntComparator {
        private final S2Point xPoint;

        public SiteIdDistanceComparator(S2Point p) {
            this.xPoint = p;
        }

        @Override
        public int compare(int a, int b) {
            S2Point aPoint = S2Builder.this.sites.get(a);
            S2Point bPoint = S2Builder.this.sites.get(b);
            return S2Predicates.compareDistances(this.xPoint, aPoint, bPoint);
        }
    }
}

