/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.base.Preconditions;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multisets;
import com.google.common.collect.TreeMultiset;
import com.google.common.geometry.Platform;
import com.google.common.geometry.S2Builder;
import com.google.common.geometry.S2BuilderGraph;
import com.google.common.geometry.S2BuilderLayer;
import com.google.common.geometry.S2BuilderShapesLayer;
import com.google.common.geometry.S2Edge;
import com.google.common.geometry.S2Error;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Shape;
import com.google.common.geometry.S2ShapeIndex;
import com.google.common.geometry.S2TextFormat;
import com.google.common.geometry.primitives.IdSetLexicon;
import com.google.common.geometry.primitives.IntVector;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;

public class S2BuilderUtil {
    private S2BuilderUtil() {
    }

    static <T extends Comparable<T>> void deduplicateSortedList(List<T> input) {
        if (input.size() < 2) {
            return;
        }
        int dst = 0;
        for (int src = 1; src < input.size(); ++src) {
            if (((Comparable)input.get(src)).compareTo((Comparable)input.get(dst)) == 0) continue;
            input.set(++dst, (Comparable)input.get(src));
        }
        input.subList(dst + 1, input.size()).clear();
    }

    static class IndexMatchingLayer
    implements S2BuilderLayer {
        private final S2Builder.GraphOptions graphOptions;
        private final S2ShapeIndex expectedIndex;
        private final int dimension;
        private final double tolerance;
        private final Multiset<S2Edge> expectedEdges;
        private final Multiset<S2Edge> actualEdges;

        public IndexMatchingLayer(S2Builder.GraphOptions graphOptions, S2ShapeIndex index, int dimension, double tolerance) {
            this.graphOptions = new S2Builder.GraphOptions(graphOptions);
            this.expectedIndex = index;
            this.dimension = dimension;
            this.tolerance = tolerance;
            this.expectedEdges = TreeMultiset.create(this.tolerantEdgeComparator(tolerance));
            this.actualEdges = TreeMultiset.create(this.tolerantEdgeComparator(tolerance));
            this.computeExpectedEdges();
        }

        public IndexMatchingLayer(S2Builder.GraphOptions graphOptions, S2ShapeIndex index, int dimension) {
            this(graphOptions, index, dimension, 2.220446049250313E-16);
        }

        public IndexMatchingLayer(S2Builder.GraphOptions graphOptions, S2ShapeIndex index) {
            this(graphOptions, index, -1);
        }

        public Comparator<S2Edge> tolerantEdgeComparator(double tolerance) {
            return (left, right) -> {
                if (left.getStart().getDistance(right.getStart()) > tolerance) {
                    return left.getStart().compareTo(right.getStart());
                }
                if (left.getEnd().getDistance(right.getEnd()) > tolerance) {
                    return left.getEnd().compareTo(right.getEnd());
                }
                return 0;
            };
        }

        private void computeExpectedEdges() {
            for (S2Shape expectedShape : this.expectedIndex.getShapes()) {
                if (this.dimension >= 0 && expectedShape.dimension() != this.dimension) continue;
                S2Shape.MutableEdge mutableEdge = new S2Shape.MutableEdge();
                int e = expectedShape.numEdges();
                while (--e >= 0) {
                    expectedShape.getEdge(e, mutableEdge);
                    this.expectedEdges.add((Object)new S2Edge(mutableEdge.getStart(), mutableEdge.getEnd()));
                }
            }
        }

        private String toString(Collection<S2Edge> edges) {
            StringBuilder msg = new StringBuilder();
            for (S2Edge edge : edges) {
                if (msg.length() > 0) {
                    msg.append("; ");
                }
                msg.append("[").append(S2TextFormat.toString(edge.getStart())).append(", ").append(S2TextFormat.toString(edge.getEnd())).append("]");
            }
            return msg.toString();
        }

        public String toString() {
            return "IndexMatchingLayer for dimension " + this.dimension + " with tolerance=" + this.tolerance + ", and " + this.expectedEdges.size() + " expected edges: " + this.toString((Collection<S2Edge>)this.expectedEdges);
        }

        @Override
        public S2Builder.GraphOptions graphOptions() {
            return this.graphOptions;
        }

        @Override
        public boolean build(S2BuilderGraph g, S2Error error) {
            this.actualEdges.clear();
            for (int e = 0; e < g.numEdges(); ++e) {
                this.actualEdges.add((Object)new S2Edge(g.vertex(g.edges().getSrcId(e)), g.vertex(g.edges().getDstId(e))));
            }
            Multiset missingEdges = Multisets.difference(this.expectedEdges, this.actualEdges);
            Multiset extraEdges = Multisets.difference(this.actualEdges, this.expectedEdges);
            if (!missingEdges.isEmpty() || !extraEdges.isEmpty()) {
                String label = "";
                if (this.dimension >= 0) {
                    label = Platform.formatString("Dimension %d: ", this.dimension);
                }
                error.init(S2Error.Code.FAILED_PRECONDITION, "%s%s\n   Missing edges: %s\n   Extra edges: %s\n", error.text(), label, this.toString((Collection<S2Edge>)missingEdges), this.toString((Collection<S2Edge>)extraEdges));
            }
            return error.ok();
        }
    }

    public static class GraphCloningLayer
    implements S2BuilderLayer {
        private final S2Builder.GraphOptions graphOptions;
        private final GraphClone graphClone;

        public GraphCloningLayer(S2Builder.GraphOptions graphOptions, GraphClone graphClone) {
            this.graphOptions = graphOptions;
            this.graphClone = graphClone;
        }

        @Override
        public S2Builder.GraphOptions graphOptions() {
            return this.graphOptions;
        }

        @Override
        public boolean build(S2BuilderGraph graph, S2Error error) {
            this.graphClone.init(graph);
            return true;
        }

        public String toString() {
            return "GraphCloningLayer with GraphOptions " + this.graphOptions;
        }
    }

    public static class IndexedLayer<L extends S2BuilderShapesLayer>
    implements S2BuilderShapesLayer {
        private final S2ShapeIndex index;
        private final L layer;

        public IndexedLayer(S2ShapeIndex index, L wrappedLayer) {
            this.index = index;
            this.layer = wrappedLayer;
        }

        @Override
        public S2Builder.GraphOptions graphOptions() {
            return this.layer.graphOptions();
        }

        @Override
        public boolean build(S2BuilderGraph graph, S2Error error) {
            boolean success = this.layer.build(graph, error);
            if (success) {
                for (S2Shape s2Shape : this.layer.shapes()) {
                    if (s2Shape.isEmpty()) continue;
                    this.index.add(s2Shape);
                }
            }
            return success;
        }

        @Override
        public Iterable<? extends S2Shape> shapes() {
            return this.layer.shapes();
        }

        public String toString() {
            return "IndexedLayer wrapping a " + this.layer;
        }
    }

    static class GraphClone {
        private S2Builder.GraphOptions options;
        private ArrayList<S2Point> vertices;
        private S2BuilderGraph.EdgeList edges;
        private IntVector inputEdgeIdSetIds;
        private IdSetLexicon inputEdgeIdSetLexicon;
        private IntVector labelSetIds;
        private IdSetLexicon labelSetLexicon;
        private S2Builder.IsFullPolygonPredicate isFullPolygonPredicate;
        private S2BuilderGraph graph;

        public GraphClone() {
        }

        public GraphClone(S2BuilderGraph graph) {
            this.init(graph);
        }

        public void init(S2BuilderGraph graph) {
            this.options = new S2Builder.GraphOptions(graph.options());
            this.vertices = new ArrayList<S2Point>(graph.vertices());
            this.edges = new S2BuilderGraph.EdgeList(graph.edges());
            this.inputEdgeIdSetIds = IntVector.copyOf(graph.inputEdgeIdSetIds());
            this.inputEdgeIdSetLexicon = new IdSetLexicon(graph.inputEdgeIdSetLexicon());
            this.labelSetIds = IntVector.copyOf(graph.labelSetIds());
            this.labelSetLexicon = new IdSetLexicon(graph.labelSetLexicon());
            this.isFullPolygonPredicate = graph.isFullPolygonPredicate();
            this.graph = new S2BuilderGraph(this.options, this.vertices, this.edges, this.inputEdgeIdSetIds, this.inputEdgeIdSetLexicon, this.labelSetIds, this.labelSetLexicon, this.isFullPolygonPredicate);
        }

        public S2BuilderGraph graph() {
            return this.graph;
        }
    }

    static class GraphShape
    implements S2Shape {
        private final List<S2Point> vertices;
        private final S2BuilderGraph.EdgeList edges;

        public GraphShape(S2BuilderGraph graph) {
            this.edges = graph.edges();
            this.vertices = graph.vertices();
        }

        public GraphShape(S2BuilderGraph.EdgeList edges, List<S2Point> vertices) {
            this.edges = edges;
            this.vertices = vertices;
        }

        @Override
        public int dimension() {
            return 1;
        }

        @Override
        public int numEdges() {
            return this.edges.size();
        }

        @Override
        public void getEdge(int edgeId, S2Shape.MutableEdge result) {
            result.set(this.vertices.get(this.edges.getSrcId(edgeId)), this.vertices.get(this.edges.getDstId(edgeId)));
        }

        @Override
        public boolean hasInterior() {
            return false;
        }

        @Override
        public boolean containsOrigin() {
            return false;
        }

        @Override
        public int numChains() {
            return this.edges.size();
        }

        @Override
        public int getChainStart(int chainId) {
            Preconditions.checkElementIndex((int)chainId, (int)this.numChains());
            return chainId;
        }

        @Override
        public int getChainLength(int chainId) {
            Preconditions.checkElementIndex((int)chainId, (int)this.numChains());
            return 1;
        }

        @Override
        public void getChainEdge(int chainId, int offset, S2Shape.MutableEdge result) {
            Preconditions.checkElementIndex((int)offset, (int)this.getChainLength(chainId));
            this.getEdge(chainId, result);
        }

        @Override
        public S2Point getChainVertex(int chainId, int edgeOffset) {
            Preconditions.checkElementIndex((int)chainId, (int)this.numChains());
            Preconditions.checkElementIndex((int)edgeOffset, (int)2);
            return edgeOffset == 0 ? this.vertices.get(this.edges.getSrcId(chainId)) : this.vertices.get(this.edges.getDstId(chainId));
        }

        @Override
        public void getChainPosition(int edgeId, S2Shape.ChainPosition result) {
            Preconditions.checkElementIndex((int)edgeId, (int)this.numEdges());
            result.set(edgeId, 0);
        }
    }
}

