/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Streams;
import com.google.common.geometry.S2CellId;
import com.google.common.geometry.S2CellUnion;
import com.google.common.geometry.S2DensityTree;
import com.google.common.geometry.S2Region;
import com.google.common.geometry.S2RegionCoverer;
import com.google.common.geometry.S2RegionIntersection;
import java.util.Collection;
import java.util.Iterator;
import java.util.stream.Stream;

public class S2DensityClusterQuery {
    public static final S2RegionCoverer DEFAULT_COVERER = S2RegionCoverer.builder().setMaxCells(256).build();
    private final long minClusterWeight;
    private final long maxClusterWeight;

    public S2DensityClusterQuery(long desiredClusterWeight) {
        this(Math.max(1L, desiredClusterWeight * 8L / 10L), Math.max(1L, desiredClusterWeight * 12L / 10L));
    }

    public S2DensityClusterQuery(long minClusterWeight, long maxClusterWeight) {
        Preconditions.checkArgument((minClusterWeight > 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((minClusterWeight <= maxClusterWeight ? 1 : 0) != 0);
        this.minClusterWeight = minClusterWeight;
        this.maxClusterWeight = maxClusterWeight;
    }

    public long minClusterWeight() {
        return this.minClusterWeight;
    }

    public long maxClusterWeight() {
        return this.maxClusterWeight;
    }

    public ImmutableList<S2CellUnion> coverings(S2DensityTree density) {
        return (ImmutableList)Streams.stream(this.clusters(density)).map(Cluster::covering).collect(ImmutableList.toImmutableList());
    }

    public ImmutableList<S2CellUnion> tightCoverings(S2DensityTree density) {
        S2CellUnion leaves = density.getLeaves();
        leaves.normalize();
        return (ImmutableList)Streams.stream(this.clusters(density)).map(c -> c.intersection(leaves)).collect(ImmutableList.toImmutableList());
    }

    public ImmutableList<S2CellUnion> approximateCoverings(S2DensityTree density) {
        return this.approximateCoverings(DEFAULT_COVERER, density);
    }

    public ImmutableList<S2CellUnion> approximateCoverings(S2RegionCoverer coverer, S2DensityTree density) {
        return (ImmutableList)this.regions(density).map(coverer::getCovering).collect(ImmutableList.toImmutableList());
    }

    public ImmutableList<S2Region> clusterRegions(S2DensityTree density) {
        return (ImmutableList)this.regions(density).collect(ImmutableList.toImmutableList());
    }

    private Stream<S2Region> regions(S2DensityTree density) {
        S2Region region = density.asRegion();
        return Streams.stream(this.clusters(density)).map(c -> new S2RegionIntersection((Collection<S2Region>)ImmutableList.of((Object)c.covering(), (Object)region)));
    }

    public Iterable<Cluster> clusters(S2DensityTree density) {
        final S2DensityTree tree = density.normalize();
        return () -> new Iterator<Cluster>(){
            Cluster c;
            {
                this.c = S2DensityClusterQuery.this.next(tree, S2CellId.begin(30));
            }

            @Override
            public boolean hasNext() {
                return this.c.weight > 0L;
            }

            @Override
            public Cluster next() {
                Cluster result = this.c;
                this.c = S2DensityClusterQuery.this.next(tree, this.c.end);
                return result;
            }
        };
    }

    private Cluster next(S2DensityTree tree, S2CellId start) {
        Cluster cluster = new Cluster(start);
        tree.visitCells((cell, node) -> {
            if (cell.rangeMax().lessThan(cluster.begin)) {
                return S2DensityTree.CellVisitor.Action.SKIP_CELL;
            }
            CellInterpolator range = new CellInterpolator(cell);
            double ratio = node.hasChildren() ? 0.0 : Math.max(0.0, Math.min(1.0, range.uninterpolate(cluster.begin)));
            long sum = cluster.weight + (long)Math.ceil((1.0 - ratio) * (double)node.weight());
            if (sum < this.minClusterWeight) {
                cluster.weight = sum;
                return S2DensityTree.CellVisitor.Action.SKIP_CELL;
            }
            if (sum <= this.maxClusterWeight) {
                cluster.weight = sum;
                cluster.end = cell.rangeMax().next();
                return S2DensityTree.CellVisitor.Action.STOP;
            }
            if (node.hasChildren()) {
                return S2DensityTree.CellVisitor.Action.ENTER_CELL;
            }
            long missingWeight = (this.minClusterWeight + this.maxClusterWeight + 1L) / 2L - cluster.weight;
            cluster.weight += missingWeight;
            cluster.end = range.interpolate(ratio += 1.0 * (double)missingWeight / (double)node.weight()).rangeMin();
            return S2DensityTree.CellVisitor.Action.STOP;
        });
        return cluster;
    }

    public static final class Cluster {
        private S2CellId begin;
        private S2CellId end = S2CellId.end(30);
        private long weight;

        private Cluster(S2CellId begin) {
            Preconditions.checkArgument((begin.level() == 30 ? 1 : 0) != 0);
            this.begin = begin;
        }

        public S2CellId begin() {
            return this.begin;
        }

        public S2CellId end() {
            return this.end;
        }

        public S2CellUnion covering() {
            S2CellUnion covering = new S2CellUnion();
            covering.initFromBeginEnd(this.begin, this.end);
            return covering;
        }

        public S2CellUnion intersection(S2CellUnion covering) {
            S2CellUnion intersection = new S2CellUnion();
            intersection.getIntersection(covering, this.covering());
            return intersection;
        }

        public long weight() {
            return this.weight;
        }
    }

    @VisibleForTesting
    static class CellInterpolator {
        private final S2CellId parent;
        private final int level;
        private final long begin;
        private final long length;

        CellInterpolator(S2CellId parent) {
            this.parent = parent;
            this.level = Math.min(30, parent.level() + 26);
            this.begin = parent.childBegin(this.level).distanceFromBegin();
            this.length = parent.childEnd(this.level).distanceFromBegin() - this.begin;
        }

        public S2CellId interpolate(double ratio) {
            long steps = (long)Math.ceil(ratio * (double)this.length);
            return this.parent.childBegin(this.level).advance(steps);
        }

        public double uninterpolate(S2CellId child) {
            Preconditions.checkArgument((child.level() >= this.parent.level() ? 1 : 0) != 0);
            S2CellId adjusted = this.level <= child.level() ? child.parent(this.level) : child.childBegin(this.level);
            return 1.0 * (double)(adjusted.distanceFromBegin() - this.begin) / (double)this.length;
        }
    }
}

