/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.geometry.EncodedInts;
import com.google.common.geometry.Platform;
import com.google.common.geometry.PrimitiveArrays;
import com.google.common.geometry.S1Angle;
import com.google.common.geometry.S2Cap;
import com.google.common.geometry.S2Cell;
import com.google.common.geometry.S2CellId;
import com.google.common.geometry.S2CellIndex;
import com.google.common.geometry.S2CellUnion;
import com.google.common.geometry.S2Coder;
import com.google.common.geometry.S2LatLngRect;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Projections;
import com.google.common.geometry.S2Region;
import com.google.common.geometry.S2Shape;
import com.google.common.geometry.S2ShapeIndex;
import com.google.common.geometry.S2ShapeIndexRegion;
import com.google.common.geometry.S2ShapeUtil;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.ToLongFunction;
import jsinterop.annotations.JsEnum;
import jsinterop.annotations.JsIgnore;
import jsinterop.annotations.JsType;

@JsType
public class S2DensityTree {
    public static final String VERSION = "S2DensityTree0";
    private static final byte[] VERSION_BYTES = "S2DensityTree0".getBytes(StandardCharsets.ISO_8859_1);
    private static final byte[] REVERSED_VERSION_BYTES = new byte[VERSION_BYTES.length];
    public static final S2Coder<S2DensityTree> CODER;
    private static final int CHILD_MASK_BITS = 4;
    private static final int CHILD_MASK = 15;
    private static final long MAX_WEIGHT = 0x7FFFFFFFFFFFFFFL;
    private final PrimitiveArrays.Bytes encoded;
    private final long[] facePositions;

    private S2DensityTree(PrimitiveArrays.Bytes encoded, PrimitiveArrays.Cursor cursor) throws IOException {
        this.encoded = encoded;
        this.facePositions = S2DensityTree.decodeHeader(encoded, cursor);
    }

    public long size() {
        return this.encoded.length();
    }

    public PrimitiveArrays.Bytes encoded() {
        return this.encoded;
    }

    @JsIgnore
    public static S2DensityTree decode(PrimitiveArrays.Bytes bytes) throws IOException {
        return new S2DensityTree(bytes, bytes.cursor());
    }

    public static S2DensityTree decode(PrimitiveArrays.Bytes bytes, PrimitiveArrays.Cursor cursor) throws IOException {
        return new S2DensityTree(bytes, cursor);
    }

    public Map<S2CellId, Long> decode() {
        TreeMap<S2CellId, Long> results = new TreeMap<S2CellId, Long>();
        this.visitAll(results::put);
        return results;
    }

    public void visitAll(CellVisitor.All visitor) {
        this.visitCells(visitor);
    }

    public boolean visitWhile(CellVisitor.While visitor) {
        return this.visitCells((cell, node) -> visitor.visit(cell, node.weight) ? CellVisitor.Action.ENTER_CELL : CellVisitor.Action.STOP);
    }

    public S2Region asRegion() {
        final DecodedPath path = new DecodedPath(this);
        return new S2Region(){

            @Override
            public S2Cap getCapBound() {
                return S2Cap.full();
            }

            @Override
            public S2LatLngRect getRectBound() {
                return S2LatLngRect.full();
            }

            @Override
            public boolean mayIntersect(S2Cell cell) {
                Cell node = path.cell(cell.id());
                return !node.isEmpty();
            }

            @Override
            public boolean contains(S2Cell cell) {
                return this.contains(cell.id());
            }

            @Override
            public boolean contains(S2Point p) {
                return this.contains(S2CellId.fromPoint(p));
            }

            private boolean contains(S2CellId id) {
                Cell node = path.cell(id);
                return !node.isEmpty() && !node.hasChildren();
            }
        };
    }

    public S2DensityTree normalize() {
        HashMap<S2CellId, Long> normalizedWeights = new HashMap<S2CellId, Long>();
        DecodedPath path = new DecodedPath(this);
        this.visitAll((cell, weight) -> {
            if (!cell.isFace()) {
                S2CellId parent = cell.parent();
                long parentWeight = (Long)normalizedWeights.get(parent);
                long siblingWeight = 0L;
                for (int i = 0; i < 4; ++i) {
                    siblingWeight += path.weight(parent.child(i));
                }
                weight = parentWeight < Long.MAX_VALUE / weight ? (weight * parentWeight - 1L) / siblingWeight + 1L : BigInteger.valueOf(weight).multiply(BigInteger.valueOf(parentWeight)).subtract(BigInteger.ONE).divide(BigInteger.valueOf(siblingWeight)).add(BigInteger.ONE).longValue();
            }
            normalizedWeights.put(cell, weight);
        });
        TreeEncoder encoder = new TreeEncoder();
        normalizedWeights.forEach(encoder::put);
        return encoder.build();
    }

    public List<S2CellId> getPartitioning(int maxWeight) {
        ArrayList<S2CellId> clusters = new ArrayList<S2CellId>();
        this.visitCells((cell, node) -> {
            if (node.weight > (long)maxWeight && node.hasChildren()) {
                return CellVisitor.Action.ENTER_CELL;
            }
            clusters.add(cell);
            return CellVisitor.Action.SKIP_CELL;
        });
        return clusters;
    }

    public S2CellUnion getLeaves() {
        S2CellUnion result = new S2CellUnion();
        result.initRawCellIds(this.getLeavesAsList());
        return result;
    }

    public ArrayList<S2CellId> getLeavesAsList() {
        ArrayList<S2CellId> leaves = new ArrayList<S2CellId>();
        this.visitCells((cell, node) -> {
            if (node.hasChildren()) {
                return CellVisitor.Action.ENTER_CELL;
            }
            leaves.add(cell);
            return CellVisitor.Action.SKIP_CELL;
        });
        return leaves;
    }

    public int getMaxLevel() {
        int[] maxLevel = new int[1];
        this.visitCells((cell, node) -> {
            maxLevel[0] = Math.max(maxLevel[0], cell.level());
            return CellVisitor.Action.ENTER_CELL;
        });
        return maxLevel[0];
    }

    @CanIgnoreReturnValue
    public boolean visitCells(CellVisitor visitor) {
        PrimitiveArrays.Cursor cursor = this.encoded.cursor();
        Cell cell = new Cell();
        for (int i = 0; i < S2CellId.FACE_CELLS.length; ++i) {
            cursor.position = this.facePositions[i];
            if (cursor.position < 0L || this.visitCells(cell, cursor, S2CellId.FACE_CELLS[i], visitor)) continue;
            return false;
        }
        return true;
    }

    private boolean visitCells(Cell decoded, PrimitiveArrays.Cursor cursor, S2CellId cell, CellVisitor visitor) {
        decoded.decode(this.encoded, cursor);
        switch (visitor.visit(cell, decoded)) {
            case SKIP_CELL: {
                return true;
            }
            case ENTER_CELL: {
                int p0 = decoded.positions[0];
                int p1 = decoded.positions[1];
                int p2 = decoded.positions[2];
                int p3 = decoded.positions[3];
                return !(p0 >= 0 && !this.visitCells(decoded, cursor.seek(p0), cell.child(0), visitor) || p1 >= 0 && !this.visitCells(decoded, cursor.seek(p1), cell.child(1), visitor) || p2 >= 0 && !this.visitCells(decoded, cursor.seek(p2), cell.child(2), visitor) || p3 >= 0 && !this.visitCells(decoded, cursor.seek(p3), cell.child(3), visitor));
            }
            case STOP: {
                return false;
            }
        }
        throw new IllegalStateException("Unexpected next state");
    }

    public static S2DensityTree shapeDensity(S2ShapeIndex index, ShapeWeightFunction weigher, int approximateSizeBytes, int maxLevel) {
        return S2DensityTree.createShapeDensityOp(approximateSizeBytes, maxLevel).apply(index, weigher);
    }

    public static S2DensityTree vertexDensity(S2ShapeIndex index, int approximateSizeBytes, int maxLevel) {
        return S2DensityTree.createVertexDensityOp(approximateSizeBytes, maxLevel).apply(index);
    }

    public static <F> S2DensityTree featureDensity(int approximateSizeBytes, int maxLevel, S2ShapeIndex shapes, FeatureLookup<F> features, FeatureWeigher<F> weights) {
        FeatureDensityOp<F> op = S2DensityTree.createFeatureDensityOp(approximateSizeBytes, maxLevel);
        return op.apply(shapes, features, weights);
    }

    public static S2DensityTree dilate(S2DensityTree tree, S1Angle radius, int maxLevelDiff) {
        S2CellUnion leaves = tree.getLeaves();
        HashMap<S2CellId, Long> weights = new HashMap<S2CellId, Long>();
        int radiusLevel = S2Projections.MIN_WIDTH.getMaxLevel(radius.radians());
        S2CellUnion expandedLeaves = S2CellUnion.copyFrom(leaves);
        expandedLeaves.expand(radius, maxLevelDiff);
        S2CellUnion dilationCells = new S2CellUnion();
        dilationCells.getDifference(expandedLeaves, leaves);
        tree.visitCells((cellId, node) -> {
            long dilatedWeight = Math.max(weights.getOrDefault(cellId, 0L), node.weight());
            weights.put(cellId, dilatedWeight);
            if (node.hasChildren() && cellId.level() < radiusLevel) {
                return CellVisitor.Action.ENTER_CELL;
            }
            int dilateLevel = Math.min(radiusLevel, maxLevelDiff + cellId.level());
            cellId.visitNeighbors(dilateLevel, nbr -> {
                if (!dilationCells.intersects(nbr)) {
                    return true;
                }
                if (weights.getOrDefault(nbr, 0L) < dilatedWeight) {
                    weights.put(nbr, dilatedWeight);
                    while (nbr.level() > 0 && weights.getOrDefault(nbr.parent(), 0L) < dilatedWeight) {
                        weights.put(nbr.parent(), dilatedWeight);
                        nbr = nbr.parent();
                    }
                }
                return true;
            });
            return CellVisitor.Action.SKIP_CELL;
        });
        TreeEncoder encoder = new TreeEncoder();
        weights.forEach(encoder::put);
        return encoder.build();
    }

    @JsIgnore
    public static S2DensityTree sumDensity(Iterable<S2DensityTree> trees, int approximateSizeBytes, int maxLevel) {
        return S2DensityTree.createSumDensityOp(approximateSizeBytes, maxLevel).apply(trees);
    }

    @JsIgnore
    public static S2DensityTree intersectionDensity(Iterable<S2DensityTree> trees, int approximateSizeBytes, int maxLevel) {
        return S2DensityTree.createIntersectionDensityOp(trees, approximateSizeBytes, maxLevel).apply(trees);
    }

    public static ShapeDensityOp createShapeDensityOp(int approximateSizeBytes, int maxLevel) {
        BreadthFirstTreeBuilder acc = new BreadthFirstTreeBuilder(approximateSizeBytes, maxLevel);
        return (index, weigher) -> acc.build(new IndexCellWeightFunction(index, weigher));
    }

    public static VertexDensityOp createVertexDensityOp(int approximateSizeBytes, int maxLevel) {
        BreadthFirstTreeBuilder acc = new BreadthFirstTreeBuilder(approximateSizeBytes, maxLevel);
        IdentityHashMap vertices = new IdentityHashMap();
        return index -> {
            vertices.clear();
            for (S2Shape s : index.getShapes()) {
                vertices.put(s, S2ShapeUtil.numVertices(s));
            }
            return acc.build(new IndexCellWeightFunction(index, vertices::get));
        };
    }

    public static <F> FeatureDensityOp<F> createFeatureDensityOp(int approximateSizeBytes, int maxLevel) {
        BreadthFirstTreeBuilder acc = new BreadthFirstTreeBuilder(approximateSizeBytes, maxLevel);
        FeatureDensityWeigher weigher = new FeatureDensityWeigher();
        return (shapes, features, weights) -> {
            weigher.init(shapes, features, weights);
            return acc.build(weigher);
        };
    }

    public static AggregateDensityOp createSumDensityOp(int approximateSizeBytes, int maxLevel) {
        BreadthFirstTreeBuilder acc = new BreadthFirstTreeBuilder(approximateSizeBytes, maxLevel);
        SumDensity sum = new SumDensity();
        return trees -> {
            sum.set(trees);
            return acc.build(sum);
        };
    }

    public static AggregateDensityOp createIntersectionDensityOp(Iterable<S2DensityTree> intersectionTrees, int approximateSizeBytes, int maxLevel) {
        BreadthFirstTreeBuilder acc = new BreadthFirstTreeBuilder(approximateSizeBytes, maxLevel);
        IntersectionDensity intersection = IntersectionDensity.create(intersectionTrees);
        return trees -> {
            intersection.set(trees);
            return acc.build(intersection);
        };
    }

    static long[] decodeHeader(PrimitiveArrays.Bytes encoded, PrimitiveArrays.Cursor cursor) throws IOException {
        int i;
        try {
            for (byte expected : VERSION_BYTES) {
                byte actual;
                if (expected == (actual = encoded.get(cursor.position++))) continue;
                throw new IOException("Invalid version byte, expected " + expected + ", got " + actual);
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new IOException("Insufficient or invalid input bytes: ", e);
        }
        long[] offsets = new long[6];
        long faceMask = encoded.readVarint64(cursor);
        int lengthsRemaining = Long.bitCount(faceMask) - 1;
        int offset = 0;
        for (i = 0; i < S2CellId.FACE_CELLS.length; ++i) {
            if ((faceMask & 1L << i) == 0L) {
                offsets[i] = -1L;
                continue;
            }
            offsets[i] = offset;
            if (lengthsRemaining-- <= 0) continue;
            offset += S2DensityTree.checkLength(encoded.readVarint64(cursor));
        }
        for (i = 0; i < S2CellId.FACE_CELLS.length; ++i) {
            if (offsets[i] < 0L) continue;
            int n = i;
            offsets[n] = offsets[n] + cursor.position;
        }
        return offsets;
    }

    private static int checkLength(long length) {
        Preconditions.checkState((length > 0L ? 1 : 0) != 0, (Object)"Invalid length, corrupt bytes");
        Preconditions.checkState((length <= Integer.MAX_VALUE ? 1 : 0) != 0, (Object)"Invalid length, corrupt bytes");
        return (int)length;
    }

    public static int estimateSize(long weight) {
        int weightSize = EncodedInts.varIntSize(weight << 4 | 0xFL);
        return weightSize + 2 * EncodedInts.varIntSize(weightSize);
    }

    static {
        int i = 0;
        for (int j = VERSION_BYTES.length - 1; j >= 0; --j) {
            S2DensityTree.REVERSED_VERSION_BYTES[i] = VERSION_BYTES[j];
            ++i;
        }
        CODER = new S2Coder<S2DensityTree>(){

            @Override
            @JsIgnore
            public void encode(S2DensityTree value, OutputStream output) throws IOException {
                value.encoded.writeTo(output);
            }

            @Override
            public S2DensityTree decode(PrimitiveArrays.Bytes data, PrimitiveArrays.Cursor cursor) throws IOException {
                return S2DensityTree.decode(data, cursor);
            }

            @Override
            public boolean isLazy() {
                return true;
            }
        };
    }

    @JsType
    static class TreeEncoder {
        private final ReversibleBytes output = new ReversibleBytes();
        private WeightedCell[] weights = new WeightedCell[8];
        private int size;
        private static final Comparator<WeightedCell> ENCODER_ORDER = (a, b) -> {
            if (b.cell.contains(a.cell)) {
                return -1;
            }
            if (a.cell.contains(b.cell)) {
                return 1;
            }
            return a.cell.compareTo(b.cell);
        };

        TreeEncoder() {
            for (int i = 0; i < this.weights.length; ++i) {
                this.weights[i] = new WeightedCell();
            }
            this.clear();
        }

        public void clear() {
            this.size = 0;
            this.output.clear();
        }

        public void put(S2CellId cell, long weight) {
            if (this.size == this.weights.length) {
                this.weights = Arrays.copyOf(this.weights, this.weights.length * 2);
                for (int i = this.size; i < this.weights.length; ++i) {
                    this.weights[i] = new WeightedCell();
                }
            }
            this.weights[this.size++].set(cell, weight);
        }

        public S2DensityTree build() {
            try {
                Arrays.sort(this.weights, 0, this.size, ENCODER_ORDER);
                this.output.clear();
                ReversedLengthsWriter lengths = new ReversedLengthsWriter(this.output);
                int faceMask = 0;
                while (this.size > 0) {
                    WeightedCell weighted = this.weights[--this.size];
                    assert (weighted.cell.level() == 0) : "First cell of a face must be a face cell";
                    int face = weighted.cell.face();
                    this.encodeCellReverse(0, weighted);
                    lengths.next();
                    faceMask |= 1 << face;
                }
                lengths.write(faceMask);
                this.output.write(REVERSED_VERSION_BYTES);
                S2DensityTree s2DensityTree = S2DensityTree.decode(PrimitiveArrays.Bytes.fromByteArray(this.output.reversedCopy()));
                return s2DensityTree;
            }
            catch (IOException e) {
                throw new IllegalStateException("ByteStack EOF", e);
            }
            finally {
                this.clear();
            }
        }

        private void encodeCellReverse(int level, WeightedCell parent) {
            int childMask = 0;
            ReversedLengthsWriter lengths = new ReversedLengthsWriter(this.output);
            while (this.size > 0 && parent.cell.contains(this.weights[this.size - 1].cell)) {
                WeightedCell child = this.weights[--this.size];
                assert (parent.cell.equals(child.cell.parent())) : "Must be a direct descendant";
                int childLevel = level + 1;
                childMask |= 1 << child.cell.childPosition(childLevel);
                this.encodeCellReverse(childLevel, child);
                lengths.next();
            }
            lengths.write(parent.weight << 4 | (long)childMask);
            parent.cell = null;
        }

        @VisibleForTesting
        static final class ReversibleBytes
        extends OutputStream {
            private byte[] bytes = new byte[256];
            private int size;

            ReversibleBytes() {
            }

            public void clear() {
                this.size = 0;
            }

            public int size() {
                return this.size;
            }

            @Override
            public void write(int b) {
                this.maybeResize(1);
                this.bytes[this.size++] = (byte)b;
            }

            @Override
            public void write(byte[] b) {
                this.write(b, 0, b.length);
            }

            @Override
            public void write(byte[] b, int off, int len) {
                this.maybeResize(len);
                for (int i = 0; i < len; ++i) {
                    this.bytes[this.size++] = b[off++];
                }
            }

            private void maybeResize(int increase) {
                while (this.size + increase > this.bytes.length) {
                    this.bytes = Arrays.copyOf(this.bytes, this.bytes.length * 2);
                }
            }

            public void reverseFrom(int start) {
                int i = start;
                for (int j = this.size - 1; i < j; ++i, --j) {
                    byte b = this.bytes[i];
                    this.bytes[i] = this.bytes[j];
                    this.bytes[j] = b;
                }
            }

            public byte[] toByteArray() {
                return Arrays.copyOf(this.bytes, this.size);
            }

            public byte[] reversedCopy() {
                byte[] reversed = new byte[this.size];
                int i = 0;
                int j = this.size - 1;
                while (i < this.size) {
                    reversed[i] = this.bytes[j];
                    ++i;
                    --j;
                }
                return reversed;
            }
        }

        @VisibleForTesting
        static class ReversedLengthsWriter {
            final ReversibleBytes output;
            final int[] lengths = new int[6];
            int size;
            int start;

            ReversedLengthsWriter(ReversibleBytes output) {
                this.output = output;
                this.start = output.size;
            }

            void next() {
                this.lengths[this.size++] = this.output.size - this.start;
                this.start = this.output.size;
            }

            void write(long mask) {
                try {
                    EncodedInts.writeVarint64(this.output, mask);
                    for (int i = this.size - 1; i > 0; --i) {
                        EncodedInts.writeVarint64(this.output, this.lengths[i]);
                    }
                }
                catch (IOException e) {
                    throw new IllegalStateException("Can't be thrown by ReversibleBytes: ", e);
                }
                this.output.reverseFrom(this.start);
                this.start = this.output.size;
            }
        }

        private static class WeightedCell {
            S2CellId cell;
            long weight;

            private WeightedCell() {
            }

            void set(S2CellId cell, long weight) {
                this.cell = cell;
                this.weight = weight;
            }
        }
    }

    @JsType
    public static class Cell {
        private static final int[] NO_CHILDREN = new int[]{-1, -1, -1, -1};
        long weight;
        final int[] positions = new int[4];

        public boolean isEmpty() {
            return this.weight == 0L;
        }

        public void setNoChildren() {
            System.arraycopy(NO_CHILDREN, 0, this.positions, 0, this.positions.length);
        }

        public boolean hasChildren() {
            return !Arrays.equals(this.positions, NO_CHILDREN);
        }

        public void decode(PrimitiveArrays.Bytes encoded, PrimitiveArrays.Cursor cursor) {
            int i;
            long bits = encoded.readVarint64(cursor);
            this.weight = bits >>> 4;
            int childMask = (int)bits & 0xF;
            int offset = 0;
            for (i = 0; i < 4; ++i) {
                if ((childMask & 1) != 0) {
                    this.positions[i] = offset;
                    if (childMask > 1) {
                        offset += S2DensityTree.checkLength(encoded.readVarint64(cursor));
                    }
                } else {
                    this.positions[i] = -1;
                }
                childMask >>>= 1;
            }
            for (i = 0; i < 4; ++i) {
                if (this.positions[i] < 0) continue;
                this.positions[i] = S2DensityTree.checkLength((long)this.positions[i] + cursor.position);
            }
        }

        public long weight() {
            return this.weight;
        }

        public String toString() {
            return Platform.formatString("Cell[weight=%d, positions=%s]", this.weight, Arrays.toString(this.positions));
        }
    }

    @JsType
    public static class BreadthFirstTreeBuilder {
        private final int approximateSizeBytes;
        private final int maxLevel;
        private final TreeEncoder encoder;
        private long[] ranges = new long[8];
        private int rangesSize;
        private long[] nextLevelRanges = new long[8];
        private int nextLevelRangesSize;

        @JsIgnore
        public BreadthFirstTreeBuilder(int approximateSizeBytes, int maxLevel) {
            this(approximateSizeBytes, maxLevel, new TreeEncoder());
        }

        public BreadthFirstTreeBuilder(int approximateSizeBytes, int maxLevel, TreeEncoder encoder) {
            this.approximateSizeBytes = approximateSizeBytes;
            this.maxLevel = maxLevel;
            this.encoder = encoder;
            this.clear();
        }

        public void clear() {
            this.rangesSize = 0;
            this.nextLevelRangesSize = 0;
            this.encoder.clear();
        }

        public S2DensityTree build(CellWeightFunction weigher) {
            this.clear();
            int sizeEstimateBytes = 0;
            this.ranges[this.rangesSize++] = S2CellId.begin(30).id();
            this.ranges[this.rangesSize++] = S2CellId.end(30).id();
            for (int level = 0; level <= this.maxLevel && sizeEstimateBytes < this.approximateSizeBytes; ++level) {
                long lastHilbertEnd = S2CellId.sentinel().id();
                for (int i = 0; i < this.rangesSize; i += 2) {
                    S2CellId start = new S2CellId(this.ranges[i]);
                    S2CellId end = new S2CellId(this.ranges[i + 1]);
                    S2CellId cell = start.parent(level);
                    while (cell.lessThan(end)) {
                        long weight = weigher.applyAsLong(cell);
                        if (weight != 0L) {
                            if (weight < 0L) {
                                weight = -weight;
                            } else {
                                long hilbertStart = cell.childBegin(30).id();
                                long hilbertEnd = cell.childEnd(30).id();
                                if (hilbertStart == lastHilbertEnd) {
                                    this.nextLevelRanges[this.nextLevelRangesSize - 1] = hilbertEnd;
                                } else {
                                    if (this.nextLevelRangesSize == this.nextLevelRanges.length) {
                                        this.nextLevelRanges = Arrays.copyOf(this.nextLevelRanges, this.nextLevelRanges.length * 2);
                                    }
                                    this.nextLevelRanges[this.nextLevelRangesSize++] = hilbertStart;
                                    this.nextLevelRanges[this.nextLevelRangesSize++] = hilbertEnd;
                                }
                                lastHilbertEnd = hilbertEnd;
                            }
                            Preconditions.checkArgument((weight <= 0x7FFFFFFFFFFFFFFL ? 1 : 0) != 0, (Object)"Weigher exceeded max weight");
                            this.encoder.put(cell, weight);
                            sizeEstimateBytes += S2DensityTree.estimateSize(weight);
                        }
                        cell = cell.next();
                    }
                }
                long[] tempArray = this.ranges;
                this.ranges = this.nextLevelRanges;
                this.nextLevelRanges = tempArray;
                this.rangesSize = this.nextLevelRangesSize;
                this.nextLevelRangesSize = 0;
            }
            S2DensityTree tree = this.encoder.build();
            this.clear();
            return tree;
        }

        @JsType
        public static interface CellWeightFunction
        extends ToLongFunction<S2CellId> {
            @Override
            public long applyAsLong(S2CellId var1);
        }
    }

    public static class DecodedPath {
        private S2DensityTree tree;
        private PrimitiveArrays.Cursor cursor;
        private final List<Cell> stack = new ArrayList<Cell>();
        private S2CellId lastDecoded;
        private static final Cell EMPTY_CELL = new Cell();

        public DecodedPath() {
        }

        public DecodedPath(S2DensityTree tree) {
            this.set(tree);
        }

        public void set(S2DensityTree tree) {
            this.tree = tree;
            this.cursor = tree.encoded.cursor();
            this.lastDecoded = null;
        }

        public long weight(S2CellId id) {
            return this.cell((S2CellId)id).weight;
        }

        public Cell cell(S2CellId cell) {
            int decodedLevel = this.decodedLevel(cell);
            if (decodedLevel < 0) {
                Cell node = this.loadFace(cell.face());
                if (node.isEmpty()) {
                    return node;
                }
                decodedLevel = 0;
            }
            return this.loadCell(decodedLevel, cell);
        }

        private int decodedLevel(S2CellId cell) {
            if (this.lastDecoded == null) {
                return -1;
            }
            if (this.lastDecoded.contains(cell)) {
                return this.lastDecoded.level();
            }
            return this.lastDecoded.getCommonAncestorLevel(cell);
        }

        private Cell loadFace(int face) {
            Cell cell = this.get(0);
            if (this.tree.facePositions[face] < 0L) {
                cell.weight = 0L;
                cell.setNoChildren();
                this.lastDecoded = null;
            } else {
                this.cursor.position = this.tree.facePositions[face];
                cell.decode(this.tree.encoded, this.cursor);
                this.lastDecoded = S2CellId.FACE_CELLS[face];
            }
            return cell;
        }

        private Cell loadCell(int decodedLevel, S2CellId cell) {
            Cell decodedCell = this.get(decodedLevel);
            for (int i = decodedLevel + 1; i <= cell.level(); ++i) {
                this.cursor.position = decodedCell.positions[cell.childPosition(i)];
                if (this.cursor.position < 0L) {
                    return decodedCell.hasChildren() ? EMPTY_CELL : decodedCell;
                }
                decodedCell = this.get(i);
                decodedCell.decode(this.tree.encoded, this.cursor);
                this.lastDecoded = cell.parent(i);
            }
            return decodedCell;
        }

        private Cell get(int level) {
            if (this.stack.size() == level) {
                this.stack.add(new Cell());
            }
            return this.stack.get(level);
        }

        static {
            EMPTY_CELL.setNoChildren();
        }
    }

    @VisibleForTesting
    static class IntersectionDensity
    extends SumDensity {
        private final S2CellUnion intersectingLeaves;

        public static IntersectionDensity create(Iterable<S2DensityTree> trees) {
            S2CellUnion leaves;
            Iterator<S2DensityTree> iterator = trees.iterator();
            S2CellUnion s2CellUnion = leaves = iterator.hasNext() ? iterator.next().getLeaves() : new S2CellUnion();
            while (iterator.hasNext()) {
                S2CellUnion temp = new S2CellUnion();
                temp.getIntersection(leaves, iterator.next().getLeaves());
                leaves = temp;
            }
            return new IntersectionDensity(leaves);
        }

        private IntersectionDensity(S2CellUnion intersectingLeaves) {
            this.intersectingLeaves = intersectingLeaves;
        }

        @VisibleForTesting
        public S2CellUnion getIntersectingLeaves() {
            return this.intersectingLeaves;
        }

        @Override
        public long applyAsLong(S2CellId id) {
            return this.intersectingLeaves.intersects(id) ? super.applyAsLong(id) : 0L;
        }
    }

    public static class SumDensity
    extends AggregateDensity {
        @Override
        public long applyAsLong(S2CellId id) {
            long sum = 0L;
            boolean allContained = true;
            for (int i = 0; i < this.size; ++i) {
                Cell node = ((DecodedPath)this.decodedPaths.get(i)).cell(id);
                sum += node.weight;
                allContained &= !node.hasChildren();
            }
            sum = Math.min(0x7FFFFFFFFFFFFFFL, sum);
            return allContained ? -sum : sum;
        }
    }

    static abstract class AggregateDensity
    implements BreadthFirstTreeBuilder.CellWeightFunction {
        protected final List<DecodedPath> decodedPaths = new ArrayList<DecodedPath>();
        protected int size;

        AggregateDensity() {
        }

        public void set(Iterable<S2DensityTree> trees) {
            this.size = Iterables.size(trees);
            while (this.decodedPaths.size() < this.size) {
                this.decodedPaths.add(new DecodedPath());
            }
            int i = 0;
            for (S2DensityTree tree : trees) {
                this.decodedPaths.get(i++).set(tree);
            }
        }
    }

    public static class CellDensityFunction
    implements BreadthFirstTreeBuilder.CellWeightFunction {
        private final S2CellIndex index;
        private final S2CellUnion[] coverings;
        private final long[] weights;
        private final S2CellUnion query = new S2CellUnion();

        public CellDensityFunction(S2CellIndex index, S2CellUnion[] coverings, long[] weights) {
            Preconditions.checkArgument((coverings.length == weights.length ? 1 : 0) != 0);
            this.index = index;
            this.coverings = coverings;
            this.weights = weights;
        }

        public static CellDensityFunction fromArrays(S2CellUnion[] coverings, long[] weights) {
            S2CellIndex index = new S2CellIndex();
            for (int i = 0; i < coverings.length; ++i) {
                index.add(coverings[i], i);
            }
            index.build();
            return new CellDensityFunction(index, coverings, weights);
        }

        @Override
        public long applyAsLong(S2CellId cell) {
            this.query.cellIds().add(cell);
            long sum = 0L;
            boolean contained = true;
            Iterator iterator = this.index.getIntersectingLabels(this.query).iterator();
            while (iterator.hasNext()) {
                int label = (Integer)iterator.next();
                sum += this.weights[label];
                contained &= this.coverings[label].contains(this.query);
            }
            this.query.cellIds().clear();
            return contained ? -sum : sum;
        }

        public static <T> BreadthFirstTreeBuilder.CellWeightFunction fromFeatures(List<T> features, Function<T, S2CellUnion> covering, ToLongFunction<T> weight) {
            S2CellUnion[] coverings = new S2CellUnion[features.size()];
            long[] weights = new long[features.size()];
            for (int i = 0; i < features.size(); ++i) {
                T feature = features.get(i);
                coverings[i] = covering.apply(feature);
                weights[i] = weight.applyAsLong(feature);
            }
            return CellDensityFunction.fromArrays(coverings, weights);
        }

        public static <T> S2DensityTree density(List<T> features, Function<T, S2CellUnion> covering, ToLongFunction<T> weight, BreadthFirstTreeBuilder builder) {
            return builder.build(CellDensityFunction.fromFeatures(features, covering, weight));
        }
    }

    public static class IndexCellWeightFunction
    implements BreadthFirstTreeBuilder.CellWeightFunction {
        private final S2ShapeIndex index;
        private final S2ShapeIndexRegion region;
        private final ShapeWeightFunction weigher;

        public IndexCellWeightFunction(S2ShapeIndex index, ShapeWeightFunction weigher) {
            this.index = index;
            this.region = new S2ShapeIndexRegion(index);
            this.weigher = weigher;
        }

        @Override
        public long applyAsLong(S2CellId cell) {
            class IntersectingVisitor
            implements S2ShapeIndexRegion.ShapeVisitor {
                long sum = 0L;
                long limit = 0x7FFFFFFFFFFFFFFL;
                boolean allContained = true;

                IntersectingVisitor() {
                }

                @Override
                public boolean test(int shapeId, boolean containsTarget) {
                    long weight = IndexCellWeightFunction.this.weigher.applyAsLong(IndexCellWeightFunction.this.index.getShapes().get(shapeId));
                    Preconditions.checkState((weight >= 0L ? 1 : 0) != 0, (Object)"Weight must not be negative.");
                    this.limit -= weight;
                    Preconditions.checkState((this.limit >= 0L ? 1 : 0) != 0, (Object)"Weight must be at most 576460752303423487");
                    this.sum += weight;
                    this.allContained &= containsTarget;
                    return true;
                }

                long finalWeight() {
                    return this.allContained ? -this.sum : this.sum;
                }
            }
            IntersectingVisitor visitor = new IntersectingVisitor();
            this.region.visitIntersectingShapes(new S2Cell(cell), visitor);
            this.weigher.reset();
            return visitor.finalWeight();
        }
    }

    @JsType
    public static class FeatureDensityWeigher<T>
    implements BreadthFirstTreeBuilder.CellWeightFunction {
        private S2ShapeIndexRegion region;
        private int[] ids;
        private int[] weights;
        private int[] lastCall;
        private int nextCall;
        private long sum;
        private boolean allContained;

        public void init(S2ShapeIndex shapes, FeatureLookup<T> featureLookup, FeatureWeigher<T> weightLookup) {
            this.region = new S2ShapeIndexRegion(shapes);
            this.ids = new int[shapes.shapes.size()];
            IdentityHashMap map = Maps.newIdentityHashMap();
            for (int i = 0; i < this.ids.length; ++i) {
                T feature2 = featureLookup.feature(shapes.shapes.get(i));
                this.ids[i] = map.computeIfAbsent(feature2, s -> map.size());
            }
            this.weights = new int[map.size()];
            this.lastCall = new int[this.weights.length];
            this.nextCall = 1;
            map.forEach((feature, id) -> {
                this.weights[id.intValue()] = Math.toIntExact(weightLookup.weight(feature));
            });
        }

        @Override
        public long applyAsLong(S2CellId cell) {
            this.sum = 0L;
            this.allContained = true;
            this.region.visitIntersectingShapes(new S2Cell(cell), this::sumFeatures);
            if (++this.nextCall == Integer.MAX_VALUE) {
                this.nextCall = 1;
                Arrays.fill(this.lastCall, 0);
            }
            return this.allContained ? -this.sum : this.sum;
        }

        private boolean sumFeatures(int shapeId, boolean containsTarget) {
            int featureId = this.ids[shapeId];
            if (this.lastCall[featureId] != this.nextCall) {
                this.lastCall[featureId] = this.nextCall;
                this.sum += (long)this.weights[featureId];
                this.allContained &= containsTarget;
            }
            return true;
        }
    }

    @JsType
    public static interface FeatureWeigher<T> {
        public long weight(T var1);
    }

    @JsType
    public static interface FeatureLookup<T> {
        public T feature(S2Shape var1);
    }

    @JsType
    public static interface AggregateDensityOp
    extends Function<Iterable<S2DensityTree>, S2DensityTree> {
        @Override
        public S2DensityTree apply(Iterable<S2DensityTree> var1);
    }

    @JsType
    public static interface FeatureDensityOp<F> {
        public S2DensityTree apply(S2ShapeIndex var1, FeatureLookup<F> var2, FeatureWeigher<F> var3);
    }

    @JsType
    public static interface VertexDensityOp
    extends Function<S2ShapeIndex, S2DensityTree> {
        @Override
        public S2DensityTree apply(S2ShapeIndex var1);
    }

    @JsType
    public static interface ShapeDensityOp
    extends BiFunction<S2ShapeIndex, ShapeWeightFunction, S2DensityTree> {
        @Override
        public S2DensityTree apply(S2ShapeIndex var1, ShapeWeightFunction var2);
    }

    @JsType
    public static interface ShapeWeightFunction
    extends ToLongFunction<S2Shape> {
        @Override
        public long applyAsLong(S2Shape var1);

        default public void reset() {
        }
    }

    @JsType
    public static interface CellVisitor {
        public Action visit(S2CellId var1, Cell var2);

        @JsType
        public static interface While
        extends CellVisitor {
            @Override
            default public Action visit(S2CellId cell, Cell node) {
                return this.visit(cell, node.weight) ? Action.ENTER_CELL : Action.STOP;
            }

            @JsIgnore
            public boolean visit(S2CellId var1, long var2);
        }

        @JsType
        public static interface All
        extends CellVisitor {
            @Override
            default public Action visit(S2CellId cell, Cell node) {
                this.visit(cell, node.weight);
                return Action.ENTER_CELL;
            }

            @JsIgnore
            public void visit(S2CellId var1, long var2);
        }

        @JsEnum
        public static enum Action {
            SKIP_CELL,
            ENTER_CELL,
            STOP;

        }
    }
}

