/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.base.Preconditions;
import com.google.common.geometry.BigPoint;
import com.google.common.geometry.Platform;
import com.google.common.geometry.R1Interval;
import com.google.common.geometry.R2Rect;
import com.google.common.geometry.R2Vector;
import com.google.common.geometry.S1Angle;
import com.google.common.geometry.S1ChordAngle;
import com.google.common.geometry.S1Interval;
import com.google.common.geometry.S2;
import com.google.common.geometry.S2Edge;
import com.google.common.geometry.S2LatLng;
import com.google.common.geometry.S2LatLngRect;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Predicates;
import com.google.common.geometry.S2Projections;
import com.google.common.geometry.S2RobustCrossProd;
import com.google.common.geometry.S2Shape;
import com.google.errorprone.annotations.InlineMe;

public class S2EdgeUtil {
    public static final S1Angle DEFAULT_INTERSECTION_TOLERANCE = S1Angle.radians(1.538415276616845E-15);
    private static final double MAX_DET_ERROR = 1.0E-14;
    public static final double FACE_CLIP_ERROR_RADIANS = 6.661338147750939E-16;
    public static final double FACE_CLIP_ERROR_UV_DIST = 1.9984014443252818E-15;
    public static final double FACE_CLIP_ERROR_UV_COORD = 9.0 * S2.M_SQRT1_2 * 2.220446049250313E-16;
    public static final double INTERSECTS_RECT_ERROR_UV_DIST = 3.0 * S2.M_SQRT2 * 2.220446049250313E-16;
    public static final double EDGE_CLIP_ERROR_UV_COORD = 4.996003610813204E-16;
    public static final double EDGE_CLIP_ERROR_UV_DIST = 4.996003610813204E-16;
    public static final double MAX_CELL_EDGE_ERROR = FACE_CLIP_ERROR_UV_COORD + INTERSECTS_RECT_ERROR_UV_DIST;
    public static final double INTERSECTION_ERROR = 1.7763568394002505E-15;
    public static final S1Angle GET_POINT_ON_LINE_ERROR = S1Angle.radians((4.0 + 2.0 / S2.M_SQRT3) * (double)1.110223E-16f).add(S2.ROBUST_CROSS_PROD_ERROR);
    public static final S1Angle GET_POINT_ON_RAY_PERPENDICULAR_ERROR = S1Angle.radians(3.330669E-16f);
    public static final S1Angle INTERSECTION_MERGE_RADIUS = S1Angle.radians(3.552713678800501E-15);

    public static WedgeRelation getWedgeRelation(S2Point a0, S2Point ab1, S2Point a2, S2Point b0, S2Point b2) {
        if (a0.equalsPoint(b0) && a2.equalsPoint(b2)) {
            return WedgeRelation.WEDGE_EQUALS;
        }
        if (S2Predicates.orderedCCW(a0, a2, b2, ab1)) {
            if (S2Predicates.orderedCCW(b2, b0, a0, ab1)) {
                return WedgeRelation.WEDGE_PROPERLY_CONTAINS;
            }
            if (a2.equalsPoint(b2)) {
                return WedgeRelation.WEDGE_IS_PROPERLY_CONTAINED;
            }
            return WedgeRelation.WEDGE_PROPERLY_OVERLAPS;
        }
        if (S2Predicates.orderedCCW(a0, b0, b2, ab1)) {
            return WedgeRelation.WEDGE_IS_PROPERLY_CONTAINED;
        }
        if (S2Predicates.orderedCCW(a0, b0, a2, ab1)) {
            return WedgeRelation.WEDGE_IS_DISJOINT;
        }
        return WedgeRelation.WEDGE_PROPERLY_OVERLAPS;
    }

    static boolean sumEquals(double u, double v, double w) {
        return u + v == w && u == w - v && v == w - u;
    }

    static boolean intersectsFace(S2Point n) {
        double w;
        double u = Math.abs(n.x);
        double v = Math.abs(n.y);
        return v >= (w = Math.abs(n.z)) - u && u >= w - v;
    }

    static boolean intersectsOppositeEdges(S2Point n) {
        double u = Math.abs(n.x);
        double v = Math.abs(n.y);
        double w = Math.abs(n.z);
        if (Math.abs(u - v) != w) {
            return Math.abs(u - v) >= w;
        }
        return u >= v ? u - w >= v : v - w >= u;
    }

    static int getExitAxis(S2Point n) {
        assert (S2EdgeUtil.intersectsFace(n));
        if (S2EdgeUtil.intersectsOppositeEdges(n)) {
            return Math.abs(n.x) >= Math.abs(n.y) ? 1 : 0;
        }
        assert (n.x != 0.0 && n.y != 0.0 && n.z != 0.0);
        return n.x < 0.0 ^ n.y < 0.0 ^ n.z < 0.0 ? 0 : 1;
    }

    static void getExitPoint(S2Point n, int axis, R2Vector result) {
        if (axis == 0) {
            result.x = n.y > 0.0 ? 1.0 : -1.0;
            result.y = (-result.x * n.x - n.z) / n.y;
        } else {
            result.y = n.x < 0.0 ? 1.0 : -1.0;
            result.x = (-result.y * n.y - n.z) / n.x;
        }
    }

    static int moveOriginToValidFace(int face, S2Point a, S2Point ab, R2Vector aUv) {
        double kMaxSafeUVCoord = 1.0 - FACE_CLIP_ERROR_UV_COORD;
        double au = aUv.x;
        double av = aUv.y;
        if (Math.max(Math.abs(au), Math.abs(av)) <= kMaxSafeUVCoord) {
            return face;
        }
        S2Point n = S2Projections.faceXyzToUvw(face, ab);
        if (S2EdgeUtil.intersectsFace(n)) {
            S2EdgeUtil.getExitPoint(n, S2EdgeUtil.getExitAxis(n), aUv);
            S2Point exit = S2Projections.faceUvToXyz(face, aUv);
            S2Point aTangent = ab.normalize().crossProd(a);
            if (exit.sub(a).dotProd(aTangent) >= -6.661338147750939E-16) {
                aUv.x = au;
                aUv.y = av;
                return face;
            }
        }
        face = Math.abs(au) >= Math.abs(av) ? S2Projections.getUVWFace(face, 0, au > 0.0 ? 1 : 0) : S2Projections.getUVWFace(face, 1, av > 0.0 ? 1 : 0);
        assert (S2EdgeUtil.intersectsFace(S2Projections.faceXyzToUvw(face, ab)));
        S2Projections.validFaceXyzToUv(face, a, aUv);
        aUv.set(Math.max(-1.0, Math.min(1.0, aUv.x)), Math.max(-1.0, Math.min(1.0, aUv.y)));
        return face;
    }

    static int getNextFace(int face, R2Vector exit, int axis, S2Point n, int targetFace) {
        if (Math.abs(exit.get(1 - axis)) == 1.0 && S2Projections.getUVWFace(face, 1 - axis, exit.get(1 - axis) > 0.0 ? 1 : 0) == targetFace && S2EdgeUtil.sumEquals(exit.x * n.x, exit.y * n.y, -n.z)) {
            return targetFace;
        }
        return S2Projections.getUVWFace(face, axis, exit.get(axis) > 0.0 ? 1 : 0);
    }

    static int getFaceSegments(S2Point a, S2Point b, FaceSegment[] segments) {
        assert (S2.isUnitLength(a));
        assert (S2.isUnitLength(b));
        FaceSegment seg = segments[0];
        seg.face = S2Projections.xyzToFace(a);
        S2Projections.validFaceXyzToUv(seg.face, a, seg.a);
        int bFace = S2Projections.xyzToFace(b);
        S2Projections.validFaceXyzToUv(bFace, b, seg.b);
        if (seg.face == bFace) {
            return 1;
        }
        S2Point ab = S2RobustCrossProd.robustCrossProd(a, b);
        seg.face = S2EdgeUtil.moveOriginToValidFace(seg.face, a, ab, seg.a);
        bFace = S2EdgeUtil.moveOriginToValidFace(bFace, b, ab.neg(), seg.b);
        segments[5].b.set(seg.b);
        int size = 1;
        while (seg.face != bFace) {
            S2Point n = S2Projections.faceXyzToUvw(seg.face, ab);
            int exitAxis = S2EdgeUtil.getExitAxis(n);
            S2EdgeUtil.getExitPoint(n, exitAxis, seg.b);
            int newFace = S2EdgeUtil.getNextFace(seg.face, seg.b, exitAxis, n, bFace);
            S2Point oldExitXyz = S2Projections.faceUvToXyz(seg.face, seg.b);
            S2Point newExitUvw = S2Projections.faceXyzToUvw(newFace, oldExitXyz);
            seg = segments[size++];
            seg.face = newFace;
            seg.a.set(newExitUvw.x, newExitUvw.y);
        }
        seg.b.set(segments[5].b);
        return size;
    }

    static int clipDestination(S2Point a, S2Point b, S2Point nScaled, S2Point aTangent, S2Point bTangent, double uvScale, R2Vector uv) {
        assert (S2EdgeUtil.intersectsFace(nScaled));
        double kMaxSafeUVCoord = 1.0 - FACE_CLIP_ERROR_UV_COORD;
        if (b.z > 0.0) {
            uv.set(b.x / b.z, b.y / b.z);
            if (Math.max(Math.abs(uv.x), Math.abs(uv.y)) <= kMaxSafeUVCoord) {
                return 0;
            }
        }
        S2EdgeUtil.getExitPoint(nScaled, S2EdgeUtil.getExitAxis(nScaled), uv);
        uv.x *= uvScale;
        uv.y *= uvScale;
        S2Point p = new S2Point(uv.x, uv.y, 1.0);
        int score = 0;
        if (p.sub(a).dotProd(aTangent) < 0.0) {
            score = 2;
        } else if (p.sub(b).dotProd(bTangent) < 0.0) {
            score = 1;
        }
        if (score > 0) {
            if (b.z <= 0.0) {
                score = 3;
            } else {
                uv.set(b.x / b.z, b.y / b.z);
            }
        }
        return score;
    }

    public static boolean clipToPaddedFace(S2Point aXyz, S2Point bXyz, int face, double padding, R2Vector aUv, R2Vector bUv) {
        int bScore;
        assert (padding >= 0.0);
        if (S2Projections.xyzToFace(aXyz) == face && S2Projections.xyzToFace(bXyz) == face) {
            S2Projections.validFaceXyzToUv(face, aXyz, aUv);
            S2Projections.validFaceXyzToUv(face, bXyz, bUv);
            return true;
        }
        S2Point n = S2Projections.faceXyzToUvw(face, S2RobustCrossProd.robustCrossProd(aXyz, bXyz));
        S2Point a = S2Projections.faceXyzToUvw(face, aXyz);
        S2Point b = S2Projections.faceXyzToUvw(face, bXyz);
        double uvScale = 1.0 + padding;
        S2Point nScaled = new S2Point(uvScale * n.x, uvScale * n.y, n.z);
        if (!S2EdgeUtil.intersectsFace(nScaled)) {
            return false;
        }
        n = n.normalize();
        S2Point aTangent = n.crossProd(a);
        S2Point bTangent = b.crossProd(n);
        int aScore = S2EdgeUtil.clipDestination(b, a, nScaled.neg(), bTangent, aTangent, uvScale, aUv);
        return aScore + (bScore = S2EdgeUtil.clipDestination(a, b, nScaled, aTangent, bTangent, uvScale, bUv)) < 3;
    }

    static boolean intersectsRect(R2Vector a, R2Vector b, R2Rect rect) {
        R2Rect bound = R2Rect.fromPointPair(a, b);
        if (!rect.intersects(bound)) {
            return false;
        }
        R2Vector n = R2Vector.sub(b, a).ortho();
        int i = n.x >= 0.0 ? 1 : 0;
        int j = n.y >= 0.0 ? 1 : 0;
        double max = n.dotProd(R2Vector.sub(rect.getVertex(i, j), a));
        double min = n.dotProd(R2Vector.sub(rect.getVertex(1 - i, 1 - j), a));
        return max >= 0.0 && min <= 0.0;
    }

    static boolean updateEndpoint(R1Interval bound, boolean slopeNegative, double value) {
        if (!slopeNegative) {
            if (bound.hi() < value) {
                return false;
            }
            if (bound.lo() < value) {
                bound.setLo(value);
            }
        } else {
            if (bound.lo() > value) {
                return false;
            }
            if (bound.hi() > value) {
                bound.setHi(value);
            }
        }
        return true;
    }

    static boolean clipBoundAxis(double a0, double b0, R1Interval bound0, double a1, double b1, R1Interval bound1, boolean slopeNegative, R1Interval clip0) {
        if (bound0.lo() < clip0.lo()) {
            if (bound0.hi() < clip0.lo()) {
                return false;
            }
            bound0.setLo(clip0.lo());
            if (!S2EdgeUtil.updateEndpoint(bound1, slopeNegative, S2EdgeUtil.interpolateDouble(clip0.lo(), a0, b0, a1, b1))) {
                return false;
            }
        }
        if (bound0.hi() > clip0.hi()) {
            if (bound0.lo() > clip0.hi()) {
                return false;
            }
            bound0.setHi(clip0.hi());
            if (!S2EdgeUtil.updateEndpoint(bound1, !slopeNegative, S2EdgeUtil.interpolateDouble(clip0.hi(), a0, b0, a1, b1))) {
                return false;
            }
        }
        return true;
    }

    static R2Rect getClippedEdgeBound(R2Vector a, R2Vector b, R2Rect clip) {
        R2Rect bound = R2Rect.fromPointPair(a, b);
        if (S2EdgeUtil.clipEdgeBound(a, b, clip, bound)) {
            return bound;
        }
        return R2Rect.empty();
    }

    static boolean clipEdgeBound(R2Vector a, R2Vector b, R2Rect clip, R2Rect bound) {
        boolean slopeNegative = a.x > b.x != a.y > b.y;
        return S2EdgeUtil.clipBoundAxis(a.x, b.x, bound.x(), a.y, b.y, bound.y(), slopeNegative, clip.x()) && S2EdgeUtil.clipBoundAxis(a.y, b.y, bound.y(), a.x, b.x, bound.x(), slopeNegative, clip.y());
    }

    static boolean clipEdge(R2Vector a, R2Vector b, R2Rect clip, R2Vector aClipped, R2Vector bClipped) {
        R2Rect bound = R2Rect.fromPointPair(a, b);
        if (S2EdgeUtil.clipEdgeBound(a, b, clip, bound)) {
            int iEnd = a.x > b.x ? 1 : 0;
            int jEnd = a.y > b.y ? 1 : 0;
            aClipped.set(bound.getVertex(iEnd, jEnd));
            bClipped.set(bound.getVertex(1 - iEnd, 1 - jEnd));
            return true;
        }
        return false;
    }

    static double interpolateDouble(double x, double a, double b, double a1, double b1) {
        assert (a != b);
        if (Math.abs(a - x) <= Math.abs(b - x)) {
            return a1 + (b1 - a1) * ((x - a) / (b - a));
        }
        return b1 + (a1 - b1) * ((x - b) / (a - b));
    }

    public static boolean clipToFace(S2Point a, S2Point b, int face, R2Vector aUv, R2Vector bUv) {
        return S2EdgeUtil.clipToPaddedFace(a, b, face, 0.0, aUv, bUv);
    }

    public static boolean simpleCrossing(S2Point a, S2Point b, S2Point c, S2Point d) {
        double bda;
        S2Point ab = a.crossProd(b);
        double acb = -ab.dotProd(c);
        if (acb * (bda = ab.dotProd(d)) <= 0.0) {
            return false;
        }
        S2Point cd = c.crossProd(d);
        double cbd = -cd.dotProd(b);
        if (acb * cbd <= 0.0) {
            return false;
        }
        double dac = cd.dotProd(a);
        return acb * dac > 0.0;
    }

    public static int robustCrossing(S2Point a, S2Point b, S2Point c, S2Point d) {
        EdgeCrosser crosser = new EdgeCrosser(a, b, c);
        return crosser.robustCrossing(d);
    }

    public static boolean vertexCrossing(S2Point a, S2Point b, S2Point c, S2Point d) {
        if (a.equalsPoint(b) || c.equalsPoint(d)) {
            return false;
        }
        if (a.equalsPoint(d)) {
            return S2Predicates.orderedCCW(S2.refDir(a), c, b, a);
        }
        if (b.equalsPoint(c)) {
            return S2Predicates.orderedCCW(S2.refDir(b), d, a, b);
        }
        if (a.equalsPoint(c)) {
            return S2Predicates.orderedCCW(S2.refDir(a), d, b, a);
        }
        if (b.equalsPoint(d)) {
            return S2Predicates.orderedCCW(S2.refDir(b), c, a, b);
        }
        assert (false) : "vertexCrossing called with 4 distinct vertices, which is not allowed";
        return false;
    }

    public static int signedVertexCrossing(S2Point a, S2Point b, S2Point c, S2Point d) {
        if (a.equalsPoint(b) || c.equalsPoint(d)) {
            return 0;
        }
        if (a.equalsPoint(c)) {
            return b.equalsPoint(d) || S2Predicates.orderedCCW(S2.refDir(a), d, b, a) ? 1 : 0;
        }
        if (b.equalsPoint(d)) {
            return S2Predicates.orderedCCW(S2.refDir(b), c, a, b) ? 1 : 0;
        }
        if (a.equalsPoint(d)) {
            return b.equalsPoint(c) || S2Predicates.orderedCCW(S2.refDir(a), c, b, a) ? -1 : 0;
        }
        if (b.equalsPoint(c)) {
            return S2Predicates.orderedCCW(S2.refDir(b), d, a, b) ? -1 : 0;
        }
        assert (false);
        return 0;
    }

    public static boolean edgeOrVertexCrossing(S2Point a, S2Point b, S2Point c, S2Point d) {
        int crossing = S2EdgeUtil.robustCrossing(a, b, c, d);
        if (crossing < 0) {
            return false;
        }
        if (crossing > 0) {
            return true;
        }
        return S2EdgeUtil.vertexCrossing(a, b, c, d);
    }

    static S2Point closestAcceptableEndpoint(S2Point a0, S2Point a1, S2Point aNorm, S2Point b0, S2Point b1, S2Point bNorm, S2Point x) {
        CloserResult r = new CloserResult(Double.POSITIVE_INFINITY, x);
        if (S2Predicates.orderedCCW(b0, a0, b1, bNorm)) {
            r.replaceIfCloser(x, a0);
        }
        if (S2Predicates.orderedCCW(b0, a1, b1, bNorm)) {
            r.replaceIfCloser(x, a1);
        }
        if (S2Predicates.orderedCCW(a0, b0, a1, aNorm)) {
            r.replaceIfCloser(x, b0);
        }
        if (S2Predicates.orderedCCW(a0, b1, a1, aNorm)) {
            r.replaceIfCloser(x, b1);
        }
        return r.getVmin();
    }

    public static final boolean lenientCrossing(S2Point a, S2Point b, S2Point c, S2Point d) {
        assert (S2.isUnitLength(a));
        assert (S2.isUnitLength(b));
        assert (S2.isUnitLength(c));
        double acb = S2Point.scalarTripleProduct(b, a, c);
        if (Math.abs(acb) < 1.0E-14) {
            return true;
        }
        double bda = S2Point.scalarTripleProduct(a, b, d);
        if (Math.abs(bda) < 1.0E-14) {
            return true;
        }
        if (acb * bda < 0.0) {
            return false;
        }
        double cbd = S2Point.scalarTripleProduct(d, c, b);
        if (Math.abs(cbd) < 1.0E-14) {
            return true;
        }
        double dac = S2Point.scalarTripleProduct(c, d, a);
        if (Math.abs(dac) < 1.0E-14) {
            return true;
        }
        return acb * cbd >= 0.0 && acb * dac >= 0.0;
    }

    public static S2Point getIntersection(S2Point a, S2Point b, S2Point c, S2Point d) {
        return S2EdgeUtil.getIntersection(a, b, c, d, new ResultError());
    }

    static S2Point getIntersection(S2Point a0, S2Point a1, S2Point b0, S2Point b1, ResultError resultError) {
        Preconditions.checkArgument((S2EdgeUtil.robustCrossing(a0, a1, b0, b1) > 0 ? 1 : 0) != 0, (Object)"Input edges a0a1 and b0b1 must have a true robustCrossing.");
        S2Point result = S2EdgeUtil.getIntersectionApprox(a0, a1, b0, b1, resultError);
        if (resultError.error > 1.7763568394002505E-15) {
            result = S2EdgeUtil.getIntersectionExact(a0, a1, b0, b1);
        }
        return S2EdgeUtil.correctIntersectionSign(a0, a1, b0, b1, result);
    }

    static S2Point correctIntersectionSign(S2Point a0, S2Point a1, S2Point b0, S2Point b1, S2Point intersectionResult) {
        if (intersectionResult.dotProd(a0.add(a1).add(b0.add(b1))) < 0.0) {
            intersectionResult = intersectionResult.neg();
        }
        return intersectionResult;
    }

    public static double getDistanceFraction(S2Point x, S2Point a, S2Point b) {
        Preconditions.checkArgument((!a.equalsPoint(b) ? 1 : 0) != 0);
        double d0 = x.angle(a);
        double d1 = x.angle(b);
        return d0 / (d0 + d1);
    }

    public static S1Angle getDistance(S2Point x, S2Point a, S2Point b) {
        Preconditions.checkArgument((boolean)S2.isUnitLength(x), (String)"S2Point not normalized: %s", (Object)x);
        Preconditions.checkArgument((boolean)S2.isUnitLength(a), (String)"S2Point not normalized: %s", (Object)a);
        Preconditions.checkArgument((boolean)S2.isUnitLength(b), (String)"S2Point not normalized: %s", (Object)b);
        return S1Angle.radians(S2EdgeUtil.getDistanceRadians(x, a, b, S2RobustCrossProd.robustCrossProd(a, b)));
    }

    public static S1Angle getDistance(S2Point x, S2Point a, S2Point b, S2Point aCrossB) {
        Preconditions.checkArgument((boolean)S2.isUnitLength(x), (String)"S2Point not normalized: %s", (Object)x);
        Preconditions.checkArgument((boolean)S2.isUnitLength(a), (String)"S2Point not normalized: %s", (Object)a);
        Preconditions.checkArgument((boolean)S2.isUnitLength(b), (String)"S2Point not normalized: %s", (Object)b);
        return S1Angle.radians(S2EdgeUtil.getDistanceRadians(x, a, b, aCrossB));
    }

    public static S1ChordAngle getDistance(S2Point p, S2Edge e) {
        return S2EdgeUtil.updateMinDistance(p, e, S1ChordAngle.INFINITY);
    }

    public static S1ChordAngle updateMinDistance(S2Point p, S2Edge e, S1ChordAngle minDistance) {
        return S2EdgeUtil.updateMinDistance(p, e.getStart(), e.getEnd(), minDistance);
    }

    public static S1ChordAngle updateMinDistance(S2Point x, S2Point a, S2Point b, S1ChordAngle minDistance) {
        Preconditions.checkArgument((boolean)S2.isUnitLength(x), (String)"S2Point not normalized: %s", (Object)x);
        Preconditions.checkArgument((boolean)S2.isUnitLength(a), (String)"S2Point not normalized: %s", (Object)a);
        Preconditions.checkArgument((boolean)S2.isUnitLength(b), (String)"S2Point not normalized: %s", (Object)b);
        double xa2 = x.getDistance2(a);
        double xb2 = x.getDistance2(b);
        double ab2 = a.getDistance2(b);
        double dist2 = Math.min(xa2, xb2);
        if (xa2 + xb2 < ab2 + 2.0 * dist2) {
            S2Point c = S2RobustCrossProd.robustCrossProd(a, b);
            double c2 = c.norm2();
            double xDotC = x.dotProd(c);
            double xDotC2 = xDotC * xDotC;
            if (xDotC2 > c2 * minDistance.getLength2()) {
                return minDistance;
            }
            S2Point cx = c.crossProd(x);
            if (a.dotProd(cx) < 0.0 && b.dotProd(cx) > 0.0) {
                double qr = 1.0 - Math.sqrt(cx.norm2() / c2);
                dist2 = xDotC2 / c2 + qr * qr;
            }
        }
        if (dist2 >= minDistance.getLength2()) {
            return minDistance;
        }
        return S1ChordAngle.fromLength2(dist2);
    }

    public static S1ChordAngle updateMaxDistance(S2Point x, S2Point a, S2Point b, S1ChordAngle maxDistance) {
        S1ChordAngle dist = S1ChordAngle.max(new S1ChordAngle(x, a), new S1ChordAngle(x, b));
        if (dist.compareTo(S1ChordAngle.RIGHT) > 0) {
            dist = S2EdgeUtil.updateMinDistance(x.neg(), a, b, S1ChordAngle.INFINITY);
            dist = S1ChordAngle.sub(S1ChordAngle.STRAIGHT, dist);
        }
        if (maxDistance.compareTo(dist) >= 0) {
            return maxDistance;
        }
        return dist;
    }

    public static S1ChordAngle getEdgePairMinDistance(S2Point a0, S2Point a1, S2Point b0, S2Point b1, S1ChordAngle minDist) {
        if (minDist.equals(S1ChordAngle.ZERO)) {
            return minDist;
        }
        if (S2EdgeUtil.robustCrossing(a0, a1, b0, b1) >= 0) {
            return S1ChordAngle.ZERO;
        }
        minDist = S2EdgeUtil.updateMinDistance(a0, b0, b1, minDist);
        minDist = S2EdgeUtil.updateMinDistance(a1, b0, b1, minDist);
        minDist = S2EdgeUtil.updateMinDistance(b0, a0, a1, minDist);
        minDist = S2EdgeUtil.updateMinDistance(b1, a0, a1, minDist);
        return minDist;
    }

    public static S1ChordAngle getEdgePairDistance(S2Point a0, S2Point a1, S2Point b0, S2Point b1) {
        return S2EdgeUtil.getEdgePairMinDistance(a0, a1, b0, b1, S1ChordAngle.INFINITY);
    }

    public static boolean isEdgePairDistanceLess(S2Point a0, S2Point a1, S2Point b0, S2Point b1, S1ChordAngle distance) {
        if (distance.isZero()) {
            return false;
        }
        if (S2EdgeUtil.robustCrossing(a0, a1, b0, b1) >= 0) {
            return distance.compareTo(S1ChordAngle.ZERO) != 0;
        }
        double r2 = distance.getLength2();
        return S2Predicates.compareEdgeDistance(a0, b0, b1, r2) < 0 || S2Predicates.compareEdgeDistance(a1, b0, b1, r2) < 0 || S2Predicates.compareEdgeDistance(b0, a0, a1, r2) < 0 || S2Predicates.compareEdgeDistance(b1, a0, a1, r2) < 0;
    }

    public static void getEdgePairClosestPoints(S2Point a0, S2Point a1, S2Point b0, S2Point b1, S2Shape.MutableEdge result) {
        if (S2EdgeUtil.robustCrossing(a0, a1, b0, b1) > 0) {
            S2Point intersection;
            result.a = intersection = S2EdgeUtil.getIntersection(a0, a1, b0, b1);
            result.b = intersection;
            return;
        }
        S1ChordAngle actualMin = S1ChordAngle.INFINITY;
        ClosestPoint closest = ClosestPoint.NONE;
        S1ChordAngle newMin = S2EdgeUtil.updateMinDistance(a0, b0, b1, actualMin);
        if (newMin != actualMin) {
            closest = ClosestPoint.A0;
            actualMin = newMin;
        }
        if ((newMin = S2EdgeUtil.updateMinDistance(a1, b0, b1, actualMin)) != actualMin) {
            closest = ClosestPoint.A1;
            actualMin = newMin;
        }
        if ((newMin = S2EdgeUtil.updateMinDistance(b0, a0, a1, actualMin)) != actualMin) {
            closest = ClosestPoint.B0;
            actualMin = newMin;
        }
        if ((newMin = S2EdgeUtil.updateMinDistance(b1, a0, a1, actualMin)) != actualMin) {
            closest = ClosestPoint.B1;
        }
        switch (closest) {
            case A0: {
                result.a = a0;
                result.b = S2EdgeUtil.getClosestPoint(a0, b0, b1);
                return;
            }
            case A1: {
                result.a = a1;
                result.b = S2EdgeUtil.getClosestPoint(a1, b0, b1);
                return;
            }
            case B0: {
                result.a = S2EdgeUtil.getClosestPoint(b0, a0, a1);
                result.b = b0;
                return;
            }
            case B1: {
                result.a = S2EdgeUtil.getClosestPoint(b1, a0, a1);
                result.b = b1;
                return;
            }
        }
        throw new IllegalArgumentException(Platform.formatString("Unknown ClosestPoint case when finding closest points of %s:%s and %s:%s", a0, a1, b0, b1));
    }

    public static boolean isEdgeBNearEdgeA(S2Point a0, S2Point a1, S2Point b0, S2Point b1, S1Angle tolerance) {
        S2Point aNearestB1;
        S2Point aNearestB0;
        assert (tolerance.radians() < 1.5707963267948966);
        assert (tolerance.radians() > 0.0);
        S2Point aOrtho = S2RobustCrossProd.robustCrossProd(a0, a1).normalize();
        if (S2Predicates.sign(aOrtho, aNearestB0 = S2EdgeUtil.getClosestPoint(b0, a0, a1, aOrtho), aNearestB1 = S2EdgeUtil.getClosestPoint(b1, a0, a1, aOrtho)) < 0) {
            aOrtho = aOrtho.neg();
        }
        S1Angle b0Distance = new S1Angle(b0, aNearestB0);
        S1Angle b1Distance = new S1Angle(b1, aNearestB1);
        if (b0Distance.greaterThan(tolerance) || b1Distance.greaterThan(tolerance)) {
            return false;
        }
        S2Point bOrtho = S2RobustCrossProd.robustCrossProd(b0, b1).normalize();
        S1Angle planarAngle = new S1Angle(aOrtho, bOrtho);
        if (planarAngle.lessOrEquals(tolerance)) {
            return true;
        }
        if (planarAngle.radians() >= 3.1315926535897933) {
            boolean b1NearerA0;
            boolean b0NearerA0 = new S1Angle(b0, a0).lessThan(new S1Angle(b0, a1));
            return b0NearerA0 == (b1NearerA0 = new S1Angle(b1, a0).lessThan(new S1Angle(b1, a1)));
        }
        S2Point furthest = aOrtho.sub(bOrtho.mul(aOrtho.dotProd(bOrtho))).normalize();
        assert (S2.isUnitLength(furthest));
        S2Point furthestInv = furthest.neg();
        return !(S2Predicates.sign(bOrtho, b0, furthest) > 0 && S2Predicates.sign(furthest, b1, bOrtho) > 0 || S2Predicates.sign(bOrtho, b0, furthestInv) > 0 && S2Predicates.sign(furthestInv, b1, bOrtho) > 0);
    }

    public static S1ChordAngle getEdgePairMaxDistance(S2Point a0, S2Point a1, S2Point b0, S2Point b1, S1ChordAngle maxDist) {
        if (S1ChordAngle.STRAIGHT.equals(maxDist)) {
            return maxDist;
        }
        if (S2EdgeUtil.robustCrossing(a0, a1, b0.neg(), b1.neg()) >= 0) {
            return S1ChordAngle.STRAIGHT;
        }
        maxDist = S2EdgeUtil.updateMaxDistance(a0, b0, b1, maxDist);
        maxDist = S2EdgeUtil.updateMaxDistance(a1, b0, b1, maxDist);
        maxDist = S2EdgeUtil.updateMaxDistance(b0, a0, a1, maxDist);
        maxDist = S2EdgeUtil.updateMaxDistance(b1, a0, a1, maxDist);
        return maxDist;
    }

    public static double getDistanceRadians(S2Point x, S2Point a, S2Point b, S2Point aCrossB) {
        if (S2Point.scalarTripleProduct(a, x, aCrossB) > 0.0 && S2Point.scalarTripleProduct(b, aCrossB, x) > 0.0) {
            double sinDist = Math.abs(x.dotProd(aCrossB)) / aCrossB.norm();
            return Math.asin(Math.min(1.0, sinDist));
        }
        double linearDist2 = Math.min(S2EdgeUtil.distance2(x, a), S2EdgeUtil.distance2(x, b));
        return 2.0 * Math.asin(Math.min(1.0, 0.5 * Math.sqrt(linearDist2)));
    }

    private static final double distance2(S2Point a, S2Point b) {
        double dx = a.getX() - b.getX();
        double dy = a.getY() - b.getY();
        double dz = a.getZ() - b.getZ();
        return dx * dx + dy * dy + dz * dz;
    }

    public static S2Point getClosestPoint(S2Point x, S2Point a, S2Point b, S2Point aCrossB) {
        assert (S2.isUnitLength(a));
        assert (S2.isUnitLength(b));
        assert (S2.isUnitLength(x));
        S2Point p = x.sub(aCrossB.mul(x.dotProd(aCrossB) / aCrossB.norm2()));
        if (S2Point.scalarTripleProduct(a, p, aCrossB) > 0.0 && S2Point.scalarTripleProduct(b, aCrossB, p) > 0.0) {
            return p.normalize();
        }
        return x.getDistance2(a) <= x.getDistance2(b) ? a : b;
    }

    public static S2Point getClosestPoint(S2Point x, S2Point a, S2Point b) {
        return S2EdgeUtil.getClosestPoint(x, a, b, S2RobustCrossProd.robustCrossProd(a, b));
    }

    public static S2Point project(S2Point x, S2Point a, S2Point b) {
        return S2EdgeUtil.project(x, a, b, S2RobustCrossProd.robustCrossProd(a, b));
    }

    public static S2Point project(S2Point x, S2Point a, S2Point b, S2Point aCrossB) {
        S2Point pn;
        assert (S2.isUnitLength(a));
        assert (S2.isUnitLength(b));
        assert (S2.isUnitLength(x));
        if (x.equalsPoint(a) || x.equalsPoint(b)) {
            return x;
        }
        S2Point n = aCrossB.normalize();
        S2Point p = S2RobustCrossProd.robustCrossProd(n, x).crossProd(n).normalize();
        if (S2Predicates.sign(p, n, a, pn = p.crossProd(n)) > 0 && S2Predicates.sign(p, n, b, pn) < 0) {
            return p;
        }
        return x.getDistance2(a) <= x.getDistance2(b) ? a : b;
    }

    public static S2Point getPointOnRay(S2Point origin, S2Point dir, S1ChordAngle r) {
        assert (S2.isUnitLength(origin));
        assert (S2.isUnitLength(dir));
        assert (origin.dotProd(dir) <= S2.ROBUST_CROSS_PROD_ERROR.radians() + 1.6653345369377348E-16);
        return origin.mul(S1ChordAngle.cos(r)).add(dir.mul(S1ChordAngle.sin(r))).normalize();
    }

    public static S2Point getPointOnRay(S2Point origin, S2Point dir, S1Angle r) {
        assert (S2.isUnitLength(origin));
        assert (S2.isUnitLength(dir));
        assert (origin.dotProd(dir) <= S2.ROBUST_CROSS_PROD_ERROR.radians() + 1.6653345369377348E-16);
        return origin.mul(r.cos()).add(dir.mul(r.sin())).normalize();
    }

    public static S2Point getPointOnLine(S2Point a, S2Point b, S1Angle r) {
        S2Point dir = S2RobustCrossProd.robustCrossProd(a, b).crossProd(a).normalize();
        return S2EdgeUtil.getPointOnRay(a, dir, r);
    }

    public static S2Point getPointOnLine(S2Point a, S2Point b, S1ChordAngle r) {
        S2Point dir = S2RobustCrossProd.robustCrossProd(a, b).crossProd(a).normalize();
        return S2EdgeUtil.getPointOnRay(a, dir, r);
    }

    public static S2Point getPointToLeft(S2Point a, S2Point b, S1Angle r) {
        return S2EdgeUtil.getPointOnRay(a, S2RobustCrossProd.robustCrossProd(a, b).normalize(), r);
    }

    public static S2Point getPointToLeft(S2Point a, S2Point b, S1ChordAngle r) {
        return S2EdgeUtil.getPointOnRay(a, S2RobustCrossProd.robustCrossProd(a, b).normalize(), r);
    }

    public static S2Point getPointToRight(S2Point a, S2Point b, S1Angle r) {
        return S2EdgeUtil.getPointOnRay(a, S2RobustCrossProd.robustCrossProd(b, a).normalize(), r);
    }

    public static S2Point getPointToRight(S2Point a, S2Point b, S1ChordAngle r) {
        return S2EdgeUtil.getPointOnRay(a, S2RobustCrossProd.robustCrossProd(b, a).normalize(), r);
    }

    @Deprecated
    public static S2Point interpolateAtDistance(S1Angle ax, S2Point a, S2Point b, S1Angle ab) {
        assert (S2.isUnitLength(a));
        assert (S2.isUnitLength(b));
        double axRadians = ax.radians();
        double abRadians = ab.radians();
        if (axRadians == 0.0) {
            return a;
        }
        double f = Math.sin(axRadians) / Math.sin(abRadians);
        double e = Math.cos(axRadians) - f * Math.cos(abRadians);
        return a.mul(e).add(b.mul(f)).normalize();
    }

    @Deprecated
    @InlineMe(replacement="S2EdgeUtil.getPointOnLine(a, b, ax)", imports={"com.google.common.geometry.S2EdgeUtil"})
    public static S2Point interpolateAtDistance(S1Angle ax, S2Point a, S2Point b) {
        return S2EdgeUtil.getPointOnLine(a, b, ax);
    }

    public static S2Point interpolate(S2Point a, S2Point b, double t) {
        if (t == 0.0) {
            return a;
        }
        if (t == 1.0) {
            return b;
        }
        S1Angle ab = new S1Angle(a, b);
        return S2EdgeUtil.getPointOnLine(a, b, S1Angle.radians(t * ab.radians()));
    }

    @Deprecated
    @InlineMe(replacement="S2EdgeUtil.interpolate(a, b, t)", imports={"com.google.common.geometry.S2EdgeUtil"})
    public static S2Point interpolate(double t, S2Point a, S2Point b) {
        return S2EdgeUtil.interpolate(a, b, t);
    }

    static double getMinInteriorDistanceMaxError(S1ChordAngle distance) {
        if (distance.compareTo(S1ChordAngle.RIGHT) > 0) {
            return 0.0;
        }
        double x = distance.getLength2();
        double b = 0.5 * x * x;
        double a = x * Math.sqrt(1.0 - 0.5 * b);
        return ((2.5 + 2.0 * Math.sqrt(3.0) + 8.5 * a) * a + (2.0 + 2.0 * Math.sqrt(3.0) / 3.0 + 6.5 * (1.0 - b)) * b + (23.0 + 16.0 / Math.sqrt(3.0)) * 2.220446049250313E-16) * 2.220446049250313E-16;
    }

    static double getMinDistanceMaxError(S1ChordAngle distance) {
        return Math.max(S2EdgeUtil.getMinInteriorDistanceMaxError(distance), distance.getS2PointConstructorMaxError());
    }

    static S2Point getIntersectionExact(S2Point a0, S2Point a1, S2Point b0, S2Point b1) {
        ExactIntersection exact = ExactIntersection.create(a0, a1, b0, b1);
        S2Point x = exact.result.toS2Point().normalize();
        if (x.equals(S2Point.ORIGIN)) {
            x = new S2Point(10.0, 10.0, 10.0);
            S2Point aNorm = exact.aNorm.toS2Point().normalize();
            S2Point bNorm = exact.bNorm.toS2Point().normalize();
            Preconditions.checkArgument((!aNorm.equals(S2Point.ORIGIN) && !bNorm.equals(S2Point.ORIGIN) ? 1 : 0) != 0, (Object)"Exactly antipodal edges not supported by getIntersectionExact");
            x = S2EdgeUtil.closestAcceptableEndpoint(a0, a1, aNorm, b0, b1, bNorm, x);
        }
        assert (S2.isUnitLength(x));
        return x;
    }

    static S2Point getIntersectionApprox(S2Point a0, S2Point a1, S2Point b0, S2Point b1, ResultError resultError) {
        double bLen2;
        double aLen2 = a1.getDistance2(a0);
        if (aLen2 < (bLen2 = b1.getDistance2(b0)) || aLen2 == bLen2 && S2EdgeUtil.compareEdges(a0, a1, b0, b1)) {
            return S2EdgeUtil.getIntersectionApproxSorted(b0, b1, a0, a1, resultError);
        }
        return S2EdgeUtil.getIntersectionApproxSorted(a0, a1, b0, b1, resultError);
    }

    private static boolean compareEdges(S2Point a0, S2Point a1, S2Point b0, S2Point b1) {
        S2Point temp;
        if (a1.lessThan(a0)) {
            temp = a0;
            a0 = a1;
            a1 = temp;
        }
        if (b1.lessThan(b0)) {
            temp = b0;
            b0 = b1;
            b1 = temp;
        }
        return a0.lessThan(b0) || a0.equalsPoint(b0) && b0.lessThan(b1);
    }

    private static S2Point getIntersectionApproxSorted(S2Point a0, S2Point a1, S2Point b0, S2Point b1, ResultError resultError) {
        double errorSum;
        double b1Dist;
        assert (a1.getDistance2(a0) >= b1.getDistance2(b0));
        S2Point aNormal = S2RobustCrossProd.robustCrossProd(a0, a1);
        double aNormalLen = aNormal.norm();
        double bLen = b1.getDistance(b0);
        ResultError b0ResultError = new ResultError();
        ResultError b1ResultError = new ResultError();
        double b0Dist = S2EdgeUtil.getProjection(b0, aNormal, aNormalLen, a0, a1, b0ResultError);
        double distSum = Math.abs(b0Dist - (b1Dist = S2EdgeUtil.getProjection(b1, aNormal, aNormalLen, a0, a1, b1ResultError)));
        if (distSum <= (errorSum = b0ResultError.error + b1ResultError.error)) {
            resultError.error = Double.POSITIVE_INFINITY;
            return S2Point.ORIGIN;
        }
        S2Point x = b1.mul(b0Dist).sub(b0.mul(b1Dist));
        double xLen2 = x.norm2();
        if (xLen2 < Double.MIN_NORMAL) {
            resultError.error = Double.POSITIVE_INFINITY;
            return S2Point.ORIGIN;
        }
        double xLen = Math.sqrt(xLen2);
        double scaledInterpFactor = Math.abs(b0Dist * b1ResultError.error - b1Dist * b0ResultError.error) / (distSum - errorSum);
        resultError.error = (bLen * scaledInterpFactor + 4.440892098500626E-16 * distSum) / xLen + 2.220446049250313E-16;
        return x.mul(1.0 / xLen);
    }

    static double getProjection(S2Point x, S2Point aNormal, double aNormalLen, S2Point a0, S2Point a1, ResultError resultError) {
        double result;
        double dist;
        double x1Dist2;
        S2Point x0 = x.sub(a0);
        S2Point x1 = x.sub(a1);
        double x0Dist2 = x0.norm2();
        if (x0Dist2 < (x1Dist2 = x1.norm2()) || x0Dist2 == x1Dist2 && x0.lessThan(x1)) {
            dist = Math.sqrt(x0Dist2);
            result = x0.dotProd(aNormal);
        } else {
            dist = Math.sqrt(x1Dist2);
            result = x1.dotProd(aNormal);
        }
        resultError.error = 2.220446049250313E-16 * (dist * ((3.5 + 2.0 * Math.sqrt(3.0)) * aNormalLen + 32.0 * Math.sqrt(3.0) * 2.220446049250313E-16) + 1.5 * Math.abs(result));
        return result;
    }

    private S2EdgeUtil() {
    }

    static final class ResultError {
        double error;

        ResultError() {
        }
    }

    static final class ExactIntersection {
        public final BigPoint result;
        public final BigPoint aNorm;
        public final BigPoint bNorm;

        ExactIntersection(BigPoint result, BigPoint aNorm, BigPoint bNorm) {
            this.result = result;
            this.aNorm = aNorm;
            this.bNorm = bNorm;
        }

        public static ExactIntersection create(S2Point a0, S2Point a1, S2Point b0, S2Point b1) {
            BigPoint aNormBp = new BigPoint(a0).crossProd(new BigPoint(a1));
            BigPoint bNormBp = new BigPoint(b0).crossProd(new BigPoint(b1));
            BigPoint xBp = aNormBp.crossProd(bNormBp);
            return new ExactIntersection(xBp, aNormBp, bNormBp);
        }
    }

    static class CloserResult {
        private double dmin2;
        private S2Point vmin;

        public double getDmin2() {
            return this.dmin2;
        }

        public S2Point getVmin() {
            return this.vmin;
        }

        public CloserResult(double dmin2, S2Point vmin) {
            this.dmin2 = dmin2;
            this.vmin = vmin;
        }

        public void replaceIfCloser(S2Point x, S2Point y) {
            double d2 = x.sub(y).norm2();
            if (d2 < this.dmin2 || d2 == this.dmin2 && y.lessThan(this.vmin)) {
                this.dmin2 = d2;
                this.vmin = y;
            }
        }
    }

    static final class FaceSegment {
        int face;
        final R2Vector a = new R2Vector();
        final R2Vector b = new R2Vector();

        FaceSegment() {
        }

        static FaceSegment[] allFaces() {
            FaceSegment[] faces = new FaceSegment[6];
            for (int i = 0; i < faces.length; ++i) {
                faces[i] = new FaceSegment();
            }
            return faces;
        }
    }

    public static class WedgeContainsOrCrosses
    implements WedgeProcessor {
        @Override
        public int test(S2Point a0, S2Point ab1, S2Point a2, S2Point b0, S2Point b2) {
            if (S2Predicates.orderedCCW(a0, a2, b2, ab1)) {
                if (S2Predicates.orderedCCW(b2, b0, a0, ab1)) {
                    return 1;
                }
                return a2.equalsPoint(b2) ? 0 : -1;
            }
            return S2Predicates.orderedCCW(a0, b0, a2, ab1) ? 0 : -1;
        }
    }

    public static class WedgeContainsOrIntersects
    implements WedgeProcessor {
        @Override
        public int test(S2Point a0, S2Point ab1, S2Point a2, S2Point b0, S2Point b2) {
            if (S2Predicates.orderedCCW(a0, a2, b2, ab1)) {
                return S2Predicates.orderedCCW(b2, b0, a0, ab1) ? 1 : -1;
            }
            if (!S2Predicates.orderedCCW(a2, b0, b2, ab1)) {
                return 0;
            }
            return a2.equalsPoint(b0) ? 0 : -1;
        }
    }

    public static class WedgeIntersects
    implements WedgeProcessor {
        @Override
        public int test(S2Point a0, S2Point ab1, S2Point a2, S2Point b0, S2Point b2) {
            return S2Predicates.orderedCCW(a0, b2, b0, ab1) && S2Predicates.orderedCCW(b0, a2, a0, ab1) ? 0 : -1;
        }
    }

    public static class WedgeContains
    implements WedgeProcessor {
        @Override
        public int test(S2Point a0, S2Point ab1, S2Point a2, S2Point b0, S2Point b2) {
            return S2Predicates.orderedCCW(a2, b2, b0, ab1) && S2Predicates.orderedCCW(b0, a0, a2, ab1) ? 1 : 0;
        }
    }

    public static interface WedgeProcessor {
        public int test(S2Point var1, S2Point var2, S2Point var3, S2Point var4, S2Point var5);
    }

    static enum WedgeRelation {
        WEDGE_EQUALS,
        WEDGE_PROPERLY_CONTAINS,
        WEDGE_IS_PROPERLY_CONTAINED,
        WEDGE_PROPERLY_OVERLAPS,
        WEDGE_IS_DISJOINT;

    }

    public static class LongitudePruner {
        private S1Interval interval;
        private double lng0;

        public LongitudePruner(S1Interval interval, S2Point v0) {
            this.interval = interval;
            this.lng0 = S2LatLng.longitude(v0).radians();
        }

        public boolean intersects(S2Point v1) {
            double lng1 = S2LatLng.longitude(v1).radians();
            boolean result = this.interval.intersects(S1Interval.fromPointPair(this.lng0, lng1));
            this.lng0 = lng1;
            return result;
        }
    }

    public static class XYZPruner {
        private S2Point lastVertex;
        private boolean boundSet = false;
        private double xmin;
        private double ymin;
        private double zmin;
        private double xmax;
        private double ymax;
        private double zmax;
        private double maxDeformation;

        public void addEdgeToBounds(S2Point from, S2Point to) {
            if (!this.boundSet) {
                this.boundSet = true;
                this.xmin = this.xmax = from.x;
                this.ymin = this.ymax = from.y;
                this.zmin = this.zmax = from.z;
            }
            this.xmin = Math.min(this.xmin, Math.min(to.x, from.x));
            this.ymin = Math.min(this.ymin, Math.min(to.y, from.y));
            this.zmin = Math.min(this.zmin, Math.min(to.z, from.z));
            this.xmax = Math.max(this.xmax, Math.max(to.x, from.x));
            this.ymax = Math.max(this.ymax, Math.max(to.y, from.y));
            this.zmax = Math.max(this.zmax, Math.max(to.z, from.z));
            double approxArcLen = Math.abs(from.x - to.x) + Math.abs(from.y - to.y) + Math.abs(from.z - to.z);
            this.maxDeformation = approxArcLen < 0.025 ? Math.max(this.maxDeformation, approxArcLen * 0.0025) : (approxArcLen < 1.0 ? Math.max(this.maxDeformation, approxArcLen * 0.11) : approxArcLen * 0.5);
        }

        public void setFirstIntersectPoint(S2Point v0) {
            this.xmin -= this.maxDeformation;
            this.ymin -= this.maxDeformation;
            this.zmin -= this.maxDeformation;
            this.xmax += this.maxDeformation;
            this.ymax += this.maxDeformation;
            this.zmax += this.maxDeformation;
            this.lastVertex = v0;
        }

        public boolean intersects(S2Point v1) {
            boolean result = true;
            if (v1.x < this.xmin && this.lastVertex.x < this.xmin || v1.x > this.xmax && this.lastVertex.x > this.xmax) {
                result = false;
            } else if (v1.y < this.ymin && this.lastVertex.y < this.ymin || v1.y > this.ymax && this.lastVertex.y > this.ymax) {
                result = false;
            } else if (v1.z < this.zmin && this.lastVertex.z < this.zmin || v1.z > this.zmax && this.lastVertex.z > this.zmax) {
                result = false;
            }
            this.lastVertex = v1;
            return result;
        }
    }

    public static class RectBounder {
        private S2LatLngRect.Builder builder = S2LatLngRect.Builder.empty();
        private S2Point a;
        private S2LatLng aLatLng;
        private final S1Interval lngAB = new S1Interval();
        private final R1Interval latAB = new R1Interval();

        public void addPoint(S2LatLng b) {
            this.addPoint(b.toPoint(), b);
        }

        public void addPoint(S2Point b) {
            this.addPoint(b, new S2LatLng(b));
        }

        private void addPoint(S2Point b, S2LatLng bLatLng) {
            assert (S2.skipAssertions || S2.isUnitLength(b));
            if (this.builder.isEmpty()) {
                this.builder.addPoint(bLatLng);
            } else {
                S2Point n = this.a.sub(b).crossProd(this.a.add(b));
                double nNorm = n.norm();
                if (nNorm < 1.91346E-15) {
                    if (this.a.dotProd(b) < 0.0) {
                        this.builder.setFull();
                    } else {
                        this.builder.union(S2LatLngRect.fromPointPair(this.aLatLng, bLatLng));
                    }
                } else {
                    this.lngAB.initFromPointPair(this.aLatLng.lng().radians(), bLatLng.lng().radians());
                    if (this.lngAB.getLength() >= 3.1415926535897927) {
                        this.lngAB.setFull();
                    }
                    this.latAB.initFromPointPair(this.aLatLng.lat().radians(), bLatLng.lat().radians());
                    S2Point m = n.crossProd(S2Point.Z_POS);
                    double mDotA = m.dotProd(this.a);
                    double mDotB = m.dotProd(b);
                    double mError = 6.06638E-16 * nNorm + 6.83174E-31;
                    if (mDotA * mDotB < 0.0 || Math.abs(mDotA) <= mError || Math.abs(mDotB) <= mError) {
                        double maxLat = Math.min(1.5707963267948966, 6.661338147750939E-16 + Math.atan2(Math.sqrt(n.getX() * n.getX() + n.getY() * n.getY()), Math.abs(n.getZ())));
                        double latBudget = 2.0 * Math.asin(0.5 * this.a.sub(b).norm() * Math.sin(maxLat));
                        double maxDelta = 0.5 * (latBudget - this.latAB.getLength()) + 2.220446049250313E-16;
                        if (mDotA <= mError && mDotB >= -mError) {
                            this.latAB.setHi(Math.min(maxLat, this.latAB.hi() + maxDelta));
                        }
                        if (mDotB <= mError && mDotA >= -mError) {
                            this.latAB.setLo(Math.max(-maxLat, this.latAB.lo() - maxDelta));
                        }
                    }
                    this.builder.union(new S2LatLngRect(this.latAB, this.lngAB));
                }
            }
            this.a = b;
            this.aLatLng = bLatLng;
        }

        public S2LatLngRect getBound() {
            S2LatLng expansion = S2LatLng.fromRadians(4.440892098500626E-16, 0.0);
            return this.builder.build().expanded(expansion).polarClosure();
        }

        static S2LatLng maxErrorForTests() {
            return S2LatLng.fromRadians(2.220446049250313E-15, 2.220446049250313E-16);
        }

        static S2LatLngRect expandForSubregions(S2LatLngRect bound) {
            if (bound.isEmpty()) {
                return bound;
            }
            double lngGap = Math.max(0.0, Math.PI - bound.lng().getLength() - 5.551115123125783E-16);
            double minAbsLat = Math.max(bound.lat().lo(), -bound.lat().hi());
            double latGap1 = 1.5707963267948966 + bound.lat().lo();
            double latGap2 = 1.5707963267948966 - bound.lat().hi();
            if (minAbsLat >= 0.0 ? 2.0 * minAbsLat + lngGap < 1.354E-15 : (lngGap >= 1.5707963267948966 ? latGap1 + latGap2 < 1.687E-15 : Math.max(latGap1, latGap2) * lngGap < 1.765E-15)) {
                return S2LatLngRect.full();
            }
            double latExpansion = 1.9984014443252818E-15;
            double lngExpansion = lngGap <= 0.0 ? Math.PI : 0.0;
            return bound.expanded(S2LatLng.fromRadians(latExpansion, lngExpansion)).polarClosure();
        }
    }

    public static final class EdgeCrosser {
        private S2Point a;
        private S2Point b;
        private S2Point aCrossB;
        private S2Point c;
        private int acb;
        int bdaReturn;
        private boolean haveTangents;
        private S2Point aTangent;
        private S2Point bTangent;

        public EdgeCrosser() {
        }

        public EdgeCrosser(S2Point a, S2Point b) {
            this.init(a, b);
        }

        public EdgeCrosser(S2Point a, S2Point b, S2Point c) {
            this(a, b);
            this.restartAt(c);
        }

        public void init(S2Point a, S2Point b) {
            assert (S2.skipAssertions || S2.isUnitLength(a));
            assert (S2.skipAssertions || S2.isUnitLength(b));
            this.a = a;
            this.b = b;
            this.c = null;
            this.aCrossB = a.crossProd(b);
            this.haveTangents = false;
        }

        public S2Point a() {
            return this.a;
        }

        public S2Point b() {
            return this.b;
        }

        public S2Point c() {
            return this.c;
        }

        public S2Point normal() {
            return this.aCrossB;
        }

        public void restartAt(S2Point c) {
            assert (S2.skipAssertions || S2.isUnitLength(c));
            this.c = c;
            this.acb = -S2Predicates.Sign.triage(this.aCrossB, c);
        }

        public int robustCrossing(S2Point d) {
            assert (S2.skipAssertions || S2.isUnitLength(d));
            int bda = S2Predicates.Sign.triage(this.aCrossB, d);
            if (this.acb == -bda && bda != 0) {
                this.c = d;
                this.acb = -bda;
                return -1;
            }
            this.bdaReturn = bda;
            return this.robustCrossingInternal(d);
        }

        public int robustCrossing(S2Point c, S2Point d) {
            if (this.c != c) {
                this.restartAt(c);
            }
            return this.robustCrossing(d);
        }

        public boolean edgeOrVertexCrossing(S2Point d) {
            S2Point c2 = this.c;
            int crossing = this.robustCrossing(d);
            if (crossing < 0) {
                return false;
            }
            if (crossing > 0) {
                return true;
            }
            return S2EdgeUtil.vertexCrossing(this.a, this.b, c2, d);
        }

        public boolean edgeOrVertexCrossing(S2Point c, S2Point d) {
            if (this.c != c) {
                this.restartAt(c);
            }
            return this.edgeOrVertexCrossing(d);
        }

        public int signedEdgeOrVertexCrossing(S2Point c, S2Point d) {
            if (this.c != c) {
                this.restartAt(c);
            }
            return this.signedEdgeOrVertexCrossing(d);
        }

        public int signedEdgeOrVertexCrossing(S2Point d) {
            S2Point c2 = this.c;
            int crossing = this.robustCrossing(d);
            if (crossing < 0) {
                return 0;
            }
            if (crossing > 0) {
                return this.acb;
            }
            return S2EdgeUtil.signedVertexCrossing(this.a, this.b, c2, d);
        }

        public int lastInteriorCrossingSign() {
            return this.acb;
        }

        private int robustCrossingInternal(S2Point d) {
            int result = this.robustCrossingInternal2(d);
            this.c = d;
            this.acb = -this.bdaReturn;
            return result;
        }

        private int robustCrossingInternal2(S2Point d) {
            if (!this.haveTangents) {
                S2Point norm = S2RobustCrossProd.robustCrossProd(this.a, this.b).normalize();
                this.aTangent = this.a.crossProd(norm);
                this.bTangent = norm.crossProd(this.b);
                this.haveTangents = true;
            }
            double kError = (1.5 + 1.0 / Math.sqrt(3.0)) * 2.220446049250313E-16;
            if (this.c.dotProd(this.aTangent) > kError && d.dotProd(this.aTangent) > kError || this.c.dotProd(this.bTangent) > kError && d.dotProd(this.bTangent) > kError) {
                return -1;
            }
            if (this.a.equalsPoint(this.c) || this.a.equalsPoint(d) || this.b.equalsPoint(this.c) || this.b.equalsPoint(d)) {
                return 0;
            }
            if (this.a.equalsPoint(this.b) || this.c.equalsPoint(d)) {
                return -1;
            }
            if (this.acb == 0) {
                this.acb = -S2Predicates.Sign.expensive(this.a, this.b, this.c, true);
                assert (this.acb != 0);
            }
            if (this.bdaReturn == 0) {
                this.bdaReturn = S2Predicates.Sign.expensive(this.a, this.b, d, true);
                assert (this.bdaReturn != 0);
            }
            if (this.bdaReturn != this.acb) {
                return -1;
            }
            S2Point cCrossD = this.c.crossProd(d);
            int cbd = -EdgeCrosser.sign(this.c, d, this.b, cCrossD);
            assert (cbd != 0);
            if (cbd != this.acb) {
                return -1;
            }
            int dac = EdgeCrosser.sign(this.c, d, this.a, cCrossD);
            assert (dac != 0);
            return dac == this.acb ? 1 : -1;
        }

        private static int sign(S2Point a, S2Point b, S2Point c, S2Point aCrossB) {
            int ccw = S2Predicates.Sign.triage(aCrossB, c);
            if (ccw == 0) {
                ccw = S2Predicates.Sign.expensive(a, b, c, true);
            }
            return ccw;
        }
    }

    private static enum ClosestPoint {
        A0,
        A1,
        B0,
        B1,
        NONE;

    }
}

