/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.geometry.EncodedInts;
import com.google.common.geometry.PrimitiveArrays;
import com.google.common.geometry.S2CellId;
import com.google.common.geometry.S2Coder;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Projections;
import com.google.common.geometry.UintVectorCoder;
import com.google.common.geometry.VectorCoder;
import com.google.common.primitives.ImmutableLongArray;
import com.google.common.primitives.Ints;
import com.google.common.primitives.UnsignedInts;
import com.google.common.primitives.UnsignedLongs;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

public class S2PointVectorCoder
implements S2Coder<List<S2Point>> {
    public static final S2PointVectorCoder FAST = new S2PointVectorCoder(Format.FAST);
    public static final S2PointVectorCoder COMPACT = new S2PointVectorCoder(Format.COMPACT);
    private final Format type;
    private static final int FORMAT_FAST = 0;
    private static final int FORMAT_COMPACT = 1;
    private static final int ENCODING_FORMAT_BITS = 3;
    private static final byte ENCODING_FORMAT_MASK = 7;
    private static final int SIZEOF_S2POINT = 24;
    private static final int BLOCK_SHIFT = 4;
    private static final int BLOCK_SIZE = 16;
    private static final long EXCEPTION = S2CellId.sentinel().id();

    private S2PointVectorCoder(Format type) {
        this.type = type;
    }

    @Override
    public void encode(List<S2Point> values, OutputStream output) throws IOException {
        switch (this.type) {
            case FAST: {
                S2PointVectorCoder.encodeFast(values, output);
                break;
            }
            case COMPACT: {
                S2PointVectorCoder.encodeCompact(values, output);
            }
        }
    }

    @Override
    public List<S2Point> decode(PrimitiveArrays.Bytes data, PrimitiveArrays.Cursor cursor) throws IOException {
        int format;
        try {
            format = data.get(cursor.position) & 7;
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException("Insufficient input data: ", e);
        }
        switch (format) {
            case 0: {
                return S2PointVectorCoder.decodeFast(data, cursor);
            }
            case 1: {
                return S2PointVectorCoder.decodeCompact(data, cursor);
            }
        }
        throw new IOException("Invalid encoding format: " + format);
    }

    @Override
    public boolean isLazy() {
        return true;
    }

    private static void encodeFast(List<S2Point> values, OutputStream output) throws IOException {
        long sizeFormat = (long)values.size() << 3 | 0L;
        EncodedInts.writeVarint64(output, sizeFormat);
        for (S2Point point : values) {
            point.encode(output);
        }
    }

    private static List<S2Point> decodeFast(final PrimitiveArrays.Bytes data, PrimitiveArrays.Cursor cursor) throws IOException {
        int size;
        long tmpSize;
        try {
            tmpSize = data.readVarint64(cursor);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new IOException("Insufficient input data: ", e);
        }
        catch (IllegalArgumentException e) {
            throw new IOException("Invalid input data: ", e);
        }
        tmpSize >>= 3;
        try {
            size = Ints.checkedCast((long)tmpSize);
        }
        catch (IllegalArgumentException e) {
            throw new IOException("Invalid input data: ", e);
        }
        final long offset = cursor.position;
        cursor.position += (long)size * 24L;
        return new AbstractList<S2Point>(){

            @Override
            public S2Point get(int index) {
                long position = offset + (long)index * 24L;
                return new S2Point(data.readLittleEndianDouble(position), data.readLittleEndianDouble(position + 8L), data.readLittleEndianDouble(position + 16L));
            }

            @Override
            public int size() {
                return size;
            }
        };
    }

    private static void encodeCompact(List<S2Point> values, OutputStream output) throws IOException {
        ArrayList cellPoints = Lists.newArrayListWithCapacity((int)values.size());
        int level = S2PointVectorCoder.chooseBestLevel(values, cellPoints);
        if (level < 0) {
            S2PointVectorCoder.encodeFast(values, output);
            return;
        }
        ImmutableLongArray cellPointValues = S2PointVectorCoder.convertCellsToValues(cellPoints, level);
        boolean haveExceptions = cellPointValues.contains(EXCEPTION);
        Base base = S2PointVectorCoder.chooseBase(cellPointValues, level, haveExceptions);
        int numBlocks = cellPointValues.length() + 16 - 1 >> 4;
        int baseBytes = base.baseBits >> 3;
        int lastBlockCount = cellPointValues.length() - 16 * (numBlocks - 1);
        Preconditions.checkArgument((lastBlockCount >= 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((lastBlockCount <= 16 ? 1 : 0) != 0);
        Preconditions.checkArgument((baseBytes <= 7 ? 1 : 0) != 0);
        Preconditions.checkArgument((level <= 30 ? 1 : 0) != 0);
        output.write(1 | (haveExceptions ? 1 : 0) << 3 | lastBlockCount - 1 << 4);
        output.write(baseBytes | level << 3);
        int baseShift = S2PointVectorCoder.baseShift(level, base.baseBits);
        EncodedInts.encodeUintWithLength(output, base.base >> baseShift, baseBytes);
        ArrayList<byte[]> blocks = new ArrayList<byte[]>();
        ArrayList<S2Point> exceptions = new ArrayList<S2Point>();
        MutableBlockCode code = new MutableBlockCode();
        for (int i = 0; i < cellPointValues.length(); i += 16) {
            int offsetShift;
            int blockSize = Math.min(16, cellPointValues.length() - i);
            S2PointVectorCoder.getBlockCode(code, cellPointValues.subArray(i, i + blockSize), base.base, haveExceptions);
            ByteArrayOutput block = new ByteArrayOutput();
            int offsetBytes = code.offsetBits >> 3;
            int deltaNibbles = code.deltaBits >> 2;
            int overlapNibbles = code.overlapBits >> 2;
            Preconditions.checkArgument((offsetBytes - overlapNibbles <= 7 ? 1 : 0) != 0);
            Preconditions.checkArgument((overlapNibbles <= 1 ? 1 : 0) != 0);
            Preconditions.checkArgument((deltaNibbles <= 16 ? 1 : 0) != 0);
            block.write(offsetBytes - overlapNibbles | overlapNibbles << 3 | deltaNibbles - 1 << 4);
            long offset = -1L;
            int numExceptions = 0;
            for (int j = 0; j < blockSize; ++j) {
                if (cellPointValues.get(i + j) == EXCEPTION) {
                    ++numExceptions;
                    continue;
                }
                Preconditions.checkArgument((cellPointValues.get(i + j) >= base.base ? 1 : 0) != 0);
                offset = UnsignedLongs.min((long[])new long[]{offset, cellPointValues.get(i + j) - base.base});
            }
            if (numExceptions == blockSize) {
                offset = 0L;
            }
            Preconditions.checkArgument(((offset &= S2PointVectorCoder.bitMask(offsetShift = code.deltaBits - code.overlapBits) ^ 0xFFFFFFFFFFFFFFFFL) == 0L == (offsetBytes == 0) ? 1 : 0) != 0);
            if (offset > 0L) {
                EncodedInts.encodeUintWithLength(block, offset >>> offsetShift, offsetBytes);
            }
            int deltaBytes = deltaNibbles + 1 >> 1;
            exceptions.clear();
            for (int j = 0; j < blockSize; ++j) {
                long delta;
                if (cellPointValues.get(i + j) == EXCEPTION) {
                    delta = exceptions.size();
                    exceptions.add(values.get(i + j));
                } else {
                    Preconditions.checkArgument((UnsignedLongs.compare((long)cellPointValues.get(i + j), (long)(offset + base.base)) >= 0 ? 1 : 0) != 0);
                    delta = cellPointValues.get(i + j) - (offset + base.base);
                    if (haveExceptions) {
                        Preconditions.checkArgument((UnsignedLongs.compare((long)delta, (long)-17L) <= 0 ? 1 : 0) != 0);
                        delta += 16L;
                    }
                }
                Preconditions.checkArgument((UnsignedLongs.compare((long)delta, (long)S2PointVectorCoder.bitMask(code.deltaBits)) <= 0 ? 1 : 0) != 0);
                if ((deltaNibbles & 1) != 0 && (j & 1) != 0) {
                    int lastByte = block.removeLast();
                    delta = delta << 4 | (long)(lastByte & 0xF);
                }
                EncodedInts.encodeUintWithLength(block, delta, deltaBytes);
            }
            if (numExceptions > 0) {
                for (S2Point p : exceptions) {
                    p.encode(block);
                }
            }
            blocks.add(block.toByteArray());
        }
        VectorCoder.BYTE_ARRAY.encode(blocks, output);
    }

    private static List<S2Point> decodeCompact(final PrimitiveArrays.Bytes data, PrimitiveArrays.Cursor cursor) throws IOException {
        long base;
        int level;
        int lastBlockCount;
        boolean haveExceptions;
        try {
            int header1 = data.get(cursor.position++) & 0xFF;
            int header2 = data.get(cursor.position++) & 0xFF;
            if ((header1 & 7) != 1) {
                throw new IOException("Invalid encoding format.");
            }
            haveExceptions = (header1 & 8) != 0;
            lastBlockCount = (header1 >> 4) + 1;
            int baseBytes = header2 & 7;
            level = header2 >> 3;
            long tmpBase = data.readUintWithLength(cursor, baseBytes);
            base = tmpBase << S2PointVectorCoder.baseShift(level, baseBytes << 3);
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException("Insufficient or invalid input bytes: ", e);
        }
        final PrimitiveArrays.Longs blockOffsets = UintVectorCoder.UINT64.decode(data, cursor);
        final long offset = cursor.position;
        final int size = 16 * (blockOffsets.length() - 1) + lastBlockCount;
        cursor.position = cursor.position + (blockOffsets.length() > 0 ? blockOffsets.get(blockOffsets.length() - 1) : 0L);
        return new AbstractList<S2Point>(){

            @Override
            public S2Point get(int index) {
                int iShifted = index >> 4;
                long position = offset + (iShifted == 0 ? 0L : blockOffsets.get(iShifted - 1));
                int header = data.get(position++) & 0xFF;
                int overlapNibbles = header >> 3 & 1;
                int offsetBytes = (header & 7) + overlapNibbles;
                int deltaNibbles = (header >> 4) + 1;
                long offset2 = 0L;
                if (offsetBytes > 0) {
                    int offsetShift = deltaNibbles - overlapNibbles << 2;
                    offset2 = data.readUintWithLength(data.cursor(position), offsetBytes) << offsetShift;
                    position += (long)offsetBytes;
                }
                int deltaNibbleOffset = (index & 0xF) * deltaNibbles;
                int deltaBytes = deltaNibbles + 1 >> 1;
                long deltaPosition = position + (long)(deltaNibbleOffset >> 1);
                long delta = data.readUintWithLength(data.cursor(deltaPosition), deltaBytes);
                delta >>>= (deltaNibbleOffset & 1) << 2;
                delta &= S2PointVectorCoder.bitMask(deltaNibbles << 2);
                if (haveExceptions) {
                    if (delta < 16L) {
                        int blockSize = Math.min(16, size - (index & 0xFFFFFFF0));
                        position += (long)(blockSize * deltaNibbles + 1 >> 1);
                        position += delta * 24L;
                        try {
                            return S2Point.decode(data.toInputStream(position));
                        }
                        catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    }
                    delta -= 16L;
                }
                long value = base + offset2 + delta;
                int shift = 30 - level;
                int sj = EncodedInts.deinterleaveBitPairs1(value);
                int tj = EncodedInts.deinterleaveBitPairs2(value);
                int si = (sj << 1 | 1) << shift & Integer.MAX_VALUE;
                int ti = (tj << 1 | 1) << shift & Integer.MAX_VALUE;
                int face = sj << shift >>> 30 | tj << shift + 1 >>> 29 & 4;
                return S2Projections.faceUvToXyz(face, S2Projections.stToUV(S2Projections.siTiToSt(si)), S2Projections.stToUV(S2Projections.siTiToSt(ti))).normalize();
            }

            @Override
            public int size() {
                return size;
            }
        };
    }

    private static long bitMask(int n) {
        return n == 0 ? 0L : -1L >>> 64 - n;
    }

    private static int maxBitsForLevel(int level) {
        return 2 * level + 3;
    }

    private static int baseShift(int level, int baseBits) {
        return Math.max(0, S2PointVectorCoder.maxBitsForLevel(level) - baseBits);
    }

    private static int chooseBestLevel(List<S2Point> points, List<CellPoint> cellPoints) {
        int[] levelCounts = new int[31];
        for (S2Point point : points) {
            S2Projections.FaceSiTi faceSiTi = S2Projections.xyzToFaceSiTi(point);
            int level = S2Projections.levelIfCenter(faceSiTi, point);
            cellPoints.add(new CellPoint(level, faceSiTi));
            if (level < 0) continue;
            int n = level;
            levelCounts[n] = levelCounts[n] + 1;
        }
        int bestLevel = 0;
        for (int level = 1; level <= 30; ++level) {
            if (levelCounts[level] <= levelCounts[bestLevel]) continue;
            bestLevel = level;
        }
        double minEncodableFraction = 0.05;
        if ((double)levelCounts[bestLevel] <= minEncodableFraction * (double)points.size()) {
            return -1;
        }
        return bestLevel;
    }

    private static ImmutableLongArray convertCellsToValues(List<CellPoint> cellPoints, int level) {
        ImmutableLongArray.Builder builder = ImmutableLongArray.builder((int)cellPoints.size());
        int shift = 30 - level;
        for (CellPoint cp : cellPoints) {
            if (cp.level != level) {
                builder.add(EXCEPTION);
                continue;
            }
            int sj = ((cp.face & 3) << 30 | cp.si >>> 1) >>> shift;
            int tj = ((cp.face & 4) << 29 | cp.ti) >>> shift + 1;
            long v = EncodedInts.interleaveBitPairs(sj, tj);
            Preconditions.checkArgument((UnsignedLongs.compare((long)v, (long)S2PointVectorCoder.bitMask(S2PointVectorCoder.maxBitsForLevel(level))) <= 0 ? 1 : 0) != 0);
            builder.add(v);
        }
        return builder.build();
    }

    private static Base chooseBase(ImmutableLongArray values, int level, boolean haveExceptions) {
        long vMin = EXCEPTION;
        long vMax = 0L;
        for (int i = 0; i < values.length(); ++i) {
            long v = values.get(i);
            if (v == EXCEPTION) continue;
            vMin = UnsignedLongs.min((long[])new long[]{vMin, v});
            vMax = UnsignedLongs.max((long[])new long[]{vMax, v});
        }
        if (vMin == EXCEPTION) {
            return new Base(0L, 0);
        }
        int minDeltaBits = haveExceptions || values.length() == 1 ? 8 : 4;
        int excludedBits = Ints.max((int[])new int[]{63 - Long.numberOfLeadingZeros(vMin ^ vMax) + 1, minDeltaBits, S2PointVectorCoder.baseShift(level, 56)});
        long base = vMin & (S2PointVectorCoder.bitMask(excludedBits) ^ 0xFFFFFFFFFFFFFFFFL);
        int baseBits = 0;
        if (base != 0L) {
            int lowBit = Long.numberOfTrailingZeros(base);
            baseBits = S2PointVectorCoder.maxBitsForLevel(level) - lowBit + 7 & 0xFFFFFFF8;
        }
        return new Base(vMin & (S2PointVectorCoder.bitMask(S2PointVectorCoder.baseShift(level, baseBits)) ^ 0xFFFFFFFFFFFFFFFFL), baseBits);
    }

    private static boolean canEncode(long dMin, long dMax, int deltaBits, int overlapBits, boolean haveExceptions) {
        dMin &= S2PointVectorCoder.bitMask(deltaBits - overlapBits) ^ 0xFFFFFFFFFFFFFFFFL;
        long maxDelta = S2PointVectorCoder.bitMask(deltaBits);
        if (haveExceptions) {
            if (UnsignedLongs.compare((long)maxDelta, (long)16L) < 0) {
                return false;
            }
            maxDelta -= 16L;
        }
        return UnsignedLongs.compare((long)dMin, (long)(maxDelta ^ 0xFFFFFFFFFFFFFFFFL)) > 0 || UnsignedLongs.compare((long)(dMin + maxDelta), (long)dMax) >= 0;
    }

    private static void getBlockCode(MutableBlockCode code, ImmutableLongArray values, long base, boolean haveExceptions) {
        long bMin = EXCEPTION;
        long bMax = 0L;
        for (int i = 0; i < values.length(); ++i) {
            long v = values.get(i);
            if (v == EXCEPTION) continue;
            bMin = UnsignedLongs.min((long[])new long[]{bMin, v});
            bMax = UnsignedLongs.max((long[])new long[]{bMax, v});
        }
        if (bMin == EXCEPTION) {
            code.set(4, 0, 0);
            return;
        }
        int deltaBits = Math.max(1, 63 - Long.numberOfLeadingZeros((bMax -= base) - (bMin -= base))) + 3 & 0xFFFFFFFC;
        int overlapBits = 0;
        if (!S2PointVectorCoder.canEncode(bMin, bMax, deltaBits, 0, haveExceptions)) {
            if (S2PointVectorCoder.canEncode(bMin, bMax, deltaBits, 4, haveExceptions)) {
                overlapBits = 4;
            } else {
                Preconditions.checkArgument((deltaBits <= 60 ? 1 : 0) != 0);
                if (!S2PointVectorCoder.canEncode(bMin, bMax, deltaBits += 4, 0, haveExceptions)) {
                    Preconditions.checkArgument((boolean)S2PointVectorCoder.canEncode(bMin, bMax, deltaBits, 4, haveExceptions));
                    overlapBits = 4;
                }
            }
        }
        if (values.length() == 1 && !haveExceptions) {
            Preconditions.checkArgument((deltaBits == 4 && overlapBits == 0 ? 1 : 0) != 0);
            deltaBits = 8;
        }
        long maxDelta = S2PointVectorCoder.bitMask(deltaBits) - (long)(haveExceptions ? 16 : 0);
        int offsetBits = 0;
        if (UnsignedLongs.compare((long)bMax, (long)maxDelta) > 0) {
            int offsetShift = deltaBits - overlapBits;
            long mask = S2PointVectorCoder.bitMask(offsetShift);
            long minOffset = bMax - maxDelta + mask & (mask ^ 0xFFFFFFFFFFFFFFFFL);
            Preconditions.checkArgument((minOffset != 0L ? 1 : 0) != 0);
            offsetBits = 63 - Long.numberOfLeadingZeros(minOffset) + 1 - offsetShift + 7 & 0xFFFFFFF8;
            if (offsetBits == 64) {
                overlapBits = 4;
            }
        }
        code.set(deltaBits, offsetBits, overlapBits);
    }

    private static class ByteArrayOutput
    extends ByteArrayOutputStream {
        private ByteArrayOutput() {
        }

        int removeLast() {
            Preconditions.checkState((this.count > 0 ? 1 : 0) != 0);
            return this.buf[--this.count];
        }
    }

    private static final class CellPoint {
        short level;
        short face;
        int si;
        int ti;

        CellPoint(int level, S2Projections.FaceSiTi faceSiTi) {
            this.level = (short)level;
            Preconditions.checkArgument((faceSiTi.face >> 8 == 0 ? 1 : 0) != 0);
            this.face = (byte)faceSiTi.face;
            this.si = UnsignedInts.checkedCast((long)faceSiTi.si);
            this.ti = UnsignedInts.checkedCast((long)faceSiTi.ti);
        }
    }

    private static final class Base {
        long base;
        int baseBits;

        Base(long base, int baseBits) {
            this.base = base;
            this.baseBits = baseBits;
        }
    }

    private static final class MutableBlockCode {
        int deltaBits;
        int offsetBits;
        int overlapBits;

        MutableBlockCode() {
        }

        public void set(int deltaBits, int offsetBits, int overlapBits) {
            this.deltaBits = deltaBits;
            this.offsetBits = offsetBits;
            this.overlapBits = overlapBits;
        }
    }

    private static enum Format {
        FAST,
        COMPACT;

    }
}

