/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.base.Preconditions;
import com.google.common.geometry.S2;
import com.google.common.geometry.S2Builder;
import com.google.common.geometry.S2BuilderGraph;
import com.google.common.geometry.S2BuilderUtil;
import com.google.common.geometry.S2ContainsVertexQuery;
import com.google.common.geometry.S2EdgeQuery;
import com.google.common.geometry.S2EdgeUtil;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2Predicates;
import com.google.common.geometry.S2ShapeIndex;
import com.google.common.geometry.primitives.IntVector;
import com.google.common.geometry.primitives.Ints;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.List;

final class S2PolygonDegeneracyFinder {
    private final S2BuilderGraph g;
    private final S2BuilderGraph.VertexInMap in;
    private final S2BuilderGraph.VertexOutMap out;
    private boolean[] isVertexUsed;
    private final boolean[] isEdgeDegeneracy;
    private final boolean[] isVertexUnbalanced;
    private final S2BuilderGraph.Edge reverseInEdge = new S2BuilderGraph.Edge();
    private final S2BuilderGraph.Edge outEdge = new S2BuilderGraph.Edge();

    public static PolygonDegeneracyList findPolygonDegeneracies(S2BuilderGraph g) {
        S2PolygonDegeneracyFinder.checkGraphOptions(g);
        if (g.options().degenerateEdges() == S2Builder.GraphOptions.DegenerateEdges.DISCARD && g.options().siblingPairs() == S2Builder.GraphOptions.SiblingPairs.DISCARD) {
            return new PolygonDegeneracyList();
        }
        S2PolygonDegeneracyFinder df = new S2PolygonDegeneracyFinder(g);
        return df.run();
    }

    public static boolean isFullyDegenerate(S2BuilderGraph g) {
        S2PolygonDegeneracyFinder.checkGraphOptions(g);
        S2BuilderGraph.EdgeList edges = g.edges();
        for (int edgeId = 0; edgeId < g.numEdges(); ++edgeId) {
            if (edges.getSrcId(edgeId) == edges.getDstId(edgeId) || edges.containsSibling(edgeId)) continue;
            return false;
        }
        return true;
    }

    private S2PolygonDegeneracyFinder(S2BuilderGraph g) {
        this.g = g;
        this.in = new S2BuilderGraph.VertexInMap(g);
        this.out = new S2BuilderGraph.VertexOutMap(g);
        this.isEdgeDegeneracy = new boolean[g.numEdges()];
        this.isVertexUnbalanced = new boolean[g.numVertices()];
    }

    private PolygonDegeneracyList run() {
        int numDegeneracies = this.computeDegeneracies();
        if (numDegeneracies == 0) {
            return new PolygonDegeneracyList();
        }
        if (numDegeneracies == this.g.numEdges()) {
            return new PolygonDegeneracyList(this.g.numEdges(), this.g.isFullPolygon());
        }
        ArrayList<Component> components = new ArrayList<Component>();
        int knownVertexId = -1;
        int knownVertexSign = 0;
        int numUnknownSigns = 0;
        this.isVertexUsed = new boolean[this.g.numVertices()];
        for (int edgeId = 0; edgeId < this.g.numEdges(); ++edgeId) {
            int rootVertexId;
            if (!this.isEdgeDegeneracy[edgeId] || this.isVertexUsed[rootVertexId = this.g.edges().getSrcId(edgeId)]) continue;
            Component component = this.buildComponent(rootVertexId);
            if (component.rootSign == 0) {
                ++numUnknownSigns;
            } else {
                knownVertexId = rootVertexId;
                knownVertexSign = component.rootSign;
            }
            components.add(component);
        }
        if (numUnknownSigns > 0) {
            int kMaxUnindexedSignComputations;
            if (knownVertexSign == 0) {
                knownVertexId = this.findUnbalancedVertex();
                knownVertexSign = this.containsVertexSign(knownVertexId);
            }
            if (numUnknownSigns <= (kMaxUnindexedSignComputations = 25)) {
                this.computeUnknownSignsBruteForce(knownVertexId, knownVertexSign, components);
            } else {
                this.computeUnknownSignsIndexed(knownVertexId, knownVertexSign, components);
            }
        }
        return S2PolygonDegeneracyFinder.mergeDegeneracies(components);
    }

    private S2BuilderGraph.Edge getReverse(S2BuilderGraph.EdgeList edges, Ints.IntList inEdgeIds, int inIndex) {
        edges.getReverse(inEdgeIds.get(inIndex), this.reverseInEdge);
        return this.reverseInEdge;
    }

    private int computeDegeneracies() {
        int numDegeneracies = 0;
        Ints.IntList inEdgeIds = this.in.inEdgeIds();
        S2BuilderGraph.EdgeList edges = this.g.edges();
        int numEdges = this.g.numEdges();
        int inIndex = 0;
        for (int outEdgeId = 0; outEdgeId < numEdges; ++outEdgeId) {
            edges.get(outEdgeId, this.outEdge);
            if (this.outEdge.srcId == this.outEdge.dstId) {
                this.isEdgeDegeneracy[outEdgeId] = true;
                ++numDegeneracies;
                continue;
            }
            while (inIndex < numEdges && this.getReverse(edges, inEdgeIds, inIndex).isLessThan(this.outEdge)) {
                ++inIndex;
            }
            if (inIndex < numEdges && this.getReverse(edges, inEdgeIds, inIndex).equals(this.outEdge)) {
                this.isEdgeDegeneracy[outEdgeId] = true;
                ++numDegeneracies;
                continue;
            }
            this.isVertexUnbalanced[this.outEdge.srcId] = true;
        }
        return numDegeneracies;
    }

    private Component buildComponent(int rootVertexId) {
        Component result = new Component();
        result.rootVertexId = rootVertexId;
        ArrayDeque<VertexIdContained> frontier = new ArrayDeque<VertexIdContained>();
        frontier.push(VertexIdContained.of(rootVertexId, true));
        this.isVertexUsed[rootVertexId] = true;
        while (!frontier.isEmpty()) {
            VertexIdContained vertexIdContained = (VertexIdContained)frontier.pop();
            int v0 = vertexIdContained.vertexId;
            boolean v0SameInside = vertexIdContained.contained;
            if (result.rootSign == 0 && this.isVertexUnbalanced[v0]) {
                int v0Sign = this.containsVertexSign(v0);
                Preconditions.checkState((v0Sign != 0 ? 1 : 0) != 0);
                result.rootSign = v0SameInside ? v0Sign : -v0Sign;
            }
            Ints.OfInt outEdges = this.out.edgeIds(v0).intIterator();
            while (outEdges.hasNext()) {
                int edgeId = outEdges.nextInt();
                int v1 = this.g.edges().getDstId(edgeId);
                boolean sameInside = v0SameInside ^ this.crossingParity(v0, v1, false);
                if (this.isEdgeDegeneracy[edgeId]) {
                    result.degeneracies.add(edgeId, sameInside);
                }
                if (this.isVertexUsed[v1]) continue;
                frontier.push(VertexIdContained.of(v1, sameInside ^= this.crossingParity(v1, v0, true)));
                this.isVertexUsed[v1] = true;
            }
        }
        return result;
    }

    private boolean crossingParity(int v0, int v1, boolean includeSame) {
        int crossings = 0;
        S2Point p0 = this.g.vertex(v0);
        S2Point p1 = this.g.vertex(v1);
        S2Point p0Ref = S2.ortho(p0);
        S2BuilderGraph.VertexOutEdges outEdges = this.out.edges(v0);
        for (int i = 0; i < outEdges.size(); ++i) {
            int dstId = outEdges.getDstId(i);
            if (dstId == v1) {
                if (!includeSame) continue;
                ++crossings;
                continue;
            }
            if (!S2Predicates.orderedCCW(p0Ref, this.g.vertex(dstId), p1, p0)) continue;
            ++crossings;
        }
        Ints.OfInt i = this.in.edgeIds(v0).intIterator();
        while (i.hasNext()) {
            int srcId = this.g.edges().getSrcId(i.nextInt());
            if (srcId == v1) {
                if (!includeSame) continue;
                ++crossings;
                continue;
            }
            if (!S2Predicates.orderedCCW(p0Ref, this.g.vertex(srcId), p1, p0)) continue;
            ++crossings;
        }
        return (crossings & 1) != 0;
    }

    private int findUnbalancedVertex() {
        for (int vertexId = 0; vertexId < this.g.numVertices(); ++vertexId) {
            if (!this.isVertexUnbalanced[vertexId]) continue;
            return vertexId;
        }
        throw new IllegalStateException("Could not find previously marked unbalanced vertex");
    }

    private int containsVertexSign(int v0) {
        S2ContainsVertexQuery query = new S2ContainsVertexQuery(this.g.vertex(v0));
        S2BuilderGraph.VertexOutEdges outEdges = this.out.edges(v0);
        for (int i = 0; i < outEdges.size(); ++i) {
            int dstId = outEdges.getDstId(i);
            query.addOutgoing(this.g.vertex(dstId));
        }
        Ints.OfInt i = this.in.edgeIds(v0).intIterator();
        while (i.hasNext()) {
            int srcId = this.g.edges().getSrcId(i.nextInt());
            query.addIncoming(this.g.vertex(srcId));
        }
        return query.containsSign();
    }

    private void computeUnknownSignsBruteForce(int knownVertexId, int knownVertexSign, List<Component> components) {
        S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser();
        for (Component component : components) {
            if (component.rootSign != 0) continue;
            boolean inside = knownVertexSign > 0;
            crosser.init(this.g.vertex(knownVertexId), this.g.vertex(component.rootVertexId));
            for (int edgeId = 0; edgeId < this.g.numEdges(); ++edgeId) {
                if (this.isEdgeDegeneracy[edgeId]) continue;
                inside ^= crosser.edgeOrVertexCrossing(this.g.vertex(this.g.edges().getSrcId(edgeId)), this.g.vertex(this.g.edges().getDstId(edgeId)));
            }
            component.rootSign = inside ? 1 : -1;
        }
    }

    private void computeUnknownSignsIndexed(int knownVertexId, int knownVertexSign, List<Component> components) {
        S2ShapeIndex index = new S2ShapeIndex();
        index.add(new S2BuilderUtil.GraphShape(this.g));
        S2EdgeQuery query = new S2EdgeQuery(index);
        S2EdgeUtil.EdgeCrosser crosser = new S2EdgeUtil.EdgeCrosser();
        for (Component component : components) {
            if (component.rootSign != 0) continue;
            boolean inside = knownVertexSign > 0;
            crosser.init(this.g.vertex(knownVertexId), this.g.vertex(component.rootVertexId));
            S2EdgeQuery.Edges crossingEdges = query.getCandidates(this.g.vertex(knownVertexId), this.g.vertex(component.rootVertexId), 0);
            while (!crossingEdges.isEmpty()) {
                int edgeId = crossingEdges.nextEdge();
                if (this.isEdgeDegeneracy[edgeId]) continue;
                inside ^= crosser.edgeOrVertexCrossing(this.g.vertex(this.g.edges().getSrcId(edgeId)), this.g.vertex(this.g.edges().getDstId(edgeId)));
            }
            component.rootSign = inside ? 1 : -1;
        }
    }

    private static PolygonDegeneracyList mergeDegeneracies(List<Component> components) {
        PolygonDegeneracyList result = new PolygonDegeneracyList();
        for (Component component : components) {
            Preconditions.checkState((component.rootSign != 0 ? 1 : 0) != 0);
            boolean invert = component.rootSign < 0;
            for (int d = 0; d < component.degeneracies.size(); ++d) {
                result.add(component.degeneracies.edgeId(d), component.degeneracies.isHole(d) ^ invert);
            }
        }
        result.sort();
        return result;
    }

    private static void checkGraphOptions(S2BuilderGraph g) {
        Preconditions.checkState((g.options().edgeType() == S2Builder.EdgeType.DIRECTED ? 1 : 0) != 0);
        Preconditions.checkState((g.options().degenerateEdges() == S2Builder.GraphOptions.DegenerateEdges.DISCARD || g.options().degenerateEdges() == S2Builder.GraphOptions.DegenerateEdges.DISCARD_EXCESS ? 1 : 0) != 0);
        Preconditions.checkState((g.options().siblingPairs() == S2Builder.GraphOptions.SiblingPairs.DISCARD || g.options().siblingPairs() == S2Builder.GraphOptions.SiblingPairs.DISCARD_EXCESS ? 1 : 0) != 0);
    }

    private static class Component {
        public int rootVertexId;
        public int rootSign = 0;
        public final PolygonDegeneracyList degeneracies = new PolygonDegeneracyList();

        private Component() {
        }
    }

    public static class PolygonDegeneracyList {
        private final IntVector encoded = IntVector.empty();

        public PolygonDegeneracyList() {
        }

        public PolygonDegeneracyList(int numEdges, boolean isHole) {
            for (int i = 0; i < numEdges; ++i) {
                this.encoded.add(this.encode(i, isHole));
            }
        }

        public void add(int edgeId, boolean isHole) {
            this.encoded.add(this.encode(edgeId, isHole));
        }

        public int size() {
            return this.encoded.size();
        }

        public int edgeId(int index) {
            this.checkIndex(index);
            return this.decodeEdgeId(this.encoded.get(index));
        }

        public boolean isHole(int index) {
            this.checkIndex(index);
            return this.encoded.get(index) < 0;
        }

        public void sort() {
            this.encoded.sort((a, b) -> {
                int edgeIdB;
                int edgeIdA = this.decodeEdgeId(a);
                if (edgeIdA < (edgeIdB = this.decodeEdgeId(b))) {
                    return -1;
                }
                if (edgeIdA > edgeIdB) {
                    return 1;
                }
                return Boolean.compare(this.decodeIsHole(b), this.decodeIsHole(a));
            });
        }

        private void checkIndex(int index) {
            if (index < 0 || index >= this.encoded.size()) {
                throw new IndexOutOfBoundsException("index " + index + " out of bounds on list of size " + this.encoded.size());
            }
        }

        private int encode(int edgeId, boolean isHole) {
            if (isHole) {
                edgeId |= Integer.MIN_VALUE;
            }
            return edgeId;
        }

        private int decodeEdgeId(int encoded) {
            return encoded & Integer.MAX_VALUE;
        }

        private boolean decodeIsHole(int encoded) {
            return encoded < 0;
        }
    }

    private static class VertexIdContained {
        final int vertexId;
        final boolean contained;

        public VertexIdContained(int vertexId, boolean contained) {
            this.vertexId = vertexId;
            this.contained = contained;
        }

        public static VertexIdContained of(int vertexId, boolean contained) {
            return new VertexIdContained(vertexId, contained);
        }
    }
}

