/*
 * Decompiled with CFR 0.152.
 */
package org.datatransferproject.datatransfer.google.photos;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.client.auth.oauth2.Credential;
import com.google.api.client.json.JsonFactory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import org.datatransferproject.api.launcher.Monitor;
import org.datatransferproject.datatransfer.google.common.GoogleCredentialFactory;
import org.datatransferproject.datatransfer.google.common.GoogleErrorLogger;
import org.datatransferproject.datatransfer.google.common.InvalidExportedItemException;
import org.datatransferproject.datatransfer.google.mediaModels.AlbumListResponse;
import org.datatransferproject.datatransfer.google.mediaModels.GoogleAlbum;
import org.datatransferproject.datatransfer.google.mediaModels.GoogleMediaItem;
import org.datatransferproject.datatransfer.google.mediaModels.MediaItemSearchResponse;
import org.datatransferproject.datatransfer.google.photos.GooglePhotosInterface;
import org.datatransferproject.spi.cloud.storage.JobStore;
import org.datatransferproject.spi.transfer.provider.ExportResult;
import org.datatransferproject.spi.transfer.provider.Exporter;
import org.datatransferproject.spi.transfer.types.ContinuationData;
import org.datatransferproject.spi.transfer.types.InvalidTokenException;
import org.datatransferproject.spi.transfer.types.PermissionDeniedException;
import org.datatransferproject.spi.transfer.types.TempMediaData;
import org.datatransferproject.spi.transfer.types.UploadErrorException;
import org.datatransferproject.types.common.ExportInformation;
import org.datatransferproject.types.common.PaginationData;
import org.datatransferproject.types.common.StringPaginationToken;
import org.datatransferproject.types.common.models.ContainerResource;
import org.datatransferproject.types.common.models.DataModel;
import org.datatransferproject.types.common.models.IdOnlyContainerResource;
import org.datatransferproject.types.common.models.photos.PhotoAlbum;
import org.datatransferproject.types.common.models.photos.PhotoModel;
import org.datatransferproject.types.common.models.photos.PhotosContainerResource;
import org.datatransferproject.types.transfer.auth.TokensAndUrlAuthData;
import org.datatransferproject.types.transfer.errors.ErrorDetail;

public class GooglePhotosExporter
implements Exporter<TokensAndUrlAuthData, PhotosContainerResource> {
    static final String ALBUM_TOKEN_PREFIX = "album:";
    static final String PHOTO_TOKEN_PREFIX = "media:";
    private final GoogleCredentialFactory credentialFactory;
    private final JobStore jobStore;
    private final JsonFactory jsonFactory;
    private volatile GooglePhotosInterface photosInterface;
    private final Monitor monitor;

    public GooglePhotosExporter(GoogleCredentialFactory credentialFactory, JobStore jobStore, JsonFactory jsonFactory, Monitor monitor) {
        this.credentialFactory = credentialFactory;
        this.jobStore = jobStore;
        this.jsonFactory = jsonFactory;
        this.monitor = monitor;
    }

    @VisibleForTesting
    GooglePhotosExporter(GoogleCredentialFactory credentialFactory, JobStore jobStore, JsonFactory jsonFactory, GooglePhotosInterface photosInterface, Monitor monitor) {
        this.credentialFactory = credentialFactory;
        this.jobStore = jobStore;
        this.jsonFactory = jsonFactory;
        this.photosInterface = photosInterface;
        this.monitor = monitor;
    }

    public ExportResult<PhotosContainerResource> export(UUID jobId, TokensAndUrlAuthData authData, Optional<ExportInformation> exportInformation) throws IOException, InvalidTokenException, PermissionDeniedException, UploadErrorException {
        boolean paginationDataPresent;
        if (!exportInformation.isPresent()) {
            this.populateContainedPhotosList(jobId, authData);
            return this.exportAlbums(authData, Optional.empty(), jobId);
        }
        if (exportInformation.get().getContainerResource() instanceof PhotosContainerResource) {
            return this.exportPhotosContainer((PhotosContainerResource)exportInformation.get().getContainerResource(), authData, jobId);
        }
        StringPaginationToken paginationToken = (StringPaginationToken)exportInformation.get().getPaginationData();
        IdOnlyContainerResource idOnlyContainerResource = (IdOnlyContainerResource)exportInformation.get().getContainerResource();
        boolean containerResourcePresent = idOnlyContainerResource != null;
        boolean bl = paginationDataPresent = paginationToken != null;
        if (!containerResourcePresent && paginationDataPresent && paginationToken.getToken().startsWith(ALBUM_TOKEN_PREFIX)) {
            return this.exportAlbums(authData, Optional.of(paginationToken), jobId);
        }
        return this.exportPhotos(authData, Optional.ofNullable(idOnlyContainerResource), Optional.ofNullable(paginationToken), jobId);
    }

    private ExportResult<PhotosContainerResource> exportPhotosContainer(PhotosContainerResource container, TokensAndUrlAuthData authData, UUID jobId) throws IOException, InvalidTokenException, PermissionDeniedException {
        ImmutableList.Builder albumBuilder = ImmutableList.builder();
        ImmutableList.Builder photosBuilder = ImmutableList.builder();
        ArrayList<IdOnlyContainerResource> subResources = new ArrayList<IdOnlyContainerResource>();
        for (Object album : container.getAlbums()) {
            GoogleAlbum googleAlbum = this.getOrCreatePhotosInterface(authData).getAlbum(album.getId());
            albumBuilder.add((Object)new PhotoAlbum(googleAlbum.getId(), googleAlbum.getTitle(), null));
            subResources.add(new IdOnlyContainerResource(googleAlbum.getId()));
        }
        ImmutableList.Builder errors = ImmutableList.builder();
        for (PhotoModel photo : container.getPhotos()) {
            GoogleMediaItem googleMediaItem = this.getOrCreatePhotosInterface(authData).getMediaItem(photo.getDataId());
            try {
                photosBuilder.add((Object)GoogleMediaItem.convertToPhotoModel(Optional.empty(), googleMediaItem));
            }
            catch (ParseException e) {
                this.monitor.info(() -> "Parse exception occurred while converting photo, skipping this item. Failure message : %s ", new Object[]{e.getMessage()});
                errors.add((Object)GoogleErrorLogger.createErrorDetail(googleMediaItem.getId(), googleMediaItem.getFilename(), e, true));
            }
        }
        GoogleErrorLogger.logFailedItemErrors(this.jobStore, jobId, (ImmutableList<ErrorDetail>)errors.build());
        PhotosContainerResource photosContainerResource = new PhotosContainerResource((Collection)albumBuilder.build(), (Collection)photosBuilder.build());
        ContinuationData continuationData = new ContinuationData(null);
        subResources.forEach(resource -> continuationData.addContainerResource((ContainerResource)resource));
        return new ExportResult(ExportResult.ResultType.CONTINUE, (DataModel)photosContainerResource, continuationData);
    }

    @VisibleForTesting
    ExportResult<PhotosContainerResource> exportAlbums(TokensAndUrlAuthData authData, Optional<PaginationData> paginationData, UUID jobId) throws IOException, InvalidTokenException, PermissionDeniedException {
        Optional<String> paginationToken = Optional.empty();
        if (paginationData.isPresent()) {
            String token = ((StringPaginationToken)paginationData.get()).getToken();
            Preconditions.checkArgument((boolean)token.startsWith(ALBUM_TOKEN_PREFIX), (Object)("Invalid pagination token " + token));
            paginationToken = Optional.of(token.substring(ALBUM_TOKEN_PREFIX.length()));
        }
        AlbumListResponse albumListResponse = this.getOrCreatePhotosInterface(authData).listAlbums(paginationToken);
        String token = albumListResponse.getNextPageToken();
        ArrayList<PhotoAlbum> albums = new ArrayList<PhotoAlbum>();
        GoogleAlbum[] googleAlbums = albumListResponse.getAlbums();
        StringPaginationToken nextPageData = Strings.isNullOrEmpty((String)token) ? new StringPaginationToken(PHOTO_TOKEN_PREFIX) : new StringPaginationToken(ALBUM_TOKEN_PREFIX + token);
        ContinuationData continuationData = new ContinuationData((PaginationData)nextPageData);
        if (googleAlbums != null && googleAlbums.length > 0) {
            for (GoogleAlbum googleAlbum : googleAlbums) {
                PhotoAlbum photoAlbum = new PhotoAlbum(googleAlbum.getId(), googleAlbum.getTitle(), null);
                albums.add(photoAlbum);
                this.monitor.debug(() -> String.format("%s: Google Photos exporting album: %s", jobId, photoAlbum.getId()), new Object[0]);
                continuationData.addContainerResource((ContainerResource)new IdOnlyContainerResource(googleAlbum.getId()));
            }
        }
        ExportResult.ResultType resultType = ExportResult.ResultType.CONTINUE;
        PhotosContainerResource containerResource = new PhotosContainerResource(albums, null);
        return new ExportResult(resultType, (DataModel)containerResource, continuationData);
    }

    @VisibleForTesting
    ExportResult<PhotosContainerResource> exportPhotos(TokensAndUrlAuthData authData, Optional<IdOnlyContainerResource> albumData, Optional<PaginationData> paginationData, UUID jobId) throws IOException, InvalidTokenException, PermissionDeniedException, UploadErrorException {
        Optional<String> albumId = Optional.empty();
        if (albumData.isPresent()) {
            albumId = Optional.of(albumData.get().getId());
        }
        Optional<String> paginationToken = this.getPhotosPaginationToken(paginationData);
        MediaItemSearchResponse mediaItemSearchResponse = this.getOrCreatePhotosInterface(authData).listMediaItems(albumId, paginationToken);
        StringPaginationToken nextPageData = null;
        if (!Strings.isNullOrEmpty((String)mediaItemSearchResponse.getNextPageToken())) {
            nextPageData = new StringPaginationToken(PHOTO_TOKEN_PREFIX + mediaItemSearchResponse.getNextPageToken());
        }
        ContinuationData continuationData = new ContinuationData(nextPageData);
        PhotosContainerResource containerResource = null;
        GoogleMediaItem[] mediaItems = mediaItemSearchResponse.getMediaItems();
        if (mediaItems != null && mediaItems.length > 0) {
            List<PhotoModel> photos = this.convertPhotosList(albumId, mediaItems, jobId);
            containerResource = new PhotosContainerResource(null, photos);
        }
        ExportResult.ResultType resultType = ExportResult.ResultType.CONTINUE;
        if (nextPageData == null) {
            resultType = ExportResult.ResultType.END;
        }
        return new ExportResult(resultType, containerResource, continuationData);
    }

    @VisibleForTesting
    void populateContainedPhotosList(UUID jobId, TokensAndUrlAuthData authData) throws IOException, InvalidTokenException, PermissionDeniedException, UploadErrorException {
        TempMediaData tempMediaData = new TempMediaData(jobId);
        String albumToken = null;
        do {
            AlbumListResponse albumListResponse = this.getOrCreatePhotosInterface(authData).listAlbums(Optional.ofNullable(albumToken));
            albumToken = albumListResponse.getNextPageToken();
            if (albumListResponse.getAlbums() == null) continue;
            for (GoogleAlbum album : albumListResponse.getAlbums()) {
                MediaItemSearchResponse containedMediaSearchResponse;
                String albumId = album.getId();
                String photoToken = null;
                do {
                    if ((containedMediaSearchResponse = this.getOrCreatePhotosInterface(authData).listMediaItems(Optional.of(albumId), Optional.ofNullable(photoToken))).getMediaItems() == null) continue;
                    for (GoogleMediaItem mediaItem : containedMediaSearchResponse.getMediaItems()) {
                        tempMediaData.addContainedPhotoId(mediaItem.getId());
                    }
                } while ((photoToken = containedMediaSearchResponse.getNextPageToken()) != null);
            }
        } while (albumToken != null);
        InputStream stream = GooglePhotosExporter.convertJsonToInputStream(tempMediaData);
        this.jobStore.create(jobId, GooglePhotosExporter.createCacheKey(), stream);
    }

    @VisibleForTesting
    static InputStream convertJsonToInputStream(Object jsonObject) throws JsonProcessingException {
        String tempString = new ObjectMapper().writeValueAsString(jsonObject);
        return new ByteArrayInputStream(tempString.getBytes(StandardCharsets.UTF_8));
    }

    private Optional<String> getPhotosPaginationToken(Optional<PaginationData> paginationData) {
        Optional<String> paginationToken = Optional.empty();
        if (paginationData.isPresent()) {
            String token = ((StringPaginationToken)paginationData.get()).getToken();
            Preconditions.checkArgument((boolean)token.startsWith(PHOTO_TOKEN_PREFIX), (Object)("Invalid pagination token " + token));
            if (token.length() > PHOTO_TOKEN_PREFIX.length()) {
                paginationToken = Optional.of(token.substring(PHOTO_TOKEN_PREFIX.length()));
            }
        }
        return paginationToken;
    }

    private List<PhotoModel> convertPhotosList(Optional<String> albumId, GoogleMediaItem[] mediaItems, UUID jobId) throws IOException {
        TempMediaData tempMediaData = null;
        InputStream stream = this.jobStore.getStream(jobId, GooglePhotosExporter.createCacheKey()).getStream();
        if (stream != null) {
            tempMediaData = (TempMediaData)new ObjectMapper().readValue(stream, TempMediaData.class);
            stream.close();
        }
        return this.convertMediaItemsToPhotoModels(jobId, albumId, mediaItems, tempMediaData);
    }

    private List<PhotoModel> convertMediaItemsToPhotoModels(UUID jobId, Optional<String> albumId, GoogleMediaItem[] mediaItems, TempMediaData tempMediaData) throws IOException {
        ArrayList<PhotoModel> photos = new ArrayList<PhotoModel>(mediaItems.length);
        ImmutableList.Builder errors = ImmutableList.builder();
        for (GoogleMediaItem mediaItem : mediaItems) {
            try {
                PhotoModel photoModel = GooglePhotosExporter.convertToPhotoModel(mediaItem, albumId, tempMediaData);
                photos.add(photoModel);
                this.monitor.debug(() -> String.format("%s: Google exporting photo: %s", jobId, photoModel.getDataId()), new Object[0]);
            }
            catch (ParseException | InvalidExportedItemException e) {
                this.monitor.info(() -> String.format("%s: MediaItem (id: %s) failed to be converted to PhotoModel, and is being skipped: %s", jobId, mediaItem.getId(), e), new Object[0]);
                errors.add((Object)GoogleErrorLogger.createErrorDetail(mediaItem.getId(), mediaItem.getFilename(), e, true));
            }
        }
        GoogleErrorLogger.logFailedItemErrors(this.jobStore, jobId, (ImmutableList<ErrorDetail>)errors.build());
        return photos;
    }

    private static PhotoModel convertToPhotoModel(GoogleMediaItem mediaItem, Optional<String> albumId, TempMediaData tempMediaData) throws InvalidExportedItemException, ParseException {
        boolean shouldUpload = albumId.isPresent();
        if (tempMediaData != null) {
            boolean bl = shouldUpload = shouldUpload || !tempMediaData.isContainedPhotoId(mediaItem.getId());
        }
        if (!shouldUpload) {
            throw new InvalidExportedItemException(String.format("Failed to convert media item (id: %s) into a PhotoModel", mediaItem.getId()));
        }
        try {
            return GoogleMediaItem.convertToPhotoModel(albumId, mediaItem);
        }
        catch (IllegalArgumentException | ParseException e) {
            throw new InvalidExportedItemException(String.format("Failed to convert media item (id: %s) into a PhotoModel : %s", mediaItem.getId(), e));
        }
    }

    private synchronized GooglePhotosInterface getOrCreatePhotosInterface(TokensAndUrlAuthData authData) {
        return this.photosInterface == null ? this.makePhotosInterface(authData) : this.photosInterface;
    }

    private synchronized GooglePhotosInterface makePhotosInterface(TokensAndUrlAuthData authData) {
        Credential credential = this.credentialFactory.createCredential(authData);
        return new GooglePhotosInterface(this.credentialFactory, credential, this.jsonFactory, this.monitor, 1.0);
    }

    private static String createCacheKey() {
        return "tempMediaData";
    }
}

