/*
 * Decompiled with CFR 0.152.
 */
package org.datatransferproject.transfer.microsoft.media;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.client.auth.oauth2.Credential;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.RateLimiter;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Optional;
import java.util.UUID;
import javax.annotation.Nonnull;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.apache.commons.lang3.tuple.Pair;
import org.datatransferproject.api.launcher.Monitor;
import org.datatransferproject.spi.api.transport.DiscardingStreamCounter;
import org.datatransferproject.spi.api.transport.JobFileStream;
import org.datatransferproject.spi.cloud.storage.TemporaryPerJobDataStore;
import org.datatransferproject.spi.transfer.idempotentexecutor.IdempotentImportExecutor;
import org.datatransferproject.spi.transfer.provider.ImportResult;
import org.datatransferproject.spi.transfer.provider.Importer;
import org.datatransferproject.spi.transfer.types.CopyExceptionWithFailureReason;
import org.datatransferproject.spi.transfer.types.DestinationMemoryFullException;
import org.datatransferproject.spi.transfer.types.PermissionDeniedException;
import org.datatransferproject.transfer.microsoft.DataChunk;
import org.datatransferproject.transfer.microsoft.MicrosoftApiResponse;
import org.datatransferproject.transfer.microsoft.MicrosoftTransmogrificationConfig;
import org.datatransferproject.transfer.microsoft.StreamChunker;
import org.datatransferproject.transfer.microsoft.common.MicrosoftCredentialFactory;
import org.datatransferproject.types.common.DownloadableFile;
import org.datatransferproject.types.common.models.TransmogrificationConfig;
import org.datatransferproject.types.common.models.media.MediaAlbum;
import org.datatransferproject.types.common.models.media.MediaContainerResource;
import org.datatransferproject.types.transfer.auth.TokensAndUrlAuthData;

public class MicrosoftMediaImporter
implements Importer<TokensAndUrlAuthData, MediaContainerResource> {
    private static final int MICROSOFT_UPLOAD_CHUNK_BYTE_SIZE = 32768000;
    private final OkHttpClient client;
    private final ObjectMapper objectMapper;
    private final TemporaryPerJobDataStore jobStore;
    private final Monitor monitor;
    private final MicrosoftCredentialFactory credentialFactory;
    private final JobFileStream jobFileStream;
    private final RateLimiter writeRateLimiter;
    private final MicrosoftTransmogrificationConfig transmogrificationConfig = new MicrosoftTransmogrificationConfig();
    private Credential credential;
    private final String createFolderUrl;
    private final String uploadMediaUrlTemplate;
    private final String albumlessMediaUrlTemplate;
    private static final String UPLOAD_PARAMS = "?@microsoft.graph.conflictBehavior=rename";

    public MicrosoftMediaImporter(String baseUrl, OkHttpClient client, ObjectMapper objectMapper, TemporaryPerJobDataStore jobStore, Monitor monitor, MicrosoftCredentialFactory credentialFactory, JobFileStream jobFileStream, double maxWritesPerSecond) {
        this.createFolderUrl = baseUrl + "/v1.0/me/drive/special/photos/children";
        this.albumlessMediaUrlTemplate = baseUrl + "/v1.0/me/drive/special/photos:/%s:/createUploadSession%s";
        this.uploadMediaUrlTemplate = baseUrl + "/v1.0/me/drive/items/%s:/%s:/createUploadSession%s";
        this.client = client;
        this.objectMapper = objectMapper;
        this.jobStore = jobStore;
        this.monitor = monitor;
        this.credentialFactory = credentialFactory;
        this.credential = null;
        this.jobFileStream = jobFileStream;
        this.writeRateLimiter = RateLimiter.create((double)maxWritesPerSecond);
    }

    public ImportResult importItem(UUID jobId, IdempotentImportExecutor idempotentImportExecutor, TokensAndUrlAuthData authData, MediaContainerResource resource) throws Exception {
        this.getOrCreateCredential(authData);
        this.logDebugJobStatus("%s before transmogrification", jobId, resource);
        resource.transmogrify((TransmogrificationConfig)this.transmogrificationConfig);
        this.logDebugJobStatus("%s after transmogrification", jobId, resource);
        for (MediaAlbum album : resource.getAlbums()) {
            idempotentImportExecutor.executeAndSwallowIOExceptions(album.getId(), album.getName(), () -> this.createOneDriveFolder(album));
        }
        this.executeIdempotentImport(jobId, idempotentImportExecutor, resource.getVideos());
        this.executeIdempotentImport(jobId, idempotentImportExecutor, resource.getPhotos());
        return ImportResult.OK;
    }

    private void logDebugJobStatus(String format, UUID jobId, MediaContainerResource resource) {
        String statusMessage = String.format("%s: Importing %s albums, %s photos, and %s videos", jobId, resource.getAlbums().size(), resource.getPhotos().size(), resource.getVideos().size());
        this.monitor.debug(() -> String.format(format, statusMessage), new Object[0]);
    }

    private String createOneDriveFolder(MediaAlbum album) throws IOException, CopyExceptionWithFailureReason {
        LinkedHashMap<String, Object> rawFolder = new LinkedHashMap<String, Object>();
        String albumName = Strings.isNullOrEmpty((String)album.getName()) ? "Untitled" : album.getName();
        rawFolder.put("name", albumName);
        rawFolder.put("folder", new LinkedHashMap());
        rawFolder.put("@microsoft.graph.conflictBehavior", "rename");
        Request.Builder requestBuilder = new Request.Builder().url(this.createFolderUrl);
        requestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
        requestBuilder.post(RequestBody.create((MediaType)MediaType.parse((String)"application/json"), (String)this.objectMapper.writeValueAsString(rawFolder)));
        return this.tryWithCredsOrFail(requestBuilder, "id", "creating empty folder");
    }

    private void executeIdempotentImport(UUID jobId, IdempotentImportExecutor idempotentImportExecutor, Collection<? extends DownloadableFile> downloadableFiles) throws Exception {
        for (DownloadableFile downloadableFile : downloadableFiles) {
            idempotentImportExecutor.executeAndSwallowIOExceptions(downloadableFile.getIdempotentId(), downloadableFile.getName(), () -> this.importDownloadableItem(downloadableFile, jobId, idempotentImportExecutor));
        }
    }

    private String importDownloadableItem(DownloadableFile item, UUID jobId, IdempotentImportExecutor idempotentImportExecutor) throws Exception {
        long totalFileSize = DiscardingStreamCounter.discardForLength((InputStream)this.jobFileStream.streamFile(item, jobId, this.jobStore));
        if (totalFileSize <= 0L) {
            throw new IOException(String.format("jobid %s hit empty unexpectedly empty (bytes=%d) download for file %s", jobId, totalFileSize, item.getFetchableUrl()));
        }
        try (InputStream fileStream = this.jobFileStream.streamFile(item, jobId, this.jobStore);){
            String itemUploadUrl = this.createUploadSession(item, idempotentImportExecutor);
            MicrosoftApiResponse finalChunkResponse = this.uploadStreamInChunks(totalFileSize, itemUploadUrl, item.getMimeType(), fileStream);
            Preconditions.checkState((boolean)finalChunkResponse.isOkay(), (String)"final chunk-upload response should have had an ID, but a non-OK response came back: %s", (Object)finalChunkResponse.toString());
            String string = finalChunkResponse.getJsonValue(this.objectMapper, "id", "final chunk-upload response should have had ID, but got empty HTTP response-body");
            return string;
        }
    }

    private MicrosoftApiResponse uploadStreamInChunks(long totalFileSize, String itemUploadUrl, String itemMimeType, InputStream inputStream) throws IOException, DestinationMemoryFullException, PermissionDeniedException {
        Optional<DataChunk> currentChunk;
        MicrosoftApiResponse lastChunkResponse = null;
        StreamChunker streamChunker = new StreamChunker(32768000, inputStream);
        while (!(currentChunk = streamChunker.nextChunk()).isEmpty()) {
            lastChunkResponse = this.uploadChunk(currentChunk.get(), itemUploadUrl, totalFileSize, itemMimeType);
            DataChunk lastChunksent = currentChunk.get();
            int httpStatus = lastChunkResponse.httpStatus();
            this.monitor.info(() -> String.format("Uploaded chunk range %d-%d (of total bytesize: %d) successfuly, HTTP status %d", lastChunksent.streamByteOffset(), lastChunksent.finalByteOffset(), totalFileSize, httpStatus), new Object[0]);
        }
        return (MicrosoftApiResponse)Preconditions.checkNotNull(lastChunkResponse, (Object)"bug: empty-stream already checked for yet stream empty now?");
    }

    private String createUploadSession(DownloadableFile item, IdempotentImportExecutor idempotentImportExecutor) throws IOException, CopyExceptionWithFailureReason {
        Request.Builder createSessionRequestBuilder = this.buildCreateUploadSessionPath(item, idempotentImportExecutor);
        createSessionRequestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
        createSessionRequestBuilder.header("Content-Type", "application/json");
        createSessionRequestBuilder.post(RequestBody.create((MediaType)MediaType.parse((String)"application/json"), (String)this.objectMapper.writeValueAsString((Object)ImmutableMap.of())));
        return this.tryWithCredsOrFail(createSessionRequestBuilder, "uploadUrl", "creating initial upload session");
    }

    private Request.Builder buildCreateUploadSessionPath(DownloadableFile item, IdempotentImportExecutor idempotentImportExecutor) {
        String createSessionUrl;
        if (Strings.isNullOrEmpty((String)item.getFolderId())) {
            createSessionUrl = String.format(this.albumlessMediaUrlTemplate, item.getName(), UPLOAD_PARAMS);
        } else {
            String oneDriveFolderId = (String)((Object)idempotentImportExecutor.getCachedValue(item.getFolderId()));
            createSessionUrl = String.format(this.uploadMediaUrlTemplate, oneDriveFolderId, item.getName(), UPLOAD_PARAMS);
        }
        return new Request.Builder().url(createSessionUrl);
    }

    private MicrosoftApiResponse uploadChunk(DataChunk chunk, String photoUploadUrl, long totalFileSize, String mediaType) throws IOException, DestinationMemoryFullException, PermissionDeniedException {
        Request.Builder uploadRequestBuilder = new Request.Builder().url(photoUploadUrl);
        RequestBody uploadChunkBody = RequestBody.create((MediaType)MediaType.parse((String)mediaType), (byte[])chunk.chunk(), (int)0, (int)chunk.size());
        uploadRequestBuilder.put(uploadChunkBody);
        String contentRange = String.format("bytes %d-%d/%d", chunk.streamByteOffset(), chunk.finalByteOffset(), totalFileSize);
        uploadRequestBuilder.header("Content-Range", contentRange);
        uploadRequestBuilder.header("Content-Length", String.format("%d", chunk.size()));
        return this.tryWithCredsOrFail(uploadRequestBuilder, String.format("uploading one chunk (%s) mediaType=%s amid %d total bytes", contentRange, mediaType, totalFileSize));
    }

    private Credential getOrCreateCredential(TokensAndUrlAuthData authData) {
        if (this.credential == null) {
            this.credential = this.credentialFactory.createCredential(authData);
        }
        return this.credential;
    }

    private MicrosoftApiResponse sendMicrosoftRequest(Request.Builder requestBuilder) throws IOException {
        this.writeRateLimiter.acquire();
        return MicrosoftApiResponse.ofResponse((Response)Preconditions.checkNotNull((Object)this.client.newCall(requestBuilder.build()).execute(), (String)"null microsoft server response for %s", (Object)requestBuilder.build().url()));
    }

    private Pair<Request, MicrosoftApiResponse> tryWithCreds(Request.Builder requestBuilder) throws IOException {
        MicrosoftApiResponse response = this.sendMicrosoftRequest(requestBuilder);
        if (response.isTokenRefreshRequired()) {
            this.credentialFactory.refreshCredential(this.credential);
            this.monitor.info(() -> "Refreshed Microsoft authorization token successfuly", new Object[0]);
            requestBuilder.header("Authorization", "Bearer " + this.credential.getAccessToken());
            response = this.sendMicrosoftRequest(requestBuilder);
        }
        return Pair.of((Object)requestBuilder.build(), (Object)response);
    }

    private MicrosoftApiResponse tryWithCredsOrFail(Request.Builder req, String causeMessage) throws IOException, DestinationMemoryFullException, PermissionDeniedException {
        Pair<Request, MicrosoftApiResponse> reqResp = this.tryWithCreds(req);
        MicrosoftApiResponse response = (MicrosoftApiResponse)reqResp.getRight();
        Optional<MicrosoftApiResponse.RecoverableState> recovery = response.recoverableState();
        if (recovery.isPresent()) {
            switch (recovery.get()) {
                case RECOVERABLE_STATE_OKAY: {
                    return response;
                }
                case RECOVERABLE_STATE_NEEDS_TOKEN_REFRESH: {
                    throw response.toIoException(String.format("bug! microsoft server needs token refresh immediately after a refreshing: %s", causeMessage));
                }
            }
            throw new AssertionError((Object)"exhaustive switch");
        }
        return response.returnConvertDtpException(String.format("%s: for request url \"%s\" and bearer token \"%s\"\n", causeMessage, ((Request)reqResp.getLeft()).url(), this.credential.getAccessToken()));
    }

    @Nonnull
    private String tryWithCredsOrFail(Request.Builder requestBuilder, String jsonResponseKey, String causeMessage) throws IOException, DestinationMemoryFullException, PermissionDeniedException {
        MicrosoftApiResponse resp = this.tryWithCredsOrFail(requestBuilder, causeMessage);
        return resp.getJsonValue(this.objectMapper, jsonResponseKey, causeMessage);
    }
}

