/*-
 *  * Copyright 2016 Skymind, Inc.
 *  *
 *  *    Licensed under the Apache License, Version 2.0 (the "License");
 *  *    you may not use this file except in compliance with the License.
 *  *    You may obtain a copy of the License at
 *  *
 *  *        http://www.apache.org/licenses/LICENSE-2.0
 *  *
 *  *    Unless required by applicable law or agreed to in writing, software
 *  *    distributed under the License is distributed on an "AS IS" BASIS,
 *  *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  *    See the License for the specific language governing permissions and
 *  *    limitations under the License.
 */
package org.datavec.image.transform;

import org.bytedeco.javacv.OpenCVFrameConverter;
import org.datavec.image.data.ImageWritable;
import org.nd4j.shade.jackson.annotation.JsonInclude;
import org.nd4j.shade.jackson.annotation.JsonProperty;

import java.util.Random;

import static org.bytedeco.javacpp.opencv_core.Mat;
import static org.bytedeco.javacpp.opencv_core.Rect;

/**
 * Crops images deterministically or randomly.
 *
 * @author saudet
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class CropImageTransform extends BaseImageTransform<Mat> {

    private int cropTop;
    private int cropLeft;
    private int cropBottom;
    private int cropRight;

    /** Calls {@code this(null, crop, crop, crop, crop)}. */
    public CropImageTransform(int crop) {
        this(null, crop, crop, crop, crop);
    }

    /** Calls {@code this(random, crop, crop, crop, crop)}. */
    public CropImageTransform(Random random, int crop) {
        this(random, crop, crop, crop, crop);
    }

    /** Calls {@code this(random, cropTop, cropLeft, cropBottom, cropRight)}. */
    public CropImageTransform(@JsonProperty("cropTop") int cropTop, @JsonProperty("cropLeft") int cropLeft,
                    @JsonProperty("cropBottom") int cropBottom, @JsonProperty("cropRight") int cropRight) {
        this(null, cropTop, cropLeft, cropBottom, cropRight);
    }

    /**
     * Constructs an instance of the ImageTransform.
     *
     * @param random     object to use (or null for deterministic)
     * @param cropTop    maximum cropping of the top of the image (pixels)
     * @param cropLeft   maximum cropping of the left of the image (pixels)
     * @param cropBottom maximum cropping of the bottom of the image (pixels)
     * @param cropRight  maximum cropping of the right of the image (pixels)
     */
    public CropImageTransform(Random random, int cropTop, int cropLeft, int cropBottom, int cropRight) {
        super(random);
        this.cropTop = cropTop;
        this.cropLeft = cropLeft;
        this.cropBottom = cropBottom;
        this.cropRight = cropRight;
        this.converter = new OpenCVFrameConverter.ToMat();
    }

    /**
     * Takes an image and returns a transformed image.
     * Uses the random object in the case of random transformations.
     *
     * @param image  to transform, null == end of stream
     * @param random object to use (or null for deterministic)
     * @return transformed image
     */
    @Override
    public ImageWritable transform(ImageWritable image, Random random) {
        if (image == null) {
            return null;
        }
        Mat mat = converter.convert(image.getFrame());
        int top = random != null ? random.nextInt(cropTop + 1) : cropTop;
        int left = random != null ? random.nextInt(cropLeft + 1) : cropLeft;
        int bottom = random != null ? random.nextInt(cropBottom + 1) : cropBottom;
        int right = random != null ? random.nextInt(cropRight + 1) : cropRight;

        int y = Math.min(top, mat.rows() - 1);
        int x = Math.min(left, mat.cols() - 1);
        int h = Math.max(1, mat.rows() - bottom - y);
        int w = Math.max(1, mat.cols() - right - x);
        Mat result = mat.apply(new Rect(x, y, w, h));

        return new ImageWritable(converter.convert(result));
    }

}
