/*-
 *  * Copyright 2016 Skymind, Inc.
 *  *
 *  *    Licensed under the Apache License, Version 2.0 (the "License");
 *  *    you may not use this file except in compliance with the License.
 *  *    You may obtain a copy of the License at
 *  *
 *  *        http://www.apache.org/licenses/LICENSE-2.0
 *  *
 *  *    Unless required by applicable law or agreed to in writing, software
 *  *    distributed under the License is distributed on an "AS IS" BASIS,
 *  *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  *    See the License for the specific language governing permissions and
 *  *    limitations under the License.
 */
package org.datavec.image.transform;

import org.bytedeco.javacv.OpenCVFrameConverter;
import org.datavec.image.data.ImageWritable;
import org.nd4j.shade.jackson.annotation.JsonInclude;
import org.nd4j.shade.jackson.annotation.JsonProperty;

import java.util.Random;

import static org.bytedeco.javacpp.opencv_core.Mat;
import static org.bytedeco.javacpp.opencv_core.Size;
import static org.bytedeco.javacpp.opencv_imgproc.resize;

/**
 * ScaleImageTransform is aim to scale by a certain <b>random factor</b>,
 * <b>not</b> the <b>final size</b> of the image.
 *
 * @author saudet
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class ScaleImageTransform extends BaseImageTransform<Mat> {

    private float dx;
    private float dy;

    /** Calls {@code this(null, delta, delta)}. */
    public ScaleImageTransform(float delta) {
        this(null, delta, delta);
    }

    /** Calls {@code this(random, delta, delta)}. */
    public ScaleImageTransform(Random random, float delta) {
        this(random, delta, delta);
    }

    /** Calls {@code this(null, dx, dy)}. */
    public ScaleImageTransform(@JsonProperty("dx") float dx, @JsonProperty("dy") float dy) {
        this(null, dx, dy);
    }

    /**
     * Constructs an instance of the ImageTransform.
     *
     * @param random object to use (or null for deterministic)
     * @param dx     maximum scaling in width of the image (pixels)
     * @param dy     maximum scaling in height of the image (pixels)
     */
    public ScaleImageTransform(Random random, float dx, float dy) {
        super(random);
        this.dx = dx;
        this.dy = dy;
        this.converter = new OpenCVFrameConverter.ToMat();
    }

    @Override
    public ImageWritable transform(ImageWritable image, Random random) {
        if (image == null) {
            return null;
        }
        Mat mat = converter.convert(image.getFrame());
        int h = Math.round(mat.rows() + dy * (random != null ? 2 * random.nextFloat() - 1 : 1));
        int w = Math.round(mat.cols() + dx * (random != null ? 2 * random.nextFloat() - 1 : 1));

        Mat result = new Mat();
        resize(mat, result, new Size(w, h));
        return new ImageWritable(converter.convert(result));
    }

}
