declare type CallCallback = (item: Phaser.GameObjects.GameObject)=>void;

declare type GridAlignConfig = {
    /**
     * The width of the grid in items (not pixels). -1 means lay all items out horizontally, regardless of quantity.
     *                                  If both this value and height are set to -1 then this value overrides it and the `height` value is ignored.
     */
    width?: integer;
    /**
     * The height of the grid in items (not pixels). -1 means lay all items out vertically, regardless of quantity.
     *                                   If both this value and `width` are set to -1 then `width` overrides it and this value is ignored.
     */
    height?: integer;
    /**
     * The width of the cell, in pixels, in which the item is positioned.
     */
    cellWidth?: integer;
    /**
     * The height of the cell, in pixels, in which the item is positioned.
     */
    cellHeight?: integer;
    /**
     * The alignment position. One of the Phaser.Display.Align consts such as `TOP_LEFT` or `RIGHT_CENTER`.
     */
    position?: integer;
    /**
     * Optionally place the top-left of the final grid at this coordinate.
     */
    x?: number;
    /**
     * Optionally place the top-left of the final grid at this coordinate.
     */
    y?: number;
};

declare type JSONAnimation = {
    /**
     * The key that the animation will be associated with. i.e. sprite.animations.play(key)
     */
    key: string;
    /**
     * A frame based animation (as opposed to a bone based animation)
     */
    type: string;
    /**
     * [description]
     */
    frames: JSONAnimationFrame[];
    /**
     * The frame rate of playback in frames per second (default 24 if duration is null)
     */
    frameRate: integer;
    /**
     * How long the animation should play for in milliseconds. If not given its derived from frameRate.
     */
    duration: integer;
    /**
     * Skip frames if the time lags, or always advanced anyway?
     */
    skipMissedFrames: boolean;
    /**
     * Delay before starting playback. Value given in milliseconds.
     */
    delay: integer;
    /**
     * Number of times to repeat the animation (-1 for infinity)
     */
    repeat: integer;
    /**
     * Delay before the animation repeats. Value given in milliseconds.
     */
    repeatDelay: integer;
    /**
     * Should the animation yoyo? (reverse back down to the start) before repeating?
     */
    yoyo: boolean;
    /**
     * Should sprite.visible = true when the animation starts to play?
     */
    showOnStart: boolean;
    /**
     * Should sprite.visible = false when the animation finishes?
     */
    hideOnComplete: boolean;
};

declare type AnimationFrameConfig = {
    /**
     * The key that the animation will be associated with. i.e. sprite.animations.play(key)
     */
    key: string;
    /**
     * [description]
     */
    frame: string | number;
    /**
     * [description]
     */
    duration?: number;
    /**
     * [description]
     */
    visible?: boolean;
};

declare type AnimationConfig = {
    /**
     * The key that the animation will be associated with. i.e. sprite.animations.play(key)
     */
    key?: string;
    /**
     * An object containing data used to generate the frames for the animation
     */
    frames?: AnimationFrameConfig[];
    /**
     * The key of the texture all frames of the animation will use. Can be overridden on a per frame basis.
     */
    defaultTextureKey?: string;
    /**
     * The frame rate of playback in frames per second (default 24 if duration is null)
     */
    frameRate?: integer;
    /**
     * How long the animation should play for in milliseconds. If not given its derived from frameRate.
     */
    duration?: integer;
    /**
     * Skip frames if the time lags, or always advanced anyway?
     */
    skipMissedFrames?: boolean;
    /**
     * Delay before starting playback. Value given in milliseconds.
     */
    delay?: integer;
    /**
     * Number of times to repeat the animation (-1 for infinity)
     */
    repeat?: integer;
    /**
     * Delay before the animation repeats. Value given in milliseconds.
     */
    repeatDelay?: integer;
    /**
     * Should the animation yoyo? (reverse back down to the start) before repeating?
     */
    yoyo?: boolean;
    /**
     * Should sprite.visible = true when the animation starts to play?
     */
    showOnStart?: boolean;
    /**
     * Should sprite.visible = false when the animation finishes?
     */
    hideOnComplete?: boolean;
};

declare type JSONAnimationFrame = {
    /**
     * The key of the Texture this AnimationFrame uses.
     */
    key: string;
    /**
     * The key of the Frame within the Texture that this AnimationFrame uses.
     */
    frame: string | integer;
    /**
     * Additional time (in ms) that this frame should appear for during playback.
     */
    duration: number;
};

declare type JSONAnimationManager = {
    /**
     * [description]
     */
    anims: JSONAnimation[];
    /**
     * [description]
     */
    globalTimeScale: number;
};

declare type GenerateFrameNamesConfig = {
    /**
     * [description]
     */
    prefix?: string;
    /**
     * [description]
     */
    start?: integer;
    /**
     * [description]
     */
    end?: integer;
    /**
     * [description]
     */
    suffix?: string;
    /**
     * [description]
     */
    zeroPad?: integer;
    /**
     * [description]
     */
    outputArray?: AnimationFrameConfig[];
    /**
     * [description]
     */
    frames?: boolean;
};

declare type GenerateFrameNumbersConfig = {
    /**
     * The starting frame of the animation.
     */
    start?: integer;
    /**
     * The ending frame of the animation.
     */
    end?: integer;
    /**
     * A frame to put at the beginning of the animation, before `start` or `outputArray` or `frames`.
     */
    first?: boolean | integer;
    /**
     * An array to concatenate the output onto.
     */
    outputArray?: AnimationFrameConfig[];
    /**
     * A custom sequence of frames.
     */
    frames?: boolean | integer[];
};

/**
 * This callback type is completely empty, a no-operation.
 */
declare type NOOP = ()=>void;

declare type BootCallback = (game: Phaser.Game)=>void;

declare type InputConfig = {
    /**
     * Keyboard input configuration. `true` uses the default configuration and `false` disables keyboard input.
     */
    keyboard?: boolean | KeyboardInputConfig;
    /**
     * Mouse input configuration. `true` uses the default configuration and `false` disables mouse input.
     */
    mouse?: boolean | MouseInputConfig;
    /**
     * Touch input configuration. `true` uses the default configuration and `false` disables touch input.
     */
    touch?: boolean | TouchInputConfig;
    /**
     * Gamepad input configuration. `true` enables gamepad input.
     */
    gamepad?: boolean | GamepadInputConfig;
    /**
     * The maximum number of touch pointers. See {@link Phaser.Input.InputManager#pointers}.
     */
    activePointers?: integer;
};

declare type MouseInputConfig = {
    /**
     * Where the Mouse Manager listens for mouse input events. The default is the game canvas.
     */
    target?: any;
    /**
     * Whether mouse input events have preventDefault() called on them.
     */
    capture?: boolean;
};

declare type KeyboardInputConfig = {
    /**
     * Where the Keyboard Manager listens for keyboard input events.
     */
    target?: any;
};

declare type TouchInputConfig = {
    /**
     * Where the Touch Manager listens for touch input events. The default is the game canvas.
     */
    target?: any;
    /**
     * Whether touch input events have preventDefault() called on them.
     */
    capture?: boolean;
};

declare type GamepadInputConfig = {
    /**
     * Where the Gamepad Manager listens for gamepad input events.
     */
    target?: any;
};

declare type BannerConfig = {
    /**
     * Omit Phaser's name and version from the banner.
     */
    hidePhaser?: boolean;
    /**
     * The color of the banner text.
     */
    text?: string;
    /**
     * The background colors of the banner.
     */
    background?: string[];
};

declare type FPSConfig = {
    /**
     * The minimum acceptable rendering rate, in frames per second.
     */
    min?: integer;
    /**
     * The optimum rendering rate, in frames per second.
     */
    target?: integer;
    /**
     * Use setTimeout instead of requestAnimationFrame to run the game loop.
     */
    forceSetTimeOut?: boolean;
    /**
     * Calculate the average frame delta from this many consecutive frame intervals.
     */
    deltaHistory?: integer;
    /**
     * [description]
     */
    panicMax?: integer;
};

declare type RenderConfig = {
    /**
     * When set to `true`, WebGL uses linear interpolation to draw scaled or rotated textures, giving a smooth appearance. When set to `false`, WebGL uses nearest-neighbor interpolation, giving a crisper appearance. `false` also disables antialiasing of the game canvas itself, if the browser supports it, when the game canvas is scaled.
     */
    antialias?: boolean;
    /**
     * Sets `antialias` and `roundPixels` to true. This is the best setting for pixel-art games.
     */
    pixelArt?: boolean;
    /**
     * Automatically resize the Game Canvas if you resize the renderer.
     */
    autoResize?: boolean;
    /**
     * Draw texture-based Game Objects at only whole-integer positions. Game Objects without textures, like Graphics, ignore this property.
     */
    roundPixels?: boolean;
    /**
     * Whether the game canvas will be transparent.
     */
    transparent?: boolean;
    /**
     * Whether the game canvas will be cleared between each rendering frame.
     */
    clearBeforeRender?: boolean;
    /**
     * In WebGL mode, the drawing buffer contains colors with pre-multiplied alpha.
     */
    premultipliedAlpha?: boolean;
    /**
     * In WebGL mode, the drawing buffer won't be cleared automatically each frame.
     */
    preserveDrawingBuffer?: boolean;
    /**
     * Let the browser abort creating a WebGL context if it judges performance would be unacceptable.
     */
    failIfMajorPerformanceCaveat?: boolean;
    /**
     * "high-performance", "low-power" or "default". A hint to the browser on how much device power the game might use.
     */
    powerPreference?: string;
    /**
     * The default WebGL batch size.
     */
    batchSize?: integer;
};

declare type ScaleConfig = {
    /**
     * The base width of your game.
     */
    width?: integer | string;
    /**
     * The base height of your game.
     */
    height?: integer | string;
    /**
     * The zoom value of the game canvas.
     */
    zoom?: integer;
    /**
     * The rendering resolution of the canvas.
     */
    resolution?: number;
    /**
     * The DOM element that will contain the game canvas, or its `id`. If null (the default) or if the named element doesn't exist, the game canvas is inserted directly into the document body.
     */
    parent?: HTMLElement | string;
    /**
     * The scale mode to apply to the canvas. SHOW_ALL, EXACT_FIT, USER_SCALE, or RESIZE.
     */
    mode?: integer;
    /**
     * The minimum width the canvas can be scaled down to.
     */
    minWidth?: integer;
    /**
     * The minimum height the canvas can be scaled down to.
     */
    minHeight?: integer;
    /**
     * The maximum width the canvas can be scaled up to.
     */
    maxWidth?: integer;
    /**
     * The maximum height the canvas can be scaled up to.
     */
    maxHeight?: integer;
};

declare type CallbacksConfig = {
    /**
     * A function to run at the start of the boot sequence.
     */
    preBoot?: BootCallback;
    /**
     * A function to run at the end of the boot sequence. At this point, all the game systems have started and plugins have been loaded.
     */
    postBoot?: BootCallback;
};

declare type LoaderConfig = {
    /**
     * An URL used to resolve paths given to the loader. Example: 'http://labs.phaser.io/assets/'.
     */
    baseURL?: string;
    /**
     * An URL path used to resolve relative paths given to the loader. Example: 'images/sprites/'.
     */
    path?: string;
    /**
     * The maximum number of resources the loader will start loading at once.
     */
    maxParallelDownloads?: integer;
    /**
     * 'anonymous', 'use-credentials', or `undefined`. If you're not making cross-origin requests, leave this as `undefined`. See {@link https://developer.mozilla.org/en-US/docs/Web/HTML/CORS_settings_attributes}.
     */
    crossOrigin?: string | undefined;
    /**
     * The response type of the XHR request, e.g. `blob`, `text`, etc.
     */
    responseType?: string;
    /**
     * Should the XHR request use async or not?
     */
    async?: boolean;
    /**
     * Optional username for the XHR request.
     */
    user?: string;
    /**
     * Optional password for the XHR request.
     */
    password?: string;
    /**
     * Optional XHR timeout value, in ms.
     */
    timeout?: integer;
};

declare type DOMContainerConfig = {
    /**
     * Create a div element in which DOM Elements will be contained. You must also provide a parent.
     */
    createContainer?: boolean;
    /**
     * Place the DOM Container behind the Phaser Canvas. The default is to place it over the Canvas.
     */
    behindCanvas?: boolean;
};

declare type ImagesConfig = {
    /**
     * URL to use for the 'default' texture.
     */
    default?: string;
    /**
     * URL to use for the 'missing' texture.
     */
    missing?: string;
};

declare type PhysicsConfig = {
    /**
     * The default physics system. It will be started for each scene. Phaser provides 'arcade', 'impact', and 'matter'.
     */
    default?: string;
    /**
     * Arcade Physics configuration.
     */
    arcade?: ArcadeWorldConfig;
    /**
     * Impact Physics configuration.
     */
    impact?: Phaser.Physics.Impact.WorldConfig;
    /**
     * Matter Physics configuration.
     */
    matter?: object;
};

declare type PluginObjectItem = {
    /**
     * A key to identify the plugin in the Plugin Manager.
     */
    key?: string;
    /**
     * The plugin itself. Usually a class/constructor.
     */
    plugin?: any;
    /**
     * Whether the plugin should be started automatically.
     */
    start?: boolean;
    /**
     * For a scene plugin, add the plugin to the scene's systems object under this key (`this.sys.KEY`, from the scene).
     */
    systemKey?: string;
    /**
     * For a scene plugin, add the plugin to the scene object under this key (`this.KEY`, from the scene).
     */
    sceneKey?: string;
    /**
     * Arbitrary data passed to the plugin's init() method.
     */
    data?: any;
};

declare type PluginObject = {
    /**
     * Global plugins to install.
     */
    global?: PluginObjectItem[];
    /**
     * Scene plugins to install.
     */
    scene?: PluginObjectItem[];
    /**
     * The default set of scene plugins (names).
     */
    default?: string[];
    /**
     * Plugins to *add* to the default set of scene plugins.
     */
    defaultMerge?: string[];
};

declare type GameConfig = {
    /**
     * The width of the game, in game pixels.
     */
    width?: integer | string;
    /**
     * The height of the game, in game pixels.
     */
    height?: integer | string;
    /**
     * Simple scale applied to the game canvas. 2 is double size, 0.5 is half size, etc.
     */
    zoom?: number;
    /**
     * The size of each game pixel, in canvas pixels. Values larger than 1 are "high" resolution.
     */
    resolution?: number;
    /**
     * Which renderer to use. Phaser.AUTO, Phaser.CANVAS, Phaser.HEADLESS, or Phaser.WEBGL. AUTO picks WEBGL if available, otherwise CANVAS.
     */
    type?: number;
    /**
     * The DOM element that will contain the game canvas, or its `id`. If null (the default) or if the named element doesn't exist, the game canvas is inserted directly into the document body.
     */
    parent?: HTMLElement | string;
    /**
     * Provide your own Canvas element for Phaser to use instead of creating one.
     */
    canvas?: HTMLCanvasElement;
    /**
     * CSS styles to apply to the game canvas instead of Phaser's default styles.
     */
    canvasStyle?: string;
    /**
     * Provide your own Canvas Context for Phaser to use, instead of creating one.
     */
    context?: CanvasRenderingContext2D;
    /**
     * A scene or scenes to add to the game. If several are given, the first is started; the remainder are started only if they have { active: true }.
     */
    scene?: object;
    /**
     * Seed for the random number generator.
     */
    seed?: string[];
    /**
     * The title of the game. Shown in the browser console.
     */
    title?: string;
    /**
     * The URL of the game. Shown in the browser console.
     */
    url?: string;
    /**
     * The version of the game. Shown in the browser console.
     */
    version?: string;
    /**
     * Automatically call window.focus() when the game boots. Usually necessary to capture input events if the game is in a separate frame.
     */
    autoFocus?: boolean;
    /**
     * Input configuration, or `false` to disable all game input.
     */
    input?: boolean | InputConfig;
    /**
     * Disable the browser's default 'contextmenu' event (usually triggered by a right-button mouse click).
     */
    disableContextMenu?: boolean;
    /**
     * Configuration for the banner printed in the browser console when the game starts.
     */
    banner?: boolean | BannerConfig;
    /**
     * The DOM Container configuration object.
     */
    dom?: DOMContainerConfig;
    /**
     * Game loop configuration.
     */
    fps?: FPSConfig;
    /**
     * Game renderer configuration.
     */
    render?: RenderConfig;
    /**
     * The background color of the game canvas. The default is black.
     */
    backgroundColor?: string | number;
    /**
     * Optional callbacks to run before or after game boot.
     */
    callbacks?: CallbacksConfig;
    /**
     * Loader configuration.
     */
    loader?: LoaderConfig;
    /**
     * Images configuration.
     */
    images?: ImagesConfig;
    /**
     * Physics configuration.
     */
    physics?: object;
    /**
     * Plugins to install.
     */
    plugins?: PluginObject | PluginObjectItem[];
};

declare type TimeStepCallback = (time: number, average: number, interpolation: number)=>void;

declare type JSONCameraBounds = {
    /**
     * The horizontal position of camera
     */
    x: number;
    /**
     * The vertical position of camera
     */
    y: number;
    /**
     * The width size of camera
     */
    width: number;
    /**
     * The height size of camera
     */
    height: number;
};

declare type JSONCamera = {
    /**
     * The name of the camera
     */
    name: string;
    /**
     * The horizontal position of camera
     */
    x: number;
    /**
     * The vertical position of camera
     */
    y: number;
    /**
     * The width size of camera
     */
    width: number;
    /**
     * The height size of camera
     */
    height: number;
    /**
     * The zoom of camera
     */
    zoom: number;
    /**
     * The rotation of camera
     */
    rotation: number;
    /**
     * The round pixels st status of camera
     */
    roundPixels: boolean;
    /**
     * The horizontal scroll of camera
     */
    scrollX: number;
    /**
     * The vertical scroll of camera
     */
    scrollY: number;
    /**
     * The background color of camera
     */
    backgroundColor: string;
    /**
     * The bounds of camera
     */
    bounds?: JSONCameraBounds | undefined;
};

declare type InputJSONCameraObject = {
    /**
     * The name of the Camera.
     */
    name?: string;
    /**
     * The horizontal position of the Camera viewport.
     */
    x?: integer;
    /**
     * The vertical position of the Camera viewport.
     */
    y?: integer;
    /**
     * The width of the Camera viewport.
     */
    width?: integer;
    /**
     * The height of the Camera viewport.
     */
    height?: integer;
    /**
     * The default zoom level of the Camera.
     */
    zoom?: number;
    /**
     * The rotation of the Camera, in radians.
     */
    rotation?: number;
    /**
     * Should the Camera round pixels before rendering?
     */
    roundPixels?: boolean;
    /**
     * The horizontal scroll position of the Camera.
     */
    scrollX?: number;
    /**
     * The vertical scroll position of the Camera.
     */
    scrollY?: number;
    /**
     * A CSS color string controlling the Camera background color.
     */
    backgroundColor?: false | string;
    /**
     * Defines the Camera bounds.
     */
    bounds?: object;
    /**
     * The top-left extent of the Camera bounds.
     */
    "bounds.x"?: number;
    /**
     * The top-left extent of the Camera bounds.
     */
    "bounds.y"?: number;
    /**
     * The width of the Camera bounds.
     */
    "bounds.width"?: number;
    /**
     * The height of the Camera bounds.
     */
    "bounds.height"?: number;
};

declare type CameraFadeCallback = (camera: Phaser.Cameras.Scene2D.Camera, progress: number)=>void;

declare type CameraFlashCallback = (camera: Phaser.Cameras.Scene2D.Camera, progress: number)=>void;

declare type CameraPanCallback = (camera: Phaser.Cameras.Scene2D.Camera, progress: number, x: number, y: number)=>void;

declare type CameraShakeCallback = (camera: Phaser.Cameras.Scene2D.Camera, progress: number)=>void;

declare type CameraZoomCallback = (camera: Phaser.Cameras.Scene2D.Camera, progress: number, zoom: number)=>void;

declare type FixedKeyControlConfig = {
    /**
     * The Camera that this Control will update.
     */
    camera?: Phaser.Cameras.Scene2D.Camera;
    /**
     * The Key to be pressed that will move the Camera left.
     */
    left?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will move the Camera right.
     */
    right?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will move the Camera up.
     */
    up?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will move the Camera down.
     */
    down?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will zoom the Camera in.
     */
    zoomIn?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will zoom the Camera out.
     */
    zoomOut?: Phaser.Input.Keyboard.Key;
    /**
     * The speed at which the camera will zoom if the `zoomIn` or `zoomOut` keys are pressed.
     */
    zoomSpeed?: number;
    /**
     * The horizontal and vertical speed the camera will move.
     */
    speed?: number | Object;
};

declare type SmoothedKeyControlConfig = {
    /**
     * The Camera that this Control will update.
     */
    camera?: Phaser.Cameras.Scene2D.Camera;
    /**
     * The Key to be pressed that will move the Camera left.
     */
    left?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will move the Camera right.
     */
    right?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will move the Camera up.
     */
    up?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will zoom the Camera in.
     */
    zoomIn?: Phaser.Input.Keyboard.Key;
    /**
     * The Key to be pressed that will zoom the Camera out.
     */
    zoomOut?: Phaser.Input.Keyboard.Key;
    /**
     * The speed at which the camera will zoom if the `zoomIn` or `zoomOut` keys are pressed.
     */
    zoomSpeed?: number;
    /**
     * The horizontal and vertical acceleration the camera will move.
     */
    acceleration?: number | Object;
    /**
     * The horizontal and vertical drag applied to the camera when it is moving.
     */
    drag?: number | Object;
    /**
     * The maximum horizontal and vertical speed the camera will move.
     */
    maxSpeed?: number | Object;
};

declare type GenerateTextureRendererCallback = (canvas: HTMLCanvasElement, context: CanvasRenderingContext2D)=>void;

declare type GenerateTextureConfig = {
    /**
     * [description]
     */
    data?: any[];
    /**
     * [description]
     */
    canvas?: HTMLCanvasElement;
    /**
     * [description]
     */
    palette?: Palette;
    /**
     * The width of each 'pixel' in the generated texture.
     */
    pixelWidth?: number;
    /**
     * The height of each 'pixel' in the generated texture.
     */
    pixelHeight?: number;
    /**
     * [description]
     */
    resizeCanvas?: boolean;
    /**
     * [description]
     */
    clearCanvas?: boolean;
    /**
     * [description]
     */
    preRender?: GenerateTextureRendererCallback;
    /**
     * [description]
     */
    postRender?: GenerateTextureRendererCallback;
};

declare type Palette = {
    /**
     * Color value 1.
     */
    "0": string;
    /**
     * Color value 2.
     */
    "1": string;
    /**
     * Color value 3.
     */
    "2": string;
    /**
     * Color value 4.
     */
    "3": string;
    /**
     * Color value 5.
     */
    "4": string;
    /**
     * Color value 6.
     */
    "5": string;
    /**
     * Color value 7.
     */
    "6": string;
    /**
     * Color value 8.
     */
    "7": string;
    /**
     * Color value 9.
     */
    "8": string;
    /**
     * Color value 10.
     */
    "9": string;
    /**
     * Color value 11.
     */
    A: string;
    /**
     * Color value 12.
     */
    B: string;
    /**
     * Color value 13.
     */
    C: string;
    /**
     * Color value 14.
     */
    D: string;
    /**
     * Color value 15.
     */
    E: string;
    /**
     * Color value 16.
     */
    F: string;
};

declare type JSONEllipseCurve = {
    /**
     * The of the curve.
     */
    type: string;
    /**
     * The x coordinate of the ellipse.
     */
    x: number;
    /**
     * The y coordinate of the ellipse.
     */
    y: number;
    /**
     * The horizontal radius of ellipse.
     */
    xRadius: number;
    /**
     * The vertical radius of ellipse.
     */
    yRadius: number;
    /**
     * The start angle of the ellipse, in degrees.
     */
    startAngle: integer;
    /**
     * The end angle of the ellipse, in degrees.
     */
    endAngle: integer;
    /**
     * Sets if the the ellipse rotation is clockwise (true) or anti-clockwise (false)
     */
    clockwise: boolean;
    /**
     * The rotation of ellipse, in degrees.
     */
    rotation: integer;
};

declare type EllipseCurveConfig = {
    /**
     * The x coordinate of the ellipse.
     */
    x?: number;
    /**
     * The y coordinate of the ellipse.
     */
    y?: number;
    /**
     * The horizontal radius of the ellipse.
     */
    xRadius?: number;
    /**
     * The vertical radius of the ellipse.
     */
    yRadius?: number;
    /**
     * The start angle of the ellipse, in degrees.
     */
    startAngle?: integer;
    /**
     * The end angle of the ellipse, in degrees.
     */
    endAngle?: integer;
    /**
     * Sets if the the ellipse rotation is clockwise (true) or anti-clockwise (false)
     */
    clockwise?: boolean;
    /**
     * The rotation of the ellipse, in degrees.
     */
    rotation?: integer;
};

declare type JSONCurve = {
    /**
     * The of the curve
     */
    type: string;
    /**
     * The arrays of points like `[x1, y1, x2, y2]`
     */
    points: number[];
};

declare type JSONPath = {
    /**
     * The of the curve.
     */
    type: string;
    /**
     * [description]
     */
    x: number;
    /**
     * [description]
     */
    y: number;
    /**
     * The path is auto closed.
     */
    autoClose: boolean;
    /**
     * The list of the curves
     */
    curves: JSONCurve[];
};

declare type DataEachCallback = (parent: any, key: string, value: any, ...args: any[])=>void;

/**
 * Checks for support of the Full Screen API.
 */
declare function init(): void;

declare type InputColorObject = {
    /**
     * The red color value in the range 0 to 255.
     */
    r?: number;
    /**
     * The green color value in the range 0 to 255.
     */
    g?: number;
    /**
     * The blue color value in the range 0 to 255.
     */
    b?: number;
    /**
     * The alpha color value in the range 0 to 255.
     */
    a?: number;
};

declare type ColorObject = ()=>void;

declare namespace Phaser.Display.Color {
    /**
     * Interpolates color values
     */
    namespace Interpolate {
        /**
         * Interpolates between the two given color ranges over the length supplied.
         * @param r1 Red value.
         * @param g1 Blue value.
         * @param b1 Green value.
         * @param r2 Red value.
         * @param g2 Blue value.
         * @param b2 Green value.
         * @param length Distance to interpolate over. Default 100.
         * @param index Index to start from. Default 0.
         */
        function RGBWithRGB(r1: number, g1: number, b1: number, r2: number, g2: number, b2: number, length?: number, index?: number): ColorObject;

        /**
         * Interpolates between the two given color objects over the length supplied.
         * @param color1 The first Color object.
         * @param color2 The second Color object.
         * @param length Distance to interpolate over. Default 100.
         * @param index Index to start from. Default 0.
         */
        function ColorWithColor(color1: Phaser.Display.Color, color2: Phaser.Display.Color, length?: number, index?: number): ColorObject;

        /**
         * Interpolates between the Color object and color values over the length supplied.
         * @param color1 The first Color object.
         * @param r Red value.
         * @param g Blue value.
         * @param b Green value.
         * @param length Distance to interpolate over. Default 100.
         * @param index Index to start from. Default 0.
         */
        function ColorWithRGB(color1: Phaser.Display.Color, r: number, g: number, b: number, length?: number, index?: number): ColorObject;

    }

}

declare type HSVColorObject = {
    /**
     * The hue color value. A number between 0 and 1
     */
    h: number;
    /**
     * The saturation color value. A number between 0 and 1
     */
    s: number;
    /**
     * The lightness color value. A number between 0 and 1
     */
    v: number;
};

declare type ContentLoadedCallback = ()=>void;

declare type DisplayCallbackConfig = {
    /**
     * The tint of the character being rendered.
     */
    tint: Object;
    /**
     * The index of the character being rendered.
     */
    index: number;
    /**
     * The character code of the character being rendered.
     */
    charCode: number;
    /**
     * The x position of the character being rendered.
     */
    x: number;
    /**
     * The y position of the character being rendered.
     */
    y: number;
    /**
     * The scale of the character being rendered.
     */
    scale: number;
    /**
     * The rotation of the character being rendered.
     */
    rotation: number;
    /**
     * Custom data stored with the character being rendered.
     */
    data: any;
};

declare type DisplayCallback = (display: DisplayCallbackConfig)=>void;

declare type BitmapTextConfig = GameObjectConfig & {
    /**
     * The key of the font to use from the BitmapFont cache.
     */
    font?: string;
    /**
     * The string, or array of strings, to be set as the content of this Bitmap Text.
     */
    text?: string;
    /**
     * The font size to set.
     */
    size?: number | false;
};

declare type BitmapTextSize = {
    /**
     * The position and size of the BitmapText, taking into account the position and scale of the Game Object.
     */
    global: GlobalBitmapTextSize;
    /**
     * The position and size of the BitmapText, taking just the font size into account.
     */
    local: LocalBitmapTextSize;
};

/**
 * The position and size of the Bitmap Text in global space, taking into account the Game Object's scale and world position.
 */
declare type GlobalBitmapTextSize = {
    /**
     * The x position of the BitmapText, taking into account the x position and scale of the Game Object.
     */
    x: number;
    /**
     * The y position of the BitmapText, taking into account the y position and scale of the Game Object.
     */
    y: number;
    /**
     * The width of the BitmapText, taking into account the x scale of the Game Object.
     */
    width: number;
    /**
     * The height of the BitmapText, taking into account the y scale of the Game Object.
     */
    height: number;
};

/**
 * The position and size of the Bitmap Text in local space, taking just the font size into account.
 */
declare type LocalBitmapTextSize = {
    /**
     * The x position of the BitmapText.
     */
    x: number;
    /**
     * The y position of the BitmapText.
     */
    y: number;
    /**
     * The width of the BitmapText.
     */
    width: number;
    /**
     * The height of the BitmapText.
     */
    height: number;
};

/**
 * The font data for an individual character of a Bitmap Font.
 *
 * Describes the character's position, size, offset and kerning.
 */
declare type BitmapFontCharacterData = {
    /**
     * The x position of the character.
     */
    x: number;
    /**
     * The y position of the character.
     */
    y: number;
    /**
     * The width of the character.
     */
    width: number;
    /**
     * The height of the character.
     */
    height: number;
    /**
     * The center x position of the character.
     */
    centerX: number;
    /**
     * The center y position of the character.
     */
    centerY: number;
    /**
     * The x offset of the character.
     */
    xOffset: number;
    /**
     * The y offset of the character.
     */
    yOffset: number;
    /**
     * Extra data for the character.
     */
    data: object;
    /**
     * Kerning values, keyed by character code.
     */
    kerning: {[key: string]: number};
};

/**
 * Bitmap Font data that can be used by a BitmapText Game Object.
 */
declare type BitmapFontData = {
    /**
     * The name of the font.
     */
    font: string;
    /**
     * The size of the font.
     */
    size: number;
    /**
     * The line height of the font.
     */
    lineHeight: number;
    /**
     * Whether this font is a retro font (monospace).
     */
    retroFont: boolean;
    /**
     * The character data of the font, keyed by character code. Each character datum includes a position, size, offset and more.
     */
    chars: {[key: number]:  BitmapFontCharacterData};
};

declare type JSONBitmapText = JSONGameObject & {
    /**
     * The name of the font.
     */
    font: string;
    /**
     * The text that this Bitmap Text displays.
     */
    text: string;
    /**
     * The size of the font.
     */
    fontSize: number;
    /**
     * Adds / Removes spacing between characters.
     */
    letterSpacing: number;
    /**
     * The alignment of the text in a multi-line BitmapText object.
     */
    align: integer;
};

declare namespace Phaser.GameObjects.Blitter {
    type CreateCallback = (bob: Phaser.GameObjects.Blitter.Bob, index: integer)=>void;

    /**
     * A Bob Game Object.
     *
     * A Bob belongs to a Blitter Game Object. The Blitter is responsible for managing and rendering this object.
     *
     * A Bob has a position, alpha value and a frame from a texture that it uses to render with. You can also toggle
     * the flipped and visible state of the Bob. The Frame the Bob uses to render can be changed dynamically, but it
     * must be a Frame within the Texture used by the parent Blitter.
     *
     * Bob positions are relative to the Blitter parent. So if you move the Blitter parent, all Bob children will
     * have their positions impacted by this change as well.
     *
     * You can manipulate Bob objects directly from your game code, but the creation and destruction of them should be
     * handled via the Blitter parent.
     */
    class Bob {
        /**
         *
         * @param blitter The parent Blitter object is responsible for updating this Bob.
         * @param x The horizontal position of this Game Object in the world, relative to the parent Blitter position.
         * @param y The vertical position of this Game Object in the world, relative to the parent Blitter position.
         * @param frame The Frame this Bob will render with, as defined in the Texture the parent Blitter is using.
         * @param visible Should the Bob render visible or not to start with?
         */
        constructor(blitter: Phaser.GameObjects.Blitter, x: number, y: number, frame: string | integer, visible: boolean);

        /**
         * The Blitter object that this Bob belongs to.
         */
        parent: Phaser.GameObjects.Blitter;

        /**
         * The x position of this Bob, relative to the x position of the Blitter.
         */
        x: number;

        /**
         * The y position of this Bob, relative to the y position of the Blitter.
         */
        y: number;

        /**
         * The frame that the Bob uses to render with.
         * To change the frame use the `Bob.setFrame` method.
         */
        protected frame: Phaser.Textures.Frame;

        /**
         * A blank object which can be used to store data related to this Bob in.
         */
        data: object;

        /**
         * The horizontally flipped state of the Bob.
         * A Bob that is flipped horizontally will render inversed on the horizontal axis.
         * Flipping always takes place from the middle of the texture.
         */
        flipX: boolean;

        /**
         * The vertically flipped state of the Bob.
         * A Bob that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
         * Flipping always takes place from the middle of the texture.
         */
        flipY: boolean;

        /**
         * Changes the Texture Frame being used by this Bob.
         * The frame must be part of the Texture the parent Blitter is using.
         * If no value is given it will use the default frame of the Blitter parent.
         * @param frame The frame to be used during rendering.
         */
        setFrame(frame?: string | integer | Phaser.Textures.Frame): Phaser.GameObjects.Blitter.Bob;

        /**
         * Resets the horizontal and vertical flipped state of this Bob back to their default un-flipped state.
         */
        resetFlip(): Phaser.GameObjects.Blitter.Bob;

        /**
         * Resets this Bob.
         *
         * Changes the position to the values given, and optionally changes the frame.
         *
         * Also resets the flipX and flipY values, sets alpha back to 1 and visible to true.
         * @param x The x position of the Bob. Bob coordinate are relative to the position of the Blitter object.
         * @param y The y position of the Bob. Bob coordinate are relative to the position of the Blitter object.
         * @param frame The Frame the Bob will use. It _must_ be part of the Texture the parent Blitter object is using.
         */
        reset(x: number, y: number, frame?: string | integer | Phaser.Textures.Frame): Phaser.GameObjects.Blitter.Bob;

        /**
         * Sets the horizontal flipped state of this Bob.
         * @param value The flipped state. `false` for no flip, or `true` to be flipped.
         */
        setFlipX(value: boolean): Phaser.GameObjects.Blitter.Bob;

        /**
         * Sets the vertical flipped state of this Bob.
         * @param value The flipped state. `false` for no flip, or `true` to be flipped.
         */
        setFlipY(value: boolean): Phaser.GameObjects.Blitter.Bob;

        /**
         * Sets the horizontal and vertical flipped state of this Bob.
         * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
         * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
         */
        setFlip(x: boolean, y: boolean): Phaser.GameObjects.Blitter.Bob;

        /**
         * Sets the visibility of this Bob.
         *
         * An invisible Bob will skip rendering.
         * @param value The visible state of the Game Object.
         */
        setVisible(value: boolean): Phaser.GameObjects.Blitter.Bob;

        /**
         * Set the Alpha level of this Bob. The alpha controls the opacity of the Game Object as it renders.
         * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
         *
         * A Bob with alpha 0 will skip rendering.
         * @param value The alpha value used for this Bob. Between 0 and 1.
         */
        setAlpha(value: number): Phaser.GameObjects.Blitter.Bob;

        /**
         * Destroys this Bob instance.
         * Removes itself from the Blitter and clears the parent, frame and data properties.
         */
        destroy(): void;

        /**
         * The visible state of the Bob.
         *
         * An invisible Bob will skip rendering.
         */
        visible: boolean;

        /**
         * The alpha value of the Bob, between 0 and 1.
         *
         * A Bob with alpha 0 will skip rendering.
         */
        alpha: number;

    }

}

declare type GameObjectConfig = {
    /**
     * The x position of the Game Object.
     */
    x?: number;
    /**
     * The y position of the Game Object.
     */
    y?: number;
    /**
     * The depth of the GameObject.
     */
    depth?: number;
    /**
     * The horizontally flipped state of the Game Object.
     */
    flipX?: boolean;
    /**
     * The vertically flipped state of the Game Object.
     */
    flipY?: boolean;
    /**
     * The scale of the GameObject.
     */
    scale?: number | object;
    /**
     * The scroll factor of the GameObject.
     */
    scrollFactor?: number | object;
    /**
     * The rotation angle of the Game Object, in radians.
     */
    rotation?: number;
    /**
     * The rotation angle of the Game Object, in degrees.
     */
    angle?: number;
    /**
     * The alpha (opacity) of the Game Object.
     */
    alpha?: number;
    /**
     * The origin of the Game Object.
     */
    origin?: number | object;
    /**
     * The scale mode of the GameObject.
     */
    scaleMode?: number;
    /**
     * The blend mode of the GameObject.
     */
    blendMode?: number;
    /**
     * The visible state of the Game Object.
     */
    visible?: boolean;
    /**
     * Add the GameObject to the scene.
     */
    add?: boolean;
};

declare type JSONGameObject = {
    /**
     * The name of this Game Object.
     */
    name: string;
    /**
     * A textual representation of this Game Object, i.e. `sprite`.
     */
    type: string;
    /**
     * The x position of this Game Object.
     */
    x: number;
    /**
     * The y position of this Game Object.
     */
    y: number;
    /**
     * The scale of this Game Object
     */
    scale: object;
    /**
     * The horizontal scale of this Game Object.
     */
    "scale.x": number;
    /**
     * The vertical scale of this Game Object.
     */
    "scale.y": number;
    /**
     * The origin of this Game Object.
     */
    origin: object;
    /**
     * The horizontal origin of this Game Object.
     */
    "origin.x": number;
    /**
     * The vertical origin of this Game Object.
     */
    "origin.y": number;
    /**
     * The horizontally flipped state of the Game Object.
     */
    flipX: boolean;
    /**
     * The vertically flipped state of the Game Object.
     */
    flipY: boolean;
    /**
     * The angle of this Game Object in radians.
     */
    rotation: number;
    /**
     * The alpha value of the Game Object.
     */
    alpha: number;
    /**
     * The visible state of the Game Object.
     */
    visible: boolean;
    /**
     * The Scale Mode being used by this Game Object.
     */
    scaleMode: integer;
    /**
     * Sets the Blend Mode being used by this Game Object.
     */
    blendMode: integer | string;
    /**
     * The texture key of this Game Object.
     */
    textureKey: string;
    /**
     * The frame key of this Game Object.
     */
    frameKey: string;
    /**
     * The data of this Game Object.
     */
    data: object;
};

declare type EachContainerCallback<I> = (item: any, ...args: any[])=>void;

declare namespace Phaser.GameObjects.Graphics {
    namespace Commands {
    }

}

/**
 * Graphics line style (or stroke style) settings.
 */
declare type GraphicsLineStyle = {
    /**
     * The stroke width.
     */
    width?: number;
    /**
     * The stroke color.
     */
    color?: number;
    /**
     * The stroke alpha.
     */
    alpha?: number;
};

/**
 * Graphics fill style settings.
 */
declare type GraphicsFillStyle = {
    /**
     * The fill color.
     */
    color?: number;
    /**
     * The fill alpha.
     */
    alpha?: number;
};

/**
 * Graphics style settings.
 */
declare type GraphicsStyles = {
    /**
     * The style applied to shape outlines.
     */
    lineStyle?: GraphicsLineStyle;
    /**
     * The style applied to shape areas.
     */
    fillStyle?: GraphicsFillStyle;
};

/**
 * Options for the Graphics game Object.
 */
declare type GraphicsOptions = GraphicsStyles & {
    /**
     * The x coordinate of the Graphics.
     */
    x?: number;
    /**
     * The y coordinate of the Graphics.
     */
    y?: number;
};

declare type GroupCallback = (item: Phaser.GameObjects.GameObject)=>void;

declare type GroupMultipleCreateCallback = (items: Phaser.GameObjects.GameObject[])=>void;

declare type GroupConfig = {
    /**
     * Sets {@link Phaser.GameObjects.Group#classType}.
     */
    classType?: object;
    /**
     * Sets {@link Phaser.GameObjects.Group#active}.
     */
    active?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Group#maxSize}.
     */
    maxSize?: number;
    /**
     * Sets {@link Phaser.GameObjects.Group#defaultKey}.
     */
    defaultKey?: string;
    /**
     * Sets {@link Phaser.GameObjects.Group#defaultFrame}.
     */
    defaultFrame?: string | integer;
    /**
     * Sets {@link Phaser.GameObjects.Group#runChildUpdate}.
     */
    runChildUpdate?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Group#createCallback}.
     */
    createCallback?: GroupCallback;
    /**
     * Sets {@link Phaser.GameObjects.Group#removeCallback}.
     */
    removeCallback?: GroupCallback;
    /**
     * Sets {@link Phaser.GameObjects.Group#createMultipleCallback}.
     */
    createMultipleCallback?: GroupMultipleCreateCallback;
};

/**
 * The total number of objects created will be
 *
 *     key.length * frame.length * frameQuantity * (yoyo ? 2 : 1) * (1 + repeat)
 *
 * In the simplest case, 1 + `repeat` objects will be created.
 *
 * If `max` is positive, then the total created will not exceed `max`.
 *
 * `key` is required. {@link Phaser.GameObjects.Group#defaultKey} is not used.
 */
declare type GroupCreateConfig = {
    /**
     * The class of each new Game Object.
     */
    classType?: object;
    /**
     * The texture key of each new Game Object.
     */
    key?: string;
    /**
     * The texture frame of each new Game Object.
     */
    frame?: string | integer;
    /**
     * The visible state of each new Game Object.
     */
    visible?: boolean;
    /**
     * The active state of each new Game Object.
     */
    active?: boolean;
    /**
     * The number of times each `key` × `frame` combination will be *repeated* (after the first combination).
     */
    repeat?: number;
    /**
     * Select a `key` at random.
     */
    randomKey?: boolean;
    /**
     * Select a `frame` at random.
     */
    randomFrame?: boolean;
    /**
     * Select keys and frames by moving forward then backward through `key` and `frame`.
     */
    yoyo?: boolean;
    /**
     * The number of times each `frame` should be combined with one `key`.
     */
    frameQuantity?: number;
    /**
     * The maximum number of new Game Objects to create. 0 is no maximum.
     */
    max?: number;
    setXY?: object;
    /**
     * The horizontal position of each new Game Object.
     */
    "setXY.x"?: number;
    /**
     * The vertical position of each new Game Object.
     */
    "setXY.y"?: number;
    /**
     * Increment each Game Object's horizontal position from the previous by this amount, starting from `setXY.x`.
     */
    "setXY.stepX"?: number;
    /**
     * Increment each Game Object's vertical position from the previous by this amount, starting from `setXY.y`.
     */
    "setXY.stepY"?: number;
    setRotation?: object;
    /**
     * Rotation of each new Game Object.
     */
    "setRotation.value"?: number;
    /**
     * Increment each Game Object's rotation from the previous by this amount, starting at `setRotation.value`.
     */
    "setRotation.step"?: number;
    setScale?: object;
    /**
     * The horizontal scale of each new Game Object.
     */
    "setScale.x"?: number;
    /**
     * The vertical scale of each new Game Object.
     */
    "setScale.y"?: number;
    /**
     * Increment each Game Object's horizontal scale from the previous by this amount, starting from `setScale.x`.
     */
    "setScale.stepX"?: number;
    /**
     * Increment each Game object's vertical scale from the previous by this amount, starting from `setScale.y`.
     */
    "setScale.stepY"?: number;
    setAlpha?: object;
    /**
     * The alpha value of each new Game Object.
     */
    "setAlpha.value"?: number;
    /**
     * Increment each Game Object's alpha from the previous by this amount, starting from `setAlpha.value`.
     */
    "setAlpha.step"?: number;
    /**
     * A geometric shape that defines the hit area for the Game Object.
     */
    hitArea?: any;
    /**
     * A callback to be invoked when the Game Object is interacted with.
     */
    hitAreaCallback?: HitAreaCallback;
    /**
     * Align the new Game Objects in a grid using these settings.
     */
    gridAlign?: false | GridAlignConfig;
};

declare type LightForEach = (light: Phaser.GameObjects.Light)=>void;

/**
 * The returned value sets what the property will be at the START of the particle's life, on emit.
 */
declare type EmitterOpOnEmitCallback = (particle: Phaser.GameObjects.Particles.Particle, key: string, value: number)=>void;

/**
 * The returned value updates the property for the duration of the particle's life.
 */
declare type EmitterOpOnUpdateCallback = (particle: Phaser.GameObjects.Particles.Particle, key: string, t: number, value: number)=>void;

/**
 * Defines an operation yielding a random value within a range.
 */
declare type EmitterOpRandomConfig = {
    /**
     * The minimum and maximum values, as [min, max].
     */
    random: number[];
};

/**
 * Defines an operation yielding a random value within a range.
 */
declare type EmitterOpRandomMinMaxConfig = {
    /**
     * The minimum value.
     */
    min: number;
    /**
     * The maximum value.
     */
    max: number;
};

/**
 * Defines an operation yielding a random value within a range.
 */
declare type EmitterOpRandomStartEndConfig = {
    /**
     * The starting value.
     */
    start: number;
    /**
     * The ending value.
     */
    end: number;
    /**
     * If false, this becomes {@link EmitterOpEaseConfig}.
     */
    random: boolean;
};

/**
 * Defines an operation yielding a value incremented continuously across a range.
 */
declare type EmitterOpEaseConfig = {
    /**
     * The starting value.
     */
    start: number;
    /**
     * The ending value.
     */
    end: number;
    /**
     * The name of the easing function.
     */
    ease?: string;
};

/**
 * Defines an operation yielding a value incremented by steps across a range.
 */
declare type EmitterOpSteppedConfig = {
    /**
     * The starting value.
     */
    start: number;
    /**
     * The ending value.
     */
    end: number;
    /**
     * The number of steps between start and end.
     */
    steps: number;
};

declare type EmitterOpCustomEmitConfig = {
    /**
     * [description]
     */
    onEmit: EmitterOpOnEmitCallback;
};

declare type EmitterOpCustomUpdateConfig = {
    /**
     * [description]
     */
    onEmit?: EmitterOpOnEmitCallback;
    /**
     * [description]
     */
    onUpdate: EmitterOpOnUpdateCallback;
};

declare type GravityWellConfig = {
    /**
     * The x coordinate of the Gravity Well, in world space.
     */
    x?: number;
    /**
     * The y coordinate of the Gravity Well, in world space.
     */
    y?: number;
    /**
     * The power of the Gravity Well.
     */
    power?: number;
    /**
     * [description]
     */
    epsilon?: number;
    /**
     * The gravitational force of this Gravity Well.
     */
    gravity?: number;
};

declare type ParticleEmitterCallback = (particle: Phaser.GameObjects.Particles.Particle, emitter: Phaser.GameObjects.Particles.ParticleEmitter)=>void;

declare type ParticleDeathCallback = (particle: Phaser.GameObjects.Particles.Particle)=>void;

declare type ParticleEmitterBounds = {
    /**
     * The left edge of the rectangle.
     */
    x: number;
    /**
     * The top edge of the rectangle.
     */
    y: number;
    /**
     * The width of the rectangle.
     */
    width: number;
    /**
     * The height of the rectangle.
     */
    height: number;
};

declare type ParticleEmitterBoundsAlt = {
    /**
     * The left edge of the rectangle.
     */
    x: number;
    /**
     * The top edge of the rectangle.
     */
    y: number;
    /**
     * The width of the rectangle.
     */
    w: number;
    /**
     * The height of the rectangle.
     */
    h: number;
};

declare type ParticleEmitterDeathZoneConfig = {
    /**
     * A shape representing the zone. See {@link Phaser.GameObjects.Particles.Zones.DeathZone#source}.
     */
    source: DeathZoneSource;
    /**
     * 'onEnter' or 'onLeave'.
     */
    type?: string;
};

declare type ParticleEmitterEdgeZoneConfig = {
    /**
     * A shape representing the zone. See {@link Phaser.GameObjects.Particles.Zones.EdgeZone#source}.
     */
    source: EdgeZoneSource;
    /**
     * 'edge'.
     */
    type: string;
    /**
     * The number of particles to place on the source edge. Set to 0 to use `stepRate` instead.
     */
    quantity: integer;
    /**
     * The distance between each particle. When set, `quantity` is implied and should be set to 0.
     */
    stepRate?: number;
    /**
     * Whether particles are placed from start to end and then end to start.
     */
    yoyo?: boolean;
    /**
     * Whether one endpoint will be removed if it's identical to the other.
     */
    seamless?: boolean;
};

declare type ParticleEmitterRandomZoneConfig = {
    /**
     * A shape representing the zone. See {@link Phaser.GameObjects.Particles.Zones.RandomZone#source}.
     */
    source: RandomZoneSource;
    /**
     * 'random'.
     */
    type?: string;
};

declare type ParticleEmitterConfig = {
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#active}.
     */
    active?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#blendMode}.
     */
    blendMode?: integer;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#deathCallbackScope} and {@link Phaser.GameObjects.Particles.ParticleEmitter#emitCallbackScope}.
     */
    callbackScope?: any;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#collideBottom}.
     */
    collideBottom?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#collideLeft}.
     */
    collideLeft?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#collideRight}.
     */
    collideRight?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#collideTop}.
     */
    collideTop?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#deathCallback}.
     */
    deathCallback?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#deathCallbackScope}.
     */
    deathCallbackScope?: any;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#emitCallback}.
     */
    emitCallback?: Function;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#emitCallbackScope}.
     */
    emitCallbackScope?: any;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#follow}.
     */
    follow?: Phaser.GameObjects.GameObject;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#frequency}.
     */
    frequency?: number;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#gravityX}.
     */
    gravityX?: number;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#gravityY}.
     */
    gravityY?: number;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#maxParticles}.
     */
    maxParticles?: integer;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#name}.
     */
    name?: string;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#on}.
     */
    on?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#particleBringToTop}.
     */
    particleBringToTop?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#particleClass}.
     */
    particleClass?: Phaser.GameObjects.Particles.Particle;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#radial}.
     */
    radial?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#timeScale}.
     */
    timeScale?: number;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#trackVisible}.
     */
    trackVisible?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#visible}.
     */
    visible?: boolean;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#accelerationX} (emit only).
     */
    accelerationX?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#accelerationY} (emit only).
     */
    accelerationY?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#alpha}.
     */
    alpha?: number | number[] | EmitterOpOnUpdateCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#angle} (emit only)
     */
    angle?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#bounce} (emit only).
     */
    bounce?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#delay} (emit only).
     */
    delay?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#lifespan} (emit only).
     */
    lifespan?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#maxVelocityX} (emit only).
     */
    maxVelocityX?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#maxVelocityY} (emit only).
     */
    maxVelocityY?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#moveToX} (emit only).
     */
    moveToX?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#moveToY} (emit only).
     */
    moveToY?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#quantity} (emit only).
     */
    quantity?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#rotate}.
     */
    rotate?: number | number[] | EmitterOpOnUpdateCallback | object;
    /**
     * As {@link Phaser.GameObjects.Particles.ParticleEmitter#setScale}.
     */
    scale?: number | number[] | EmitterOpOnUpdateCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#scaleX}.
     */
    scaleX?: number | number[] | EmitterOpOnUpdateCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#scaleY}.
     */
    scaleY?: number | number[] | EmitterOpOnUpdateCallback | object;
    /**
     * As {@link Phaser.GameObjects.Particles.ParticleEmitter#setSpeed} (emit only).
     */
    speed?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#speedX} (emit only).
     */
    speedX?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#speedY} (emit only).
     */
    speedY?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#tint}.
     */
    tint?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#x} (emit only).
     */
    x?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#y} (emit only).
     */
    y?: number | number[] | EmitterOpOnEmitCallback | object;
    /**
     * As {@link Phaser.GameObjects.Particles.ParticleEmitter#setEmitZone}.
     */
    emitZone?: object;
    /**
     * As {@link Phaser.GameObjects.Particles.ParticleEmitter#setBounds}.
     */
    bounds?: ParticleEmitterBounds | ParticleEmitterBoundsAlt;
    /**
     * Assigns to {@link Phaser.GameObjects.Particles.ParticleEmitter#followOffset}.
     */
    followOffset?: object;
    /**
     * x-coordinate of the offset.
     */
    "followOffset.x"?: number;
    /**
     * y-coordinate of the offset.
     */
    "followOffset.y"?: number;
    /**
     * Sets {@link Phaser.GameObjects.Particles.ParticleEmitter#frames}.
     */
    frames?: number | number[] | string | string[] | Phaser.Textures.Frame | Phaser.Textures.Frame[] | ParticleEmitterFrameConfig;
};

declare type ParticleEmitterFrameConfig = {
    /**
     * One or more texture frames.
     */
    frames?: number | number[] | string | string[] | Phaser.Textures.Frame | Phaser.Textures.Frame[];
    /**
     * Whether texture frames will be assigned consecutively (true) or at random (false).
     */
    cycle?: boolean;
    /**
     * The number of consecutive particles receiving each texture frame, when `cycle` is true.
     */
    quantity?: integer;
};

declare type DeathZoneSourceCallback = (x: number, y: number)=>void;

declare type DeathZoneSource = {
    contains: DeathZoneSourceCallback;
};

declare type EdgeZoneSourceCallback = (quantity: integer, stepRate?: number)=>void;

declare type EdgeZoneSource = {
    /**
     * A function placing points on the source's edge or edges.
     */
    getPoints: EdgeZoneSourceCallback;
};

declare type RandomZoneSourceCallback = (point: Phaser.Math.Vector2)=>void;

declare type RandomZoneSource = {
    /**
     * A function modifying its point argument.
     */
    getRandomPoint: RandomZoneSourceCallback;
};

/**
 * Settings for a PathFollower.
 */
declare type PathConfig = {
    /**
     * The duration of the path follow.
     */
    duration: number;
    /**
     * The start position of the path follow, between 0 and 1.
     */
    from: number;
    /**
     * The end position of the path follow, between 0 and 1.
     */
    to: number;
    /**
     * Whether to position the PathFollower on the Path using its path offset.
     */
    positionOnPath?: boolean;
    /**
     * Should the PathFollower automatically rotate to point in the direction of the Path?
     */
    rotateToPath?: boolean;
    /**
     * If the PathFollower is rotating to match the Path, this value is added to the rotation value. This allows you to rotate objects to a path but control the angle of the rotation as well.
     */
    rotationOffset?: number;
    /**
     * [description]
     */
    verticalAdjust?: boolean;
};

declare type RenderTextureConfig = {
    /**
     * The x coordinate of the RenderTexture's position.
     */
    x?: number;
    /**
     * The y coordinate of the RenderTexture's position.
     */
    y?: number;
    /**
     * The width of the RenderTexture.
     */
    width?: number;
    /**
     * The height of the RenderTexture.
     */
    height?: number;
};

declare type SpriteConfig = GameObjectConfig & {
    /**
     * The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
     */
    key?: string;
    /**
     * An optional frame from the Texture this Game Object is rendering with.
     */
    frame?: number | string;
};

/**
 * A custom function that will be responsible for wrapping the text.
 */
declare type TextStyleWordWrapCallback = (text: string, textObject: Phaser.GameObjects.Text)=>void;

/**
 * Font metrics for a Text Style object.
 */
declare type BitmapTextMetrics = {
    /**
     * The ascent of the font.
     */
    ascent: number;
    /**
     * The descent of the font.
     */
    descent: number;
    /**
     * The size of the font.
     */
    fontSize: number;
};

declare namespace Phaser.GameObjects.Text {
    /**
     * Style settings for a Text object.
     */
    class TextStyle {
        /**
         *
         * @param text The Text object that this TextStyle is styling.
         * @param style The style settings to set.
         */
        constructor(text: Phaser.GameObjects.Text, style: object);

        /**
         * The Text object that this TextStyle is styling.
         */
        parent: Phaser.GameObjects.Text;

        /**
         * The font family.
         */
        fontFamily: string;

        /**
         * The font size.
         */
        fontSize: string;

        /**
         * The font style.
         */
        fontStyle: string;

        /**
         * The background color.
         */
        backgroundColor: string;

        /**
         * The text fill color.
         */
        color: string;

        /**
         * The text stroke color.
         */
        stroke: string;

        /**
         * The text stroke thickness.
         */
        strokeThickness: number;

        /**
         * The horizontal shadow offset.
         */
        shadowOffsetX: number;

        /**
         * The vertical shadow offset.
         */
        shadowOffsetY: number;

        /**
         * The shadow color.
         */
        shadowColor: string;

        /**
         * The shadow blur radius.
         */
        shadowBlur: number;

        /**
         * Whether shadow stroke is enabled or not.
         */
        shadowStroke: boolean;

        /**
         * Whether shadow fill is enabled or not.
         */
        shadowFill: boolean;

        /**
         * The text alignment.
         */
        align: string;

        /**
         * The maximum number of lines to draw.
         */
        maxLines: integer;

        /**
         * The fixed width of the text.
         *
         * `0` means no fixed with.
         */
        fixedWidth: number;

        /**
         * The fixed height of the text.
         *
         * `0` means no fixed height.
         */
        fixedHeight: number;

        /**
         * The resolution the text is rendered to its internal canvas at.
         * The default is 0, which means it will use the resolution set in the Game Config.
         */
        resolution: number;

        /**
         * Whether the text should render right to left.
         */
        rtl: boolean;

        /**
         * The test string to use when measuring the font.
         */
        testString: string;

        /**
         * The amount of horizontal padding adding to the width of the text when calculating the font metrics.
         */
        baselineX: number;

        /**
         * The amount of vertical padding adding to the width of the text when calculating the font metrics.
         */
        baselineY: number;

        /**
         * Set the text style.
         * @param style The style settings to set.
         * @param updateText Whether to update the text immediately. Default true.
         * @param setDefaults Use the default values is not set, or the local values. Default false.
         */
        setStyle(style: object, updateText?: boolean, setDefaults?: boolean): Phaser.GameObjects.Text;

        /**
         * Synchronize the font settings to the given Canvas Rendering Context.
         * @param canvas The Canvas Element.
         * @param context The Canvas Rendering Context.
         */
        syncFont(canvas: HTMLCanvasElement, context: CanvasRenderingContext2D): void;

        /**
         * Synchronize the text style settings to the given Canvas Rendering Context.
         * @param canvas The Canvas Element.
         * @param context The Canvas Rendering Context.
         */
        syncStyle(canvas: HTMLCanvasElement, context: CanvasRenderingContext2D): void;

        /**
         * Synchronize the shadow settings to the given Canvas Rendering Context.
         * @param context The Canvas Rendering Context.
         * @param enabled Whether shadows are enabled or not.
         */
        syncShadow(context: CanvasRenderingContext2D, enabled: boolean): void;

        /**
         * Update the style settings for the parent Text object.
         * @param recalculateMetrics Whether to recalculate font and text metrics.
         */
        update(recalculateMetrics: boolean): Phaser.GameObjects.Text;

        /**
         * Set the font.
         *
         * If a string is given, the font family is set.
         *
         * If an object is given, the `fontFamily`, `fontSize` and `fontStyle`
         * properties of that object are set.
         * @param font The font family or font settings to set.
         */
        setFont(font: string | object): Phaser.GameObjects.Text;

        /**
         * Set the font family.
         * @param family The font family.
         */
        setFontFamily(family: string): Phaser.GameObjects.Text;

        /**
         * Set the font style.
         * @param style The font style.
         */
        setFontStyle(style: string): Phaser.GameObjects.Text;

        /**
         * Set the font size.
         * @param size The font size.
         */
        setFontSize(size: number | string): Phaser.GameObjects.Text;

        /**
         * Set the test string to use when measuring the font.
         * @param string The test string to use when measuring the font.
         */
        setTestString(string: string): Phaser.GameObjects.Text;

        /**
         * Set a fixed width and height for the text.
         *
         * Pass in `0` for either of these parameters to disable fixed width or height respectively.
         * @param width The fixed width to set.
         * @param height The fixed height to set.
         */
        setFixedSize(width: number, height: number): Phaser.GameObjects.Text;

        /**
         * Set the background color.
         * @param color The background color.
         */
        setBackgroundColor(color: string): Phaser.GameObjects.Text;

        /**
         * Set the text fill color.
         * @param color The text fill color.
         */
        setFill(color: string): Phaser.GameObjects.Text;

        /**
         * Set the text fill color.
         * @param color The text fill color.
         */
        setColor(color: string): Phaser.GameObjects.Text;

        /**
         * Set the resolution used by the Text object.
         *
         * By default it will be set to match the resolution set in the Game Config,
         * but you can override it via this method. It allows for much clearer text on High DPI devices,
         * at the cost of memory because it uses larger internal Canvas textures for the Text.
         *
         * Please use with caution, as the more high res Text you have, the more memory it uses up.
         * @param value The resolution for this Text object to use.
         */
        setResolution(value: number): Phaser.GameObjects.Text;

        /**
         * Set the stroke settings.
         * @param color The stroke color.
         * @param thickness The stroke thickness.
         */
        setStroke(color: string, thickness: number): Phaser.GameObjects.Text;

        /**
         * Set the shadow settings.
         *
         * Calling this method always re-measures the parent Text object,
         * so only call it when you actually change the shadow settings.
         * @param x The horizontal shadow offset. Default 0.
         * @param y The vertical shadow offset. Default 0.
         * @param color The shadow color. Default '#000'.
         * @param blur The shadow blur radius. Default 0.
         * @param shadowStroke Whether to stroke the shadow. Default false.
         * @param shadowFill Whether to fill the shadow. Default true.
         */
        setShadow(x?: number, y?: number, color?: string, blur?: number, shadowStroke?: boolean, shadowFill?: boolean): Phaser.GameObjects.Text;

        /**
         * Set the shadow offset.
         * @param x The horizontal shadow offset. Default 0.
         * @param y The vertical shadow offset. Default 0.
         */
        setShadowOffset(x?: number, y?: number): Phaser.GameObjects.Text;

        /**
         * Set the shadow color.
         * @param color The shadow color. Default '#000'.
         */
        setShadowColor(color?: string): Phaser.GameObjects.Text;

        /**
         * Set the shadow blur radius.
         * @param blur The shadow blur radius. Default 0.
         */
        setShadowBlur(blur?: number): Phaser.GameObjects.Text;

        /**
         * Enable or disable shadow stroke.
         * @param enabled Whether shadow stroke is enabled or not.
         */
        setShadowStroke(enabled: boolean): Phaser.GameObjects.Text;

        /**
         * Enable or disable shadow fill.
         * @param enabled Whether shadow fill is enabled or not.
         */
        setShadowFill(enabled: boolean): Phaser.GameObjects.Text;

        /**
         * Set the width (in pixels) to use for wrapping lines.
         *
         * Pass in null to remove wrapping by width.
         * @param width The maximum width of a line in pixels. Set to null to remove wrapping.
         * @param useAdvancedWrap Whether or not to use the advanced wrapping
         * algorithm. If true, spaces are collapsed and whitespace is trimmed from lines. If false,
         * spaces and whitespace are left as is. Default false.
         */
        setWordWrapWidth(width: number, useAdvancedWrap?: boolean): Phaser.GameObjects.Text;

        /**
         * Set a custom callback for wrapping lines.
         *
         * Pass in null to remove wrapping by callback.
         * @param callback A custom function that will be responsible for wrapping the
         * text. It will receive two arguments: text (the string to wrap), textObject (this Text
         * instance). It should return the wrapped lines either as an array of lines or as a string with
         * newline characters in place to indicate where breaks should happen.
         * @param scope The scope that will be applied when the callback is invoked. Default null.
         */
        setWordWrapCallback(callback: TextStyleWordWrapCallback, scope?: object): Phaser.GameObjects.Text;

        /**
         * Set the text alignment.
         *
         * Expects values like `'left'`, `'right'`, `'center'` or `'justified'`.
         * @param align The text alignment.
         */
        setAlign(align: string): Phaser.GameObjects.Text;

        /**
         * Set the maximum number of lines to draw.
         * @param max The maximum number of lines to draw. Default 0.
         */
        setMaxLines(max?: integer): Phaser.GameObjects.Text;

        /**
         * Get the current text metrics.
         */
        getTextMetrics(): BitmapTextMetrics;

        /**
         * Build a JSON representation of this Text Style.
         */
        toJSON(): object;

        /**
         * Destroy this Text Style.
         */
        destroy(): void;

    }

}

declare type TileSprite = GameObjectConfig & {
    /**
     * The x coordinate of the Tile Sprite.
     */
    x?: number;
    /**
     * The y coordinate of the Tile Sprite.
     */
    y?: number;
    /**
     * The width of the Tile Sprite.
     */
    width?: number;
    /**
     * The height of the Tile Sprite.
     */
    height?: number;
    /**
     * The key of the Texture this Tile Sprite will use to render with, as stored in the Texture Manager.
     */
    key?: string;
    /**
     * An optional frame from the Texture this Tile Sprite is rendering with.
     */
    frame?: string;
};

declare type CenterFunction = (triangle: Phaser.Geom.Triangle)=>void;

declare type HitAreaCallback = (hitArea: any, x: number, y: number, gameObject: Phaser.GameObjects.GameObject)=>void;

declare type Pad = {
    /**
     * The ID of the Gamepad.
     */
    id: string;
    /**
     * The index of the Gamepad.
     */
    index: integer;
};

declare type InputPluginContainer = {
    /**
     * The unique name of this plugin in the input plugin cache.
     */
    key: string;
    /**
     * The plugin to be stored. Should be the source object, not instantiated.
     */
    plugin: Function;
    /**
     * If this plugin is to be injected into the Input Plugin, this is the property key map used.
     */
    mapping?: string;
};

declare namespace Phaser.Input.InputPluginCache {
    /**
     * Static method called directly by the Core internal Plugins.
     * Key is a reference used to get the plugin from the plugins object (i.e. InputPlugin)
     * Plugin is the object to instantiate to create the plugin
     * Mapping is what the plugin is injected into the Scene.Systems as (i.e. input)
     * @param key A reference used to get this plugin from the plugin cache.
     * @param plugin The plugin to be stored. Should be the core object, not instantiated.
     * @param mapping If this plugin is to be injected into the Input Plugin, this is the property key used.
     * @param settingsKey The key in the Scene Settings to check to see if this plugin should install or not.
     * @param configKey The key in the Game Config to check to see if this plugin should install or not.
     */
    function register(key: string, plugin: Function, mapping: string, settingsKey: string, configKey: string): void;

    /**
     * Returns the input plugin object from the cache based on the given key.
     * @param key The key of the input plugin to get.
     */
    function getCore(key: string): InputPluginContainer;

    /**
     * Installs all of the registered Input Plugins into the given target.
     * @param target The target InputPlugin to install the plugins into.
     */
    function install(target: Phaser.Input.InputPlugin): void;

    /**
     * Removes an input plugin based on the given key.
     * @param key The key of the input plugin to remove.
     */
    function remove(key: string): void;

}

declare type KeyboardKeydownCallback = (event: KeyboardEvent)=>void;

declare type KeyComboConfig = {
    /**
     * If they press the wrong key do we reset the combo?
     */
    resetOnWrongKey?: boolean;
    /**
     * The max delay in ms between each key press. Above this the combo is reset. 0 means disabled.
     */
    maxKeyDelay?: number;
    /**
     * If previously matched and they press the first key of the combo again, will it reset?
     */
    resetOnMatch?: boolean;
    /**
     * If the combo matches, will it delete itself?
     */
    deleteOnMatch?: boolean;
};

declare type FileConfig = {
    /**
     * The file type string (image, json, etc) for sorting within the Loader.
     */
    type: string;
    /**
     * Unique cache key (unique within its file type)
     */
    key: string;
    /**
     * The URL of the file, not including baseURL.
     */
    url?: string;
    /**
     * The path of the file, not including the baseURL.
     */
    path?: string;
    /**
     * The default extension this file uses.
     */
    extension?: string;
    /**
     * The responseType to be used by the XHR request.
     */
    responseType?: XMLHttpRequestResponseType;
    /**
     * Custom XHR Settings specific to this file and merged with the Loader defaults.
     */
    xhrSettings?: XHRSettingsObject | false;
    /**
     * A config object that can be used by file types to store transitional data.
     */
    config?: any;
};

declare namespace Phaser.Loader.AudioSpriteFile {
    /**
     * Called by each File when it finishes loading.
     * @param file The File that has completed processing.
     */
    function onFileComplete(file: Phaser.Loader.File): void;

    /**
     * Adds this file to its target cache upon successful loading and processing.
     */
    function addToCache(): void;

}

declare namespace Phaser.Loader.FileTypesManager {
    /**
     * Static method called when a LoaderPlugin is created.
     *
     * Loops through the local types object and injects all of them as
     * properties into the LoaderPlugin instance.
     * @param loader The LoaderPlugin to install the types into.
     */
    function register(loader: Phaser.Loader.LoaderPlugin): void;

    /**
     * Removed all associated file types.
     */
    function destroy(): void;

}

declare type XHRSettingsObject = {
    /**
     * The response type of the XHR request, i.e. `blob`, `text`, etc.
     */
    responseType: XMLHttpRequestResponseType;
    /**
     * Should the XHR request use async or not?
     */
    async?: boolean;
    /**
     * Optional username for the XHR request.
     */
    user?: string;
    /**
     * Optional password for the XHR request.
     */
    password?: string;
    /**
     * Optional XHR timeout value.
     */
    timeout?: integer;
    /**
     * This value is used to populate the XHR `setRequestHeader` and is undefined by default.
     */
    header?: string | undefined;
    /**
     * This value is used to populate the XHR `setRequestHeader` and is undefined by default.
     */
    headerValue?: string | undefined;
    /**
     * This value is used to populate the XHR `setRequestHeader` and is undefined by default.
     */
    requestedWith?: string | undefined;
    /**
     * Provide a custom mime-type to use instead of the default.
     */
    overrideMimeType?: string | undefined;
};

declare type SinCosTable = {
    /**
     * The sine value.
     */
    sin: number;
    /**
     * The cosine value.
     */
    cos: number;
    /**
     * The length.
     */
    length: number;
};

declare type Vector2Like = {
    /**
     * The x component.
     */
    x: number;
    /**
     * The y component.
     */
    y: number;
};

declare namespace Phaser {
    namespace Actions {
        /**
         * Takes an array of Game Objects, or any objects that have a public `angle` property,
         * and then adds the given value to each of their `angle` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `Angle(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `angle` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function Angle<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of objects and passes each of them to the given callback.
         * @param items The array of items to be updated by this action.
         * @param callback The callback to be invoked. It will be passed just one argument: the item from the array.
         * @param context The scope in which the callback will be invoked.
         */
        function Call<G extends Phaser.GameObjects.GameObject[]>(items: G, callback: CallCallback, context: any): G;

        /**
         * Takes an array of objects and returns the first element in the array that has properties which match
         * all of those specified in the `compare` object. For example, if the compare object was: `{ scaleX: 0.5, alpha: 1 }`
         * then it would return the first item which had the property `scaleX` set to 0.5 and `alpha` set to 1.
         *
         * To use this with a Group: `GetFirst(group.getChildren(), compare, index)`
         * @param items The array of items to be searched by this action.
         * @param compare The comparison object. Each property in this object will be checked against the items of the array.
         * @param index An optional offset to start searching from within the items array. Default 0.
         */
        function GetFirst<G extends Phaser.GameObjects.GameObject[]>(items: G, compare: object, index?: integer): object | Phaser.GameObjects.GameObject;

        /**
         * Takes an array of objects and returns the last element in the array that has properties which match
         * all of those specified in the `compare` object. For example, if the compare object was: `{ scaleX: 0.5, alpha: 1 }`
         * then it would return the last item which had the property `scaleX` set to 0.5 and `alpha` set to 1.
         *
         * To use this with a Group: `GetLast(group.getChildren(), compare, index)`
         * @param items The array of items to be searched by this action.
         * @param compare The comparison object. Each property in this object will be checked against the items of the array.
         * @param index An optional offset to start searching from within the items array. Default 0.
         */
        function GetLast<G extends Phaser.GameObjects.GameObject[]>(items: G, compare: object, index?: integer): object | Phaser.GameObjects.GameObject;

        /**
         * Takes an array of Game Objects, or any objects that have public `x` and `y` properties,
         * and then aligns them based on the grid configuration given to this action.
         * @param items The array of items to be updated by this action.
         * @param options The GridAlign Configuration object.
         */
        function GridAlign<G extends Phaser.GameObjects.GameObject[]>(items: G, options: GridAlignConfig): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `alpha` property,
         * and then adds the given value to each of their `alpha` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `IncAlpha(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `alpha` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncAlpha<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `x` property,
         * and then adds the given value to each of their `x` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `IncX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `x` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have public `x` and `y` properties,
         * and then adds the given value to each of them.
         *
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `IncXY(group.getChildren(), x, y, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param x The amount to be added to the `x` property.
         * @param y The amount to be added to the `y` property. If `undefined` or `null` it uses the `x` value. Default x.
         * @param stepX This is added to the `x` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `y` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncXY<G extends Phaser.GameObjects.GameObject[]>(items: G, x: number, y?: number, stepX?: number, stepY?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `y` property,
         * and then adds the given value to each of their `y` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `IncY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `y` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function IncY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the perimeter of a Circle.
         *
         * If you wish to pass a `Phaser.GameObjects.Circle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param circle The Circle to position the Game Objects on.
         * @param startAngle Optional angle to start position from, in radians. Default 0.
         * @param endAngle Optional angle to stop position at, in radians. Default 6.28.
         */
        function PlaceOnCircle<G extends Phaser.GameObjects.GameObject[]>(items: G, circle: Phaser.Geom.Circle, startAngle?: number, endAngle?: number): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the perimeter of an Ellipse.
         *
         * If you wish to pass a `Phaser.GameObjects.Ellipse` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param ellipse The Ellipse to position the Game Objects on.
         * @param startAngle Optional angle to start position from, in radians. Default 0.
         * @param endAngle Optional angle to stop position at, in radians. Default 6.28.
         */
        function PlaceOnEllipse<G extends Phaser.GameObjects.GameObject[]>(items: G, ellipse: Phaser.Geom.Ellipse, startAngle?: number, endAngle?: number): G;

        /**
         * Positions an array of Game Objects on evenly spaced points of a Line.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param line The Line to position the Game Objects on.
         */
        function PlaceOnLine<G extends Phaser.GameObjects.GameObject[]>(items: G, line: Phaser.Geom.Line): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the perimeter of a Rectangle.
         *
         * Placement starts from the top-left of the rectangle, and proceeds in a clockwise direction.
         * If the `shift` parameter is given you can offset where placement begins.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param rect The Rectangle to position the Game Objects on.
         * @param shift An optional positional offset. Default 1.
         */
        function PlaceOnRectangle<G extends Phaser.GameObjects.GameObject[]>(items: G, rect: Phaser.Geom.Rectangle, shift?: integer): G;

        /**
         * Takes an array of Game Objects and positions them on evenly spaced points around the edges of a Triangle.
         *
         * If you wish to pass a `Phaser.GameObjects.Triangle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param triangle The Triangle to position the Game Objects on.
         * @param stepRate An optional step rate, to increase or decrease the packing of the Game Objects on the lines. Default 1.
         */
        function PlaceOnTriangle<G extends Phaser.GameObjects.GameObject[]>(items: G, triangle: Phaser.Geom.Triangle, stepRate?: number): G;

        /**
         * Play an animation with the given key, starting at the given startFrame on all Game Objects in items.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param key The name of the animation to play.
         * @param startFrame The starting frame of the animation with the given key.
         */
        function PlayAnimation<G extends Phaser.GameObjects.GameObject[]>(items: G, key: string, startFrame?: string | integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public property as defined in `key`,
         * and then adds the given value to it.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `PropertyValueInc(group.getChildren(), key, value, step)`
         * @param items The array of items to be updated by this action.
         * @param key The property to be updated.
         * @param value The amount to be added to the property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function PropertyValueInc<G extends Phaser.GameObjects.GameObject[]>(items: G, key: string, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public property as defined in `key`,
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `PropertyValueSet(group.getChildren(), key, value, step)`
         * @param items The array of items to be updated by this action.
         * @param key The property to be updated.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function PropertyValueSet<G extends Phaser.GameObjects.GameObject[]>(items: G, key: string, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Circle.
         *
         * If you wish to pass a `Phaser.GameObjects.Circle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param circle The Circle to position the Game Objects within.
         */
        function RandomCircle<G extends Phaser.GameObjects.GameObject[]>(items: G, circle: Phaser.Geom.Circle): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Ellipse.
         *
         * If you wish to pass a `Phaser.GameObjects.Ellipse` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param ellipse The Ellipse to position the Game Objects within.
         */
        function RandomEllipse<G extends Phaser.GameObjects.GameObject[]>(items: G, ellipse: Phaser.Geom.Ellipse): G;

        /**
         * Takes an array of Game Objects and positions them at random locations on the Line.
         *
         * If you wish to pass a `Phaser.GameObjects.Line` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param line The Line to position the Game Objects randomly on.
         */
        function RandomLine<G extends Phaser.GameObjects.GameObject[]>(items: G, line: Phaser.Geom.Line): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Ellipse.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param rect The Rectangle to position the Game Objects within.
         */
        function RandomRectangle<G extends Phaser.GameObjects.GameObject[]>(items: G, rect: Phaser.Geom.Rectangle): G;

        /**
         * Takes an array of Game Objects and positions them at random locations within the Triangle.
         *
         * If you wish to pass a `Phaser.GameObjects.Triangle` Shape to this function, you should pass its `geom` property.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param triangle The Triangle to position the Game Objects within.
         */
        function RandomTriangle<G extends Phaser.GameObjects.GameObject[]>(items: G, triangle: Phaser.Geom.Triangle): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `rotation` property,
         * and then adds the given value to each of their `rotation` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `Rotate(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `rotation` property (in radians).
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function Rotate<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Rotates each item around the given point by the given angle.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param point Any object with public `x` and `y` properties.
         * @param angle The angle to rotate by, in radians.
         */
        function RotateAround<G extends Phaser.GameObjects.GameObject[]>(items: G, point: object, angle: number): G;

        /**
         * Rotates an array of Game Objects around a point by the given angle and distance.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param point Any object with public `x` and `y` properties.
         * @param angle The angle to rotate by, in radians.
         * @param distance The distance from the point of rotation in pixels.
         */
        function RotateAroundDistance<G extends Phaser.GameObjects.GameObject[]>(items: G, point: object, angle: number, distance: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `scaleX` property,
         * and then adds the given value to each of their `scaleX` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `ScaleX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `scaleX` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function ScaleX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have public `scaleX` and `scaleY` properties,
         * and then adds the given value to each of them.
         *
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `ScaleXY(group.getChildren(), scaleX, scaleY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param scaleX The amount to be added to the `scaleX` property.
         * @param scaleY The amount to be added to the `scaleY` property. If `undefined` or `null` it uses the `scaleX` value.
         * @param stepX This is added to the `scaleX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `y` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function ScaleXY<G extends Phaser.GameObjects.GameObject[]>(items: G, scaleX: number, scaleY?: number, stepX?: number, stepY?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have a public `scaleY` property,
         * and then adds the given value to each of their `scaleY` properties.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `ScaleY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to be added to the `scaleY` property.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function ScaleY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `alpha`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetAlpha(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetAlpha<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `blendMode`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetBlendMode(group.getChildren(), value)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetBlendMode<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `depth`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetDepth(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetDepth<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Passes all provided Game Objects to the Input Manager to enable them for input with identical areas and callbacks.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param hitArea Either an input configuration object, or a geometric shape that defines the hit area for the Game Object. If not specified a Rectangle will be used.
         * @param hitAreaCallback A callback to be invoked when the Game Object is interacted with. If you provide a shape you must also provide a callback.
         */
        function SetHitArea<G extends Phaser.GameObjects.GameObject[]>(items: G, hitArea: any, hitAreaCallback: HitAreaCallback): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `originX` and `originY`
         * and then sets them to the given values.
         *
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetOrigin(group.getChildren(), originX, originY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param originX The amount to set the `originX` property to.
         * @param originY The amount to set the `originY` property to. If `undefined` or `null` it uses the `originX` value.
         * @param stepX This is added to the `originX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `originY` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetOrigin<G extends Phaser.GameObjects.GameObject[]>(items: G, originX: number, originY?: number, stepX?: number, stepY?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `rotation`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetRotation(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetRotation<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `scaleX` and `scaleY`
         * and then sets them to the given values.
         *
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetScale(group.getChildren(), scaleX, scaleY, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param scaleX The amount to set the `scaleX` property to.
         * @param scaleY The amount to set the `scaleY` property to. If `undefined` or `null` it uses the `scaleX` value.
         * @param stepX This is added to the `scaleX` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `scaleY` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScale<G extends Phaser.GameObjects.GameObject[]>(items: G, scaleX: number, scaleY?: number, stepX?: number, stepY?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `scaleX`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetScaleX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScaleX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `scaleY`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetScaleY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetScaleY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public method setTint() and then updates it to the given value(s). You can specify tint color per corner or provide only one color value for `topLeft` parameter, in which case whole item will be tinted with that color.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param topLeft The tint being applied to top-left corner of item. If other parameters are given no value, this tint will be applied to whole item.
         * @param topRight The tint to be applied to top-right corner of item.
         * @param bottomLeft The tint to be applied to the bottom-left corner of item.
         * @param bottomRight The tint to be applied to the bottom-right corner of item.
         */
        function SetTint<G extends Phaser.GameObjects.GameObject[]>(items: G, topLeft: number, topRight?: number, bottomLeft?: number, bottomRight?: number): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `visible`
         * and then sets it to the given value.
         *
         * To use this with a Group: `SetVisible(group.getChildren(), value)`
         * @param items The array of items to be updated by this action.
         * @param value The value to set the property to.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetVisible<G extends Phaser.GameObjects.GameObject[]>(items: G, value: boolean, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `x`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetX(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetX<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public properties `x` and `y`
         * and then sets them to the given values.
         *
         * The optional `stepX` and `stepY` properties are applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetXY(group.getChildren(), x, y, stepX, stepY)`
         * @param items The array of items to be updated by this action.
         * @param x The amount to set the `x` property to.
         * @param y The amount to set the `y` property to. If `undefined` or `null` it uses the `x` value. Default x.
         * @param stepX This is added to the `x` amount, multiplied by the iteration counter. Default 0.
         * @param stepY This is added to the `y` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetXY<G extends Phaser.GameObjects.GameObject[]>(items: G, x: number, y?: number, stepX?: number, stepY?: number, index?: integer, direction?: integer): G;

        /**
         * Takes an array of Game Objects, or any objects that have the public property `y`
         * and then sets it to the given value.
         *
         * The optional `step` property is applied incrementally, multiplied by each item in the array.
         *
         * To use this with a Group: `SetY(group.getChildren(), value, step)`
         * @param items The array of items to be updated by this action.
         * @param value The amount to set the property to.
         * @param step This is added to the `value` amount, multiplied by the iteration counter. Default 0.
         * @param index An optional offset to start searching from within the items array. Default 0.
         * @param direction The direction to iterate through the array. 1 is from beginning to end, -1 from end to beginning. Default 1.
         */
        function SetY<G extends Phaser.GameObjects.GameObject[]>(items: G, value: number, step?: number, index?: integer, direction?: integer): G;

        /**
         * Iterate through the items array changing the position of each element to be that of the element that came before
         * it in the array (or after it if direction = 1)
         *
         * The first items position is set to x/y.
         *
         * The final x/y coords are returned
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param x The x coordinate to place the first item in the array at.
         * @param y The y coordinate to place the first item in the array at.
         * @param direction The iteration direction. 0 = first to last and 1 = last to first. Default 0.
         * @param output An optional objec to store the final objects position in.
         */
        function ShiftPosition<G extends Phaser.GameObjects.GameObject[], O extends Phaser.Math.Vector2>(items: G, x: number, y: number, direction?: integer, output?: O): O;

        /**
         * Shuffles the array in place. The shuffled array is both modified and returned.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         */
        function Shuffle<G extends Phaser.GameObjects.GameObject[]>(items: G): G;

        /**
         * Smootherstep is a sigmoid-like interpolation and clamping function.
         *
         * The function depends on three parameters, the input x, the "left edge" and the "right edge", with the left edge being assumed smaller than the right edge. The function receives a real number x as an argument and returns 0 if x is less than or equal to the left edge, 1 if x is greater than or equal to the right edge, and smoothly interpolates, using a Hermite polynomial, between 0 and 1 otherwise. The slope of the smoothstep function is zero at both edges. This is convenient for creating a sequence of transitions using smoothstep to interpolate each segment as an alternative to using more sophisticated or expensive interpolation techniques.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param property The property of the Game Object to interpolate.
         * @param min The minimum interpolation value.
         * @param max The maximum interpolation value.
         * @param inc Should the values be incremented? `true` or set (`false`) Default false.
         */
        function SmootherStep<G extends Phaser.GameObjects.GameObject[]>(items: G, property: string, min: number, max: number, inc?: boolean): G;

        /**
         * Smoothstep is a sigmoid-like interpolation and clamping function.
         *
         * The function depends on three parameters, the input x, the "left edge" and the "right edge", with the left edge being assumed smaller than the right edge. The function receives a real number x as an argument and returns 0 if x is less than or equal to the left edge, 1 if x is greater than or equal to the right edge, and smoothly interpolates, using a Hermite polynomial, between 0 and 1 otherwise. The slope of the smoothstep function is zero at both edges. This is convenient for creating a sequence of transitions using smoothstep to interpolate each segment as an alternative to using more sophisticated or expensive interpolation techniques.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param property The property of the Game Object to interpolate.
         * @param min The minimum interpolation value.
         * @param max The maximum interpolation value.
         * @param inc Should the values be incremented? `true` or set (`false`) Default false.
         */
        function SmoothStep<G extends Phaser.GameObjects.GameObject[]>(items: G, property: string, min: number, max: number, inc?: boolean): G;

        /**
         * Takes an array of Game Objects and then modifies their `property` so the value equals, or is incremented, the
         * calculated spread value.
         *
         * The spread value is derived from the given `min` and `max` values and the total number of items in the array.//#endregion
         *
         * For example, to cause an array of Sprites to change in alpha from 0 to 1 you could call:
         *
         * ```javascript
         * Phaser.Actions.Spread(itemsArray, 'alpha', 0, 1);
         * ```
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param property The property of the Game Object to spread.
         * @param min The minimum value.
         * @param max The maximum value.
         * @param inc Should the values be incremented? `true` or set (`false`) Default false.
         */
        function Spread<G extends Phaser.GameObjects.GameObject[]>(items: G, property: string, min: number, max: number, inc?: boolean): G;

        /**
         * Takes an array of Game Objects and toggles the visibility of each one.
         * Those previously `visible = false` will become `visible = true`, and vice versa.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         */
        function ToggleVisible<G extends Phaser.GameObjects.GameObject[]>(items: G): G;

        /**
         * Wrap each item's coordinates within a rectangle's area.
         * @param items An array of Game Objects. The contents of this array are updated by this Action.
         * @param rect The rectangle.
         * @param padding An amount added to each side of the rectangle during the operation. Default 0.
         */
        function WrapInRectangle<G extends Phaser.GameObjects.GameObject[]>(items: G, rect: Phaser.Geom.Rectangle, padding?: number): G;

    }

    namespace Animations {
        /**
         * A Frame based Animation.
         *
         * This consists of a key, some default values (like the frame rate) and a bunch of Frame objects.
         *
         * The Animation Manager creates these. Game Objects don't own an instance of these directly.
         * Game Objects have the Animation Component, which are like playheads to global Animations (these objects)
         * So multiple Game Objects can have playheads all pointing to this one Animation instance.
         */
        class Animation {
            /**
             *
             * @param manager [description]
             * @param key [description]
             * @param config [description]
             */
            constructor(manager: Phaser.Animations.AnimationManager, key: string, config: AnimationConfig);

            /**
             * A reference to the global Animation Manager
             */
            manager: Phaser.Animations.AnimationManager;

            /**
             * The unique identifying string for this animation
             */
            key: string;

            /**
             * A frame based animation (as opposed to a bone based animation)
             */
            type: string;

            /**
             * Extract all the frame data into the frames array
             */
            frames: Phaser.Animations.AnimationFrame[];

            /**
             * The frame rate of playback in frames per second (default 24 if duration is null)
             */
            frameRate: integer;

            /**
             * How long the animation should play for, in milliseconds.
             * If the `frameRate` property has been set then it overrides this value,
             * otherwise the `frameRate` is derived from `duration`.
             */
            duration: integer;

            /**
             * How many ms per frame, not including frame specific modifiers.
             */
            msPerFrame: integer;

            /**
             * Skip frames if the time lags, or always advanced anyway?
             */
            skipMissedFrames: boolean;

            /**
             * The delay in ms before the playback will begin.
             */
            delay: integer;

            /**
             * Number of times to repeat the animation. Set to -1 to repeat forever.
             */
            repeat: integer;

            /**
             * The delay in ms before the a repeat playthrough starts.
             */
            repeatDelay: integer;

            /**
             * Should the animation yoyo? (reverse back down to the start) before repeating?
             */
            yoyo: boolean;

            /**
             * Should sprite.visible = true when the animation starts to play?
             */
            showOnStart: boolean;

            /**
             * Should sprite.visible = false when the animation finishes?
             */
            hideOnComplete: boolean;

            /**
             * Global pause. All Game Objects using this Animation instance are impacted by this property.
             */
            paused: boolean;

            /**
             * Add frames to the end of the animation.
             * @param config [description]
             */
            addFrame(config: string | AnimationFrameConfig[]): Phaser.Animations.Animation;

            /**
             * Add frame/s into the animation.
             * @param index [description]
             * @param config [description]
             */
            addFrameAt(index: integer, config: string | AnimationFrameConfig[]): Phaser.Animations.Animation;

            /**
             * Check if the given frame index is valid.
             * @param index The index to be checked.
             */
            checkFrame(index: integer): boolean;

            /**
             * [description]
             * @param component [description]
             */
            protected completeAnimation(component: Phaser.GameObjects.Components.Animation): void;

            /**
             * [description]
             * @param component [description]
             * @param includeDelay [description] Default true.
             */
            protected getFirstTick(component: Phaser.GameObjects.Components.Animation, includeDelay?: boolean): void;

            /**
             * Returns the AnimationFrame at the provided index
             * @param index The index in the AnimationFrame array
             */
            protected getFrameAt(index: integer): Phaser.Animations.AnimationFrame;

            /**
             * [description]
             * @param textureManager [description]
             * @param frames [description]
             * @param defaultTextureKey [description]
             */
            getFrames(textureManager: Phaser.Textures.TextureManager, frames: string | AnimationFrameConfig[], defaultTextureKey?: string): Phaser.Animations.AnimationFrame[];

            /**
             * [description]
             * @param component [description]
             */
            getNextTick(component: Phaser.GameObjects.Components.Animation): void;

            /**
             * Returns the frame closest to the given progress value between 0 and 1.
             * @param value A value between 0 and 1.
             */
            getFrameByProgress(value: number): Phaser.Animations.AnimationFrame;

            /**
             * Advance the animation frame.
             * @param component The Animation Component to advance.
             */
            nextFrame(component: Phaser.GameObjects.Components.Animation): void;

            /**
             * Returns the animation last frame.
             */
            getLastFrame(): Phaser.Animations.AnimationFrame;

            /**
             * [description]
             * @param component [description]
             */
            previousFrame(component: Phaser.GameObjects.Components.Animation): void;

            /**
             * [description]
             * @param frame [description]
             */
            removeFrame(frame: Phaser.Animations.AnimationFrame): Phaser.Animations.Animation;

            /**
             * Removes a frame from the AnimationFrame array at the provided index
             * and updates the animation accordingly.
             * @param index The index in the AnimationFrame array
             */
            removeFrameAt(index: integer): Phaser.Animations.Animation;

            /**
             * [description]
             * @param component [description]
             */
            repeatAnimation(component: Phaser.GameObjects.Components.Animation): void;

            /**
             * [description]
             * @param component [description]
             */
            setFrame(component: Phaser.GameObjects.Components.Animation): void;

            /**
             * [description]
             */
            toJSON(): JSONAnimation;

            /**
             * [description]
             */
            updateFrameSequence(): Phaser.Animations.Animation;

            /**
             * [description]
             */
            pause(): Phaser.Animations.Animation;

            /**
             * [description]
             */
            resume(): Phaser.Animations.Animation;

            /**
             * [description]
             */
            destroy(): void;

        }

        /**
         * A single frame in an Animation sequence.
         *
         * An AnimationFrame consists of a reference to the Texture it uses for rendering, references to other
         * frames in the animation, and index data. It also has the ability to fire its own `onUpdate` callback
         * and modify the animation timing.
         *
         * AnimationFrames are generated automatically by the Animation class.
         */
        class AnimationFrame {
            /**
             *
             * @param textureKey The key of the Texture this AnimationFrame uses.
             * @param textureFrame The key of the Frame within the Texture that this AnimationFrame uses.
             * @param index The index of this AnimationFrame within the Animation sequence.
             * @param frame A reference to the Texture Frame this AnimationFrame uses for rendering.
             */
            constructor(textureKey: string, textureFrame: string | integer, index: integer, frame: Phaser.Textures.Frame);

            /**
             * The key of the Texture this AnimationFrame uses.
             */
            textureKey: string;

            /**
             * The key of the Frame within the Texture that this AnimationFrame uses.
             */
            textureFrame: string | integer;

            /**
             * The index of this AnimationFrame within the Animation sequence.
             */
            index: integer;

            /**
             * A reference to the Texture Frame this AnimationFrame uses for rendering.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Is this the first frame in an animation sequence?
             */
            readonly isFirst: boolean;

            /**
             * Is this the last frame in an animation sequence?
             */
            readonly isLast: boolean;

            /**
             * A reference to the AnimationFrame that comes before this one in the animation, if any.
             */
            readonly prevFrame: Phaser.Animations.AnimationFrame;

            /**
             * A reference to the AnimationFrame that comes after this one in the animation, if any.
             */
            readonly nextFrame: Phaser.Animations.AnimationFrame;

            /**
             * Additional time (in ms) that this frame should appear for during playback.
             * The value is added onto the msPerFrame set by the animation.
             */
            duration: number;

            /**
             * What % through the animation does this frame come?
             * This value is generated when the animation is created and cached here.
             */
            readonly progress: number;

            /**
             * Generates a JavaScript object suitable for converting to JSON.
             */
            toJSON(): JSONAnimationFrame;

            /**
             * Destroys this object by removing references to external resources and callbacks.
             */
            destroy(): void;

        }

        /**
         * The Animation Manager.
         *
         * Animations are managed by the global Animation Manager. This is a singleton class that is
         * responsible for creating and delivering animations and their corresponding data to all Game Objects.
         * Unlike plugins it is owned by the Game instance, not the Scene.
         *
         * Sprites and other Game Objects get the data they need from the AnimationManager.
         */
        class AnimationManager extends Phaser.Events.EventEmitter {
            /**
             *
             * @param game A reference to the Phaser.Game instance.
             */
            constructor(game: Phaser.Game);

            /**
             * A reference to the Phaser.Game instance.
             */
            protected game: Phaser.Game;

            /**
             * A reference to the Texture Manager.
             */
            protected textureManager: Phaser.Textures.TextureManager;

            /**
             * [description]
             */
            globalTimeScale: number;

            /**
             * [description]
             */
            protected anims: Phaser.Structs.Map<string, Phaser.Animations.Animation>;

            /**
             * [description]
             */
            paused: boolean;

            /**
             * [description]
             */
            name: string;

            /**
             * [description]
             */
            boot(): void;

            /**
             * [description]
             * @param key [description]
             * @param animation [description]
             */
            add(key: string, animation: Phaser.Animations.Animation): Phaser.Animations.AnimationManager;

            /**
             * [description]
             * @param config [description]
             */
            create(config: AnimationConfig): Phaser.Animations.Animation;

            /**
             * [description]
             * @param data [description]
             * @param clearCurrentAnimations If set to `true`, the current animations will be removed (`anims.clear()`). If set to `false` (default), the animations in `data` will be added. Default false.
             */
            fromJSON(data: string | JSONAnimationManager | JSONAnimation, clearCurrentAnimations?: boolean): Phaser.Animations.Animation[];

            /**
             * Generate an array of {@link AnimationFrameConfig} objects from a texture key and configuration object.
             *
             * Generates objects with string frame names, as configured by the given {@link AnimationFrameConfig}.
             * @param key The key for the texture containing the animation frames.
             * @param config The configuration object for the animation frame names.
             */
            generateFrameNames(key: string, config?: GenerateFrameNamesConfig): AnimationFrameConfig[];

            /**
             * Generate an array of {@link AnimationFrameConfig} objects from a texture key and configuration object.
             *
             * Generates objects with numbered frame names, as configured by the given {@link GenerateFrameNumbersConfig}.
             * @param key The key for the texture containing the animation frames.
             * @param config The configuration object for the animation frames.
             */
            generateFrameNumbers(key: string, config: GenerateFrameNumbersConfig): AnimationFrameConfig[];

            /**
             * Get an Animation.
             * @param key The key of the Animation to retrieve.
             */
            get(key: string): Phaser.Animations.Animation;

            /**
             * Load an Animation into a Game Object's Animation Component.
             * @param child The Game Object to load the animation into.
             * @param key The key of the animation to load.
             * @param startFrame The name of a start frame to set on the loaded animation.
             */
            load(child: Phaser.GameObjects.GameObject, key: string, startFrame?: string | integer): Phaser.GameObjects.GameObject;

            /**
             * Pause all animations.
             */
            pauseAll(): Phaser.Animations.AnimationManager;

            /**
             * Play an animation on the given Game Objects that have an Animation Component.
             * @param key The key of the animation to play on the Game Object.
             * @param child The Game Objects to play the animation on.
             */
            play(key: string, child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): Phaser.Animations.AnimationManager;

            /**
             * Remove an animation.
             * @param key The key of the animation to remove.
             */
            remove(key: string): Phaser.Animations.Animation;

            /**
             * Resume all paused animations.
             */
            resumeAll(): Phaser.Animations.AnimationManager;

            /**
             * Takes an array of Game Objects that have an Animation Component and then
             * starts the given animation playing on them, each one offset by the
             * `stagger` amount given to this method.
             * @param key The key of the animation to play on the Game Objects.
             * @param children An array of Game Objects to play the animation on. They must have an Animation Component.
             * @param stagger The amount of time, in milliseconds, to offset each play time by. Default 0.
             */
            staggerPlay<G extends Phaser.GameObjects.GameObject[]>(key: string, children: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[], stagger?: number): G;

            /**
             * [description]
             * @param key [description]
             */
            toJSON(key: string): JSONAnimationManager;

            /**
             * [description]
             */
            destroy(): void;

        }

    }

    /**
     * The Phaser.Game instance is the main controller for the entire Phaser game. It is responsible
     * for handling the boot process, parsing the configuration values, creating the renderer,
     * and setting-up all of the global Phaser systems, such as sound and input.
     * Once that is complete it will start the Scene Manager and then begin the main game loop.
     *
     * You should generally avoid accessing any of the systems created by Game, and instead use those
     * made available to you via the Phaser.Scene Systems class instead.
     */
    class Game {
        /**
         *
         * @param GameConfig The configuration object for your Phaser Game instance.
         */
        constructor(GameConfig?: GameConfig);

        /**
         * The parsed Game Configuration object.
         *
         * The values stored within this object are read-only and should not be changed at run-time.
         */
        readonly config: Phaser.Boot.Config;

        /**
         * A reference to either the Canvas or WebGL Renderer that this Game is using.
         */
        renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer;

        /**
         * A reference to an HTML Div Element used as a DOM Element Container.
         *
         * Only set if `createDOMContainer` is `true` in the game config (by default it is `false`) and
         * if you provide a parent element to insert the Phaser Game inside.
         *
         * See the DOM Element Game Object for more details.
         */
        domContainer: HTMLDivElement;

        /**
         * A reference to the HTML Canvas Element that Phaser uses to render the game.
         * This is created automatically by Phaser unless you provide a `canvas` property
         * in your Game Config.
         */
        canvas: HTMLCanvasElement;

        /**
         * A reference to the Rendering Context belonging to the Canvas Element this game is rendering to.
         * If the game is running under Canvas it will be a 2d Canvas Rendering Context.
         * If the game is running under WebGL it will be a WebGL Rendering Context.
         * This context is created automatically by Phaser unless you provide a `context` property
         * in your Game Config.
         */
        context: CanvasRenderingContext2D | WebGLRenderingContext;

        /**
         * A flag indicating when this Game instance has finished its boot process.
         */
        readonly isBooted: boolean;

        /**
         * A flag indicating if this Game is currently running its game step or not.
         */
        readonly isRunning: boolean;

        /**
         * An Event Emitter which is used to broadcast game-level events from the global systems.
         */
        events: Phaser.Events.EventEmitter;

        /**
         * An instance of the Animation Manager.
         *
         * The Animation Manager is a global system responsible for managing all animations used within your game.
         */
        anims: Phaser.Animations.AnimationManager;

        /**
         * An instance of the Texture Manager.
         *
         * The Texture Manager is a global system responsible for managing all textures being used by your game.
         */
        textures: Phaser.Textures.TextureManager;

        /**
         * An instance of the Cache Manager.
         *
         * The Cache Manager is a global system responsible for caching, accessing and releasing external game assets.
         */
        cache: Phaser.Cache.CacheManager;

        /**
         * An instance of the Data Manager
         */
        registry: Phaser.Data.DataManager;

        /**
         * An instance of the Input Manager.
         *
         * The Input Manager is a global system responsible for the capture of browser-level input events.
         */
        input: Phaser.Input.InputManager;

        /**
         * An instance of the Scene Manager.
         *
         * The Scene Manager is a global system responsible for creating, modifying and updating the Scenes in your game.
         */
        scene: Phaser.Scenes.SceneManager;

        /**
         * A reference to the Device inspector.
         *
         * Contains information about the device running this game, such as OS, browser vendor and feature support.
         * Used by various systems to determine capabilities and code paths.
         */
        device: Phaser.DeviceConf;

        /**
         * An instance of the Scale Manager.
         *
         * The Scale Manager is a global system responsible for handling game scaling events.
         */
        scaleManager: Phaser.Boot.ScaleManager;

        /**
         * An instance of the base Sound Manager.
         *
         * The Sound Manager is a global system responsible for the playback and updating of all audio in your game.
         */
        sound: Phaser.Sound.BaseSoundManager;

        /**
         * An instance of the Time Step.
         *
         * The Time Step is a global system responsible for setting-up and responding to the browser frame events, processing
         * them and calculating delta values. It then automatically calls the game step.
         */
        loop: Phaser.Boot.TimeStep;

        /**
         * An instance of the Plugin Manager.
         *
         * The Plugin Manager is a global system that allows plugins to register themselves with it, and can then install
         * those plugins into Scenes as required.
         */
        plugins: Phaser.Plugins.PluginManager;

        /**
         * An instance of the Facebook Instant Games Plugin.
         *
         * This will only be available if the plugin has been built into Phaser,
         * or you're using the special Facebook Instant Games custom build.
         */
        facebook: Phaser.FacebookInstantGamesPlugin;

        /**
         * Does the window the game is running in currently have focus or not?
         * This is modified by the VisibilityHandler.
         */
        readonly hasFocus: boolean;

        /**
         * Is the mouse pointer currently over the game canvas or not?
         * This is modified by the VisibilityHandler.
         */
        readonly isOver: boolean;

        /**
         * This method is called automatically when the DOM is ready. It is responsible for creating the renderer,
         * displaying the Debug Header, adding the game canvas to the DOM and emitting the 'boot' event.
         * It listens for a 'ready' event from the base systems and once received it will call `Game.start`.
         */
        protected boot(): void;

        /**
         * Called automatically by Game.boot once all of the global systems have finished setting themselves up.
         * By this point the Game is now ready to start the main loop running.
         * It will also enable the Visibility Handler.
         */
        protected start(): void;

        /**
         * The main Game Step. Called automatically by the Time Step, once per browser frame (typically as a result of
         * Request Animation Frame, or Set Timeout on very old browsers.)
         *
         * The step will update the global managers first, then proceed to update each Scene in turn, via the Scene Manager.
         *
         * It will then render each Scene in turn, via the Renderer. This process emits `prerender` and `postrender` events.
         * @param time The current time. Either a High Resolution Timer value if it comes from Request Animation Frame, or Date.now if using SetTimeout.
         * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
         */
        step(time: number, delta: number): void;

        /**
         * A special version of the Game Step for the HEADLESS renderer only.
         *
         * The main Game Step. Called automatically by the Time Step, once per browser frame (typically as a result of
         * Request Animation Frame, or Set Timeout on very old browsers.)
         *
         * The step will update the global managers first, then proceed to update each Scene in turn, via the Scene Manager.
         *
         * This process emits `prerender` and `postrender` events, even though nothing actually displays.
         * @param time The current time. Either a High Resolution Timer value if it comes from Request Animation Frame, or Date.now if using SetTimeout.
         * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
         */
        headlessStep(time: number, delta: number): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will pause the main loop and then emit a pause event.
         */
        protected onHidden(): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will resume the main loop and then emit a resume event.
         */
        protected onVisible(): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will set the main loop into a 'blurred' state, which pauses it.
         */
        protected onBlur(): void;

        /**
         * Called automatically by the Visibility Handler.
         * This will set the main loop into a 'focused' state, which resumes it.
         */
        protected onFocus(): void;

        /**
         * Updates the Game Config with the new width and height values given.
         * Then resizes the Renderer and Input Manager scale.
         * @param width The new width of the game.
         * @param height The new height of the game.
         */
        resize(width: number, height: number): void;

        /**
         * Flags this Game instance as needing to be destroyed on the next frame.
         * It will wait until the current frame has completed and then call `runDestroy` internally.
         *
         * If you **do not** need to run Phaser again on the same web page you can set the `noReturn` argument to `true` and it will free-up
         * memory being held by the core Phaser plugins. If you do need to create another game instance on the same page, leave this as `false`.
         * @param removeCanvas Set to `true` if you would like the parent canvas element removed from the DOM, or `false` to leave it in place.
         * @param noReturn If `true` all the core Phaser plugins are destroyed. You cannot create another instance of Phaser on the same web page if you do this. Default false.
         */
        destroy(removeCanvas: boolean, noReturn?: boolean): void;

    }

    namespace Boot {
        /**
         * The active game configuration settings, parsed from a {@link GameConfig} object.
         */
        class Config {
            /**
             *
             * @param GameConfig The configuration object for your Phaser Game instance.
             */
            constructor(GameConfig?: GameConfig);

            /**
             * [description]
             */
            readonly width: integer | string;

            /**
             * [description]
             */
            readonly height: integer | string;

            /**
             * [description]
             */
            readonly zoom: number;

            /**
             * [description]
             */
            readonly resolution: number;

            /**
             * [description]
             */
            readonly parent: any;

            /**
             * [description]
             */
            readonly scaleMode: any;

            /**
             * Force Phaser to use a specific renderer. Can be `CONST.CANVAS`, `CONST.WEBGL`, `CONST.HEADLESS` or `CONST.AUTO` (default)
             */
            readonly renderType: number;

            /**
             * Force Phaser to use your own Canvas element instead of creating one.
             */
            readonly canvas: HTMLCanvasElement;

            /**
             * Force Phaser to use your own Canvas context instead of creating one.
             */
            readonly context: CanvasRenderingContext2D | WebGLRenderingContext;

            /**
             * [description]
             */
            readonly canvasStyle: string;

            /**
             * [description]
             */
            readonly sceneConfig: object;

            /**
             * [description]
             */
            readonly seed: string[];

            /**
             * The title of the game.
             */
            readonly gameTitle: string;

            /**
             * The URL of the game.
             */
            readonly gameURL: string;

            /**
             * The version of the game.
             */
            readonly gameVersion: string;

            /**
             * [description]
             */
            readonly autoFocus: boolean;

            /**
             * [description]
             */
            readonly domCreateContainer: boolean;

            /**
             * [description]
             */
            readonly domBehindCanvas: boolean;

            /**
             * [description]
             */
            readonly inputKeyboard: boolean;

            /**
             * [description]
             */
            readonly inputKeyboardEventTarget: any;

            /**
             * [description]
             */
            readonly inputMouse: boolean | object;

            /**
             * [description]
             */
            readonly inputMouseEventTarget: any;

            /**
             * [description]
             */
            readonly inputMouseCapture: boolean;

            /**
             * [description]
             */
            readonly inputTouch: boolean;

            /**
             * [description]
             */
            readonly inputTouchEventTarget: any;

            /**
             * [description]
             */
            readonly inputTouchCapture: boolean;

            /**
             * [description]
             */
            readonly inputActivePointers: integer;

            /**
             * [description]
             */
            readonly inputGamepad: boolean;

            /**
             * [description]
             */
            readonly inputGamepadEventTarget: any;

            /**
             * Set to `true` to disable context menu. Default value is `false`.
             */
            readonly disableContextMenu: boolean;

            /**
             * [description]
             */
            readonly audio: any;

            /**
             * [description]
             */
            readonly hideBanner: boolean;

            /**
             * [description]
             */
            readonly hidePhaser: boolean;

            /**
             * [description]
             */
            readonly bannerTextColor: string;

            /**
             * [description]
             */
            readonly bannerBackgroundColor: string[];

            /**
             * [description]
             */
            readonly fps: FPSConfig;

            /**
             * Automatically resize the Game Canvas if you resize the renderer.
             */
            readonly autoResize: boolean;

            /**
             * [description]
             */
            readonly antialias: boolean;

            /**
             * [description]
             */
            readonly roundPixels: boolean;

            /**
             * Prevent pixel art from becoming blurred when scaled. It will remain crisp (tells the WebGL renderer to automatically create textures using a linear filter mode).
             */
            readonly pixelArt: boolean;

            /**
             * [description]
             */
            readonly transparent: boolean;

            /**
             * [description]
             */
            readonly clearBeforeRender: boolean;

            /**
             * [description]
             */
            readonly premultipliedAlpha: boolean;

            /**
             * [description]
             */
            readonly preserveDrawingBuffer: boolean;

            /**
             * [description]
             */
            readonly failIfMajorPerformanceCaveat: boolean;

            /**
             * [description]
             */
            readonly powerPreference: string;

            /**
             * The default WebGL Batch size.
             */
            readonly batchSize: integer;

            /**
             * [description]
             */
            readonly backgroundColor: Phaser.Display.Color;

            /**
             * [description]
             */
            readonly preBoot: BootCallback;

            /**
             * [description]
             */
            readonly postBoot: BootCallback;

            /**
             * [description]
             */
            readonly physics: object;

            /**
             * [description]
             */
            readonly defaultPhysicsSystem: boolean;

            /**
             * [description]
             */
            readonly loaderBaseURL: string;

            /**
             * [description]
             */
            readonly loaderPath: string;

            /**
             * Maximum parallel downloads allowed for resources (Default to 32).
             */
            readonly loaderMaxParallelDownloads: integer;

            /**
             * [description]
             */
            readonly loaderCrossOrigin: string | undefined;

            /**
             * [description]
             */
            readonly loaderResponseType: string;

            /**
             * [description]
             */
            readonly loaderAsync: boolean;

            /**
             * [description]
             */
            readonly loaderUser: string;

            /**
             * [description]
             */
            readonly loaderPassword: string;

            /**
             * [description]
             */
            readonly loaderTimeout: integer;

            /**
             * [description]
             */
            readonly installGlobalPlugins: any;

            /**
             * [description]
             */
            readonly installScenePlugins: any;

            /**
             * The plugins installed into every Scene (in addition to CoreScene and Global).
             */
            readonly defaultPlugins: any;

            /**
             * [description]
             */
            readonly defaultImage: string;

            /**
             * [description]
             */
            readonly missingImage: string;

        }

        /**
         * Called automatically by Phaser.Game and responsible for creating the renderer it will use.
         *
         * Relies upon two webpack global flags to be defined: `WEBGL_RENDERER` and `CANVAS_RENDERER` during build time, but not at run-time.
         * @param game The Phaser.Game instance on which the renderer will be set.
         */
        function CreateRenderer(game: Phaser.Game): void;

        /**
         * Called automatically by Phaser.Game and responsible for creating the console.log debug header.
         *
         * You can customize or disable the header via the Game Config object.
         * @param game The Phaser.Game instance which will output this debug header.
         */
        function DebugHeader(game: Phaser.Game): void;

        /**
         * [description]
         */
        class ScaleManager {
            /**
             *
             * @param game A reference to the Phaser.Game instance.
             */
            constructor(game: Phaser.Game, config: any);

            /**
             * A reference to the Phaser.Game instance.
             */
            readonly game: Phaser.Game;

            /**
             * Destroys the ScaleManager.
             */
            destroy(): void;

        }

        /**
         * [description]
         */
        class TimeStep {
            /**
             *
             * @param game A reference to the Phaser.Game instance that owns this Time Step.
             */
            constructor(game: Phaser.Game, config: FPSConfig);

            /**
             * A reference to the Phaser.Game instance.
             */
            readonly game: Phaser.Game;

            /**
             * [description]
             */
            readonly raf: Phaser.DOM.RequestAnimationFrame;

            /**
             * A flag that is set once the TimeStep has started running and toggled when it stops.
             */
            readonly started: boolean;

            /**
             * A flag that is set once the TimeStep has started running and toggled when it stops.
             * The difference between this value and `started` is that `running` is toggled when
             * the TimeStep is sent to sleep, where-as `started` remains `true`, only changing if
             * the TimeStep is actually stopped, not just paused.
             */
            readonly running: boolean;

            /**
             * The minimum fps rate you want the Time Step to run at.
             */
            minFps: integer;

            /**
             * The target fps rate for the Time Step to run at.
             *
             * Setting this value will not actually change the speed at which the browser runs, that is beyond
             * the control of Phaser. Instead, it allows you to determine performance issues and if the Time Step
             * is spiraling out of control.
             */
            targetFps: integer;

            /**
             * An exponential moving average of the frames per second.
             */
            readonly actualFps: integer;

            /**
             * [description]
             */
            readonly nextFpsUpdate: integer;

            /**
             * The number of frames processed this second.
             */
            readonly framesThisSecond: integer;

            /**
             * A callback to be invoked each time the Time Step steps.
             */
            callback: TimeStepCallback;

            /**
             * You can force the Time Step to use Set Timeout instead of Request Animation Frame by setting
             * the `forceSetTimeOut` property to `true` in the Game Configuration object. It cannot be changed at run-time.
             */
            readonly forceSetTimeOut: boolean;

            /**
             * [description]
             */
            time: integer;

            /**
             * [description]
             */
            startTime: integer;

            /**
             * [description]
             */
            lastTime: integer;

            /**
             * [description]
             */
            readonly frame: integer;

            /**
             * [description]
             */
            readonly inFocus: boolean;

            /**
             * [description]
             */
            delta: integer;

            /**
             * [description]
             */
            deltaIndex: integer;

            /**
             * [description]
             */
            deltaHistory: integer[];

            /**
             * [description]
             */
            deltaSmoothingMax: integer;

            /**
             * [description]
             */
            panicMax: integer;

            /**
             * The actual elapsed time in ms between one update and the next.
             * Unlike with `delta` no smoothing, capping, or averaging is applied to this value.
             * So please be careful when using this value in calculations.
             */
            rawDelta: number;

            /**
             * Called when the DOM window.onBlur event triggers.
             */
            blur(): void;

            /**
             * Called when the DOM window.onFocus event triggers.
             */
            focus(): void;

            /**
             * Called when the visibility API says the game is 'hidden' (tab switch out of view, etc)
             */
            pause(): void;

            /**
             * Called when the visibility API says the game is 'visible' again (tab switch back into view, etc)
             */
            resume(): void;

            /**
             * [description]
             */
            resetDelta(): void;

            /**
             * Starts the Time Step running, if it is not already doing so.
             * Called automatically by the Game Boot process.
             * @param callback The callback to be invoked each time the Time Step steps.
             */
            start(callback: TimeStepCallback): void;

            /**
             * The main step method. This is called each time the browser updates, either by Request Animation Frame,
             * or by Set Timeout. It is responsible for calculating the delta values, frame totals, cool down history and more.
             * You generally should never call this method directly.
             * @param time The current time. Either a High Resolution Timer value if it comes from Request Animation Frame, or Date.now if using SetTimeout.
             */
            step(time: number): void;

            /**
             * Manually calls TimeStep.step, passing in the performance.now value to it.
             */
            tick(): void;

            /**
             * Sends the TimeStep to sleep, stopping Request Animation Frame (or SetTimeout) and toggling the `running` flag to false.
             */
            sleep(): void;

            /**
             * Wakes-up the TimeStep, restarting Request Animation Frame (or SetTimeout) and toggling the `running` flag to true.
             * The `seamless` argument controls if the wake-up should adjust the start time or not.
             * @param seamless Adjust the startTime based on the lastTime values. Default false.
             */
            wake(seamless?: boolean): void;

            /**
             * Stops the TimeStep running.
             */
            stop(): Phaser.Boot.TimeStep;

            /**
             * Destroys the TimeStep. This will stop Request Animation Frame, stop the step, clear the callbacks and null
             * any objects.
             */
            destroy(): void;

        }

        /**
         * The Visibility Handler is responsible for listening out for document level visibility change events.
         * This includes `visibilitychange` if the browser supports it, and blur and focus events. It then uses
         * the provided Event Emitter and fires the related events.
         * @param game The Game instance this Visibility Handler is working on.
         */
        function VisibilityHandler(game: Phaser.Game): void;

    }

    namespace Cache {
        /**
         * The BaseCache is a base Cache class that can be used for storing references to any kind of data.
         *
         * Data can be added, retrieved and removed based on the given keys.
         *
         * Keys are string-based.
         */
        class BaseCache {
            /**
             * The Map in which the cache objects are stored.
             *
             * You can query the Map directly or use the BaseCache methods.
             */
            entries: Phaser.Structs.Map<String, any>;

            /**
             * An instance of EventEmitter used by the cache to emit related events.
             */
            events: Phaser.Events.EventEmitter;

            /**
             * Adds an item to this cache. The item is referenced by a unique string, which you are responsible
             * for setting and keeping track of. The item can only be retrieved by using this string.
             * @param key The unique key by which the data added to the cache will be referenced.
             * @param data The data to be stored in the cache.
             */
            add(key: string, data: any): Phaser.Cache.BaseCache;

            /**
             * Checks if this cache contains an item matching the given key.
             * This performs the same action as `BaseCache.exists`.
             * @param key The unique key of the item to be checked in this cache.
             */
            has(key: string): boolean;

            /**
             * Checks if this cache contains an item matching the given key.
             * This performs the same action as `BaseCache.has` and is called directly by the Loader.
             * @param key The unique key of the item to be checked in this cache.
             */
            exists(key: string): boolean;

            /**
             * Gets an item from this cache based on the given key.
             * @param key The unique key of the item to be retrieved from this cache.
             */
            get(key: string): any;

            /**
             * Removes and item from this cache based on the given key.
             *
             * If an entry matching the key is found it is removed from the cache and a `remove` event emitted.
             * No additional checks are done on the item removed. If other systems or parts of your game code
             * are relying on this item, it is up to you to sever those relationships prior to removing the item.
             * @param key The unique key of the item to remove from the cache.
             */
            remove(key: string): Phaser.Cache.BaseCache;

            /**
             * Destroys this cache and all items within it.
             */
            destroy(): void;

        }

        /**
         * The Cache Manager is the global cache owned and maintained by the Game instance.
         *
         * Various systems, such as the file Loader, rely on this cache in order to store the files
         * it has loaded. The manager itself doesn't store any files, but instead owns multiple BaseCache
         * instances, one per type of file. You can also add your own custom caches.
         */
        class CacheManager {
            /**
             *
             * @param game A reference to the Phaser.Game instance that owns this CacheManager.
             */
            constructor(game: Phaser.Game);

            /**
             * A reference to the Phaser.Game instance that owns this CacheManager.
             */
            protected game: Phaser.Game;

            /**
             * A Cache storing all binary files, typically added via the Loader.
             */
            binary: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all bitmap font data files, typically added via the Loader.
             * Only the font data is stored in this cache, the textures are part of the Texture Manager.
             */
            bitmapFont: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all JSON data files, typically added via the Loader.
             */
            json: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all physics data files, typically added via the Loader.
             */
            physics: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all shader source files, typically added via the Loader.
             */
            shader: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all non-streaming audio files, typically added via the Loader.
             */
            audio: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all text files, typically added via the Loader.
             */
            text: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all html files, typically added via the Loader.
             */
            html: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all WaveFront OBJ files, typically added via the Loader.
             */
            obj: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all tilemap data files, typically added via the Loader.
             * Only the data is stored in this cache, the textures are part of the Texture Manager.
             */
            tilemap: Phaser.Cache.BaseCache;

            /**
             * A Cache storing all xml data files, typically added via the Loader.
             */
            xml: Phaser.Cache.BaseCache;

            /**
             * An object that contains your own custom BaseCache entries.
             * Add to this via the `addCustom` method.
             */
            custom: {[key: string]: Phaser.Cache.BaseCache};

            /**
             * Add your own custom Cache for storing your own files.
             * The cache will be available under `Cache.custom.key`.
             * The cache will only be created if the key is not already in use.
             * @param key The unique key of your custom cache.
             */
            addCustom(key: string): Phaser.Cache.BaseCache;

            /**
             * Removes all entries from all BaseCaches and destroys all custom caches.
             */
            destroy(): void;

        }

    }

    namespace Cameras {
        namespace Scene2D {
            /**
             * A Base Camera class.
             *
             * The Camera is the way in which all games are rendered in Phaser. They provide a view into your game world,
             * and can be positioned, rotated, zoomed and scrolled accordingly.
             *
             * A Camera consists of two elements: The viewport and the scroll values.
             *
             * The viewport is the physical position and size of the Camera within your game. Cameras, by default, are
             * created the same size as your game, but their position and size can be set to anything. This means if you
             * wanted to create a camera that was 320x200 in size, positioned in the bottom-right corner of your game,
             * you'd adjust the viewport to do that (using methods like `setViewport` and `setSize`).
             *
             * If you wish to change where the Camera is looking in your game, then you scroll it. You can do this
             * via the properties `scrollX` and `scrollY` or the method `setScroll`. Scrolling has no impact on the
             * viewport, and changing the viewport has no impact on the scrolling.
             *
             * By default a Camera will render all Game Objects it can see. You can change this using the `ignore` method,
             * allowing you to filter Game Objects out on a per-Camera basis.
             *
             * The Base Camera is extended by the Camera class, which adds in special effects including Fade,
             * Flash and Camera Shake, as well as the ability to follow Game Objects.
             *
             * The Base Camera was introduced in Phaser 3.12. It was split off from the Camera class, to allow
             * you to isolate special effects as needed. Therefore the 'since' values for properties of this class relate
             * to when they were added to the Camera class.
             */
            class BaseCamera extends Phaser.Events.EventEmitter implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.Visible {
                /**
                 *
                 * @param x The x position of the Camera, relative to the top-left of the game canvas.
                 * @param y The y position of the Camera, relative to the top-left of the game canvas.
                 * @param width The width of the Camera, in pixels.
                 * @param height The height of the Camera, in pixels.
                 */
                constructor(x: number, y: number, width: number, height: number);

                /**
                 * A reference to the Scene this camera belongs to.
                 */
                scene: Phaser.Scene;

                /**
                 * A reference to the Game Scene Manager.
                 */
                sceneManager: Phaser.Scenes.SceneManager;

                /**
                 * A reference to the Game Config.
                 */
                readonly config: object;

                /**
                 * The Camera ID. Assigned by the Camera Manager and used to handle camera exclusion.
                 * This value is a bitmask.
                 */
                readonly id: integer;

                /**
                 * The name of the Camera. This is left empty for your own use.
                 */
                name: string;

                /**
                 * The resolution of the Game, used in most Camera calculations.
                 */
                readonly resolution: number;

                /**
                 * Should this camera round its pixel values to integers?
                 */
                roundPixels: boolean;

                /**
                 * Is this Camera visible or not?
                 *
                 * A visible camera will render and perform input tests.
                 * An invisible camera will not render anything and will skip input tests.
                 */
                visible: boolean;

                /**
                 * Is this Camera using a bounds to restrict scrolling movement?
                 *
                 * Set this property along with the bounds via `Camera.setBounds`.
                 */
                useBounds: boolean;

                /**
                 * The World View is a Rectangle that defines the area of the 'world' the Camera is currently looking at.
                 * This factors in the Camera viewport size, zoom and scroll position and is updated in the Camera preRender step.
                 * If you have enabled Camera bounds the worldview will be clamped to those bounds accordingly.
                 * You can use it for culling or intersection checks.
                 */
                readonly worldView: Phaser.Geom.Rectangle;

                /**
                 * Is this Camera dirty?
                 *
                 * A dirty Camera has had either its viewport size, bounds, scroll, rotation or zoom levels changed since the last frame.
                 *
                 * This flag is cleared during the `postRenderCamera` method of the renderer.
                 */
                dirty: boolean;

                /**
                 * Does this Camera have a transparent background?
                 */
                transparent: boolean;

                /**
                 * The background color of this Camera. Only used if `transparent` is `false`.
                 */
                backgroundColor: Phaser.Display.Color;

                /**
                 * The Camera alpha value. Setting this property impacts every single object that this Camera
                 * renders. You can either set the property directly, i.e. via a Tween, to fade a Camera in or out,
                 * or via the chainable `setAlpha` method instead.
                 */
                alpha: number;

                /**
                 * Should the camera cull Game Objects before checking them for input hit tests?
                 * In some special cases it may be beneficial to disable this.
                 */
                disableCull: boolean;

                /**
                 * The mid-point of the Camera in 'world' coordinates.
                 *
                 * Use it to obtain exactly where in the world the center of the camera is currently looking.
                 *
                 * This value is updated in the preRender method, after the scroll values and follower
                 * have been processed.
                 */
                readonly midPoint: Phaser.Math.Vector2;

                /**
                 * The horizontal origin of rotation for this Camera.
                 *
                 * By default the camera rotates around the center of the viewport.
                 *
                 * Changing the origin allows you to adjust the point in the viewport from which rotation happens.
                 * A value of 0 would rotate from the top-left of the viewport. A value of 1 from the bottom right.
                 *
                 * See `setOrigin` to set both origins in a single, chainable call.
                 */
                originX: number;

                /**
                 * The vertical origin of rotation for this Camera.
                 *
                 * By default the camera rotates around the center of the viewport.
                 *
                 * Changing the origin allows you to adjust the point in the viewport from which rotation happens.
                 * A value of 0 would rotate from the top-left of the viewport. A value of 1 from the bottom right.
                 *
                 * See `setOrigin` to set both origins in a single, chainable call.
                 */
                originY: number;

                /**
                 * Set the Alpha level of this Camera. The alpha controls the opacity of the Camera as it renders.
                 * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
                 * @param value The Camera alpha value. Default 1.
                 */
                setAlpha(value?: number): this;

                /**
                 * Sets the rotation origin of this Camera.
                 *
                 * The values are given in the range 0 to 1 and are only used when calculating Camera rotation.
                 *
                 * By default the camera rotates around the center of the viewport.
                 *
                 * Changing the origin allows you to adjust the point in the viewport from which rotation happens.
                 * A value of 0 would rotate from the top-left of the viewport. A value of 1 from the bottom right.
                 * @param x The horizontal origin value. Default 0.5.
                 * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
                 */
                setOrigin(x?: number, y?: number): this;

                /**
                 * Calculates what the Camera.scrollX and scrollY values would need to be in order to move
                 * the Camera so it is centered on the given x and y coordinates, without actually moving
                 * the Camera there. The results are clamped based on the Camera bounds, if set.
                 * @param x The horizontal coordinate to center on.
                 * @param y The vertical coordinate to center on.
                 * @param out A Vec2 to store the values in. If not given a new Vec2 is created.
                 */
                getScroll(x: number, y: number, out?: Phaser.Math.Vector2): Phaser.Math.Vector2;

                /**
                 * Moves the Camera so that it is centered on the given coordinates, bounds allowing.
                 * @param x The horizontal coordinate to center on.
                 * @param y The vertical coordinate to center on.
                 */
                centerOn(x: number, y: number): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Moves the Camera so that it is looking at the center of the Camera Bounds, if enabled.
                 */
                centerToBounds(): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Moves the Camera so that it is re-centered based on its viewport size.
                 */
                centerToSize(): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Takes an array of Game Objects and returns a new array featuring only those objects
                 * visible by this camera.
                 * @param renderableObjects An array of Game Objects to cull.
                 */
                cull<G extends Phaser.GameObjects.GameObject[]>(renderableObjects: G): G;

                /**
                 * Converts the given `x` and `y` coordinates into World space, based on this Cameras transform.
                 * You can optionally provide a Vector2, or similar object, to store the results in.
                 * @param x The x position to convert to world space.
                 * @param y The y position to convert to world space.
                 * @param output An optional object to store the results in. If not provided a new Vector2 will be created.
                 */
                getWorldPoint<O extends Phaser.Math.Vector2>(x: number, y: number, output?: O): O;

                /**
                 * Given a Game Object, or an array of Game Objects, it will update all of their camera filter settings
                 * so that they are ignored by this Camera. This means they will not be rendered by this Camera.
                 * @param entries The Game Object, or array of Game Objects, to be ignored by this Camera.
                 */
                ignore(entries: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[] | Phaser.GameObjects.Group): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Internal preRender step.
                 * @param baseScale The base scale, as set in the Camera Manager.
                 * @param resolution The game resolution.
                 */
                protected preRender(baseScale: number, resolution: number): void;

                /**
                 * Takes an x value and checks it's within the range of the Camera bounds, adjusting if required.
                 * Do not call this method if you are not using camera bounds.
                 * @param x The value to horizontally scroll clamp.
                 */
                clampX(x: number): number;

                /**
                 * Takes a y value and checks it's within the range of the Camera bounds, adjusting if required.
                 * Do not call this method if you are not using camera bounds.
                 * @param y The value to vertically scroll clamp.
                 */
                clampY(y: number): number;

                /**
                 * If this Camera has previously had movement bounds set on it, this will remove them.
                 */
                removeBounds(): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the rotation of this Camera. This causes everything it renders to appear rotated.
                 *
                 * Rotating a camera does not rotate the viewport itself, it is applied during rendering.
                 * @param value The cameras angle of rotation, given in degrees. Default 0.
                 */
                setAngle(value?: number): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Sets the background color for this Camera.
                 *
                 * By default a Camera has a transparent background but it can be given a solid color, with any level
                 * of transparency, via this method.
                 *
                 * The color value can be specified using CSS color notation, hex or numbers.
                 * @param color The color value. In CSS, hex or numeric color notation. Default 'rgba(0,0,0,0)'.
                 */
                setBackgroundColor(color?: string | number | InputColorObject): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the bounds of the Camera. The bounds are an axis-aligned rectangle.
                 *
                 * The Camera bounds controls where the Camera can scroll to, stopping it from scrolling off the
                 * edges and into blank space. It does not limit the placement of Game Objects, or where
                 * the Camera viewport can be positioned.
                 *
                 * Temporarily disable the bounds by changing the boolean `Camera.useBounds`.
                 *
                 * Clear the bounds entirely by calling `Camera.removeBounds`.
                 *
                 * If you set bounds that are smaller than the viewport it will stop the Camera from being
                 * able to scroll. The bounds can be positioned where-ever you wish. By default they are from
                 * 0x0 to the canvas width x height. This means that the coordinate 0x0 is the top left of
                 * the Camera bounds. However, you can position them anywhere. So if you wanted a game world
                 * that was 2048x2048 in size, with 0x0 being the center of it, you can set the bounds x/y
                 * to be -1024, -1024, with a width and height of 2048. Depending on your game you may find
                 * it easier for 0x0 to be the top-left of the bounds, or you may wish 0x0 to be the middle.
                 * @param x The top-left x coordinate of the bounds.
                 * @param y The top-left y coordinate of the bounds.
                 * @param width The width of the bounds, in pixels.
                 * @param height The height of the bounds, in pixels.
                 * @param centerOn If `true` the Camera will automatically be centered on the new bounds.
                 */
                setBounds(x: integer, y: integer, width: integer, height: integer, centerOn?: boolean): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Sets the name of this Camera.
                 * This value is for your own use and isn't used internally.
                 * @param value The name of the Camera. Default ''.
                 */
                setName(value?: string): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the position of the Camera viewport within the game.
                 *
                 * This does not change where the camera is 'looking'. See `setScroll` to control that.
                 * @param x The top-left x coordinate of the Camera viewport.
                 * @param y The top-left y coordinate of the Camera viewport. Default x.
                 */
                setPosition(x: number, y?: number): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the rotation of this Camera. This causes everything it renders to appear rotated.
                 *
                 * Rotating a camera does not rotate the viewport itself, it is applied during rendering.
                 * @param value The rotation of the Camera, in radians. Default 0.
                 */
                setRotation(value?: number): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Should the Camera round pixel values to whole integers when rendering Game Objects?
                 *
                 * In some types of game, especially with pixel art, this is required to prevent sub-pixel aliasing.
                 * @param value `true` to round Camera pixels, `false` to not.
                 */
                setRoundPixels(value: boolean): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Sets the Scene the Camera is bound to.
                 *
                 * Also populates the `resolution` property and updates the internal size values.
                 * @param scene The Scene the camera is bound to.
                 */
                setScene(scene: Phaser.Scene): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the position of where the Camera is looking within the game.
                 * You can also modify the properties `Camera.scrollX` and `Camera.scrollY` directly.
                 * Use this method, or the scroll properties, to move your camera around the game world.
                 *
                 * This does not change where the camera viewport is placed. See `setPosition` to control that.
                 * @param x The x coordinate of the Camera in the game world.
                 * @param y The y coordinate of the Camera in the game world. Default x.
                 */
                setScroll(x: number, y?: number): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the size of the Camera viewport.
                 *
                 * By default a Camera is the same size as the game, but can be made smaller via this method,
                 * allowing you to create mini-cam style effects by creating and positioning a smaller Camera
                 * viewport within your game.
                 * @param width The width of the Camera viewport.
                 * @param height The height of the Camera viewport. Default width.
                 */
                setSize(width: integer, height?: integer): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * This method sets the position and size of the Camera viewport in a single call.
                 *
                 * If you're trying to change where the Camera is looking at in your game, then see
                 * the method `Camera.setScroll` instead. This method is for changing the viewport
                 * itself, not what the camera can see.
                 *
                 * By default a Camera is the same size as the game, but can be made smaller via this method,
                 * allowing you to create mini-cam style effects by creating and positioning a smaller Camera
                 * viewport within your game.
                 * @param x The top-left x coordinate of the Camera viewport.
                 * @param y The top-left y coordinate of the Camera viewport.
                 * @param width The width of the Camera viewport.
                 * @param height The height of the Camera viewport. Default width.
                 */
                setViewport(x: number, y: number, width: integer, height?: integer): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Set the zoom value of the Camera.
                 *
                 * Changing to a smaller value, such as 0.5, will cause the camera to 'zoom out'.
                 * Changing to a larger value, such as 2, will cause the camera to 'zoom in'.
                 *
                 * A value of 1 means 'no zoom' and is the default.
                 *
                 * Changing the zoom does not impact the Camera viewport in any way, it is only applied during rendering.
                 * @param value The zoom value of the Camera. The minimum it can be is 0.001. Default 1.
                 */
                setZoom(value?: number): Phaser.Cameras.Scene2D.BaseCamera;

                /**
                 * Sets the visibility of this Camera.
                 *
                 * An invisible Camera will skip rendering and input tests of everything it can see.
                 * @param value The visible state of the Camera.
                 */
                setVisible(value: boolean): this;

                /**
                 * Returns an Object suitable for JSON storage containing all of the Camera viewport and rendering properties.
                 */
                toJSON(): JSONCamera;

                /**
                 * Internal method called automatically by the Camera Manager.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                protected update(time: integer, delta: number): void;

                /**
                 * Destroys this Camera instance. You rarely need to call this directly.
                 *
                 * Called by the Camera Manager. If you wish to destroy a Camera please use `CameraManager.remove` as
                 * cameras are stored in a pool, ready for recycling later, and calling this directly will prevent that.
                 */
                destroy(): void;

                /**
                 * The x position of the Camera viewport, relative to the top-left of the game canvas.
                 * The viewport is the area into which the camera renders.
                 * To adjust the position the camera is looking at in the game world, see the `scrollX` value.
                 */
                x: number;

                /**
                 * The y position of the Camera viewport, relative to the top-left of the game canvas.
                 * The viewport is the area into which the camera renders.
                 * To adjust the position the camera is looking at in the game world, see the `scrollY` value.
                 */
                y: number;

                /**
                 * The width of the Camera viewport, in pixels.
                 *
                 * The viewport is the area into which the Camera renders. Setting the viewport does
                 * not restrict where the Camera can scroll to.
                 */
                width: number;

                /**
                 * The height of the Camera viewport, in pixels.
                 *
                 * The viewport is the area into which the Camera renders. Setting the viewport does
                 * not restrict where the Camera can scroll to.
                 */
                height: number;

                /**
                 * The horizontal scroll position of this Camera.
                 *
                 * Change this value to cause the Camera to scroll around your Scene.
                 *
                 * Alternatively, setting the Camera to follow a Game Object, via the `startFollow` method,
                 * will automatically adjust the Camera scroll values accordingly.
                 *
                 * You can set the bounds within which the Camera can scroll via the `setBounds` method.
                 */
                scrollX: number;

                /**
                 * The vertical scroll position of this Camera.
                 *
                 * Change this value to cause the Camera to scroll around your Scene.
                 *
                 * Alternatively, setting the Camera to follow a Game Object, via the `startFollow` method,
                 * will automatically adjust the Camera scroll values accordingly.
                 *
                 * You can set the bounds within which the Camera can scroll via the `setBounds` method.
                 */
                scrollY: number;

                /**
                 * The Camera zoom value. Change this value to zoom in, or out of, a Scene.
                 *
                 * A value of 0.5 would zoom the Camera out, so you can now see twice as much
                 * of the Scene as before. A value of 2 would zoom the Camera in, so every pixel
                 * now takes up 2 pixels when rendered.
                 *
                 * Set to 1 to return to the default zoom level.
                 *
                 * Be careful to never set this value to zero.
                 */
                zoom: number;

                /**
                 * The x position of the center of the Camera's viewport, relative to the top-left of the game canvas.
                 */
                readonly centerX: number;

                /**
                 * The y position of the center of the Camera's viewport, relative to the top-left of the game canvas.
                 */
                readonly centerY: number;

                /**
                 * The displayed width of the camera viewport, factoring in the camera zoom level.
                 *
                 * If a camera has a viewport width of 800 and a zoom of 0.5 then its display width
                 * would be 1600, as it's displaying twice as many pixels as zoom level 1.
                 *
                 * Equally, a camera with a width of 800 and zoom of 2 would have a display width
                 * of 400 pixels.
                 */
                readonly displayWidth: number;

                /**
                 * The displayed height of the camera viewport, factoring in the camera zoom level.
                 *
                 * If a camera has a viewport height of 600 and a zoom of 0.5 then its display height
                 * would be 1200, as it's displaying twice as many pixels as zoom level 1.
                 *
                 * Equally, a camera with a height of 600 and zoom of 2 would have a display height
                 * of 300 pixels.
                 */
                readonly displayHeight: number;

                /**
                 * Clears all alpha values associated with this Game Object.
                 *
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;

                /**
                 * The alpha value starting from the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopLeft: number;

                /**
                 * The alpha value starting from the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopRight: number;

                /**
                 * The alpha value starting from the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomLeft: number;

                /**
                 * The alpha value starting from the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomRight: number;

            }

            /**
             * A Camera.
             *
             * The Camera is the way in which all games are rendered in Phaser. They provide a view into your game world,
             * and can be positioned, rotated, zoomed and scrolled accordingly.
             *
             * A Camera consists of two elements: The viewport and the scroll values.
             *
             * The viewport is the physical position and size of the Camera within your game. Cameras, by default, are
             * created the same size as your game, but their position and size can be set to anything. This means if you
             * wanted to create a camera that was 320x200 in size, positioned in the bottom-right corner of your game,
             * you'd adjust the viewport to do that (using methods like `setViewport` and `setSize`).
             *
             * If you wish to change where the Camera is looking in your game, then you scroll it. You can do this
             * via the properties `scrollX` and `scrollY` or the method `setScroll`. Scrolling has no impact on the
             * viewport, and changing the viewport has no impact on the scrolling.
             *
             * By default a Camera will render all Game Objects it can see. You can change this using the `ignore` method,
             * allowing you to filter Game Objects out on a per-Camera basis.
             *
             * A Camera also has built-in special effects including Fade, Flash and Camera Shake.
             */
            class Camera extends Phaser.Cameras.Scene2D.BaseCamera implements Phaser.GameObjects.Components.Flip, Phaser.GameObjects.Components.Tint {
                /**
                 *
                 * @param x The x position of the Camera, relative to the top-left of the game canvas.
                 * @param y The y position of the Camera, relative to the top-left of the game canvas.
                 * @param width The width of the Camera, in pixels.
                 * @param height The height of the Camera, in pixels.
                 */
                constructor(x: number, y: number, width: number, height: number);

                /**
                 * Does this Camera allow the Game Objects it renders to receive input events?
                 */
                inputEnabled: boolean;

                /**
                 * The Camera Fade effect handler.
                 * To fade this camera see the `Camera.fade` methods.
                 */
                fadeEffect: Phaser.Cameras.Scene2D.Effects.Fade;

                /**
                 * The Camera Flash effect handler.
                 * To flash this camera see the `Camera.flash` method.
                 */
                flashEffect: Phaser.Cameras.Scene2D.Effects.Flash;

                /**
                 * The Camera Shake effect handler.
                 * To shake this camera see the `Camera.shake` method.
                 */
                shakeEffect: Phaser.Cameras.Scene2D.Effects.Shake;

                /**
                 * The Camera Pan effect handler.
                 * To pan this camera see the `Camera.pan` method.
                 */
                panEffect: Phaser.Cameras.Scene2D.Effects.Pan;

                /**
                 * The Camera Zoom effect handler.
                 * To zoom this camera see the `Camera.zoom` method.
                 */
                zoomEffect: Phaser.Cameras.Scene2D.Effects.Zoom;

                /**
                 * The linear interpolation value to use when following a target.
                 *
                 * Can also be set via `setLerp` or as part of the `startFollow` call.
                 *
                 * The default values of 1 means the camera will instantly snap to the target coordinates.
                 * A lower value, such as 0.1 means the camera will more slowly track the target, giving
                 * a smooth transition. You can set the horizontal and vertical values independently, and also
                 * adjust this value in real-time during your game.
                 *
                 * Be sure to keep the value between 0 and 1. A value of zero will disable tracking on that axis.
                 */
                lerp: Phaser.Math.Vector2;

                /**
                 * The values stored in this property are subtracted from the Camera targets position, allowing you to
                 * offset the camera from the actual target x/y coordinates by this amount.
                 * Can also be set via `setFollowOffset` or as part of the `startFollow` call.
                 */
                followOffset: Phaser.Math.Vector2;

                /**
                 * The Camera dead zone.
                 *
                 * The deadzone is only used when the camera is following a target.
                 *
                 * It defines a rectangular region within which if the target is present, the camera will not scroll.
                 * If the target moves outside of this area, the camera will begin scrolling in order to follow it.
                 *
                 * The `lerp` values that you can set for a follower target also apply when using a deadzone.
                 *
                 * You can directly set this property to be an instance of a Rectangle. Or, you can use the
                 * `setDeadzone` method for a chainable approach.
                 *
                 * The rectangle you provide can have its dimensions adjusted dynamically, however, please
                 * note that its position is updated every frame, as it is constantly re-centered on the cameras mid point.
                 *
                 * Calling `setDeadzone` with no arguments will reset an active deadzone, as will setting this property
                 * to `null`.
                 */
                deadzone: Phaser.Geom.Rectangle;

                /**
                 * Is this Camera rendering directly to the canvas or to a texture?
                 *
                 * Enable rendering to texture with the method `setRenderToTexture` (just enabling this boolean won't be enough)
                 *
                 * Once enabled you can toggle it by switching this property.
                 *
                 * To properly remove a render texture you should call the `clearRenderToTexture()` method.
                 */
                renderToTexture: boolean;

                /**
                 * If this Camera has been set to render to a texture then this holds a reference
                 * to the HTML Canvas Element that the Camera is drawing to.
                 *
                 * Enable texture rendering using the method `setRenderToTexture`.
                 *
                 * This is only populated if Phaser is running with the Canvas Renderer.
                 */
                canvas: HTMLCanvasElement;

                /**
                 * If this Camera has been set to render to a texture then this holds a reference
                 * to the Rendering Context belonging to the Canvas element the Camera is drawing to.
                 *
                 * Enable texture rendering using the method `setRenderToTexture`.
                 *
                 * This is only populated if Phaser is running with the Canvas Renderer.
                 */
                context: CanvasRenderingContext2D;

                /**
                 * If this Camera has been set to render to a texture then this holds a reference
                 * to the GL Texture belonging the Camera is drawing to.
                 *
                 * Enable texture rendering using the method `setRenderToTexture`.
                 *
                 * This is only set if Phaser is running with the WebGL Renderer.
                 */
                framebuffer: WebGLTexture;

                /**
                 * If this Camera has been set to render to a texture and to use a custom pipeline,
                 * then this holds a reference to the pipeline the Camera is drawing with.
                 *
                 * Enable texture rendering using the method `setRenderToTexture`.
                 *
                 * This is only set if Phaser is running with the WebGL Renderer.
                 */
                pipeline: any;

                /**
                 * Sets the Camera to render to a texture instead of to the main canvas.
                 *
                 * The Camera will redirect all Game Objects it's asked to render to this texture.
                 *
                 * During the render sequence, the texture itself will then be rendered to the main canvas.
                 *
                 * Doing this gives you the ability to modify the texture before this happens,
                 * allowing for special effects such as Camera specific shaders, or post-processing
                 * on the texture.
                 *
                 * If running under Canvas the Camera will render to its `canvas` property.
                 *
                 * If running under WebGL the Camera will create a frame buffer, which is stored in its `framebuffer` and `glTexture` properties.
                 *
                 * If you set a camera to render to a texture then it will emit 2 events during the render loop:
                 *
                 * First, it will emit the event `prerender`. This happens right before any Game Object's are drawn to the Camera texture.
                 *
                 * Then, it will emit the event `postrender`. This happens after all Game Object's have been drawn, but right before the
                 * Camera texture is rendered to the main game canvas. It's the final point at which you can manipulate the texture before
                 * it appears in-game.
                 *
                 * You should not enable this unless you plan on actually using the texture it creates
                 * somehow, otherwise you're just doubling the work required to render your game.
                 *
                 * To temporarily disable rendering to a texture, toggle the `renderToTexture` boolean.
                 *
                 * If you no longer require the Camera to render to a texture, call the `clearRenderToTexture` method,
                 * which will delete the respective textures and free-up resources.
                 * @param pipeline An optional WebGL Pipeline to render with, can be either a string which is the name of the pipeline, or a pipeline reference.
                 */
                setRenderToTexture(pipeline?: string | Phaser.Renderer.WebGL.WebGLPipeline): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Sets the WebGL pipeline this Camera is using when rendering to a texture.
                 *
                 * You can pass either the string-based name of the pipeline, or a reference to the pipeline itself.
                 *
                 * Call this method with no arguments to clear any previously set pipeline.
                 * @param pipeline The WebGL Pipeline to render with, can be either a string which is the name of the pipeline, or a pipeline reference. Or if left empty it will clear the pipeline.
                 */
                setPipeline(pipeline?: string | Phaser.Renderer.WebGL.WebGLPipeline): Phaser.Cameras.Scene2D.Camera;

                /**
                 * If this Camera was set to render to a texture, this will clear the resources it was using and
                 * redirect it to render back to the primary Canvas again.
                 *
                 * If you only wish to temporarily disable rendering to a texture then you can toggle the
                 * property `renderToTexture` instead.
                 */
                clearRenderToTexture(): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Sets the Camera dead zone.
                 *
                 * The deadzone is only used when the camera is following a target.
                 *
                 * It defines a rectangular region within which if the target is present, the camera will not scroll.
                 * If the target moves outside of this area, the camera will begin scrolling in order to follow it.
                 *
                 * The deadzone rectangle is re-positioned every frame so that it is centered on the mid-point
                 * of the camera. This allows you to use the object for additional game related checks, such as
                 * testing if an object is within it or not via a Rectangle.contains call.
                 *
                 * The `lerp` values that you can set for a follower target also apply when using a deadzone.
                 *
                 * Calling this method with no arguments will reset an active deadzone.
                 * @param width The width of the deadzone rectangle in pixels. If not specified the deadzone is removed.
                 * @param height The height of the deadzone rectangle in pixels.
                 */
                setDeadzone(width?: number, height?: number): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Fades the Camera in from the given color over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fadeIn(duration?: integer, red?: integer, green?: integer, blue?: integer, callback?: Function, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Fades the Camera out to the given color over the duration specified.
                 * This is an alias for Camera.fade that forces the fade to start, regardless of existing fades.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fadeOut(duration?: integer, red?: integer, green?: integer, blue?: integer, callback?: Function, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Fades the Camera from the given color to transparent over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param force Force the effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fadeFrom(duration?: integer, red?: integer, green?: integer, blue?: integer, force?: boolean, callback?: Function, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Fades the Camera from transparent to the given color over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                 * @param force Force the effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                fade(duration?: integer, red?: integer, green?: integer, blue?: integer, force?: boolean, callback?: Function, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Flashes the Camera by setting it to the given color immediately and then fading it away again quickly over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 250.
                 * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 255.
                 * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 255.
                 * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 255.
                 * @param force Force the effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                flash(duration?: integer, red?: integer, green?: integer, blue?: integer, force?: boolean, callback?: Function, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Shakes the Camera by the given intensity over the duration specified.
                 * @param duration The duration of the effect in milliseconds. Default 100.
                 * @param intensity The intensity of the shake. Default 0.05.
                 * @param force Force the shake effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                shake(duration?: integer, intensity?: number, force?: boolean, callback?: Function, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * This effect will scroll the Camera so that the center of its viewport finishes at the given destination,
                 * over the duration and with the ease specified.
                 * @param x The destination x coordinate to scroll the center of the Camera viewport to.
                 * @param y The destination y coordinate to scroll the center of the Camera viewport to.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param ease The ease to use for the pan. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                 * @param force Force the shake effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                 * the current camera scroll x coordinate and the current camera scroll y coordinate.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                pan(x: number, y: number, duration?: integer, ease?: string | Function, force?: boolean, callback?: CameraPanCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * This effect will zoom the Camera to the given scale, over the duration and with the ease specified.
                 * @param zoom The target Camera zoom value.
                 * @param duration The duration of the effect in milliseconds. Default 1000.
                 * @param ease The ease to use for the pan. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                 * @param force Force the shake effect to start immediately, even if already running. Default false.
                 * @param callback This callback will be invoked every frame for the duration of the effect.
                 * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                 * the current camera scroll x coordinate and the current camera scroll y coordinate.
                 * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                 */
                zoomTo(zoom: number, duration?: integer, ease?: string | Function, force?: boolean, callback?: CameraPanCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Internal preRender step.
                 * @param baseScale The base scale, as set in the Camera Manager.
                 * @param resolution The game resolution.
                 */
                protected preRender(baseScale: number, resolution: number): void;

                /**
                 * Sets the linear interpolation value to use when following a target.
                 *
                 * The default values of 1 means the camera will instantly snap to the target coordinates.
                 * A lower value, such as 0.1 means the camera will more slowly track the target, giving
                 * a smooth transition. You can set the horizontal and vertical values independently, and also
                 * adjust this value in real-time during your game.
                 *
                 * Be sure to keep the value between 0 and 1. A value of zero will disable tracking on that axis.
                 * @param x The amount added to the horizontal linear interpolation of the follow target. Default 1.
                 * @param y The amount added to the vertical linear interpolation of the follow target. Default 1.
                 */
                setLerp(x?: number, y?: number): this;

                /**
                 * Sets the horizontal and vertical offset of the camera from its follow target.
                 * The values are subtracted from the targets position during the Cameras update step.
                 * @param x The horizontal offset from the camera follow target.x position. Default 0.
                 * @param y The vertical offset from the camera follow target.y position. Default 0.
                 */
                setFollowOffset(x?: number, y?: number): this;

                /**
                 * Sets the Camera to follow a Game Object.
                 *
                 * When enabled the Camera will automatically adjust its scroll position to keep the target Game Object
                 * in its center.
                 *
                 * You can set the linear interpolation value used in the follow code.
                 * Use low lerp values (such as 0.1) to automatically smooth the camera motion.
                 *
                 * If you find you're getting a slight "jitter" effect when following an object it's probably to do with sub-pixel
                 * rendering of the targets position. This can be rounded by setting the `roundPixels` argument to `true` to
                 * force full pixel rounding rendering. Note that this can still be broken if you have specified a non-integer zoom
                 * value on the camera. So be sure to keep the camera zoom to integers.
                 * @param target The target for the Camera to follow.
                 * @param roundPixels Round the camera position to whole integers to avoid sub-pixel rendering? Default false.
                 * @param lerpX A value between 0 and 1. This value specifies the amount of linear interpolation to use when horizontally tracking the target. The closer the value to 1, the faster the camera will track. Default 1.
                 * @param lerpY A value between 0 and 1. This value specifies the amount of linear interpolation to use when vertically tracking the target. The closer the value to 1, the faster the camera will track. Default 1.
                 * @param offsetX The horizontal offset from the camera follow target.x position. Default 0.
                 * @param offsetY The vertical offset from the camera follow target.y position. Default 0.
                 */
                startFollow(target: Phaser.GameObjects.GameObject | object, roundPixels?: boolean, lerpX?: number, lerpY?: number, offsetX?: number, offsetY?: number): this;

                /**
                 * Stops a Camera from following a Game Object, if previously set via `Camera.startFollow`.
                 */
                stopFollow(): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Resets any active FX, such as a fade, flash or shake. Useful to call after a fade in order to
                 * remove the fade.
                 */
                resetFX(): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Internal method called automatically by the Camera Manager.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                protected update(time: integer, delta: number): void;

                /**
                 * Destroys this Camera instance. You rarely need to call this directly.
                 *
                 * Called by the Camera Manager. If you wish to destroy a Camera please use `CameraManager.remove` as
                 * cameras are stored in a pool, ready for recycling later, and calling this directly will prevent that.
                 */
                destroy(): void;

                /**
                 * Clears all alpha values associated with this Game Object.
                 *
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;

                /**
                 * The alpha value starting from the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopLeft: number;

                /**
                 * The alpha value starting from the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopRight: number;

                /**
                 * The alpha value starting from the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomLeft: number;

                /**
                 * The alpha value starting from the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomRight: number;

                /**
                 * The horizontally flipped state of the Game Object.
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 */
                flipX: boolean;

                /**
                 * The vertically flipped state of the Game Object.
                 * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 */
                flipY: boolean;

                /**
                 * Toggles the horizontal flipped state of this Game Object.
                 */
                toggleFlipX(): this;

                /**
                 * Toggles the vertical flipped state of this Game Object.
                 */
                toggleFlipY(): this;

                /**
                 * Sets the horizontal flipped state of this Game Object.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipX(value: boolean): this;

                /**
                 * Sets the vertical flipped state of this Game Object.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipY(value: boolean): this;

                /**
                 * Sets the horizontal and vertical flipped state of this Game Object.
                 * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlip(x: boolean, y: boolean): this;

                /**
                 * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
                 */
                resetFlip(): this;

                /**
                 * Fill or additive?
                 */
                tintFill: boolean;

                /**
                 * Clears all tint values associated with this Game Object.
                 *
                 * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
                 * which results in no visible change to the texture.
                 */
                clearTint(): this;

                /**
                 * Sets an additive tint on this Game Object.
                 *
                 * The tint works by taking the pixel color values from the Game Objects texture, and then
                 * multiplying it by the color value of the tint. You can provide either one color value,
                 * in which case the whole Game Object will be tinted in that color. Or you can provide a color
                 * per corner. The colors are blended together across the extent of the Game Object.
                 *
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 *
                 * To remove a tint call `clearTint`.
                 *
                 * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTint(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

                /**
                 * Sets a fill-based tint on this Game Object.
                 *
                 * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
                 * with those in the tint. You can use this for effects such as making a player flash 'white'
                 * if hit by something. You can provide either one color value, in which case the whole
                 * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
                 * are blended together across the extent of the Game Object.
                 *
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 *
                 * To remove a tint call `clearTint`.
                 *
                 * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTintFill(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

                /**
                 * The tint value being applied to the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintTopLeft: integer;

                /**
                 * The tint value being applied to the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintTopRight: integer;

                /**
                 * The tint value being applied to the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintBottomLeft: integer;

                /**
                 * The tint value being applied to the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintBottomRight: integer;

                /**
                 * The tint value being applied to the whole of the Game Object.
                 */
                tint: integer;

                /**
                 * Does this Game Object have a tint applied to it or not?
                 */
                readonly isTinted: boolean;

            }

            /**
             * The Camera Manager is a plugin that belongs to a Scene and is responsible for managing all of the Scene Cameras.
             *
             * By default you can access the Camera Manager from within a Scene using `this.cameras`, although this can be changed
             * in your game config.
             *
             * Create new Cameras using the `add` method. Or extend the Camera class with your own addition code and then add
             * the new Camera in using the `addExisting` method.
             *
             * Cameras provide a view into your game world, and can be positioned, rotated, zoomed and scrolled accordingly.
             *
             * A Camera consists of two elements: The viewport and the scroll values.
             *
             * The viewport is the physical position and size of the Camera within your game. Cameras, by default, are
             * created the same size as your game, but their position and size can be set to anything. This means if you
             * wanted to create a camera that was 320x200 in size, positioned in the bottom-right corner of your game,
             * you'd adjust the viewport to do that (using methods like `setViewport` and `setSize`).
             *
             * If you wish to change where the Camera is looking in your game, then you scroll it. You can do this
             * via the properties `scrollX` and `scrollY` or the method `setScroll`. Scrolling has no impact on the
             * viewport, and changing the viewport has no impact on the scrolling.
             *
             * By default a Camera will render all Game Objects it can see. You can change this using the `ignore` method,
             * allowing you to filter Game Objects out on a per-Camera basis. The Camera Manager can manage up to 31 unique
             * 'Game Object ignore capable' Cameras. Any Cameras beyond 31 that you create will all be given a Camera ID of
             * zero, meaning that they cannot be used for Game Object exclusion. This means if you need your Camera to ignore
             * Game Objects, make sure it's one of the first 31 created.
             *
             * A Camera also has built-in special effects including Fade, Flash, Camera Shake, Pan and Zoom.
             */
            class CameraManager {
                /**
                 *
                 * @param scene The Scene that owns the Camera Manager plugin.
                 */
                constructor(scene: Phaser.Scene);

                /**
                 * The Scene that owns the Camera Manager plugin.
                 */
                scene: Phaser.Scene;

                /**
                 * A reference to the Scene.Systems handler for the Scene that owns the Camera Manager.
                 */
                systems: Phaser.Scenes.Systems;

                /**
                 * All Cameras created by, or added to, this Camera Manager, will have their `roundPixels`
                 * property set to match this value. By default it is set to match the value set in the
                 * game configuration, but can be changed at any point. Equally, individual cameras can
                 * also be changed as needed.
                 */
                roundPixels: boolean;

                /**
                 * An Array of the Camera objects being managed by this Camera Manager.
                 * The Cameras are updated and rendered in the same order in which they appear in this array.
                 * Do not directly add or remove entries to this array. However, you can move the contents
                 * around the array should you wish to adjust the display order.
                 */
                cameras: Phaser.Cameras.Scene2D.Camera[];

                /**
                 * A handy reference to the 'main' camera. By default this is the first Camera the
                 * Camera Manager creates. You can also set it directly, or use the `makeMain` argument
                 * in the `add` and `addExisting` methods. It allows you to access it from your game:
                 *
                 * ```javascript
                 * var cam = this.cameras.main;
                 * ```
                 *
                 * Also see the properties `camera1`, `camera2` and so on.
                 */
                main: Phaser.Cameras.Scene2D.Camera;

                /**
                 * This scale affects all cameras. It's used by the Scale Manager.
                 */
                baseScale: number;

                /**
                 * Adds a new Camera into the Camera Manager. The Camera Manager can support up to 31 different Cameras.
                 *
                 * Each Camera has its own viewport, which controls the size of the Camera and its position within the canvas.
                 *
                 * Use the `Camera.scrollX` and `Camera.scrollY` properties to change where the Camera is looking, or the
                 * Camera methods such as `centerOn`. Cameras also have built in special effects, such as fade, flash, shake,
                 * pan and zoom.
                 *
                 * By default Cameras are transparent and will render anything that they can see based on their `scrollX`
                 * and `scrollY` values. Game Objects can be set to be ignored by a Camera by using the `Camera.ignore` method.
                 *
                 * The Camera will have its `roundPixels` propery set to whatever `CameraManager.roundPixels` is. You can change
                 * it after creation if required.
                 *
                 * See the Camera class documentation for more details.
                 * @param x The horizontal position of the Camera viewport. Default 0.
                 * @param y The vertical position of the Camera viewport. Default 0.
                 * @param width The width of the Camera viewport. If not given it'll be the game config size.
                 * @param height The height of the Camera viewport. If not given it'll be the game config size.
                 * @param makeMain Set this Camera as being the 'main' camera. This just makes the property `main` a reference to it. Default false.
                 * @param name The name of the Camera. Default ''.
                 */
                add(x?: integer, y?: integer, width?: integer, height?: integer, makeMain?: boolean, name?: string): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Adds an existing Camera into the Camera Manager.
                 *
                 * The Camera should either be a `Phaser.Cameras.Scene2D.Camera` instance, or a class that extends from it.
                 *
                 * The Camera will have its `roundPixels` propery set to whatever `CameraManager.roundPixels` is. You can change
                 * it after addition if required.
                 *
                 * The Camera will be assigned an ID, which is used for Game Object exclusion and then added to the
                 * manager. As long as it doesn't already exist in the manager it will be added then returned.
                 *
                 * If this method returns `null` then the Camera already exists in this Camera Manager.
                 * @param camera The Camera to be added to the Camera Manager.
                 * @param makeMain Set this Camera as being the 'main' camera. This just makes the property `main` a reference to it. Default false.
                 */
                addExisting(camera: Phaser.Cameras.Scene2D.Camera, makeMain?: boolean): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Gets the total number of Cameras in this Camera Manager.
                 *
                 * If the optional `isVisible` argument is set it will only count Cameras that are currently visible.
                 * @param isVisible Set the `true` to only include visible Cameras in the total. Default false.
                 */
                getTotal(isVisible?: boolean): integer;

                /**
                 * Populates this Camera Manager based on the given configuration object, or an array of config objects.
                 *
                 * See the `InputJSONCameraObject` documentation for details of the object structure.
                 * @param config A Camera configuration object, or an array of them, to be added to this Camera Manager.
                 */
                fromJSON(config: InputJSONCameraObject | InputJSONCameraObject[]): Phaser.Cameras.Scene2D.CameraManager;

                /**
                 * Gets a Camera based on its name.
                 *
                 * Camera names are optional and don't have to be set, so this method is only of any use if you
                 * have given your Cameras unique names.
                 * @param name The name of the Camera.
                 */
                getCamera(name: string): Phaser.Cameras.Scene2D.Camera;

                /**
                 * Returns an array of all cameras below the given Pointer.
                 *
                 * The first camera in the array is the top-most camera in the camera list.
                 * @param pointer The Pointer to check against.
                 */
                getCamerasBelowPointer(pointer: Phaser.Input.Pointer): Phaser.Cameras.Scene2D.Camera[];

                /**
                 * Removes the given Camera, or an array of Cameras, from this Camera Manager.
                 *
                 * If found in the Camera Manager it will be immediately removed from the local cameras array.
                 * If also currently the 'main' camera, 'main' will be reset to be camera 0.
                 *
                 * The removed Camera is not destroyed. If you also wish to destroy the Camera, you should call
                 * `Camera.destroy` on it, so that it clears all references to the Camera Manager.
                 * @param camera The Camera, or an array of Cameras, to be removed from this Camera Manager.
                 */
                remove(camera: Phaser.Cameras.Scene2D.Camera | Phaser.Cameras.Scene2D.Camera[]): integer;

                /**
                 * The internal render method. This is called automatically by the Scene and should not be invoked directly.
                 *
                 * It will iterate through all local cameras and render them in turn, as long as they're visible and have
                 * an alpha level > 0.
                 * @param renderer The Renderer that will render the children to this camera.
                 * @param children An array of renderable Game Objects.
                 * @param interpolation Interpolation value. Reserved for future use.
                 */
                protected render(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer, children: Phaser.GameObjects.GameObject[], interpolation: number): void;

                /**
                 * Resets this Camera Manager.
                 *
                 * This will iterate through all current Cameras, destroying them all, then it will reset the
                 * cameras array, reset the ID counter and create 1 new single camera using the default values.
                 */
                resetAll(): Phaser.Cameras.Scene2D.Camera;

                /**
                 * The main update loop. Called automatically when the Scene steps.
                 * @param timestep The timestep value.
                 * @param delta The delta value since the last frame.
                 */
                protected update(timestep: number, delta: number): void;

                /**
                 * Resizes all cameras to the given dimensions.
                 * @param width The new width of the camera.
                 * @param height The new height of the camera.
                 */
                resize(width: number, height: number): void;

            }

            namespace Effects {
                /**
                 * A Camera Fade effect.
                 *
                 * This effect will fade the camera viewport to the given color, over the duration specified.
                 *
                 * Only the camera viewport is faded. None of the objects it is displaying are impacted, i.e. their colors do
                 * not change.
                 *
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect, if required.
                 */
                class Fade {
                    /**
                     *
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * Has this effect finished running?
                     *
                     * This is different from `isRunning` because it remains set to `true` when the effect is over,
                     * until the effect is either reset or started again.
                     */
                    readonly isComplete: boolean;

                    /**
                     * The direction of the fade.
                     * `true` = fade out (transparent to color), `false` = fade in (color to transparent)
                     */
                    readonly direction: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: integer;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * Fades the Camera to or from the given color over the duration specified.
                     * @param direction The direction of the fade. `true` = fade out (transparent to color), `false` = fade in (color to transparent) Default true.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 0.
                     * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 0.
                     * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 0.
                     * @param force Force the effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(direction?: boolean, duration?: integer, red?: integer, green?: integer, blue?: integer, force?: boolean, callback?: CameraFadeCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: integer, delta: number): void;

                    /**
                     * Called internally by the Canvas Renderer.
                     * @param ctx The Canvas context to render to.
                     */
                    postRenderCanvas(ctx: CanvasRenderingContext2D): boolean;

                    /**
                     * Called internally by the WebGL Renderer.
                     * @param pipeline The WebGL Pipeline to render to.
                     * @param getTintFunction A function that will return the gl safe tint colors.
                     */
                    postRenderWebGL(pipeline: Phaser.Renderer.WebGL.Pipelines.TextureTintPipeline, getTintFunction: Function): boolean;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Flash effect.
                 *
                 * This effect will flash the camera viewport to the given color, over the duration specified.
                 *
                 * Only the camera viewport is flashed. None of the objects it is displaying are impacted, i.e. their colors do
                 * not change.
                 *
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect, if required.
                 */
                class Flash {
                    /**
                     *
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: integer;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * Flashes the Camera to or from the given color over the duration specified.
                     * @param duration The duration of the effect in milliseconds. Default 250.
                     * @param red The amount to fade the red channel towards. A value between 0 and 255. Default 255.
                     * @param green The amount to fade the green channel towards. A value between 0 and 255. Default 255.
                     * @param blue The amount to fade the blue channel towards. A value between 0 and 255. Default 255.
                     * @param force Force the effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(duration?: integer, red?: integer, green?: integer, blue?: integer, force?: boolean, callback?: CameraFlashCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: integer, delta: number): void;

                    /**
                     * Called internally by the Canvas Renderer.
                     * @param ctx The Canvas context to render to.
                     */
                    postRenderCanvas(ctx: CanvasRenderingContext2D): boolean;

                    /**
                     * Called internally by the WebGL Renderer.
                     * @param pipeline The WebGL Pipeline to render to.
                     * @param getTintFunction A function that will return the gl safe tint colors.
                     */
                    postRenderWebGL(pipeline: Phaser.Renderer.WebGL.Pipelines.TextureTintPipeline, getTintFunction: Function): boolean;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Pan effect.
                 *
                 * This effect will scroll the Camera so that the center of its viewport finishes at the given destination,
                 * over the duration and with the ease specified.
                 *
                 * Only the camera scroll is moved. None of the objects it is displaying are impacted, i.e. their positions do
                 * not change.
                 *
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class Pan {
                    /**
                     *
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: integer;

                    /**
                     * The starting scroll coordinates to pan the camera from.
                     */
                    source: Phaser.Math.Vector2;

                    /**
                     * The constantly updated value based on zoom.
                     */
                    current: Phaser.Math.Vector2;

                    /**
                     * The destination scroll coordinates to pan the camera to.
                     */
                    destination: Phaser.Math.Vector2;

                    /**
                     * The ease function to use during the pan.
                     */
                    ease: Function;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * This effect will scroll the Camera so that the center of its viewport finishes at the given destination,
                     * over the duration and with the ease specified.
                     * @param x The destination x coordinate to scroll the center of the Camera viewport to.
                     * @param y The destination y coordinate to scroll the center of the Camera viewport to.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param ease The ease to use for the pan. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                     * @param force Force the shake effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent four arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                     * the current camera scroll x coordinate and the current camera scroll y coordinate.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(x: number, y: number, duration?: integer, ease?: string | Function, force?: boolean, callback?: CameraPanCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: integer, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Shake effect.
                 *
                 * This effect will shake the camera viewport by a random amount, bounded by the specified intensity, each frame.
                 *
                 * Only the camera viewport is moved. None of the objects it is displaying are impacted, i.e. their positions do
                 * not change.
                 *
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class Shake {
                    /**
                     *
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: integer;

                    /**
                     * The intensity of the effect. Use small float values. The default when the effect starts is 0.05.
                     * This is a Vector2 object, allowing you to control the shake intensity independently across x and y.
                     * You can modify this value while the effect is active to create more varied shake effects.
                     */
                    intensity: Phaser.Math.Vector2;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * Shakes the Camera by the given intensity over the duration specified.
                     * @param duration The duration of the effect in milliseconds. Default 100.
                     * @param intensity The intensity of the shake. Default 0.05.
                     * @param force Force the shake effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent two arguments: A reference to the camera and a progress amount between 0 and 1 indicating how complete the effect is.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(duration?: integer, intensity?: number, force?: boolean, callback?: CameraShakeCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The pre-render step for this effect. Called automatically by the Camera.
                     */
                    preRender(): void;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: integer, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

                /**
                 * A Camera Zoom effect.
                 *
                 * This effect will zoom the Camera to the given scale, over the duration and with the ease specified.
                 *
                 * The effect will dispatch several events on the Camera itself and you can also specify an `onUpdate` callback,
                 * which is invoked each frame for the duration of the effect if required.
                 */
                class Zoom {
                    /**
                     *
                     * @param camera The camera this effect is acting upon.
                     */
                    constructor(camera: Phaser.Cameras.Scene2D.Camera);

                    /**
                     * The Camera this effect belongs to.
                     */
                    readonly camera: Phaser.Cameras.Scene2D.Camera;

                    /**
                     * Is this effect actively running?
                     */
                    readonly isRunning: boolean;

                    /**
                     * The duration of the effect, in milliseconds.
                     */
                    readonly duration: integer;

                    /**
                     * The starting zoom value;
                     */
                    source: number;

                    /**
                     * The destination zoom value.
                     */
                    destination: number;

                    /**
                     * The ease function to use during the zoom.
                     */
                    ease: Function;

                    /**
                     * If this effect is running this holds the current percentage of the progress, a value between 0 and 1.
                     */
                    progress: number;

                    /**
                     * This effect will zoom the Camera to the given scale, over the duration and with the ease specified.
                     * @param zoom The target Camera zoom value.
                     * @param duration The duration of the effect in milliseconds. Default 1000.
                     * @param ease The ease to use for the Zoom. Can be any of the Phaser Easing constants or a custom function. Default 'Linear'.
                     * @param force Force the shake effect to start immediately, even if already running. Default false.
                     * @param callback This callback will be invoked every frame for the duration of the effect.
                     * It is sent three arguments: A reference to the camera, a progress amount between 0 and 1 indicating how complete the effect is,
                     * and the current camera zoom value.
                     * @param context The context in which the callback is invoked. Defaults to the Scene to which the Camera belongs.
                     */
                    start(zoom: number, duration?: integer, ease?: string | Function, force?: boolean, callback?: CameraZoomCallback, context?: any): Phaser.Cameras.Scene2D.Camera;

                    /**
                     * The main update loop for this effect. Called automatically by the Camera.
                     * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                     * @param delta The delta time, in ms, elapsed since the last frame.
                     */
                    update(time: integer, delta: number): void;

                    /**
                     * Called internally when the effect completes.
                     */
                    effectComplete(): void;

                    /**
                     * Resets this camera effect.
                     * If it was previously running, it stops instantly without calling its onComplete callback or emitting an event.
                     */
                    reset(): void;

                    /**
                     * Destroys this effect, releasing it from the Camera.
                     */
                    destroy(): void;

                }

            }

        }

        namespace Controls {
            /**
             * [description]
             */
            class FixedKeyControl {
                /**
                 *
                 * @param config [description]
                 */
                constructor(config: FixedKeyControlConfig);

                /**
                 * The Camera that this Control will update.
                 */
                camera: Phaser.Cameras.Scene2D.Camera;

                /**
                 * The Key to be pressed that will move the Camera left.
                 */
                left: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera right.
                 */
                right: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera up.
                 */
                up: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera down.
                 */
                down: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera in.
                 */
                zoomIn: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera out.
                 */
                zoomOut: Phaser.Input.Keyboard.Key;

                /**
                 * The speed at which the camera will zoom if the `zoomIn` or `zoomOut` keys are pressed.
                 */
                zoomSpeed: number;

                /**
                 * The horizontal speed the camera will move.
                 */
                speedX: number;

                /**
                 * The vertical speed the camera will move.
                 */
                speedY: number;

                /**
                 * A flag controlling if the Controls will update the Camera or not.
                 */
                active: boolean;

                /**
                 * Starts the Key Control running, providing it has been linked to a camera.
                 */
                start(): Phaser.Cameras.Controls.FixedKeyControl;

                /**
                 * Stops this Key Control from running. Call `start` to start it again.
                 */
                stop(): Phaser.Cameras.Controls.FixedKeyControl;

                /**
                 * Binds this Key Control to a camera.
                 * @param camera The camera to bind this Key Control to.
                 */
                setCamera(camera: Phaser.Cameras.Scene2D.Camera): Phaser.Cameras.Controls.FixedKeyControl;

                /**
                 * [description]
                 * @param delta The delta time in ms since the last frame. This is a smoothed and capped value based on the FPS rate.
                 */
                update(delta: number): void;

                /**
                 * Destroys this Key Control.
                 */
                destroy(): void;

            }

            /**
             * [description]
             */
            class SmoothedKeyControl {
                /**
                 *
                 * @param config [description]
                 */
                constructor(config: SmoothedKeyControlConfig);

                /**
                 * The Camera that this Control will update.
                 */
                camera: Phaser.Cameras.Scene2D.Camera;

                /**
                 * The Key to be pressed that will move the Camera left.
                 */
                left: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera right.
                 */
                right: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera up.
                 */
                up: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will move the Camera down.
                 */
                down: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera in.
                 */
                zoomIn: Phaser.Input.Keyboard.Key;

                /**
                 * The Key to be pressed that will zoom the Camera out.
                 */
                zoomOut: Phaser.Input.Keyboard.Key;

                /**
                 * The speed at which the camera will zoom if the `zoomIn` or `zoomOut` keys are pressed.
                 */
                zoomSpeed: number;

                /**
                 * The horizontal acceleration the camera will move.
                 */
                accelX: number;

                /**
                 * The vertical acceleration the camera will move.
                 */
                accelY: number;

                /**
                 * The horizontal drag applied to the camera when it is moving.
                 */
                dragX: number;

                /**
                 * The vertical drag applied to the camera when it is moving.
                 */
                dragY: number;

                /**
                 * The maximum horizontal speed the camera will move.
                 */
                maxSpeedX: number;

                /**
                 * The maximum vertical speed the camera will move.
                 */
                maxSpeedY: number;

                /**
                 * A flag controlling if the Controls will update the Camera or not.
                 */
                active: boolean;

                /**
                 * Starts the Key Control running, providing it has been linked to a camera.
                 */
                start(): Phaser.Cameras.Controls.SmoothedKeyControl;

                /**
                 * Stops this Key Control from running. Call `start` to start it again.
                 */
                stop(): Phaser.Cameras.Controls.SmoothedKeyControl;

                /**
                 * Binds this Key Control to a camera.
                 * @param camera The camera to bind this Key Control to.
                 */
                setCamera(camera: Phaser.Cameras.Scene2D.Camera): Phaser.Cameras.Controls.SmoothedKeyControl;

                /**
                 * [description]
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                update(delta: number): void;

                /**
                 * Destroys this Key Control.
                 */
                destroy(): void;

            }

        }

    }

    /**
     * Phaser Release Version
     */
    var VERSION: string;

    /**
     * AUTO Detect Renderer.
     */
    var AUTO: integer;

    /**
     * Canvas Renderer.
     */
    var CANVAS: integer;

    /**
     * WebGL Renderer.
     */
    var WEBGL: integer;

    /**
     * Headless Renderer.
     */
    var HEADLESS: integer;

    /**
     * In Phaser the value -1 means 'forever' in lots of cases, this const allows you to use it instead
     * to help you remember what the value is doing in your code.
     */
    var FOREVER: integer;

    /**
     * Direction constant.
     */
    var NONE: integer;

    /**
     * Direction constant.
     */
    var UP: integer;

    /**
     * Direction constant.
     */
    var DOWN: integer;

    /**
     * Direction constant.
     */
    var LEFT: integer;

    /**
     * Direction constant.
     */
    var RIGHT: integer;

    namespace Create {
        /**
         * [description]
         * @param config [description]
         */
        function GenerateTexture(config: GenerateTextureConfig): HTMLCanvasElement;

        namespace Palettes {
            /**
             * A 16 color palette by [Arne](http://androidarts.com/palette/16pal.htm)
             */
            var ARNE16: Palette;

            /**
             * A 16 color palette inspired by the Commodore 64.
             */
            var C64: Palette;

            /**
             * A 16 color CGA inspired palette by [Arne](http://androidarts.com/palette/16pal.htm)
             */
            var CGA: Palette;

            /**
             * A 16 color JMP palette by [Arne](http://androidarts.com/palette/16pal.htm)
             */
            var JMP: Palette;

            /**
             * A 16 color palette inspired by Japanese computers like the MSX.
             */
            var MSX: Palette;

        }

    }

    namespace Curves {
        /**
         * A higher-order Bézier curve constructed of four points.
         */
        class CubicBezier extends Phaser.Curves.Curve {
            /**
             *
             * @param p0 Start point, or an array of point pairs.
             * @param p1 Control Point 1.
             * @param p2 Control Point 2.
             * @param p3 End Point.
             */
            constructor(p0: Phaser.Math.Vector2 | Phaser.Math.Vector2[], p1: Phaser.Math.Vector2, p2: Phaser.Math.Vector2, p3: Phaser.Math.Vector2);

            /**
             * The start point of this curve.
             */
            p0: Phaser.Math.Vector2;

            /**
             * The first control point of this curve.
             */
            p1: Phaser.Math.Vector2;

            /**
             * The second control point of this curve.
             */
            p2: Phaser.Math.Vector2;

            /**
             * The end point of this curve.
             */
            p3: Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * Returns the resolution of this curve.
             * @param divisions The amount of divisions used by this curve.
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Draws this curve to the specified graphics object.
             * @param graphics The graphics object this curve should be drawn to.
             * @param pointsTotal The number of intermediary points that make up this curve. A higher number of points will result in a smoother curve. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G, pointsTotal?: integer): G;

            /**
             * Returns a JSON object that describes this curve.
             */
            toJSON(): JSONCurve;

            /**
             * Generates a curve from a JSON object.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: JSONCurve): Phaser.Curves.CubicBezier;

        }

        /**
         * A Base Curve class, which all other curve types extend.
         *
         * Based on the three.js Curve classes created by [zz85](http://www.lab4games.net/zz85/blog)
         */
        class Curve {
            /**
             *
             * @param type [description]
             */
            constructor(type: string);

            /**
             * String based identifier for the type of curve.
             */
            type: string;

            /**
             * The default number of divisions within the curve.
             */
            defaultDivisions: integer;

            /**
             * The quantity of arc length divisions within the curve.
             */
            arcLengthDivisions: integer;

            /**
             * An array of cached arc length values.
             */
            cacheArcLengths: number[];

            /**
             * Does the data of this curve need updating?
             */
            needsUpdate: boolean;

            /**
             * [description]
             */
            active: boolean;

            /**
             * Draws this curve on the given Graphics object.
             *
             * The curve is drawn using `Graphics.strokePoints` so will be drawn at whatever the present Graphics stroke color is.
             * The Graphics object is not cleared before the draw, so the curve will appear on-top of anything else already rendered to it.
             * @param graphics The Graphics instance onto which this curve will be drawn.
             * @param pointsTotal The resolution of the curve. The higher the value the smoother it will render, at the cost of rendering performance. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G, pointsTotal?: integer): G;

            /**
             * Returns a Rectangle where the position and dimensions match the bounds of this Curve.
             *
             * You can control the accuracy of the bounds. The value given is used to work out how many points
             * to plot across the curve. Higher values are more accurate at the cost of calculation speed.
             * @param out The Rectangle to store the bounds in. If falsey a new object will be created.
             * @param accuracy The accuracy of the bounds calculations. Default 16.
             */
            getBounds(out?: Phaser.Geom.Rectangle, accuracy?: integer): Phaser.Geom.Rectangle;

            /**
             * Returns an array of points, spaced out X distance pixels apart.
             * The smaller the distance, the larger the array will be.
             * @param distance The distance, in pixels, between each point along the curve.
             */
            getDistancePoints(distance: integer): Phaser.Geom.Point[];

            /**
             * [description]
             * @param out Optional Vector object to store the result in.
             */
            getEndPoint(out?: Phaser.Math.Vector2): Phaser.Math.Vector2;

            /**
             * [description]
             */
            getLength(): number;

            /**
             * [description]
             * @param divisions [description]
             */
            getLengths(divisions?: integer): number[];

            /**
             * [description]
             * @param u [description]
             * @param out [description]
             */
            getPointAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * [description]
             * @param divisions [description]
             */
            getPoints(divisions?: integer): Phaser.Math.Vector2[];

            /**
             * [description]
             * @param out [description]
             */
            getRandomPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param divisions [description]
             */
            getSpacedPoints(divisions?: integer): Phaser.Math.Vector2[];

            /**
             * [description]
             * @param out [description]
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param t [description]
             * @param out [description]
             */
            getTangent<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * [description]
             * @param u [description]
             * @param out [description]
             */
            getTangentAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * [description]
             * @param distance [description]
             * @param divisions [description]
             */
            getTFromDistance(distance: integer, divisions?: integer): number;

            /**
             * [description]
             * @param u [description]
             * @param distance [description]
             * @param divisions [description]
             */
            getUtoTmapping(u: number, distance: integer, divisions?: integer): number;

            /**
             * [description]
             */
            updateArcLengths(): void;

        }

        /**
         * An Elliptical Curve derived from the Base Curve class.
         *
         * See https://en.wikipedia.org/wiki/Elliptic_curve for more details.
         */
        class Ellipse extends Phaser.Curves.Curve {
            /**
             *
             * @param x The x coordinate of the ellipse, or an Ellipse Curve configuration object. Default 0.
             * @param y The y coordinate of the ellipse. Default 0.
             * @param xRadius The horizontal radius of ellipse. Default 0.
             * @param yRadius The vertical radius of ellipse. Default 0.
             * @param startAngle The start angle of the ellipse, in degrees. Default 0.
             * @param endAngle The end angle of the ellipse, in degrees. Default 360.
             * @param clockwise Sets if the the ellipse rotation is clockwise (true) or anti-clockwise (false) Default false.
             * @param rotation The rotation of the ellipse, in degrees. Default 0.
             */
            constructor(x?: number | EllipseCurveConfig, y?: number, xRadius?: number, yRadius?: number, startAngle?: integer, endAngle?: integer, clockwise?: boolean, rotation?: integer);

            /**
             * The center point of the ellipse. Used for calculating rotation.
             */
            p0: Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param divisions [description]
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * Sets the horizontal radius of this curve.
             * @param value The horizontal radius of this curve.
             */
            setXRadius(value: number): Phaser.Curves.Ellipse;

            /**
             * Sets the vertical radius of this curve.
             * @param value The vertical radius of this curve.
             */
            setYRadius(value: number): Phaser.Curves.Ellipse;

            /**
             * Sets the width of this curve.
             * @param value The width of this curve.
             */
            setWidth(value: number): Phaser.Curves.Ellipse;

            /**
             * Sets the height of this curve.
             * @param value The height of this curve.
             */
            setHeight(value: number): Phaser.Curves.Ellipse;

            /**
             * Sets the start angle of this curve.
             * @param value The start angle of this curve, in radians.
             */
            setStartAngle(value: number): Phaser.Curves.Ellipse;

            /**
             * Sets the end angle of this curve.
             * @param value The end angle of this curve, in radians.
             */
            setEndAngle(value: number): Phaser.Curves.Ellipse;

            /**
             * Sets if this curve extends clockwise or anti-clockwise.
             * @param value The clockwise state of this curve.
             */
            setClockwise(value: boolean): Phaser.Curves.Ellipse;

            /**
             * Sets the rotation of this curve.
             * @param value The rotation of this curve, in radians.
             */
            setRotation(value: number): Phaser.Curves.Ellipse;

            /**
             * The x coordinate of the center of the ellipse.
             */
            x: number;

            /**
             * The y coordinate of the center of the ellipse.
             */
            y: number;

            /**
             * The horizontal radius of the ellipse.
             */
            xRadius: number;

            /**
             * The vertical radius of the ellipse.
             */
            yRadius: number;

            /**
             * The start angle of the ellipse in degrees.
             */
            startAngle: number;

            /**
             * The end angle of the ellipse in degrees.
             */
            endAngle: number;

            /**
             * `true` if the ellipse rotation is clockwise or `false` if anti-clockwise.
             */
            clockwise: boolean;

            /**
             * The rotation of the ellipse, relative to the center, in degrees.
             */
            angle: number;

            /**
             * The rotation of the ellipse, relative to the center, in radians.
             */
            rotation: number;

            /**
             * JSON serialization of the curve.
             */
            toJSON(): JSONEllipseCurve;

            /**
             * Creates a curve from the provided Ellipse Curve Configuration object.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: JSONEllipseCurve): Phaser.Curves.Ellipse;

        }

        /**
         * [description]
         */
        class Line extends Phaser.Curves.Curve {
            /**
             *
             * @param p0 [description]
             * @param p1 [description]
             */
            constructor(p0: Phaser.Math.Vector2 | number[], p1?: Phaser.Math.Vector2);

            /**
             * [description]
             */
            p0: Phaser.Math.Vector2;

            /**
             * [description]
             */
            p1: Phaser.Math.Vector2;

            /**
             * Returns a Rectangle where the position and dimensions match the bounds of this Curve.
             * @param out A Rectangle object to store the bounds in. If not given a new Rectangle will be created.
             */
            getBounds<O extends Phaser.Geom.Rectangle>(out?: O): O;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param divisions [description] Default 1.
             */
            getResolution(divisions?: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * [description]
             * @param u The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPointAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * [description]
             */
            getTangent<O extends Phaser.Math.Vector2>(): O;

            /**
             * Draws this curve on the given Graphics object.
             *
             * The curve is drawn using `Graphics.lineBetween` so will be drawn at whatever the present Graphics line color is.
             * The Graphics object is not cleared before the draw, so the curve will appear on-top of anything else already rendered to it.
             * @param graphics The Graphics instance onto which this curve will be drawn.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G): G;

            /**
             * [description]
             */
            toJSON(): JSONCurve;

            /**
             * [description]
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: JSONCurve): Phaser.Curves.Line;

        }

        /**
         * [description]
         */
        class MoveTo {
            /**
             *
             * @param x [description]
             * @param y [description]
             */
            constructor(x?: number, y?: number);

            /**
             * [description]
             */
            active: boolean;

            /**
             * [description]
             */
            p0: Phaser.Math.Vector2;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * [description]
             * @param u [description]
             * @param out [description]
             */
            getPointAt<O extends Phaser.Math.Vector2>(u: number, out?: O): O;

            /**
             * Gets the resolution of this curve.
             */
            getResolution(): number;

            /**
             * Gets the length of this curve.
             */
            getLength(): number;

            /**
             * [description]
             */
            toJSON(): JSONCurve;

        }

        /**
         * [description]
         */
        class Path {
            /**
             *
             * @param x [description] Default 0.
             * @param y [description] Default 0.
             */
            constructor(x?: number, y?: number);

            /**
             * [description]
             */
            name: string;

            /**
             * [description]
             */
            curves: Phaser.Curves.Curve[];

            /**
             * [description]
             */
            cacheLengths: number[];

            /**
             * Automatically closes the path.
             */
            autoClose: boolean;

            /**
             * [description]
             */
            startPoint: Phaser.Math.Vector2;

            /**
             * [description]
             * @param curve [description]
             */
            add(curve: Phaser.Curves.Curve): Phaser.Curves.Path;

            /**
             * [description]
             * @param radius [description]
             * @param clockwise [description] Default false.
             * @param rotation [description] Default 0.
             */
            circleTo(radius: number, clockwise?: boolean, rotation?: number): Phaser.Curves.Path;

            /**
             * [description]
             */
            closePath(): Phaser.Curves.Path;

            /**
             * Creates a cubic bezier curve starting at the previous end point and ending at p3, using p1 and p2 as control points.
             * @param x The x coordinate of the end point. Or, if a Vec2, the p1 value.
             * @param y The y coordinate of the end point. Or, if a Vec2, the p2 value.
             * @param control1X The x coordinate of the first control point. Or, if a Vec2, the p3 value.
             * @param control1Y The y coordinate of the first control point. Not used if vec2s are provided as the first 3 arguments.
             * @param control2X The x coordinate of the second control point. Not used if vec2s are provided as the first 3 arguments.
             * @param control2Y The y coordinate of the second control point. Not used if vec2s are provided as the first 3 arguments.
             */
            cubicBezierTo(x: number | Phaser.Math.Vector2, y: number | Phaser.Math.Vector2, control1X: number | Phaser.Math.Vector2, control1Y?: number, control2X?: number, control2Y?: number): Phaser.Curves.Path;

            /**
             * [description]
             * @param x [description]
             * @param y [description]
             * @param controlX [description]
             * @param controlY [description]
             */
            quadraticBezierTo(x: number | Phaser.Math.Vector2[], y?: number, controlX?: number, controlY?: number): Phaser.Curves.Path;

            /**
             * [description]
             * @param graphics [description]
             * @param pointsTotal [description] Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: Phaser.GameObjects.Graphics, pointsTotal?: integer): G;

            /**
             * Creates an ellipse curve positioned at the previous end point, using the given parameters.
             * @param xRadius [description]
             * @param yRadius [description]
             * @param startAngle [description]
             * @param endAngle [description]
             * @param clockwise [description]
             * @param rotation [description]
             */
            ellipseTo(xRadius: number, yRadius: number, startAngle: number, endAngle: number, clockwise: boolean, rotation: number): Phaser.Curves.Path;

            /**
             * [description]
             * @param data [description]
             */
            fromJSON(data: object): Phaser.Curves.Path;

            /**
             * [description]
             * @param out [description]
             * @param accuracy [description] Default 16.
             */
            getBounds<O extends Phaser.Math.Vector2>(out?: O, accuracy?: integer): O;

            /**
             * [description]
             */
            getCurveLengths(): number[];

            /**
             * [description]
             * @param out [description]
             */
            getEndPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             */
            getLength(): number;

            /**
             * [description]
             * @param t [description]
             * @param out [description]
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * [description]
             * @param divisions The number of points to divide the path in to. Default 12.
             */
            getPoints(divisions?: integer): Phaser.Math.Vector2[];

            /**
             * [description]
             * @param out `Vector2` instance that should be used for storing the result. If `undefined` a new `Vector2` will be created.
             */
            getRandomPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param divisions [description] Default 40.
             */
            getSpacedPoints(divisions?: integer): Phaser.Math.Vector2[];

            /**
             * [description]
             * @param out [description]
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param x [description]
             * @param y [description]
             */
            lineTo(x: number | Phaser.Math.Vector2, y?: number): Phaser.Curves.Path;

            /**
             * [description]
             * @param points [description]
             */
            splineTo(points: Phaser.Math.Vector2[]): Phaser.Curves.Path;

            /**
             * [description]
             * @param x [description]
             * @param y [description]
             */
            moveTo(x: number, y: number): Phaser.Curves.Path;

            /**
             * [description]
             */
            toJSON(): JSONPath;

            /**
             * [description]
             */
            updateArcLengths(): void;

            /**
             * [description]
             */
            destroy(): void;

        }

        /**
         * [description]
         */
        class QuadraticBezier extends Phaser.Curves.Curve {
            /**
             *
             * @param p0 Start point, or an array of point pairs.
             * @param p1 Control Point 1.
             * @param p2 Control Point 2.
             */
            constructor(p0: Phaser.Math.Vector2 | number[], p1: Phaser.Math.Vector2, p2: Phaser.Math.Vector2);

            /**
             * [description]
             */
            p0: Phaser.Math.Vector2;

            /**
             * [description]
             */
            p1: Phaser.Math.Vector2;

            /**
             * [description]
             */
            p2: Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param divisions [description]
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * [description]
             * @param graphics `Graphics` object to draw onto.
             * @param pointsTotal Number of points to be used for drawing the curve. Higher numbers result in smoother curve but require more processing. Default 32.
             */
            draw<G extends Phaser.GameObjects.Graphics>(graphics: G, pointsTotal?: integer): G;

            /**
             * Converts the curve into a JSON compatible object.
             */
            toJSON(): JSONCurve;

            /**
             * Creates a curve from a JSON object, e. g. created by `toJSON`.
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: JSONCurve): Phaser.Curves.QuadraticBezier;

        }

        /**
         * [description]
         */
        class Spline extends Phaser.Curves.Curve {
            /**
             *
             * @param points [description]
             */
            constructor(points?: Phaser.Math.Vector2[]);

            /**
             * [description]
             */
            points: Phaser.Math.Vector2[];

            /**
             * [description]
             * @param points [description]
             */
            addPoints(points: Phaser.Math.Vector2[] | number[] | number[][]): Phaser.Curves.Spline;

            /**
             * [description]
             * @param x [description]
             * @param y [description]
             */
            addPoint(x: number, y: number): Phaser.Math.Vector2;

            /**
             * Gets the starting point on the curve.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getStartPoint<O extends Phaser.Math.Vector2>(out?: O): O;

            /**
             * [description]
             * @param divisions [description]
             */
            getResolution(divisions: number): number;

            /**
             * Get point at relative position in curve according to length.
             * @param t The position along the curve to return. Where 0 is the start and 1 is the end.
             * @param out A Vector2 object to store the result in. If not given will be created.
             */
            getPoint<O extends Phaser.Math.Vector2>(t: number, out?: O): O;

            /**
             * [description]
             */
            toJSON(): JSONCurve;

            /**
             * [description]
             * @param data The JSON object containing this curve data.
             */
            static fromJSON(data: JSONCurve): Phaser.Curves.Spline;

        }

    }

    namespace Data {
        /**
         * The Data Component features a means to store pieces of data specific to a Game Object, System or Plugin.
         * You can then search, query it, and retrieve the data. The parent must either extend EventEmitter,
         * or have a property called `events` that is an instance of it.
         */
        class DataManager {
            /**
             *
             * @param parent The object that this DataManager belongs to.
             * @param eventEmitter The DataManager's event emitter.
             */
            constructor(parent: object, eventEmitter: Phaser.Events.EventEmitter);

            /**
             * The object that this DataManager belongs to.
             */
            parent: any;

            /**
             * The DataManager's event emitter.
             */
            events: Phaser.Events.EventEmitter;

            /**
             * The data list.
             */
            list: {[key: string]:  any};

            /**
             * The public values list. You can use this to access anything you have stored
             * in this Data Manager. For example, if you set a value called `gold` you can
             * access it via:
             *
             * ```javascript
             * this.data.values.gold;
             * ```
             *
             * You can also modify it directly:
             *
             * ```javascript
             * this.data.values.gold += 1000;
             * ```
             *
             * Doing so will emit a `setdata` event from the parent of this Data Manager.
             */
            values: {[key: string]:  any};

            /**
             * Retrieves the value for the given key, or undefined if it doesn't exist.
             *
             * You can also access values via the `values` object. For example, if you had a key called `gold` you can do either:
             *
             * ```javascript
             * this.data.get('gold');
             * ```
             *
             * Or access the value directly:
             *
             * ```javascript
             * this.data.values.gold;
             * ```
             *
             * You can also pass in an array of keys, in which case an array of values will be returned:
             *
             * ```javascript
             * this.data.get([ 'gold', 'armor', 'health' ]);
             * ```
             *
             * This approach is useful for destructuring arrays in ES6.
             * @param key The key of the value to retrieve, or an array of keys.
             */
            get(key: string | string[]): any;

            /**
             * Retrieves all data values in a new object.
             */
            getAll(): {[key: string]:  any};

            /**
             * Queries the DataManager for the values of keys matching the given regular expression.
             * @param search A regular expression object. If a non-RegExp object obj is passed, it is implicitly converted to a RegExp by using new RegExp(obj).
             */
            query(search: RegExp): {[key: string]:  any};

            /**
             * Sets a value for the given key. If the key doesn't already exist in the Data Manager then it is created.
             *
             * ```javascript
             * data.set('name', 'Red Gem Stone');
             * ```
             *
             * You can also pass in an object of key value pairs as the first argument:
             *
             * ```javascript
             * data.set({ name: 'Red Gem Stone', level: 2, owner: 'Link', gold: 50 });
             * ```
             *
             * To get a value back again you can call `get`:
             *
             * ```javascript
             * data.get('gold');
             * ```
             *
             * Or you can access the value directly via the `values` property, where it works like any other variable:
             *
             * ```javascript
             * data.values.gold += 50;
             * ```
             *
             * When the value is first set, a `setdata` event is emitted.
             *
             * If the key already exists, a `changedata` event is emitted instead, along an event named after the key.
             * For example, if you updated an existing key called `PlayerLives` then it would emit the event `changedata_PlayerLives`.
             * These events will be emitted regardless if you use this method to set the value, or the direct `values` setter.
             *
             * Please note that the data keys are case-sensitive and must be valid JavaScript Object property strings.
             * This means the keys `gold` and `Gold` are treated as two unique values within the Data Manager.
             * @param key The key to set the value for. Or an object or key value pairs. If an object the `data` argument is ignored.
             * @param data The value to set for the given key. If an object is provided as the key this argument is ignored.
             */
            set(key: string | object, data: any): Phaser.Data.DataManager;

            /**
             * Passes all data entries to the given callback.
             * @param callback The function to call.
             * @param context Value to use as `this` when executing callback.
             * @param args Additional arguments that will be passed to the callback, after the game object, key, and data.
             */
            each(callback: DataEachCallback, context?: any, ...args: any[]): Phaser.Data.DataManager;

            /**
             * Merge the given object of key value pairs into this DataManager.
             *
             * Any newly created values will emit a `setdata` event. Any updated values (see the `overwrite` argument)
             * will emit a `changedata` event.
             * @param data The data to merge.
             * @param overwrite Whether to overwrite existing data. Defaults to true. Default true.
             */
            merge(data: {[key: string]:  any}, overwrite?: boolean): Phaser.Data.DataManager;

            /**
             * Remove the value for the given key.
             *
             * If the key is found in this Data Manager it is removed from the internal lists and a
             * `removedata` event is emitted.
             *
             * You can also pass in an array of keys, in which case all keys in the array will be removed:
             *
             * ```javascript
             * this.data.remove([ 'gold', 'armor', 'health' ]);
             * ```
             * @param key The key to remove, or an array of keys to remove.
             */
            remove(key: string | string[]): Phaser.Data.DataManager;

            /**
             * Retrieves the data associated with the given 'key', deletes it from this Data Manager, then returns it.
             * @param key The key of the value to retrieve and delete.
             */
            pop(key: string): any;

            /**
             * Determines whether the given key is set in this Data Manager.
             *
             * Please note that the keys are case-sensitive and must be valid JavaScript Object property strings.
             * This means the keys `gold` and `Gold` are treated as two unique values within the Data Manager.
             * @param key The key to check.
             */
            has(key: string): boolean;

            /**
             * Freeze or unfreeze this Data Manager. A frozen Data Manager will block all attempts
             * to create new values or update existing ones.
             * @param value Whether to freeze or unfreeze the Data Manager.
             */
            setFreeze(value: boolean): Phaser.Data.DataManager;

            /**
             * Delete all data in this Data Manager and unfreeze it.
             */
            reset(): Phaser.Data.DataManager;

            /**
             * Destroy this data manager.
             */
            destroy(): void;

            /**
             * Gets or sets the frozen state of this Data Manager.
             * A frozen Data Manager will block all attempts to create new values or update existing ones.
             */
            freeze: boolean;

            /**
             * Return the total number of entries in this Data Manager.
             */
            count: integer;

        }

        /**
         * The Data Component features a means to store pieces of data specific to a Game Object, System or Plugin.
         * You can then search, query it, and retrieve the data. The parent must either extend EventEmitter,
         * or have a property called `events` that is an instance of it.
         */
        class DataManagerPlugin extends Phaser.Data.DataManager {
            /**
             *
             * @param scene A reference to the Scene that this DataManager belongs to.
             */
            constructor(scene: Phaser.Scene);

            /**
             * A reference to the Scene that this DataManager belongs to.
             */
            scene: Phaser.Scene;

            /**
             * A reference to the Scene's Systems.
             */
            systems: Phaser.Scenes.Systems;

            /**
             * The Scene that owns this plugin is being destroyed.
             * We need to shutdown and then kill off all external references.
             */
            destroy(): void;

        }

    }

    namespace Device {
        /**
         * Determines the audio playback capabilities of the device running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.audio` from within any Scene.
         */
        type Audio = {
            /**
             * Can this device play HTML Audio tags?
             */
            audioData: boolean;
            /**
             * Can this device play EC-3 Dolby Digital Plus files?
             */
            dolby: boolean;
            /**
             * Can this device can play m4a files.
             */
            m4a: boolean;
            /**
             * Can this device play mp3 files?
             */
            mp3: boolean;
            /**
             * Can this device play ogg files?
             */
            ogg: boolean;
            /**
             * Can this device play opus files?
             */
            opus: boolean;
            /**
             * Can this device play wav files?
             */
            wav: boolean;
            /**
             * Does this device have the Web Audio API?
             */
            webAudio: boolean;
            /**
             * Can this device play webm files?
             */
            webm: boolean;
        };

        /**
         * Determines the browser type and version running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.browser` from within any Scene.
         */
        type Browser = {
            /**
             * Set to true if running in Chrome.
             */
            chrome: boolean;
            /**
             * Set to true if running in Microsoft Edge browser.
             */
            edge: boolean;
            /**
             * Set to true if running in Firefox.
             */
            firefox: boolean;
            /**
             * Set to true if running in Internet Explorer 11 or less (not Edge).
             */
            ie: boolean;
            /**
             * Set to true if running in Mobile Safari.
             */
            mobileSafari: boolean;
            /**
             * Set to true if running in Opera.
             */
            opera: boolean;
            /**
             * Set to true if running in Safari.
             */
            safari: boolean;
            /**
             * Set to true if running in the Silk browser (as used on the Amazon Kindle)
             */
            silk: boolean;
            /**
             * Set to true if running a Trident version of Internet Explorer (IE11+)
             */
            trident: boolean;
            /**
             * If running in Chrome this will contain the major version number.
             */
            chromeVersion: number;
            /**
             * If running in Firefox this will contain the major version number.
             */
            firefoxVersion: number;
            /**
             * If running in Internet Explorer this will contain the major version number. Beyond IE10 you should use Browser.trident and Browser.tridentVersion.
             */
            ieVersion: number;
            /**
             * If running in Safari this will contain the major version number.
             */
            safariVersion: number;
            /**
             * If running in Internet Explorer 11 this will contain the major version number. See {@link http://msdn.microsoft.com/en-us/library/ie/ms537503(v=vs.85).aspx}
             */
            tridentVersion: number;
        };

        /**
         * Determines the canvas features of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.canvasFeatures` from within any Scene.
         */
        type CanvasFeatures = {
            /**
             * Set to true if the browser supports inversed alpha.
             */
            supportInverseAlpha: boolean;
            /**
             * Set to true if the browser supports new canvas blend modes.
             */
            supportNewBlendModes: boolean;
        };

        /**
         * Determines the features of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.features` from within any Scene.
         */
        type Features = {
            /**
             * True if canvas supports a 'copy' bitblt onto itself when the source and destination regions overlap.
             */
            canvasBitBltShift: boolean;
            /**
             * Is canvas available?
             */
            canvas: boolean;
            /**
             * Is file available?
             */
            file: boolean;
            /**
             * Is fileSystem available?
             */
            fileSystem: boolean;
            /**
             * Does the device support the getUserMedia API?
             */
            getUserMedia: boolean;
            /**
             * Is the device big or little endian? (only detected if the browser supports TypedArrays)
             */
            littleEndian: boolean;
            /**
             * Is localStorage available?
             */
            localStorage: boolean;
            /**
             * Is Pointer Lock available?
             */
            pointerLock: boolean;
            /**
             * Does the device context support 32bit pixel manipulation using array buffer views?
             */
            support32bit: boolean;
            /**
             * Does the device support the Vibration API?
             */
            vibration: boolean;
            /**
             * Is webGL available?
             */
            webGL: boolean;
            /**
             * Is worker available?
             */
            worker: boolean;
        };

        /**
         * Determines the full screen support of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.fullscreen` from within any Scene.
         */
        type Fullscreen = {
            /**
             * Does the browser support the Full Screen API?
             */
            available: boolean;
            /**
             * Does the browser support access to the Keyboard during Full Screen mode?
             */
            keyboard: boolean;
            /**
             * If the browser supports the Full Screen API this holds the call you need to use to cancel it.
             */
            cancel: string;
            /**
             * If the browser supports the Full Screen API this holds the call you need to use to activate it.
             */
            request: string;
        };

        /**
         * Determines the input support of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.input` from within any Scene.
         */
        type Input = {
            /**
             * The newest type of Wheel/Scroll event supported: 'wheel', 'mousewheel', 'DOMMouseScroll'
             */
            wheelType: string;
            /**
             * Is navigator.getGamepads available?
             */
            gamepads: boolean;
            /**
             * Is mspointer available?
             */
            mspointer: boolean;
            /**
             * Is touch available?
             */
            touch: boolean;
        };

        /**
         * Determines the operating system of the device running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.os` from within any Scene.
         */
        type OS = {
            /**
             * Is running on android?
             */
            android: boolean;
            /**
             * Is running on chromeOS?
             */
            chromeOS: boolean;
            /**
             * Is the game running under CocoonJS?
             */
            cocoonJS: boolean;
            /**
             * Is this game running with CocoonJS.App?
             */
            cocoonJSApp: boolean;
            /**
             * Is the game running under Apache Cordova?
             */
            cordova: boolean;
            /**
             * Is the game running under the Intel Crosswalk XDK?
             */
            crosswalk: boolean;
            /**
             * Is running on a desktop?
             */
            desktop: boolean;
            /**
             * Is the game running under Ejecta?
             */
            ejecta: boolean;
            /**
             * Is the game running under GitHub Electron?
             */
            electron: boolean;
            /**
             * Is running on iOS?
             */
            iOS: boolean;
            /**
             * Is running on iPad?
             */
            iPad: boolean;
            /**
             * Is running on iPhone?
             */
            iPhone: boolean;
            /**
             * Is running on an Amazon Kindle?
             */
            kindle: boolean;
            /**
             * Is running on linux?
             */
            linux: boolean;
            /**
             * Is running on macOS?
             */
            macOS: boolean;
            /**
             * Is the game running under Node.js?
             */
            node: boolean;
            /**
             * Is the game running under Node-Webkit?
             */
            nodeWebkit: boolean;
            /**
             * Set to true if running as a WebApp, i.e. within a WebView
             */
            webApp: boolean;
            /**
             * Is running on windows?
             */
            windows: boolean;
            /**
             * Is running on a Windows Phone?
             */
            windowsPhone: boolean;
            /**
             * If running in iOS this will contain the major version number.
             */
            iOSVersion: number;
            /**
             * PixelRatio of the host device?
             */
            pixelRatio: number;
        };

        /**
         * Determines the video support of the browser running this Phaser Game instance.
         * These values are read-only and populated during the boot sequence of the game.
         * They are then referenced by internal game systems and are available for you to access
         * via `this.sys.game.device.video` from within any Scene.
         */
        type Video = {
            /**
             * Can this device play h264 mp4 video files?
             */
            h264Video: boolean;
            /**
             * Can this device play hls video files?
             */
            hlsVideo: boolean;
            /**
             * Can this device play h264 mp4 video files?
             */
            mp4Video: boolean;
            /**
             * Can this device play ogg video files?
             */
            oggVideo: boolean;
            /**
             * Can this device play vp9 video files?
             */
            vp9Video: boolean;
            /**
             * Can this device play webm video files?
             */
            webmVideo: boolean;
        };

    }

    type DeviceConf = {
        /**
         * The OS Device functions.
         */
        os: Phaser.Device.OS;
        /**
         * The Browser Device functions.
         */
        browser: Phaser.Device.Browser;
        /**
         * The Features Device functions.
         */
        features: Phaser.Device.Features;
        /**
         * The Input Device functions.
         */
        input: Phaser.Device.Input;
        /**
         * The Audio Device functions.
         */
        audio: Phaser.Device.Audio;
        /**
         * The Video Device functions.
         */
        video: Phaser.Device.Video;
        /**
         * The Fullscreen Device functions.
         */
        fullscreen: Phaser.Device.Fullscreen;
        /**
         * The Canvas Device functions.
         */
        canvasFeatures: Phaser.Device.CanvasFeatures;
    };

    namespace Display {
        namespace Align {
            /**
             * A constant representing a top-left alignment or position.
             */
            const TOP_LEFT: integer;

            /**
             * A constant representing a top-center alignment or position.
             */
            const TOP_CENTER: integer;

            /**
             * A constant representing a top-right alignment or position.
             */
            const TOP_RIGHT: integer;

            /**
             * A constant representing a left-top alignment or position.
             */
            const LEFT_TOP: integer;

            /**
             * A constant representing a left-center alignment or position.
             */
            const LEFT_CENTER: integer;

            /**
             * A constant representing a left-bottom alignment or position.
             */
            const LEFT_BOTTOM: integer;

            /**
             * A constant representing a center alignment or position.
             */
            const CENTER: integer;

            /**
             * A constant representing a right-top alignment or position.
             */
            const RIGHT_TOP: integer;

            /**
             * A constant representing a right-center alignment or position.
             */
            const RIGHT_CENTER: integer;

            /**
             * A constant representing a right-bottom alignment or position.
             */
            const RIGHT_BOTTOM: integer;

            /**
             * A constant representing a bottom-left alignment or position.
             */
            const BOTTOM_LEFT: integer;

            /**
             * A constant representing a bottom-center alignment or position.
             */
            const BOTTOM_CENTER: integer;

            /**
             * A constant representing a bottom-right alignment or position.
             */
            const BOTTOM_RIGHT: integer;

            namespace In {
                /**
                 * Takes given Game Object and aligns it so that it is positioned in the bottom center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the bottom left of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the bottom right of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function Center<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the left center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned relative to the other.
                 * The alignment used is based on the `position` argument, which is an `ALIGN_CONST` value, such as `LEFT_CENTER` or `TOP_RIGHT`.
                 * @param child The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param position The position to align the Game Object with. This is an align constant, such as `ALIGN_CONST.LEFT_CENTER`.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function QuickSet<G extends Phaser.GameObjects.GameObject>(child: G, alignIn: Phaser.GameObjects.GameObject, position: integer, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the right center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the top center of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the top left of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned in the top right of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignIn The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignIn: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

            }

            namespace To {
                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the bottom center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the bottom left position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the bottom right position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function BottomRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the left bottom position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftBottom<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the left center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the left top position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function LeftTop<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the right bottom position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightBottom<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the right center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the right top position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function RightTop<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the top center position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopCenter<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the top left position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

                /**
                 * Takes given Game Object and aligns it so that it is positioned next to the top right position of the other.
                 * @param gameObject The Game Object that will be positioned.
                 * @param alignTo The Game Object to base the alignment position on.
                 * @param offsetX Optional horizontal offset from the position. Default 0.
                 * @param offsetY Optional vertical offset from the position. Default 0.
                 */
                function TopRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, alignTo: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number): G;

            }

        }

        namespace Bounds {
            /**
             * Positions the Game Object so that it is centered on the given coordinates.
             * @param gameObject The Game Object that will be re-positioned.
             * @param x The horizontal coordinate to position the Game Object on.
             * @param y The vertical coordinate to position the Game Object on.
             */
            function CenterOn<G extends Phaser.GameObjects.GameObject>(gameObject: G, x: number, y: number): G;

            /**
             * Returns the bottom coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetBottom(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the center x coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetCenterX(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the center y coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetCenterY(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the left coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetLeft(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the amount the Game Object is visually offset from its x coordinate.
             * This is the same as `width * origin.x`.
             * This value will only be > 0 if `origin.x` is not equal to zero.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetOffsetX(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the amount the Game Object is visually offset from its y coordinate.
             * This is the same as `width * origin.y`.
             * This value will only be > 0 if `origin.y` is not equal to zero.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetOffsetY(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the right coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetRight(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Returns the top coordinate from the bounds of the Game Object.
             * @param gameObject The Game Object to get the bounds value from.
             */
            function GetTop(gameObject: Phaser.GameObjects.GameObject): number;

            /**
             * Positions the Game Object so that the bottom of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetBottom<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

            /**
             * Positions the Game Object so that the center top of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param x The coordinate to position the Game Object bounds on.
             */
            function SetCenterX<G extends Phaser.GameObjects.GameObject>(gameObject: G, x: number): G;

            /**
             * Positions the Game Object so that the center top of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param y The coordinate to position the Game Object bounds on.
             */
            function SetCenterY<G extends Phaser.GameObjects.GameObject>(gameObject: G, y: number): G;

            /**
             * Positions the Game Object so that the left of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetLeft<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

            /**
             * Positions the Game Object so that the left of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetRight<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

            /**
             * Positions the Game Object so that the top of its bounds aligns with the given coordinate.
             * @param gameObject The Game Object that will be re-positioned.
             * @param value The coordinate to position the Game Object bounds on.
             */
            function SetTop<G extends Phaser.GameObjects.GameObject>(gameObject: G, value: number): G;

        }

        namespace Canvas {
            namespace CanvasInterpolation {
                /**
                 * Sets the CSS image-rendering property on the given canvas to be 'crisp' (aka 'optimize contrast' on webkit).
                 * @param canvas The canvas object to have the style set on.
                 */
                function setCrisp(canvas: HTMLCanvasElement): HTMLCanvasElement;

                /**
                 * Sets the CSS image-rendering property on the given canvas to be 'bicubic' (aka 'auto').
                 * @param canvas The canvas object to have the style set on.
                 */
                function setBicubic(canvas: HTMLCanvasElement): HTMLCanvasElement;

            }

            /**
             * The CanvasPool is a global static object, that allows Phaser to recycle and pool 2D Context Canvas DOM elements.
             * It does not pool WebGL Contexts, because once the context options are set they cannot be modified again,
             * which is useless for some of the Phaser pipelines / renderer.
             *
             * This singleton is instantiated as soon as Phaser loads, before a Phaser.Game instance has even been created.
             * Which means all instances of Phaser Games on the same page can share the one single pool.
             */
            namespace CanvasPool {
                /**
                 * Creates a new Canvas DOM element, or pulls one from the pool if free.
                 * @param parent The parent of the Canvas object.
                 * @param width The width of the Canvas. Default 1.
                 * @param height The height of the Canvas. Default 1.
                 * @param canvasType The type of the Canvas. Either `Phaser.CANVAS` or `Phaser.WEBGL`. Default Phaser.CANVAS.
                 * @param selfParent Use the generated Canvas element as the parent? Default false.
                 */
                function create(parent: any, width?: integer, height?: integer, canvasType?: integer, selfParent?: boolean): HTMLCanvasElement;

                /**
                 * Creates a new Canvas DOM element, or pulls one from the pool if free.
                 * @param parent The parent of the Canvas object.
                 * @param width The width of the Canvas. Default 1.
                 * @param height The height of the Canvas. Default 1.
                 */
                function create2D(parent: any, width?: integer, height?: integer): HTMLCanvasElement;

                /**
                 * Creates a new Canvas DOM element, or pulls one from the pool if free.
                 * @param parent The parent of the Canvas object.
                 * @param width The width of the Canvas. Default 1.
                 * @param height The height of the Canvas. Default 1.
                 */
                function createWebGL(parent: any, width?: integer, height?: integer): HTMLCanvasElement;

                /**
                 * Gets the first free canvas index from the pool.
                 * @param canvasType The type of the Canvas. Either `Phaser.CANVAS` or `Phaser.WEBGL`. Default Phaser.CANVAS.
                 */
                function first(canvasType?: integer): HTMLCanvasElement;

                /**
                 * Looks up a canvas based on its parent, and if found puts it back in the pool, freeing it up for re-use.
                 * The canvas has its width and height set to 1, and its parent attribute nulled.
                 * @param parent [description]
                 */
                function remove(parent: any): void;

                /**
                 * Gets the total number of used canvas elements in the pool.
                 */
                function total(): integer;

                /**
                 * Gets the total number of free canvas elements in the pool.
                 */
                function free(): integer;

                /**
                 * Disable context smoothing on any new Canvas element created.
                 */
                function disableSmoothing(): void;

                /**
                 * Enable context smoothing on any new Canvas element created.
                 */
                function enableSmoothing(): void;

            }

            namespace Smoothing {
                /**
                 * Gets the Smoothing Enabled vendor prefix being used on the given context, or null if not set.
                 * @param context [description]
                 */
                function getPrefix(context: CanvasRenderingContext2D | WebGLRenderingContext): string;

                /**
                 * Sets the Image Smoothing property on the given context. Set to false to disable image smoothing.
                 * By default browsers have image smoothing enabled, which isn't always what you visually want, especially
                 * when using pixel art in a game. Note that this sets the property on the context itself, so that any image
                 * drawn to the context will be affected. This sets the property across all current browsers but support is
                 * patchy on earlier browsers, especially on mobile.
                 * @param context [description]
                 */
                function enable(context: CanvasRenderingContext2D | WebGLRenderingContext): CanvasRenderingContext2D | WebGLRenderingContext;

                /**
                 * Sets the Image Smoothing property on the given context. Set to false to disable image smoothing.
                 * By default browsers have image smoothing enabled, which isn't always what you visually want, especially
                 * when using pixel art in a game. Note that this sets the property on the context itself, so that any image
                 * drawn to the context will be affected. This sets the property across all current browsers but support is
                 * patchy on earlier browsers, especially on mobile.
                 * @param context [description]
                 */
                function disable(context: CanvasRenderingContext2D | WebGLRenderingContext): CanvasRenderingContext2D | WebGLRenderingContext;

                /**
                 * Returns `true` if the given context has image smoothing enabled, otherwise returns `false`.
                 * Returns null if no smoothing prefix is available.
                 * @param context [description]
                 */
                function isEnabled(context: CanvasRenderingContext2D | WebGLRenderingContext): boolean;

            }

            /**
             * Sets the touch-action property on the canvas style. Can be used to disable default browser touch actions.
             * @param canvas The canvas element to have the style applied to.
             * @param value The touch action value to set on the canvas. Set to `none` to disable touch actions. Default 'none'.
             */
            function TouchAction(canvas: HTMLCanvasElement, value?: string): HTMLCanvasElement;

            /**
             * Sets the user-select property on the canvas style. Can be used to disable default browser selection actions.
             * @param canvas The canvas element to have the style applied to.
             * @param value The touch callout value to set on the canvas. Set to `none` to disable touch callouts. Default 'none'.
             */
            function UserSelect(canvas: HTMLCanvasElement, value?: string): HTMLCanvasElement;

        }

        /**
         * The Color class holds a single color value and allows for easy modification and reading of it.
         */
        class Color {
            /**
             *
             * @param red The red color value. A number between 0 and 255. Default 0.
             * @param green The green color value. A number between 0 and 255. Default 0.
             * @param blue The blue color value. A number between 0 and 255. Default 0.
             * @param alpha The alpha value. A number between 0 and 255. Default 255.
             */
            constructor(red?: integer, green?: integer, blue?: integer, alpha?: integer);

            /**
             * An array containing the calculated color values for WebGL use.
             */
            gl: number[];

            /**
             * Sets this color to be transparent. Sets all values to zero.
             */
            transparent(): Phaser.Display.Color;

            /**
             * Sets the color of this Color component.
             * @param red The red color value. A number between 0 and 255.
             * @param green The green color value. A number between 0 and 255.
             * @param blue The blue color value. A number between 0 and 255.
             * @param alpha The alpha value. A number between 0 and 255. Default 255.
             * @param updateHSV Update the HSV values after setting the RGB values? Default true.
             */
            setTo(red: integer, green: integer, blue: integer, alpha?: integer, updateHSV?: boolean): Phaser.Display.Color;

            /**
             * Sets the red, green, blue and alpha GL values of this Color component.
             * @param red The red color value. A number between 0 and 1.
             * @param green The green color value. A number between 0 and 1.
             * @param blue The blue color value. A number between 0 and 1.
             * @param alpha The alpha value. A number between 0 and 1. Default 1.
             */
            setGLTo(red: number, green: number, blue: number, alpha?: number): Phaser.Display.Color;

            /**
             * Sets the color based on the color object given.
             * @param color An object containing `r`, `g`, `b` and optionally `a` values in the range 0 to 255.
             */
            setFromRGB(color: InputColorObject): Phaser.Display.Color;

            /**
             * Sets the color based on the hue, saturation and lightness values given.
             * @param h The hue, in the range 0 - 1. This is the base color.
             * @param s The saturation, in the range 0 - 1. This controls how much of the hue will be in the final color, where 1 is fully saturated and 0 will give you white.
             * @param v The value, in the range 0 - 1. This controls how dark the color is. Where 1 is as bright as possible and 0 is black.
             */
            setFromHSV(h: number, s: number, v: number): Phaser.Display.Color;

            /**
             * Returns a new Color component using the values from this one.
             */
            clone(): Phaser.Display.Color;

            /**
             * Sets this Color object to be grayscaled based on the shade value given.
             * @param shade A value between 0 and 255.
             */
            gray(shade: integer): Phaser.Display.Color;

            /**
             * Sets this Color object to be a random color between the `min` and `max` values given.
             * @param min The minimum random color value. Between 0 and 255. Default 0.
             * @param max The maximum random color value. Between 0 and 255. Default 255.
             */
            random(min?: integer, max?: integer): Phaser.Display.Color;

            /**
             * Sets this Color object to be a random grayscale color between the `min` and `max` values given.
             * @param min The minimum random color value. Between 0 and 255. Default 0.
             * @param max The maximum random color value. Between 0 and 255. Default 255.
             */
            randomGray(min?: integer, max?: integer): Phaser.Display.Color;

            /**
             * Increase the saturation of this Color by the percentage amount given.
             * The saturation is the amount of the base color in the hue.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            saturate(amount: integer): Phaser.Display.Color;

            /**
             * Decrease the saturation of this Color by the percentage amount given.
             * The saturation is the amount of the base color in the hue.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            desaturate(amount: integer): Phaser.Display.Color;

            /**
             * Increase the lightness of this Color by the percentage amount given.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            lighten(amount: integer): Phaser.Display.Color;

            /**
             * Decrease the lightness of this Color by the percentage amount given.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            darken(amount: integer): Phaser.Display.Color;

            /**
             * Brighten this Color by the percentage amount given.
             * @param amount The percentage amount to change this color by. A value between 0 and 100.
             */
            brighten(amount: integer): Phaser.Display.Color;

            /**
             * The color of this Color component, not including the alpha channel.
             */
            readonly color: number;

            /**
             * The color of this Color component, including the alpha channel.
             */
            readonly color32: number;

            /**
             * The color of this Color component as a string which can be used in CSS color values.
             */
            readonly rgba: string;

            /**
             * The red color value, normalized to the range 0 to 1.
             */
            redGL: number;

            /**
             * The green color value, normalized to the range 0 to 1.
             */
            greenGL: number;

            /**
             * The blue color value, normalized to the range 0 to 1.
             */
            blueGL: number;

            /**
             * The alpha color value, normalized to the range 0 to 1.
             */
            alphaGL: number;

            /**
             * The red color value, normalized to the range 0 to 255.
             */
            red: number;

            /**
             * The green color value, normalized to the range 0 to 255.
             */
            green: number;

            /**
             * The blue color value, normalized to the range 0 to 255.
             */
            blue: number;

            /**
             * The alpha color value, normalized to the range 0 to 255.
             */
            alpha: number;

            /**
             * The hue color value. A number between 0 and 1.
             * This is the base color.
             */
            h: number;

            /**
             * The saturation color value. A number between 0 and 1.
             * This controls how much of the hue will be in the final color, where 1 is fully saturated and 0 will give you white.
             */
            s: number;

            /**
             * The lightness color value. A number between 0 and 1.
             * This controls how dark the color is. Where 1 is as bright as possible and 0 is black.
             */
            v: number;

            /**
             * Converts the given color value into an Object containing r,g,b and a properties.
             * @param color A color value, optionally including the alpha value.
             */
            static ColorToRGBA(color: number): ColorObject;

            /**
             * Returns a string containing a hex representation of the given color component.
             * @param color The color channel to get the hex value for, must be a value between 0 and 255.
             */
            static ComponentToHex(color: integer): string;

            /**
             * Given 3 separate color values this will return an integer representation of it.
             * @param red The red color value. A number between 0 and 255.
             * @param green The green color value. A number between 0 and 255.
             * @param blue The blue color value. A number between 0 and 255.
             */
            static GetColor(red: integer, green: integer, blue: integer): number;

            /**
             * Given an alpha and 3 color values this will return an integer representation of it.
             * @param red The red color value. A number between 0 and 255.
             * @param green The green color value. A number between 0 and 255.
             * @param blue The blue color value. A number between 0 and 255.
             * @param alpha The alpha color value. A number between 0 and 255.
             */
            static GetColor32(red: integer, green: integer, blue: integer, alpha: integer): number;

            /**
             * Converts a hex string into a Phaser Color object.
             *
             * The hex string can supplied as `'#0033ff'` or the short-hand format of `'#03f'`; it can begin with an optional "#" or "0x", or be unprefixed.
             *
             * An alpha channel is _not_ supported.
             * @param hex The hex color value to convert, such as `#0033ff` or the short-hand format: `#03f`.
             */
            static HexStringToColor(hex: string): Phaser.Display.Color;

            /**
             * Converts HSL (hue, saturation and lightness) values to a Phaser Color object.
             * @param h The hue value in the range 0 to 1.
             * @param s The saturation value in the range 0 to 1.
             * @param l The lightness value in the range 0 to 1.
             */
            static HSLToColor(h: number, s: number, l: number): Phaser.Display.Color;

            /**
             * Get HSV color wheel values in an array which will be 360 elements in size.
             * @param s The saturation, in the range 0 - 1. Default 1.
             * @param v The value, in the range 0 - 1. Default 1.
             */
            static HSVColorWheel(s?: number, v?: number): ColorObject[];

            /**
             * Converts an HSV (hue, saturation and value) color value to RGB.
             * Conversion formula from http://en.wikipedia.org/wiki/HSL_color_space.
             * Assumes HSV values are contained in the set [0, 1].
             * Based on code by Michael Jackson (https://github.com/mjijackson)
             * @param h The hue, in the range 0 - 1. This is the base color.
             * @param s The saturation, in the range 0 - 1. This controls how much of the hue will be in the final color, where 1 is fully saturated and 0 will give you white.
             * @param v The value, in the range 0 - 1. This controls how dark the color is. Where 1 is as bright as possible and 0 is black.
             * @param out A Color object to store the results in. If not given a new ColorObject will be created.
             */
            static HSVToRGB(h: number, s: number, v: number, out?: ColorObject | Phaser.Display.Color): ColorObject | Phaser.Display.Color;

            /**
             * Converts a hue to an RGB color.
             * Based on code by Michael Jackson (https://github.com/mjijackson)
             */
            static HueToComponent(p: number, q: number, t: number): number;

            /**
             * Converts the given color value into an instance of a Color object.
             * @param input The color value to convert into a Color object.
             */
            static IntegerToColor(input: integer): Phaser.Display.Color;

            /**
             * Return the component parts of a color as an Object with the properties alpha, red, green, blue.
             *
             * Alpha will only be set if it exists in the given color (0xAARRGGBB)
             * @param input The color value to convert into a Color object.
             */
            static IntegerToRGB(input: integer): ColorObject;

            /**
             * Converts an object containing `r`, `g`, `b` and `a` properties into a Color class instance.
             * @param input An object containing `r`, `g`, `b` and `a` properties in the range 0 to 255.
             */
            static ObjectToColor(input: InputColorObject): Phaser.Display.Color;

            /**
             * Creates a new Color object where the r, g, and b values have been set to random values
             * based on the given min max values.
             * @param min The minimum value to set the random range from (between 0 and 255) Default 0.
             * @param max The maximum value to set the random range from (between 0 and 255) Default 255.
             */
            static RandomRGB(min?: integer, max?: integer): Phaser.Display.Color;

            /**
             * Converts a CSS 'web' string into a Phaser Color object.
             *
             * The web string can be in the format `'rgb(r,g,b)'` or `'rgba(r,g,b,a)'` where r/g/b are in the range [0..255] and a is in the range [0..1].
             * @param rgb The CSS format color string, using the `rgb` or `rgba` format.
             */
            static RGBStringToColor(rgb: string): Phaser.Display.Color;

            /**
             * Converts an RGB color value to HSV (hue, saturation and value).
             * Conversion forumla from http://en.wikipedia.org/wiki/HSL_color_space.
             * Assumes RGB values are contained in the set [0, 255] and returns h, s and v in the set [0, 1].
             * Based on code by Michael Jackson (https://github.com/mjijackson)
             * @param r The red color value. A number between 0 and 255.
             * @param g The green color value. A number between 0 and 255.
             * @param b The blue color value. A number between 0 and 255.
             * @param out An object to store the color values in. If not given an HSV Color Object will be created.
             */
            static RGBToHSV(r: integer, g: integer, b: integer, out?: HSVColorObject | Phaser.Display.Color): HSVColorObject | Phaser.Display.Color;

            /**
             * Converts the color values into an HTML compatible color string, prefixed with either `#` or `0x`.
             * @param r The red color value. A number between 0 and 255.
             * @param g The green color value. A number between 0 and 255.
             * @param b The blue color value. A number between 0 and 255.
             * @param a The alpha value. A number between 0 and 255. Default 255.
             * @param prefix The prefix of the string. Either `#` or `0x`. Default #.
             */
            static RGBToString(r: integer, g: integer, b: integer, a?: integer, prefix?: string): string;

            /**
             * Converts the given source color value into an instance of a Color class.
             * The value can be either a string, prefixed with `rgb` or a hex string, a number or an Object.
             * @param input The source color value to convert.
             */
            static ValueToColor(input: string | number | InputColorObject): Phaser.Display.Color;

        }

        namespace Masks {
            /**
             * [description]
             */
            class BitmapMask {
                /**
                 *
                 * @param scene [description]
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                constructor(scene: Phaser.Scene, renderable: Phaser.GameObjects.GameObject);

                /**
                 * A reference to either the Canvas or WebGL Renderer that this Mask is using.
                 */
                renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer;

                /**
                 * A renderable Game Object that uses a texture, such as a Sprite.
                 */
                bitmapMask: Phaser.GameObjects.GameObject;

                /**
                 * [description]
                 */
                maskTexture: WebGLTexture;

                /**
                 * [description]
                 */
                mainTexture: WebGLTexture;

                /**
                 * [description]
                 */
                dirty: boolean;

                /**
                 * [description]
                 */
                mainFramebuffer: WebGLFramebuffer;

                /**
                 * [description]
                 */
                maskFramebuffer: WebGLFramebuffer;

                /**
                 * [description]
                 */
                invertAlpha: boolean;

                /**
                 * [description]
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                setBitmap(renderable: Phaser.GameObjects.GameObject): void;

                /**
                 * [description]
                 * @param renderer [description]
                 * @param maskedObject [description]
                 * @param camera The Camera to render to.
                 */
                preRenderWebGL(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer, maskedObject: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * [description]
                 * @param renderer [description]
                 */
                postRenderWebGL(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer): void;

                /**
                 * [description]
                 * @param renderer [description]
                 * @param mask [description]
                 * @param camera The Camera to render to.
                 */
                preRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer, mask: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * [description]
                 * @param renderer [description]
                 */
                postRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer | Phaser.Renderer.WebGL.WebGLRenderer): void;

                /**
                 * Destroys this BitmapMask and nulls any references it holds.
                 *
                 * Note that if a Game Object is currently using this mask it will _not_ automatically detect you have destroyed it,
                 * so be sure to call `clearMask` on any Game Object using it, before destroying it.
                 */
                destroy(): void;

            }

            /**
             * A Geometry Mask can be applied to a Game Object to hide any pixels of it which don't intersect a visible pixel from the geometry mask. The mask is essentially a clipping path which can only make a masked pixel fully visible or fully invisible without changing its alpha (opacity).
             *
             * A Geometry Mask uses a Graphics Game Object to determine which pixels of the masked Game Object(s) should be clipped. For any given point of a masked Game Object's texture, the pixel will only be displayed if the Graphics Game Object of the Geometry Mask has a visible pixel at the same position. The color and alpha of the pixel from the Geometry Mask do not matter.
             *
             * The Geometry Mask's location matches the location of its Graphics object, not the location of the masked objects. Moving or transforming the underlying Graphics object will change the mask (and affect the visibility of any masked objects), whereas moving or transforming a masked object will not affect the mask. You can think of the Geometry Mask (or rather, of the its Graphics object) as an invisible curtain placed in front of all masked objects which has its own visual properties and, naturally, respects the camera's visual properties, but isn't affected by and doesn't follow the masked objects by itself.
             */
            class GeometryMask {
                /**
                 *
                 * @param scene This parameter is not used.
                 * @param graphicsGeometry The Graphics Game Object to use for the Geometry Mask. Doesn't have to be in the Display List.
                 */
                constructor(scene: Phaser.Scene, graphicsGeometry: Phaser.GameObjects.Graphics);

                /**
                 * The Graphics object which describes the Geometry Mask.
                 */
                geometryMask: Phaser.GameObjects.Graphics;

                /**
                 * Sets a new Graphics object for the Geometry Mask.
                 * @param graphicsGeometry The Graphics object which will be used for the Geometry Mask.
                 */
                setShape(graphicsGeometry: Phaser.GameObjects.Graphics): void;

                /**
                 * Renders the Geometry Mask's underlying Graphics object to the OpenGL stencil buffer and enables the stencil test, which clips rendered pixels according to the mask.
                 * @param renderer The WebGL Renderer instance to draw to.
                 * @param mask The Game Object being rendered.
                 * @param camera The camera the Game Object is being rendered through.
                 */
                preRenderWebGL(renderer: Phaser.Renderer.WebGL.WebGLRenderer, mask: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * Flushes all rendered pixels and disables the stencil test of a WebGL context, thus disabling the mask for it.
                 * @param renderer The WebGL Renderer instance to draw flush.
                 */
                postRenderWebGL(renderer: Phaser.Renderer.WebGL.WebGLRenderer): void;

                /**
                 * Sets the clipping path of a 2D canvas context to the Geometry Mask's underlying Graphics object.
                 * @param renderer The Canvas Renderer instance to set the clipping path on.
                 * @param mask The Game Object being rendered.
                 * @param camera The camera the Game Object is being rendered through.
                 */
                preRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer, mask: Phaser.GameObjects.GameObject, camera: Phaser.Cameras.Scene2D.Camera): void;

                /**
                 * Restore the canvas context's previous clipping path, thus turning off the mask for it.
                 * @param renderer The Canvas Renderer instance being restored.
                 */
                postRenderCanvas(renderer: Phaser.Renderer.Canvas.CanvasRenderer): void;

                /**
                 * Destroys this GeometryMask and nulls any references it holds.
                 *
                 * Note that if a Game Object is currently using this mask it will _not_ automatically detect you have destroyed it,
                 * so be sure to call `clearMask` on any Game Object using it, before destroying it.
                 */
                destroy(): void;

            }

        }

    }

    namespace DOM {
        /**
         * Adds the given element to the DOM. If a parent is provided the element is added as a child of the parent, providing it was able to access it.
         * If no parent was given or falls back to using `document.body`.
         * @param element The element to be added to the DOM. Usually a Canvas object.
         * @param parent The parent in which to add the element. Can be a string which is passed to `getElementById` or an actual DOM object.
         * @param overflowHidden Whether or not to hide overflowing content inside the parent. Default true.
         */
        function AddToDOM(element: HTMLElement, parent?: string | HTMLElement, overflowHidden?: boolean): HTMLElement;

        /**
         * Inspects the readyState of the document. If the document is already complete then it invokes the given callback.
         * If not complete it sets up several event listeners such as `deviceready`, and once those fire, it invokes the callback.
         * Called automatically by the Phaser.Game instance. Should not usually be accessed directly.
         * @param callback The callback to be invoked when the device is ready and the DOM content is loaded.
         */
        function DOMContentLoaded(callback: ContentLoadedCallback): void;

        /**
         * Takes the given data string and parses it as XML.
         * First tries to use the window.DOMParser and reverts to the Microsoft.XMLDOM if that fails.
         * The parsed XML object is returned, or `null` if there was an error while parsing the data.
         * @param data The XML source stored in a string.
         */
        function ParseXML(data: string): DOMParser | ActiveXObject;

        /**
         * Attempts to remove the element from its parentNode in the DOM.
         * @param element The DOM element to remove from its parent node.
         */
        function RemoveFromDOM(element: HTMLElement): void;

        /**
         * Abstracts away the use of RAF or setTimeOut for the core game update loop.
         * This is invoked automatically by the Phaser.Game instance.
         */
        class RequestAnimationFrame {
            /**
             * True if RequestAnimationFrame is running, otherwise false.
             */
            isRunning: boolean;

            /**
             * The callback to be invoked each step.
             */
            callback: FrameRequestCallback;

            /**
             * The most recent timestamp. Either a DOMHighResTimeStamp under RAF or `Date.now` under SetTimeout.
             */
            tick: number;

            /**
             * True if the step is using setTimeout instead of RAF.
             */
            isSetTimeOut: boolean;

            /**
             * The setTimeout or RAF callback ID used when canceling them.
             */
            timeOutID: number;

            /**
             * The previous time the step was called.
             */
            lastTime: number;

            /**
             * The RAF step function.
             * Updates the local tick value, invokes the callback and schedules another call to requestAnimationFrame.
             */
            step: FrameRequestCallback;

            /**
             * The SetTimeout step function.
             * Updates the local tick value, invokes the callback and schedules another call to setTimeout.
             */
            stepTimeout: Function;

            /**
             * Starts the requestAnimationFrame or setTimeout process running.
             * @param callback The callback to invoke each step.
             * @param forceSetTimeOut Should it use SetTimeout, even if RAF is available?
             */
            start(callback: FrameRequestCallback, forceSetTimeOut: boolean): void;

            /**
             * Stops the requestAnimationFrame or setTimeout from running.
             */
            stop(): void;

            /**
             * Stops the step from running and clears the callback reference.
             */
            destroy(): void;

        }

    }

    namespace Events {
        /**
         * EventEmitter is a Scene Systems plugin compatible version of eventemitter3.
         */
        class EventEmitter {
            /**
             * Removes all listeners.
             */
            shutdown(): void;

            /**
             * Removes all listeners.
             */
            destroy(): void;

            /**
             * Return an array listing the events for which the emitter has registered listeners.
             */
            eventNames(): any[];

            /**
             * Return the listeners registered for a given event.
             * @param event The event name.
             */
            listeners(event: string | symbol): any[];

            /**
             * Return the number of listeners listening to a given event.
             * @param event The event name.
             */
            listenerCount(event: string | symbol): number;

            /**
             * Calls each of the listeners registered for a given event.
             * @param event The event name.
             * @param args Additional arguments that will be passed to the event handler.
             */
            emit(event: string | symbol, ...args: any[]): boolean;

            /**
             * Add a listener for a given event.
             * @param event The event name.
             * @param fn The listener function.
             * @param context The context to invoke the listener with. Default this.
             */
            on(event: string | symbol, fn: Function, context?: any): Phaser.Events.EventEmitter;

            /**
             * Add a listener for a given event.
             * @param event The event name.
             * @param fn The listener function.
             * @param context The context to invoke the listener with. Default this.
             */
            addListener(event: string | symbol, fn: Function, context?: any): Phaser.Events.EventEmitter;

            /**
             * Add a one-time listener for a given event.
             * @param event The event name.
             * @param fn The listener function.
             * @param context The context to invoke the listener with. Default this.
             */
            once(event: string | symbol, fn: Function, context?: any): Phaser.Events.EventEmitter;

            /**
             * Remove the listeners of a given event.
             * @param event The event name.
             * @param fn Only remove the listeners that match this function.
             * @param context Only remove the listeners that have this context.
             * @param once Only remove one-time listeners.
             */
            removeListener(event: string | symbol, fn: Function, context: any, once: boolean): Phaser.Events.EventEmitter;

            /**
             * Remove the listeners of a given event.
             * @param event The event name.
             * @param fn Only remove the listeners that match this function.
             * @param context Only remove the listeners that have this context.
             * @param once Only remove one-time listeners.
             */
            off(event: string | symbol, fn: Function, context: any, once: boolean): Phaser.Events.EventEmitter;

            /**
             * Remove all listeners, or those of the specified event.
             * @param event The event name.
             */
            removeAllListeners(event?: string | symbol): Phaser.Events.EventEmitter;

        }

    }

    /**
     * Phaser Blend Modes to CSS Blend Modes Map.
     */
    enum CSSBlendModes {
    }

    namespace GameObjects {
        /**
         * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
         *
         * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
         * match the font structure.
         *
         * Dynamic Bitmap Text objects are different from Static Bitmap Text in that they invoke a callback for each
         * letter being rendered during the render pass. This callback allows you to manipulate the properties of
         * each letter being rendered, such as its position, scale or tint, allowing you to create interesting effects
         * like jiggling text, which can't be done with Static text. This means that Dynamic Text takes more processing
         * time, so only use them if you require the callback ability they have.
         *
         * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
         * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
         * processing the font texture in an image editor, applying fills and any other effects required.
         *
         * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
         *
         * To create a BitmapText data files you need a 3rd party app such as:
         *
         * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
         * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
         * Littera (Web-based, free): http://kvazars.com/littera/
         *
         * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
         * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
         */
        class DynamicBitmapText extends Phaser.GameObjects.BitmapText {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. It can only belong to one Scene at any given time.
             * @param x The x coordinate of this Game Object in world space.
             * @param y The y coordinate of this Game Object in world space.
             * @param font The key of the font to use from the Bitmap Font cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size of this Bitmap Text.
             * @param align The alignment of the text in a multi-line BitmapText object. Default 0.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, font: string, text?: string | string[], size?: number, align?: integer);

            /**
             * The horizontal scroll position of the Bitmap Text.
             */
            scrollX: number;

            /**
             * The vertical scroll position of the Bitmap Text.
             */
            scrollY: number;

            /**
             * The crop width of the Bitmap Text.
             */
            cropWidth: number;

            /**
             * The crop height of the Bitmap Text.
             */
            cropHeight: number;

            /**
             * A callback that alters how each character of the Bitmap Text is rendered.
             */
            displayCallback: DisplayCallback;

            /**
             * The data object that is populated during rendering, then passed to the displayCallback.
             * You should modify this object then return it back from the callback. It's updated values
             * will be used to render the specific glyph.
             *
             * Please note that if you need a reference to this object locally in your game code then you
             * should shallow copy it, as it's updated and re-used for every glyph in the text.
             */
            callbackData: DisplayCallbackConfig;

            /**
             * Set the crop size of this Bitmap Text.
             * @param width The width of the crop.
             * @param height The height of the crop.
             */
            setSize(width: number, height: number): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Set a callback that alters how each character of the Bitmap Text is rendered.
             *
             * The callback receives a {@link DisplayCallbackConfig} object that contains information about the character that's
             * about to be rendered.
             *
             * It should return an object with `x`, `y`, `scale` and `rotation` properties that will be used instead of the
             * usual values when rendering.
             * @param callback The display callback to set.
             */
            setDisplayCallback(callback: DisplayCallback): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Set the horizontal scroll position of this Bitmap Text.
             * @param value The horizontal scroll position to set.
             */
            setScrollX(value: number): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Set the vertical scroll position of this Bitmap Text.
             * @param value The vertical scroll position to set.
             */
            setScrollY(value: number): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             *
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             */
            scaleMode: Phaser.ScaleModes;

            /**
             * Sets the Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             * @param value The Scale Mode to be used by this Game Object.
             */
            setScaleMode(value: Phaser.ScaleModes): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string, frame?: string | integer): this;

            /**
             * Sets the frame this Game Object will use to render with.
             *
             * The Frame has to belong to the current Texture being used.
             *
             * It can be either a string or an index.
             *
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * Fill or additive?
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             *
             * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
             * which results in no visible change to the texture.
             */
            clearTint(): this;

            /**
             * Sets an additive tint on this Game Object.
             *
             * The tint works by taking the pixel color values from the Game Objects texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole Game Object will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the Game Object.
             *
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             *
             * To remove a tint call `clearTint`.
             *
             * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTint(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

            /**
             * Sets a fill-based tint on this Game Object.
             *
             * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
             * with those in the tint. You can use this for effects such as making a player flash 'white'
             * if hit by something. You can provide either one color value, in which case the whole
             * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
             * are blended together across the extent of the Game Object.
             *
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             *
             * To remove a tint call `clearTint`.
             *
             * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTintFill(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

            /**
             * The tint value being applied to the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintTopLeft: integer;

            /**
             * The tint value being applied to the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintTopRight: integer;

            /**
             * The tint value being applied to the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintBottomLeft: integer;

            /**
             * The tint value being applied to the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintBottomRight: integer;

            /**
             * The tint value being applied to the whole of the Game Object.
             */
            tint: integer;

            /**
             * Does this Game Object have a tint applied to it or not?
             */
            readonly isTinted: boolean;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        namespace RetroFont {
            /**
             * Text Set 1 =  !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~
             */
            var TEXT_SET1: string;

            /**
             * Text Set 2 =  !"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ
             */
            var TEXT_SET2: string;

            /**
             * Text Set 3 = ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789
             */
            var TEXT_SET3: string;

            /**
             * Text Set 4 = ABCDEFGHIJKLMNOPQRSTUVWXYZ 0123456789
             */
            var TEXT_SET4: string;

            /**
             * Text Set 5 = ABCDEFGHIJKLMNOPQRSTUVWXYZ.,/() '!?-*:0123456789
             */
            var TEXT_SET5: string;

            /**
             * Text Set 6 = ABCDEFGHIJKLMNOPQRSTUVWXYZ!?:;0123456789"(),-.'
             */
            var TEXT_SET6: string;

            /**
             * Text Set 7 = AGMSY+:4BHNTZ!;5CIOU.?06DJPV,(17EKQW")28FLRX-'39
             */
            var TEXT_SET7: string;

            /**
             * Text Set 8 = 0123456789 .ABCDEFGHIJKLMNOPQRSTUVWXYZ
             */
            var TEXT_SET8: string;

            /**
             * Text Set 9 = ABCDEFGHIJKLMNOPQRSTUVWXYZ()-0123456789.:,'"?!
             */
            var TEXT_SET9: string;

            /**
             * Text Set 10 = ABCDEFGHIJKLMNOPQRSTUVWXYZ
             */
            var TEXT_SET10: string;

            /**
             * Text Set 11 = ABCDEFGHIJKLMNOPQRSTUVWXYZ.,"-+!?()':;0123456789
             */
            var TEXT_SET11: string;

            /**
             * Parses a Retro Font configuration object so you can pass it to the BitmapText constructor
             * and create a BitmapText object using a fixed-width retro font.
             * @param scene A reference to the Phaser Scene.
             * @param config The font configuration object.
             */
            function Parse(scene: Phaser.Scene, config: Phaser.GameObjects.RetroFont.Config): object;

            type Config = {
                /**
                 * The key of the image containing the font.
                 */
                image: string;
                /**
                 * If the font set doesn't start at the top left of the given image, specify the X coordinate offset here.
                 */
                "offset.x": number;
                /**
                 * If the font set doesn't start at the top left of the given image, specify the Y coordinate offset here.
                 */
                "offset.y": number;
                /**
                 * The width of each character in the font set.
                 */
                width: number;
                /**
                 * The height of each character in the font set.
                 */
                height: number;
                /**
                 * The characters used in the font set, in display order. You can use the TEXT_SET consts for common font set arrangements.
                 */
                chars: string;
                /**
                 * The number of characters per row in the font set. If not given charsPerRow will be the image width / characterWidth.
                 */
                charsPerRow: number;
                /**
                 * If the characters in the font set have horizontal spacing between them set the required amount here.
                 */
                "spacing.x": number;
                /**
                 * If the characters in the font set have vertical spacing between them set the required amount here.
                 */
                "spacing.y": number;
                /**
                 * The amount of vertical space to add to the line height of the font.
                 */
                lineSpacing: number;
            };

        }

        /**
         * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
         *
         * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
         * match the font structure.
         *
         * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
         * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
         * processing the font texture in an image editor, applying fills and any other effects required.
         *
         * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
         *
         * To create a BitmapText data files you need a 3rd party app such as:
         *
         * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
         * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
         * Littera (Web-based, free): http://kvazars.com/littera/
         *
         * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
         * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
         */
        class BitmapText extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.ScaleMode, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Texture, Phaser.GameObjects.Components.Tint, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. It can only belong to one Scene at any given time.
             * @param x The x coordinate of this Game Object in world space.
             * @param y The y coordinate of this Game Object in world space.
             * @param font The key of the font to use from the Bitmap Font cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size of this Bitmap Text.
             * @param align The alignment of the text in a multi-line BitmapText object. Default 0.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, font: string, text?: string | string[], size?: number, align?: integer);

            /**
             * The key of the Bitmap Font used by this Bitmap Text.
             * To change the font after creation please use `setFont`.
             */
            readonly font: string;

            /**
             * The data of the Bitmap Font used by this Bitmap Text.
             */
            readonly fontData: BitmapFontData;

            /**
             * Set the lines of text in this BitmapText to be left-aligned.
             * This only has any effect if this BitmapText contains more than one line of text.
             */
            setLeftAlign(): this;

            /**
             * Set the lines of text in this BitmapText to be center-aligned.
             * This only has any effect if this BitmapText contains more than one line of text.
             */
            setCenterAlign(): this;

            /**
             * Set the lines of text in this BitmapText to be right-aligned.
             * This only has any effect if this BitmapText contains more than one line of text.
             */
            setRightAlign(): this;

            /**
             * Set the font size of this Bitmap Text.
             * @param size The font size to set.
             */
            setFontSize(size: number): this;

            /**
             * Sets the letter spacing between each character of this Bitmap Text.
             * Can be a positive value to increase the space, or negative to reduce it.
             * Spacing is applied after the kerning values have been set.
             * @param spacing The amount of horizontal space to add between each character. Default 0.
             */
            setLetterSpacing(spacing?: number): this;

            /**
             * Set the textual content of this BitmapText.
             *
             * An array of strings will be converted into multi-line text. Use the align methods to change multi-line alignment.
             * @param value The string, or array of strings, to be set as the content of this BitmapText.
             */
            setText(value: string | string[]): this;

            /**
             * Calculate the bounds of this Bitmap Text.
             *
             * An object is returned that contains the position, width and height of the Bitmap Text in local and global
             * contexts.
             *
             * Local size is based on just the font size and a [0, 0] position.
             *
             * Global size takes into account the Game Object's scale, world position and display origin.
             *
             * Also in the object is data regarding the length of each line, should this be a multi-line BitmapText.
             * @param round Whether to round the results to the nearest integer.
             */
            getTextBounds(round?: boolean): BitmapTextSize;

            /**
             * Changes the font this BitmapText is using to render.
             *
             * The new texture is loaded and applied to the BitmapText. The existing test, size and alignment are preserved,
             * unless overridden via the arguments.
             * @param font The key of the font to use from the Bitmap Font cache.
             * @param size The font size of this Bitmap Text. If not specified the current size will be used.
             * @param align The alignment of the text in a multi-line BitmapText object. If not specified the current alignment will be used. Default 0.
             */
            setFont(font: string, size?: number, align?: integer): this;

            /**
             * Controls the alignment of each line of text in this BitmapText object.
             *
             * Only has any effect when this BitmapText contains multiple lines of text, split with carriage-returns.
             * Has no effect with single-lines of text.
             *
             * See the methods `setLeftAlign`, `setCenterAlign` and `setRightAlign`.
             *
             * 0 = Left aligned (default)
             * 1 = Middle aligned
             * 2 = Right aligned
             *
             * The alignment position is based on the longest line of text.
             */
            align: integer;

            /**
             * The text that this Bitmap Text object displays.
             *
             * You can also use the method `setText` if you want a chainable way to change the text content.
             */
            text: string;

            /**
             * The font size of this Bitmap Text.
             *
             * You can also use the method `setFontSize` if you want a chainable way to change the font size.
             */
            fontSize: number;

            /**
             * Adds / Removes spacing between characters.
             *
             * Can be a negative or positive number.
             *
             * You can also use the method `setLetterSpacing` if you want a chainable way to change the letter spacing.
             */
            letterSpacing: number;

            /**
             * The width of this Bitmap Text.
             */
            readonly width: number;

            /**
             * The height of this bitmap text.
             */
            readonly height: number;

            /**
             * Build a JSON representation of this Bitmap Text.
             */
            toJSON(): JSONBitmapText;

            /**
             * Left align the text characters in a multi-line BitmapText object.
             */
            static ALIGN_LEFT: integer;

            /**
             * Center align the text characters in a multi-line BitmapText object.
             */
            static ALIGN_CENTER: integer;

            /**
             * Right align the text characters in a multi-line BitmapText object.
             */
            static ALIGN_RIGHT: integer;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             *
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             */
            scaleMode: Phaser.ScaleModes;

            /**
             * Sets the Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             * @param value The Scale Mode to be used by this Game Object.
             */
            setScaleMode(value: Phaser.ScaleModes): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string, frame?: string | integer): this;

            /**
             * Sets the frame this Game Object will use to render with.
             *
             * The Frame has to belong to the current Texture being used.
             *
             * It can be either a string or an index.
             *
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * Fill or additive?
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             *
             * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
             * which results in no visible change to the texture.
             */
            clearTint(): this;

            /**
             * Sets an additive tint on this Game Object.
             *
             * The tint works by taking the pixel color values from the Game Objects texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole Game Object will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the Game Object.
             *
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             *
             * To remove a tint call `clearTint`.
             *
             * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTint(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

            /**
             * Sets a fill-based tint on this Game Object.
             *
             * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
             * with those in the tint. You can use this for effects such as making a player flash 'white'
             * if hit by something. You can provide either one color value, in which case the whole
             * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
             * are blended together across the extent of the Game Object.
             *
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             *
             * To remove a tint call `clearTint`.
             *
             * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTintFill(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

            /**
             * The tint value being applied to the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintTopLeft: integer;

            /**
             * The tint value being applied to the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintTopRight: integer;

            /**
             * The tint value being applied to the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintBottomLeft: integer;

            /**
             * The tint value being applied to the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintBottomRight: integer;

            /**
             * The tint value being applied to the whole of the Game Object.
             */
            tint: integer;

            /**
             * Does this Game Object have a tint applied to it or not?
             */
            readonly isTinted: boolean;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * A Blitter Game Object.
         *
         * The Blitter Game Object is a special kind of container that creates, updates and manages Bob objects.
         * Bobs are designed for rendering speed rather than flexibility. They consist of a texture, or frame from a texture,
         * a position and an alpha value. You cannot scale or rotate them. They use a batched drawing method for speed
         * during rendering.
         *
         * A Blitter Game Object has one texture bound to it. Bobs created by the Blitter can use any Frame from this
         * Texture to render with, but they cannot use any other Texture. It is this single texture-bind that allows
         * them their speed.
         *
         * If you have a need to blast a large volume of frames around the screen then Blitter objects are well worth
         * investigating. They are especially useful for using as a base for your own special effects systems.
         */
        class Blitter extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.ScaleMode, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Size, Phaser.GameObjects.Components.Texture, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. It can only belong to one Scene at any given time.
             * @param x The x coordinate of this Game Object in world space. Default 0.
             * @param y The y coordinate of this Game Object in world space. Default 0.
             * @param texture The key of the texture this Game Object will use for rendering. The Texture must already exist in the Texture Manager. Default '__DEFAULT'.
             * @param frame The Frame of the Texture that this Game Object will use. Only set if the Texture has multiple frames, such as a Texture Atlas or Sprite Sheet. Default 0.
             */
            constructor(scene: Phaser.Scene, x?: number, y?: number, texture?: string, frame?: string | integer);

            /**
             * The children of this Blitter.
             * This List contains all of the Bob objects created by the Blitter.
             */
            children: Phaser.Structs.List<Phaser.GameObjects.Blitter.Bob>;

            /**
             * Is the Blitter considered dirty?
             * A 'dirty' Blitter has had its child count changed since the last frame.
             */
            dirty: boolean;

            /**
             * Creates a new Bob in this Blitter.
             *
             * The Bob is created at the given coordinates, relative to the Blitter and uses the given frame.
             * A Bob can use any frame belonging to the texture bound to the Blitter.
             * @param x The x position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param y The y position of the Bob. Bob coordinate are relative to the position of the Blitter object.
             * @param frame The Frame the Bob will use. It _must_ be part of the Texture the parent Blitter object is using.
             * @param visible Should the created Bob render or not? Default true.
             * @param index The position in the Blitters Display List to add the new Bob at. Defaults to the top of the list.
             */
            create(x: number, y: number, frame?: string | integer | Phaser.Textures.Frame, visible?: boolean, index?: integer): Phaser.GameObjects.Blitter.Bob;

            /**
             * Creates multiple Bob objects within this Blitter and then passes each of them to the specified callback.
             * @param callback The callback to invoke after creating a bob. It will be sent two arguments: The Bob and the index of the Bob.
             * @param quantity The quantity of Bob objects to create.
             * @param frame The Frame the Bobs will use. It must be part of the Blitter Texture.
             * @param visible Should the created Bob render or not? Default true.
             */
            createFromCallback(callback: Phaser.GameObjects.Blitter.CreateCallback, quantity: integer, frame?: string | integer | Phaser.Textures.Frame | string[] | integer[] | Phaser.Textures.Frame[], visible?: boolean): Phaser.GameObjects.Blitter.Bob[];

            /**
             * Creates multiple Bobs in one call.
             *
             * The amount created is controlled by a combination of the `quantity` argument and the number of frames provided.
             *
             * If the quantity is set to 10 and you provide 2 frames, then 20 Bobs will be created. 10 with the first
             * frame and 10 with the second.
             * @param quantity The quantity of Bob objects to create.
             * @param frame The Frame the Bobs will use. It must be part of the Blitter Texture.
             * @param visible Should the created Bob render or not? Default true.
             */
            createMultiple(quantity: integer, frame?: string | integer | Phaser.Textures.Frame | string[] | integer[] | Phaser.Textures.Frame[], visible?: boolean): Phaser.GameObjects.Blitter.Bob[];

            /**
             * Checks if the given child can render or not, by checking its `visible` and `alpha` values.
             * @param child The Bob to check for rendering.
             */
            childCanRender(child: Phaser.GameObjects.Blitter.Bob): boolean;

            /**
             * Returns an array of Bobs to be rendered.
             * If the Blitter is dirty then a new list is generated and stored in `renderList`.
             */
            getRenderList(): Phaser.GameObjects.Blitter.Bob[];

            /**
             * Removes all Bobs from the children List and clears the dirty flag.
             */
            clear(): void;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             */
            scaleMode: Phaser.ScaleModes;

            /**
             * Sets the Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             * @param value The Scale Mode to be used by this Game Object.
             */
            setScaleMode(value: Phaser.ScaleModes): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The native (un-scaled) width of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the size of this Game Object to be that of the given Frame.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param frame The frame to base the size of this Game Object on.
             */
            setSizeToFrame(frame: Phaser.Textures.Frame): this;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             *
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string, frame?: string | integer): this;

            /**
             * Sets the frame this Game Object will use to render with.
             *
             * The Frame has to belong to the current Texture being used.
             *
             * It can be either a string or an index.
             *
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * Builds a Game Object using the provided configuration object.
         * @param scene A reference to the Scene.
         * @param gameObject The initial GameObject.
         * @param config The config to build the GameObject with.
         */
        function BuildGameObject(scene: Phaser.Scene, gameObject: Phaser.GameObjects.GameObject, config: GameObjectConfig): Phaser.GameObjects.GameObject;

        /**
         * Adds an Animation component to a Sprite and populates it based on the given config.
         * @param sprite The sprite to add an Animation component to.
         * @param config The animation config.
         */
        function BuildGameObjectAnimation(sprite: Phaser.GameObjects.Sprite, config: object): Phaser.GameObjects.Sprite;

        namespace Components {
            /**
             * Provides methods used for setting the alpha properties of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Alpha {
                /**
                 * Clears all alpha values associated with this Game Object.
                 *
                 * Immediately sets the alpha levels back to 1 (fully opaque).
                 */
                clearAlpha(): this;
                /**
                 * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
                 * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
                 *
                 * If your game is running under WebGL you can optionally specify four different alpha values, each of which
                 * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
                 * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
                 * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
                 * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
                 * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
                 */
                setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;
                /**
                 * The alpha value of the Game Object.
                 *
                 * This is a global value, impacting the entire Game Object, not just a region of it.
                 */
                alpha: number;
                /**
                 * The alpha value starting from the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopLeft: number;
                /**
                 * The alpha value starting from the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaTopRight: number;
                /**
                 * The alpha value starting from the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomLeft: number;
                /**
                 * The alpha value starting from the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                alphaBottomRight: number;
            }

            interface Animation {
                /**
                 * The Game Object to which this animation controller belongs.
                 */
                parent: Phaser.GameObjects.GameObject;
                /**
                 * A reference to the global Animation Manager.
                 */
                animationManager: Phaser.Animations.AnimationManager;
                /**
                 * Is an animation currently playing or not?
                 */
                isPlaying: boolean;
                /**
                 * The current Animation loaded into this Animation Controller.
                 */
                currentAnim: Phaser.Animations.Animation;
                /**
                 * The current AnimationFrame being displayed by this Animation Controller.
                 */
                currentFrame: Phaser.Animations.AnimationFrame;
                /**
                 * The frame rate of playback in frames per second.
                 * The default is 24 if the `duration` property is `null`.
                 */
                frameRate: number;
                /**
                 * How long the animation should play for, in milliseconds.
                 * If the `frameRate` property has been set then it overrides this value,
                 * otherwise the `frameRate` is derived from `duration`.
                 */
                duration: number;
                /**
                 * ms per frame, not including frame specific modifiers that may be present in the Animation data.
                 */
                msPerFrame: number;
                /**
                 * Skip frames if the time lags, or always advanced anyway?
                 */
                skipMissedFrames: boolean;
                /**
                 * Will the playhead move forwards (`true`) or in reverse (`false`).
                 */
                forward: boolean;
                /**
                 * Internal time overflow accumulator.
                 */
                accumulator: number;
                /**
                 * The time point at which the next animation frame will change.
                 */
                nextTick: number;
                /**
                 * An internal counter keeping track of how many repeats are left to play.
                 */
                repeatCounter: number;
                /**
                 * An internal flag keeping track of pending repeats.
                 */
                pendingRepeat: boolean;
                /**
                 * Sets the amount of time, in milliseconds, that the animation will be delayed before starting playback.
                 * @param value The amount of time, in milliseconds, to wait before starting playback. Default 0.
                 */
                setDelay(value?: integer): Phaser.GameObjects.GameObject;
                /**
                 * Gets the amount of time, in milliseconds that the animation will be delayed before starting playback.
                 */
                getDelay(): integer;
                /**
                 * Waits for the specified delay, in milliseconds, then starts playback of the requested animation.
                 * @param delay The delay, in milliseconds, to wait before starting the animation playing.
                 * @param key The key of the animation to play.
                 * @param startFrame The frame of the animation to start from. Default 0.
                 */
                delayedPlay(delay: integer, key: string, startFrame?: integer): Phaser.GameObjects.GameObject;
                /**
                 * Returns the key of the animation currently loaded into this component.
                 */
                getCurrentKey(): string;
                /**
                 * Internal method used to load an animation into this component.
                 * @param key The key of the animation to load.
                 * @param startFrame The start frame of the animation to load. Default 0.
                 */
                load(key: string, startFrame?: integer): Phaser.GameObjects.GameObject;
                /**
                 * Pause the current animation and set the `isPlaying` property to `false`.
                 * You can optionally pause it at a specific frame.
                 * @param atFrame An optional frame to set after pausing the animation.
                 */
                pause(atFrame?: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;
                /**
                 * Resumes playback of a paused animation and sets the `isPlaying` property to `true`.
                 * You can optionally tell it to start playback from a specific frame.
                 * @param fromFrame An optional frame to set before restarting playback.
                 */
                resume(fromFrame?: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;
                /**
                 * `true` if the current animation is paused, otherwise `false`.
                 */
                readonly isPaused: boolean;
                /**
                 * Plays an Animation on the Game Object that owns this Animation Component.
                 * @param key The string-based key of the animation to play, as defined previously in the Animation Manager.
                 * @param ignoreIfPlaying If an animation is already playing then ignore this call. Default false.
                 * @param startFrame Optionally start the animation playing from this frame index. Default 0.
                 */
                play(key: string, ignoreIfPlaying?: boolean, startFrame?: integer): Phaser.GameObjects.GameObject;
                /**
                 * Plays an Animation (in reverse mode) on the Game Object that owns this Animation Component.
                 * @param key The string-based key of the animation to play, as defined previously in the Animation Manager.
                 * @param ignoreIfPlaying If an animation is already playing then ignore this call. Default false.
                 * @param startFrame Optionally start the animation playing from this frame index. Default 0.
                 */
                playReverse(key: string, ignoreIfPlaying?: boolean, startFrame?: integer): Phaser.GameObjects.GameObject;
                /**
                 * Load an Animation and fires 'onStartEvent' event,
                 * extracted from 'play' method
                 * @param key The string-based key of the animation to play, as defined previously in the Animation Manager.
                 * @param startFrame Optionally start the animation playing from this frame index. Default 0.
                 */
                _startAnimation(key: string, startFrame?: integer): Phaser.GameObjects.GameObject;
                /**
                 * Reverse an Animation that is already playing on the Game Object.
                 * @param key The string-based key of the animation to play, as defined previously in the Animation Manager.
                 */
                reverse(key: string): Phaser.GameObjects.GameObject;
                /**
                 * Returns a value between 0 and 1 indicating how far this animation is through, ignoring repeats and yoyos.
                 * If the animation has a non-zero repeat defined, `getProgress` and `getTotalProgress` will be different
                 * because `getProgress` doesn't include any repeats or repeat delays, whereas `getTotalProgress` does.
                 */
                getProgress(): number;
                /**
                 * Takes a value between 0 and 1 and uses it to set how far this animation is through playback.
                 * Does not factor in repeats or yoyos, but does handle playing forwards or backwards.
                 * @param value The progress value, between 0 and 1. Default 0.
                 */
                setProgress(value?: number): Phaser.GameObjects.GameObject;
                /**
                 * Handle the removal of an animation from the Animation Manager.
                 * @param key The key of the removed Animation.
                 * @param animation The removed Animation.
                 */
                remove(key?: string, animation?: Phaser.Animations.Animation): void;
                /**
                 * Gets the number of times that the animation will repeat
                 * after its first iteration. For example, if returns 1, the animation will
                 * play a total of twice (the initial play plus 1 repeat).
                 * A value of -1 means the animation will repeat indefinitely.
                 */
                getRepeat(): integer;
                /**
                 * Sets the number of times that the animation should repeat
                 * after its first iteration. For example, if repeat is 1, the animation will
                 * play a total of twice (the initial play plus 1 repeat).
                 * To repeat indefinitely, use -1. repeat should always be an integer.
                 * @param value The number of times that the animation should repeat.
                 */
                setRepeat(value: integer): Phaser.GameObjects.GameObject;
                /**
                 * Gets the amount of delay between repeats, if any.
                 */
                getRepeatDelay(): number;
                /**
                 * Sets the amount of time in seconds between repeats.
                 * For example, if `repeat` is 2 and `repeatDelay` is 10, the animation will play initially,
                 * then wait for 10 seconds before repeating, then play again, then wait another 10 seconds
                 * before doing its final repeat.
                 * @param value The delay to wait between repeats, in seconds.
                 */
                setRepeatDelay(value: number): Phaser.GameObjects.GameObject;
                /**
                 * Restarts the current animation from its beginning, optionally including its delay value.
                 * @param includeDelay Whether to include the delay value of the animation when restarting. Default false.
                 */
                restart(includeDelay?: boolean): Phaser.GameObjects.GameObject;
                /**
                 * Immediately stops the current animation from playing and dispatches the `animationcomplete` event.
                 */
                stop(): Phaser.GameObjects.GameObject;
                /**
                 * Stops the current animation from playing after the specified time delay, given in milliseconds.
                 * @param delay The number of milliseconds to wait before stopping this animation.
                 */
                stopAfterDelay(delay: integer): Phaser.GameObjects.GameObject;
                /**
                 * Stops the current animation from playing when it next repeats.
                 */
                stopOnRepeat(): Phaser.GameObjects.GameObject;
                /**
                 * Stops the current animation from playing when it next sets the given frame.
                 * If this frame doesn't exist within the animation it will not stop it from playing.
                 * @param delay The frame to check before stopping this animation.
                 */
                stopOnFrame(delay: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;
                /**
                 * Sets the Time Scale factor, allowing you to make the animation go go faster or slower than default.
                 * Where 1 = normal speed (the default), 0.5 = half speed, 2 = double speed, etc.
                 * @param value The time scale factor, where 1 is no change, 0.5 is half speed, etc. Default 1.
                 */
                setTimeScale(value?: number): Phaser.GameObjects.GameObject;
                /**
                 * Gets the Time Scale factor.
                 */
                getTimeScale(): number;
                /**
                 * Returns the total number of frames in this animation.
                 */
                getTotalFrames(): integer;
                /**
                 * The internal update loop for the Animation Component.
                 * @param time The current timestamp.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                update(time: number, delta: number): void;
                /**
                 * Sets the given Animation Frame as being the current frame
                 * and applies it to the parent Game Object, adjusting its size and origin as needed.
                 * @param animationFrame The Animation Frame to set as being current.
                 */
                setCurrentFrame(animationFrame: Phaser.Animations.AnimationFrame): Phaser.GameObjects.GameObject;
                /**
                 * Sets if the current Animation will yoyo when it reaches the end.
                 * A yoyo'ing animation will play through consecutively, and then reverse-play back to the start again.
                 * @param value `true` if the animation should yoyo, `false` to not. Default false.
                 */
                setYoyo(value?: boolean): Phaser.GameObjects.GameObject;
                /**
                 * Gets if the current Animation will yoyo when it reaches the end.
                 * A yoyo'ing animation will play through consecutively, and then reverse-play back to the start again.
                 */
                getYoyo(): boolean;
                /**
                 * Destroy this Animation component.
                 *
                 * Unregisters event listeners and cleans up its references.
                 */
                destroy(): void;
            }

            /**
             * Provides methods used for setting the blend mode of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface BlendMode {
                /**
                 * Sets the Blend Mode being used by this Game Object.
                 *
                 * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
                 *
                 * Under WebGL only the following Blend Modes are available:
                 *
                 * * ADD
                 * * MULTIPLY
                 * * SCREEN
                 *
                 * Canvas has more available depending on browser support.
                 *
                 * You can also create your own custom Blend Modes in WebGL.
                 *
                 * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
                 * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
                 * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
                 * are used.
                 */
                blendMode: Phaser.BlendModes | string;
                /**
                 * Sets the Blend Mode being used by this Game Object.
                 *
                 * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
                 *
                 * Under WebGL only the following Blend Modes are available:
                 *
                 * * ADD
                 * * MULTIPLY
                 * * SCREEN
                 *
                 * Canvas has more available depending on browser support.
                 *
                 * You can also create your own custom Blend Modes in WebGL.
                 *
                 * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
                 * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
                 * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
                 * are used.
                 * @param value The BlendMode value. Either a string or a CONST.
                 */
                setBlendMode(value: string | Phaser.BlendModes): this;
            }

            /**
             * Provides methods used for calculating and setting the size of a non-Frame based Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface ComputedSize {
                /**
                 * The native (un-scaled) width of this Game Object.
                 *
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayWidth` property.
                 */
                width: number;
                /**
                 * The native (un-scaled) height of this Game Object.
                 *
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayHeight` property.
                 */
                height: number;
                /**
                 * The displayed width of this Game Object.
                 *
                 * This value takes into account the scale factor.
                 *
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayWidth: number;
                /**
                 * The displayed height of this Game Object.
                 *
                 * This value takes into account the scale factor.
                 *
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayHeight: number;
                /**
                 * Sets the internal size of this Game Object, as used for frame or physics body creation.
                 *
                 * This will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or call the
                 * `setDisplaySize` method, which is the same thing as changing the scale but allows you
                 * to do so by giving pixel values.
                 *
                 * If you have enabled this Game Object for input, changing the size will _not_ change the
                 * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setSize(width: number, height: number): this;
                /**
                 * Sets the display size of this Game Object.
                 *
                 * Calling this will adjust the scale.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setDisplaySize(width: number, height: number): this;
            }

            /**
             * Provides methods used for getting and setting the texture of a Game Object.
             */
            interface Crop {
                /**
                 * The Texture this Game Object is using to render with.
                 */
                texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;
                /**
                 * The Texture Frame this Game Object is using to render with.
                 */
                frame: Phaser.Textures.Frame;
                /**
                 * A boolean flag indicating if this Game Object is being cropped or not.
                 * You can toggle this at any time after `setCrop` has been called, to turn cropping on or off.
                 * Equally, calling `setCrop` with no arguments will reset the crop and disable it.
                 */
                isCropped: boolean;
                /**
                 * Applies a crop to a texture based Game Object, such as a Sprite or Image.
                 *
                 * The crop is a rectangle that limits the area of the texture frame that is visible during rendering.
                 *
                 * Cropping a Game Object does not change its size, dimensions, physics body or hit area, it just
                 * changes what is shown when rendered.
                 *
                 * The crop coordinates are relative to the texture frame, not the Game Object, meaning 0 x 0 is the top-left.
                 *
                 * Therefore, if you had a Game Object that had an 800x600 sized texture, and you wanted to show only the left
                 * half of it, you could call `setCrop(0, 0, 400, 600)`.
                 *
                 * It is also scaled to match the Game Object scale automatically. Therefore a crop rect of 100x50 would crop
                 * an area of 200x100 when applied to a Game Object that had a scale factor of 2.
                 *
                 * You can either pass in numeric values directly, or you can provide a single Rectangle object as the first argument.
                 *
                 * Call this method with no arguments at all to reset the crop, or toggle the property `isCropped` to `false`.
                 *
                 * You should do this if the crop rectangle becomes the same size as the frame itself, as it will allow
                 * the renderer to skip several internal calculations.
                 * @param x The x coordinate to start the crop from. Or a Phaser.Geom.Rectangle object, in which case the rest of the arguments are ignored.
                 * @param y The y coordinate to start the crop from.
                 * @param width The width of the crop rectangle in pixels.
                 * @param height The height of the crop rectangle in pixels.
                 */
                setCrop(x?: number | Phaser.Geom.Rectangle, y?: number, width?: number, height?: number): this;
            }

            /**
             * Provides methods used for setting the depth of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Depth {
                /**
                 * The depth of this Game Object within the Scene.
                 *
                 * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
                 * of Game Objects, without actually moving their position in the display list.
                 *
                 * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
                 * value will always render in front of one with a lower value.
                 *
                 * Setting the depth will queue a depth sort event within the Scene.
                 */
                depth: number;
                /**
                 * The depth of this Game Object within the Scene.
                 *
                 * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
                 * of Game Objects, without actually moving their position in the display list.
                 *
                 * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
                 * value will always render in front of one with a lower value.
                 *
                 * Setting the depth will queue a depth sort event within the Scene.
                 * @param value The depth of this Game Object.
                 */
                setDepth(value: integer): this;
            }

            /**
             * Provides methods used for visually flipping a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Flip {
                /**
                 * The horizontally flipped state of the Game Object.
                 * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 */
                flipX: boolean;
                /**
                 * The vertically flipped state of the Game Object.
                 * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
                 * Flipping always takes place from the middle of the texture and does not impact the scale value.
                 */
                flipY: boolean;
                /**
                 * Toggles the horizontal flipped state of this Game Object.
                 */
                toggleFlipX(): this;
                /**
                 * Toggles the vertical flipped state of this Game Object.
                 */
                toggleFlipY(): this;
                /**
                 * Sets the horizontal flipped state of this Game Object.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipX(value: boolean): this;
                /**
                 * Sets the vertical flipped state of this Game Object.
                 * @param value The flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlipY(value: boolean): this;
                /**
                 * Sets the horizontal and vertical flipped state of this Game Object.
                 * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
                 */
                setFlip(x: boolean, y: boolean): this;
                /**
                 * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
                 */
                resetFlip(): this;
            }

            /**
             * Provides methods used for obtaining the bounds of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface GetBounds {
                /**
                 * Gets the center coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 */
                getCenter<O extends Phaser.Math.Vector2>(output?: O): O;
                /**
                 * Gets the top-left corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getTopLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the top-right corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getTopRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bottom-left corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getBottomLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bottom-right corner coordinate of this Game Object, regardless of origin.
                 * The returned point is calculated in local space and does not factor in any parent containers
                 * @param output An object to store the values in. If not provided a new Vector2 will be created.
                 * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
                 */
                getBottomRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;
                /**
                 * Gets the bounds of this Game Object, regardless of origin.
                 * The values are stored and returned in a Rectangle, or Rectangle-like, object.
                 * @param output An object to store the values in. If not provided a new Rectangle will be created.
                 */
                getBounds<O extends Phaser.Geom.Rectangle>(output?: O): O;
            }

            /**
             * Provides methods used for getting and setting the mask of a Game Object.
             */
            interface Mask {
                /**
                 * The Mask this Game Object is using during render.
                 */
                mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;
                /**
                 * Sets the mask that this Game Object will use to render with.
                 *
                 * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
                 * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
                 *
                 * If a mask is already set on this Game Object it will be immediately replaced.
                 *
                 * Masks are positioned in global space and are not relative to the Game Object to which they
                 * are applied. The reason for this is that multiple Game Objects can all share the same mask.
                 *
                 * Masks have no impact on physics or input detection. They are purely a rendering component
                 * that allows you to limit what is visible during the render pass.
                 * @param mask The mask this Game Object will use when rendering.
                 */
                setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;
                /**
                 * Clears the mask that this Game Object was using.
                 * @param destroyMask Destroy the mask before clearing it? Default false.
                 */
                clearMask(destroyMask?: boolean): this;
                /**
                 * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
                 * including this one.
                 *
                 * To create the mask you need to pass in a reference to a renderable Game Object.
                 * A renderable Game Object is one that uses a texture to render with, such as an
                 * Image, Sprite, Render Texture or BitmapText.
                 *
                 * If you do not provide a renderable object, and this Game Object has a texture,
                 * it will use itself as the object. This means you can call this method to create
                 * a Bitmap Mask from any renderable Game Object.
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;
                /**
                 * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
                 * including this one.
                 *
                 * To create the mask you need to pass in a reference to a Graphics Game Object.
                 *
                 * If you do not provide a graphics object, and this Game Object is an instance
                 * of a Graphics object, then it will use itself to create the mask.
                 *
                 * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
                 * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
                 */
                createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;
            }

            /**
             * Provides methods used for getting and setting the origin of a Game Object.
             * Values are normalized, given in the range 0 to 1.
             * Display values contain the calculated pixel values.
             * Should be applied as a mixin and not used directly.
             */
            interface Origin {
                /**
                 * The horizontal origin of this Game Object.
                 * The origin maps the relationship between the size and position of the Game Object.
                 * The default value is 0.5, meaning all Game Objects are positioned based on their center.
                 * Setting the value to 0 means the position now relates to the left of the Game Object.
                 */
                originX: number;
                /**
                 * The vertical origin of this Game Object.
                 * The origin maps the relationship between the size and position of the Game Object.
                 * The default value is 0.5, meaning all Game Objects are positioned based on their center.
                 * Setting the value to 0 means the position now relates to the top of the Game Object.
                 */
                originY: number;
                /**
                 * The horizontal display origin of this Game Object.
                 * The origin is a normalized value between 0 and 1.
                 * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
                 */
                displayOriginX: number;
                /**
                 * The vertical display origin of this Game Object.
                 * The origin is a normalized value between 0 and 1.
                 * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
                 */
                displayOriginY: number;
                /**
                 * Sets the origin of this Game Object.
                 *
                 * The values are given in the range 0 to 1.
                 * @param x The horizontal origin value. Default 0.5.
                 * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
                 */
                setOrigin(x?: number, y?: number): this;
                /**
                 * Sets the origin of this Game Object based on the Pivot values in its Frame.
                 */
                setOriginFromFrame(): this;
                /**
                 * Sets the display origin of this Game Object.
                 * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
                 * @param x The horizontal display origin value. Default 0.
                 * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
                 */
                setDisplayOrigin(x?: number, y?: number): this;
                /**
                 * Updates the Display Origin cached values internally stored on this Game Object.
                 * You don't usually call this directly, but it is exposed for edge-cases where you may.
                 */
                updateDisplayOrigin(): this;
            }

            /**
             * Provides methods used for setting the WebGL rendering pipeline of a Game Object.
             */
            interface Pipeline {
                /**
                 * The initial WebGL pipeline of this Game Object.
                 */
                defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;
                /**
                 * The current WebGL pipeline of this Game Object.
                 */
                pipeline: Phaser.Renderer.WebGL.WebGLPipeline;
                /**
                 * Sets the initial WebGL Pipeline of this Game Object.
                 * This should only be called during the instantiation of the Game Object.
                 * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
                 */
                initPipeline(pipelineName?: string): boolean;
                /**
                 * Sets the active WebGL Pipeline of this Game Object.
                 * @param pipelineName The name of the pipeline to set on this Game Object.
                 */
                setPipeline(pipelineName: string): this;
                /**
                 * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
                 */
                resetPipeline(): boolean;
                /**
                 * Gets the name of the WebGL Pipeline this Game Object is currently using.
                 */
                getPipelineName(): string;
            }

            /**
             * Provides methods used for getting and setting the scale of a Game Object.
             */
            interface ScaleMode {
                /**
                 * The Scale Mode being used by this Game Object.
                 * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
                 */
                scaleMode: Phaser.ScaleModes;
                /**
                 * Sets the Scale Mode being used by this Game Object.
                 * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
                 * @param value The Scale Mode to be used by this Game Object.
                 */
                setScaleMode(value: Phaser.ScaleModes): this;
            }

            /**
             * Provides methods used for getting and setting the Scroll Factor of a Game Object.
             */
            interface ScrollFactor {
                /**
                 * The horizontal scroll factor of this Game Object.
                 *
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 *
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 *
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 *
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 */
                scrollFactorX: number;
                /**
                 * The vertical scroll factor of this Game Object.
                 *
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 *
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 *
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 *
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 */
                scrollFactorY: number;
                /**
                 * Sets the scroll factor of this Game Object.
                 *
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 *
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 *
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 *
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 * @param x The horizontal scroll factor of this Game Object.
                 * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
                 */
                setScrollFactor(x: number, y?: number): this;
            }

            /**
             * Provides methods used for getting and setting the size of a Game Object.
             */
            interface Size {
                /**
                 * The native (un-scaled) width of this Game Object.
                 *
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayWidth` property.
                 */
                width: number;
                /**
                 * The native (un-scaled) height of this Game Object.
                 *
                 * Changing this value will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or use
                 * the `displayHeight` property.
                 */
                height: number;
                /**
                 * The displayed width of this Game Object.
                 *
                 * This value takes into account the scale factor.
                 *
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayWidth: number;
                /**
                 * The displayed height of this Game Object.
                 *
                 * This value takes into account the scale factor.
                 *
                 * Setting this value will adjust the Game Object's scale property.
                 */
                displayHeight: number;
                /**
                 * Sets the size of this Game Object to be that of the given Frame.
                 *
                 * This will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or call the
                 * `setDisplaySize` method, which is the same thing as changing the scale but allows you
                 * to do so by giving pixel values.
                 *
                 * If you have enabled this Game Object for input, changing the size will _not_ change the
                 * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
                 * @param frame The frame to base the size of this Game Object on.
                 */
                setSizeToFrame(frame: Phaser.Textures.Frame): this;
                /**
                 * Sets the internal size of this Game Object, as used for frame or physics body creation.
                 *
                 * This will not change the size that the Game Object is rendered in-game.
                 * For that you need to either set the scale of the Game Object (`setScale`) or call the
                 * `setDisplaySize` method, which is the same thing as changing the scale but allows you
                 * to do so by giving pixel values.
                 *
                 * If you have enabled this Game Object for input, changing the size will _not_ change the
                 * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setSize(width: number, height: number): this;
                /**
                 * Sets the display size of this Game Object.
                 *
                 * Calling this will adjust the scale.
                 * @param width The width of this Game Object.
                 * @param height The height of this Game Object.
                 */
                setDisplaySize(width: number, height: number): this;
            }

            /**
             * Provides methods used for getting and setting the texture of a Game Object.
             */
            interface Texture {
                /**
                 * The Texture this Game Object is using to render with.
                 */
                texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;
                /**
                 * The Texture Frame this Game Object is using to render with.
                 */
                frame: Phaser.Textures.Frame;
                /**
                 * Sets the texture and frame this Game Object will use to render with.
                 *
                 * Textures are referenced by their string-based keys, as stored in the Texture Manager.
                 * @param key The key of the texture to be used, as stored in the Texture Manager.
                 * @param frame The name or index of the frame within the Texture.
                 */
                setTexture(key: string, frame?: string | integer): this;
                /**
                 * Sets the frame this Game Object will use to render with.
                 *
                 * The Frame has to belong to the current Texture being used.
                 *
                 * It can be either a string or an index.
                 *
                 * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
                 * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
                 * @param frame The name or index of the frame within the Texture.
                 * @param updateSize Should this call adjust the size of the Game Object? Default true.
                 * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
                 */
                setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;
            }

            /**
             * Provides methods used for getting and setting the texture of a Game Object.
             */
            interface TextureCrop {
                /**
                 * The Texture this Game Object is using to render with.
                 */
                texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;
                /**
                 * The Texture Frame this Game Object is using to render with.
                 */
                frame: Phaser.Textures.Frame;
                /**
                 * A boolean flag indicating if this Game Object is being cropped or not.
                 * You can toggle this at any time after `setCrop` has been called, to turn cropping on or off.
                 * Equally, calling `setCrop` with no arguments will reset the crop and disable it.
                 */
                isCropped: boolean;
                /**
                 * Applies a crop to a texture based Game Object, such as a Sprite or Image.
                 *
                 * The crop is a rectangle that limits the area of the texture frame that is visible during rendering.
                 *
                 * Cropping a Game Object does not change its size, dimensions, physics body or hit area, it just
                 * changes what is shown when rendered.
                 *
                 * The crop coordinates are relative to the texture frame, not the Game Object, meaning 0 x 0 is the top-left.
                 *
                 * Therefore, if you had a Game Object that had an 800x600 sized texture, and you wanted to show only the left
                 * half of it, you could call `setCrop(0, 0, 400, 600)`.
                 *
                 * It is also scaled to match the Game Object scale automatically. Therefore a crop rect of 100x50 would crop
                 * an area of 200x100 when applied to a Game Object that had a scale factor of 2.
                 *
                 * You can either pass in numeric values directly, or you can provide a single Rectangle object as the first argument.
                 *
                 * Call this method with no arguments at all to reset the crop, or toggle the property `isCropped` to `false`.
                 *
                 * You should do this if the crop rectangle becomes the same size as the frame itself, as it will allow
                 * the renderer to skip several internal calculations.
                 * @param x The x coordinate to start the crop from. Or a Phaser.Geom.Rectangle object, in which case the rest of the arguments are ignored.
                 * @param y The y coordinate to start the crop from.
                 * @param width The width of the crop rectangle in pixels.
                 * @param height The height of the crop rectangle in pixels.
                 */
                setCrop(x?: number | Phaser.Geom.Rectangle, y?: number, width?: number, height?: number): this;
                /**
                 * Sets the texture and frame this Game Object will use to render with.
                 *
                 * Textures are referenced by their string-based keys, as stored in the Texture Manager.
                 * @param key The key of the texture to be used, as stored in the Texture Manager.
                 * @param frame The name or index of the frame within the Texture.
                 */
                setTexture(key: string, frame?: string | integer): this;
                /**
                 * Sets the frame this Game Object will use to render with.
                 *
                 * The Frame has to belong to the current Texture being used.
                 *
                 * It can be either a string or an index.
                 *
                 * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
                 * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
                 * @param frame The name or index of the frame within the Texture.
                 * @param updateSize Should this call adjust the size of the Game Object? Default true.
                 * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
                 */
                setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;
            }

            /**
             * Provides methods used for setting the tint of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Tint {
                /**
                 * Fill or additive?
                 */
                tintFill: boolean;
                /**
                 * Clears all tint values associated with this Game Object.
                 *
                 * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
                 * which results in no visible change to the texture.
                 */
                clearTint(): this;
                /**
                 * Sets an additive tint on this Game Object.
                 *
                 * The tint works by taking the pixel color values from the Game Objects texture, and then
                 * multiplying it by the color value of the tint. You can provide either one color value,
                 * in which case the whole Game Object will be tinted in that color. Or you can provide a color
                 * per corner. The colors are blended together across the extent of the Game Object.
                 *
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 *
                 * To remove a tint call `clearTint`.
                 *
                 * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTint(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;
                /**
                 * Sets a fill-based tint on this Game Object.
                 *
                 * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
                 * with those in the tint. You can use this for effects such as making a player flash 'white'
                 * if hit by something. You can provide either one color value, in which case the whole
                 * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
                 * are blended together across the extent of the Game Object.
                 *
                 * To modify the tint color once set, either call this method again with new values or use the
                 * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
                 * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
                 *
                 * To remove a tint call `clearTint`.
                 *
                 * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
                 * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
                 * @param topRight The tint being applied to the top-right of the Game Object.
                 * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
                 * @param bottomRight The tint being applied to the bottom-right of the Game Object.
                 */
                setTintFill(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;
                /**
                 * The tint value being applied to the top-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintTopLeft: integer;
                /**
                 * The tint value being applied to the top-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintTopRight: integer;
                /**
                 * The tint value being applied to the bottom-left of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintBottomLeft: integer;
                /**
                 * The tint value being applied to the bottom-right of the Game Object.
                 * This value is interpolated from the corner to the center of the Game Object.
                 */
                tintBottomRight: integer;
                /**
                 * The tint value being applied to the whole of the Game Object.
                 */
                tint: integer;
                /**
                 * Does this Game Object have a tint applied to it or not?
                 */
                readonly isTinted: boolean;
            }

            /**
             * Build a JSON representation of the given Game Object.
             *
             * This is typically extended further by Game Object specific implementations.
             */
            interface ToJSON {
            }

            /**
             * Provides methods used for getting and setting the position, scale and rotation of a Game Object.
             */
            interface Transform {
                /**
                 * The x position of this Game Object.
                 */
                x: number;
                /**
                 * The y position of this Game Object.
                 */
                y: number;
                /**
                 * The z position of this Game Object.
                 * Note: Do not use this value to set the z-index, instead see the `depth` property.
                 */
                z: number;
                /**
                 * The w position of this Game Object.
                 */
                w: number;
                /**
                 * The horizontal scale of this Game Object.
                 */
                scaleX: number;
                /**
                 * The vertical scale of this Game Object.
                 */
                scaleY: number;
                /**
                 * The angle of this Game Object as expressed in degrees.
                 *
                 * Where 0 is to the right, 90 is down, 180 is left.
                 *
                 * If you prefer to work in radians, see the `rotation` property instead.
                 */
                angle: integer;
                /**
                 * The angle of this Game Object in radians.
                 *
                 * If you prefer to work in degrees, see the `angle` property instead.
                 */
                rotation: number;
                /**
                 * Sets the position of this Game Object.
                 * @param x The x position of this Game Object. Default 0.
                 * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
                 * @param z The z position of this Game Object. Default 0.
                 * @param w The w position of this Game Object. Default 0.
                 */
                setPosition(x?: number, y?: number, z?: number, w?: number): this;
                /**
                 * Sets the position of this Game Object to be a random position within the confines of
                 * the given area.
                 *
                 * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
                 *
                 * The position does not factor in the size of this Game Object, meaning that only the origin is
                 * guaranteed to be within the area.
                 * @param x The x position of the top-left of the random area. Default 0.
                 * @param y The y position of the top-left of the random area. Default 0.
                 * @param width The width of the random area.
                 * @param height The height of the random area.
                 */
                setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;
                /**
                 * Sets the rotation of this Game Object.
                 * @param radians The rotation of this Game Object, in radians. Default 0.
                 */
                setRotation(radians?: number): this;
                /**
                 * Sets the angle of this Game Object.
                 * @param degrees The rotation of this Game Object, in degrees. Default 0.
                 */
                setAngle(degrees?: number): this;
                /**
                 * Sets the scale of this Game Object.
                 * @param x The horizontal scale of this Game Object.
                 * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
                 */
                setScale(x: number, y?: number): this;
                /**
                 * Sets the x position of this Game Object.
                 * @param value The x position of this Game Object. Default 0.
                 */
                setX(value?: number): this;
                /**
                 * Sets the y position of this Game Object.
                 * @param value The y position of this Game Object. Default 0.
                 */
                setY(value?: number): this;
                /**
                 * Sets the z position of this Game Object.
                 * @param value The z position of this Game Object. Default 0.
                 */
                setZ(value?: number): this;
                /**
                 * Sets the w position of this Game Object.
                 * @param value The w position of this Game Object. Default 0.
                 */
                setW(value?: number): this;
                /**
                 * Gets the local transform matrix for this Game Object.
                 * @param tempMatrix The matrix to populate with the values from this Game Object.
                 */
                getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;
                /**
                 * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
                 * @param tempMatrix The matrix to populate with the values from this Game Object.
                 * @param parentMatrix A temporary matrix to hold parent values during the calculations.
                 */
                getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;
            }

            /**
             * A Matrix used for display transformations for rendering.
             *
             * It is represented like so:
             *
             * ```
             * | a | c | tx |
             * | b | d | ty |
             * | 0 | 0 | 1  |
             * ```
             */
            class TransformMatrix {
                /**
                 *
                 * @param a The Scale X value. Default 1.
                 * @param b The Shear Y value. Default 0.
                 * @param c The Shear X value. Default 0.
                 * @param d The Scale Y value. Default 1.
                 * @param tx The Translate X value. Default 0.
                 * @param ty The Translate Y value. Default 0.
                 */
                constructor(a?: number, b?: number, c?: number, d?: number, tx?: number, ty?: number);

                /**
                 * The matrix values.
                 */
                matrix: Float32Array;

                /**
                 * The decomposed matrix.
                 */
                decomposedMatrix: object;

                /**
                 * The Scale X value.
                 */
                a: number;

                /**
                 * The Shear Y value.
                 */
                b: number;

                /**
                 * The Shear X value.
                 */
                c: number;

                /**
                 * The Scale Y value.
                 */
                d: number;

                /**
                 * The Translate X value.
                 */
                e: number;

                /**
                 * The Translate Y value.
                 */
                f: number;

                /**
                 * The Translate X value.
                 */
                tx: number;

                /**
                 * The Translate Y value.
                 */
                ty: number;

                /**
                 * The rotation of the Matrix.
                 */
                readonly rotation: number;

                /**
                 * The horizontal scale of the Matrix.
                 */
                readonly scaleX: number;

                /**
                 * The vertical scale of the Matrix.
                 */
                readonly scaleY: number;

                /**
                 * Reset the Matrix to an identity matrix.
                 */
                loadIdentity(): this;

                /**
                 * Translate the Matrix.
                 * @param x The horizontal translation value.
                 * @param y The vertical translation value.
                 */
                translate(x: number, y: number): this;

                /**
                 * Scale the Matrix.
                 * @param x The horizontal scale value.
                 * @param y The vertical scale value.
                 */
                scale(x: number, y: number): this;

                /**
                 * Rotate the Matrix.
                 * @param angle The angle of rotation in radians.
                 */
                rotate(angle: number): this;

                /**
                 * Multiply this Matrix by the given Matrix.
                 *
                 * If an `out` Matrix is given then the results will be stored in it.
                 * If it is not given, this matrix will be updated in place instead.
                 * Use an `out` Matrix if you do not wish to mutate this matrix.
                 * @param rhs The Matrix to multiply by.
                 * @param out An optional Matrix to store the results in.
                 */
                multiply(rhs: Phaser.GameObjects.Components.TransformMatrix, out?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

                /**
                 * Multiply this Matrix by the matrix given, including the offset.
                 *
                 * The offsetX is added to the tx value: `offsetX * a + offsetY * c + tx`.
                 * The offsetY is added to the ty value: `offsetY * b + offsetY * d + ty`.
                 * @param src The source Matrix to copy from.
                 * @param offsetX Horizontal offset to factor in to the multiplication.
                 * @param offsetY Vertical offset to factor in to the multiplication.
                 */
                multiplyWithOffset(src: Phaser.GameObjects.Components.TransformMatrix, offsetX: number, offsetY: number): this;

                /**
                 * Transform the Matrix.
                 * @param a The Scale X value.
                 * @param b The Shear Y value.
                 * @param c The Shear X value.
                 * @param d The Scale Y value.
                 * @param tx The Translate X value.
                 * @param ty The Translate Y value.
                 */
                transform(a: number, b: number, c: number, d: number, tx: number, ty: number): this;

                /**
                 * Transform a point using this Matrix.
                 * @param x The x coordinate of the point to transform.
                 * @param y The y coordinate of the point to transform.
                 * @param point The Point object to store the transformed coordinates.
                 */
                transformPoint(x: number, y: number, point: Phaser.Geom.Point | Phaser.Math.Vector2 | object): Phaser.Geom.Point | Phaser.Math.Vector2 | object;

                /**
                 * Invert the Matrix.
                 */
                invert(): this;

                /**
                 * Set the values of this Matrix to copy those of the matrix given.
                 * @param src The source Matrix to copy from.
                 */
                copyFrom(src: Phaser.GameObjects.Components.TransformMatrix): this;

                /**
                 * Set the values of this Matrix to copy those of the array given.
                 * Where array indexes 0, 1, 2, 3, 4 and 5 are mapped to a, b, c, d, e and f.
                 * @param src The array of values to set into this matrix.
                 */
                copyFromArray(src: any[]): this;

                /**
                 * Copy the values from this Matrix to the given Canvas Rendering Context.
                 * This will use the Context.transform method.
                 * @param ctx The Canvas Rendering Context to copy the matrix values to.
                 */
                copyToContext(ctx: CanvasRenderingContext2D): CanvasRenderingContext2D;

                /**
                 * Copy the values from this Matrix to the given Canvas Rendering Context.
                 * This will use the Context.setTransform method.
                 * @param ctx The Canvas Rendering Context to copy the matrix values to.
                 */
                setToContext(ctx: CanvasRenderingContext2D): CanvasRenderingContext2D;

                /**
                 * Copy the values in this Matrix to the array given.
                 *
                 * Where array indexes 0, 1, 2, 3, 4 and 5 are mapped to a, b, c, d, e and f.
                 * @param out The array to copy the matrix values in to.
                 */
                copyToArray(out?: any[]): any[];

                /**
                 * Set the values of this Matrix.
                 * @param a The Scale X value.
                 * @param b The Shear Y value.
                 * @param c The Shear X value.
                 * @param d The Scale Y value.
                 * @param tx The Translate X value.
                 * @param ty The Translate Y value.
                 */
                setTransform(a: number, b: number, c: number, d: number, tx: number, ty: number): this;

                /**
                 * Decompose this Matrix into its translation, scale and rotation values.
                 */
                decomposeMatrix(): object;

                /**
                 * Apply the identity, translate, rotate and scale operations on the Matrix.
                 * @param x The horizontal translation.
                 * @param y The vertical translation.
                 * @param rotation The angle of rotation in radians.
                 * @param scaleX The horizontal scale.
                 * @param scaleY The vertical scale.
                 */
                applyITRS(x: number, y: number, rotation: number, scaleX: number, scaleY: number): this;

                /**
                 * Takes the `x` and `y` values and returns a new position in the `output` vector that is the inverse of
                 * the current matrix with its transformation applied.
                 *
                 * Can be used to translate points from world to local space.
                 * @param x The x position to translate.
                 * @param y The y position to translate.
                 * @param output A Vector2, or point-like object, to store the results in.
                 */
                applyInverse(x: number, y: number, output?: Phaser.Math.Vector2): Phaser.Math.Vector2;

                /**
                 * Returns the X component of this matrix multiplied by the given values.
                 * This is the same as `x * a + y * c + e`.
                 * @param x The x value.
                 * @param y The y value.
                 */
                getX(x: number, y: number): number;

                /**
                 * Returns the Y component of this matrix multiplied by the given values.
                 * This is the same as `x * b + y * d + f`.
                 * @param x The x value.
                 * @param y The y value.
                 */
                getY(x: number, y: number): number;

                /**
                 * Returns a string that can be used in a CSS Transform call as a `matrix` property.
                 */
                getCSSMatrix(): string;

                /**
                 * Destroys this Transform Matrix.
                 */
                destroy(): void;

            }

            /**
             * Provides methods used for setting the visibility of a Game Object.
             * Should be applied as a mixin and not used directly.
             */
            interface Visible {
                /**
                 * The visible state of the Game Object.
                 *
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 */
                visible: boolean;
                /**
                 * Sets the visibility of this Game Object.
                 *
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 * @param value The visible state of the Game Object.
                 */
                setVisible(value: boolean): this;
            }

        }

        /**
         * A Container Game Object.
         *
         * A Container, as the name implies, can 'contain' other types of Game Object.
         * When a Game Object is added to a Container, the Container becomes responsible for the rendering of it.
         * By default it will be removed from the Display List and instead added to the Containers own internal list.
         *
         * The position of the Game Object automatically becomes relative to the position of the Container.
         *
         * When the Container is rendered, all of its children are rendered as well, in the order in which they exist
         * within the Container. Container children can be repositioned using methods such as `MoveUp`, `MoveDown` and `SendToBack`.
         *
         * If you modify a transform property of the Container, such as `Container.x` or `Container.rotation` then it will
         * automatically influence all children as well.
         *
         * Containers can include other Containers for deeply nested transforms.
         *
         * Containers can have masks set on them and can be used as a mask too. However, Container children cannot be masked.
         * The masks do not 'stack up'. Only a Container on the root of the display list will use its mask.
         *
         * Containers can be enabled for input. Because they do not have a texture you need to provide a shape for them
         * to use as their hit area. Container children can also be enabled for input, independent of the Container.
         *
         * Containers can be given a physics body for either Arcade Physics, Impact Physics or Matter Physics. However,
         * if Container _children_ are enabled for physics you may get unexpected results, such as offset bodies,
         * if the Container itself, or any of its ancestors, is positioned anywhere other than at 0 x 0. Container children
         * with physics do not factor in the Container due to the excessive extra calculations needed. Please structure
         * your game to work around this.
         *
         * It's important to understand the impact of using Containers. They add additional processing overhead into
         * every one of their children. The deeper you nest them, the more the cost escalates. This is especially true
         * for input events. You also loose the ability to set the display depth of Container children in the same
         * flexible manner as those not within them. In short, don't use them for the sake of it. You pay a small cost
         * every time you create one, try to structure your game around avoiding that where possible.
         */
        class Container extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.ComputedSize, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param children An optional array of Game Objects to add to this Container.
             */
            constructor(scene: Phaser.Scene, x?: number, y?: number, children?: Phaser.GameObjects.GameObject[]);

            /**
             * An array holding the children of this Container.
             */
            list: Phaser.GameObjects.GameObject[];

            /**
             * Does this Container exclusively manage its children?
             *
             * The default is `true` which means a child added to this Container cannot
             * belong in another Container, which includes the Scene display list.
             *
             * If you disable this then this Container will no longer exclusively manage its children.
             * This allows you to create all kinds of interesting graphical effects, such as replicating
             * Game Objects without reparenting them all over the Scene.
             * However, doing so will prevent children from receiving any kind of input event or have
             * their physics bodies work by default, as they're no longer a single entity on the
             * display list, but are being replicated where-ever this Container is.
             */
            exclusive: boolean;

            /**
             * Containers can have an optional maximum size. If set to anything above 0 it
             * will constrict the addition of new Game Objects into the Container, capping off
             * the maximum limit the Container can grow in size to.
             */
            maxSize: integer;

            /**
             * The cursor position.
             */
            position: integer;

            /**
             * Internal Transform Matrix used for local space conversion.
             */
            localTransform: Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly originX: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly originY: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly displayOriginX: number;

            /**
             * Internal value to allow Containers to be used for input and physics.
             * Do not change this value. It has no effect other than to break things.
             */
            readonly displayOriginY: number;

            /**
             * Does this Container exclusively manage its children?
             *
             * The default is `true` which means a child added to this Container cannot
             * belong in another Container, which includes the Scene display list.
             *
             * If you disable this then this Container will no longer exclusively manage its children.
             * This allows you to create all kinds of interesting graphical effects, such as replicating
             * Game Objects without reparenting them all over the Scene.
             * However, doing so will prevent children from receiving any kind of input event or have
             * their physics bodies work by default, as they're no longer a single entity on the
             * display list, but are being replicated where-ever this Container is.
             * @param value The exclusive state of this Container. Default true.
             */
            setExclusive(value?: boolean): Phaser.GameObjects.Container;

            /**
             * Gets the bounds of this Container. It works by iterating all children of the Container,
             * getting their respective bounds, and then working out a min-max rectangle from that.
             * It does not factor in if the children render or not, all are included.
             *
             * Some children are unable to return their bounds, such as Graphics objects, in which case
             * they are skipped.
             *
             * Depending on the quantity of children in this Container it could be a really expensive call,
             * so cache it and only poll it as needed.
             *
             * The values are stored and returned in a Rectangle object.
             * @param output A Geom.Rectangle object to store the values in. If not provided a new Rectangle will be created.
             */
            getBounds(output?: Phaser.Geom.Rectangle): Phaser.Geom.Rectangle;

            /**
             * Takes a Point-like object, such as a Vector2, Geom.Point or object with public x and y properties,
             * and transforms it into the space of this Container, then returns it in the output object.
             * @param source The Source Point to be transformed.
             * @param output A destination object to store the transformed point in. If none given a Vector2 will be created and returned.
             */
            pointToContainer(source: object | Phaser.Geom.Point | Phaser.Math.Vector2, output?: object | Phaser.Geom.Point | Phaser.Math.Vector2): object | Phaser.Geom.Point | Phaser.Math.Vector2;

            /**
             * Returns the world transform matrix as used for Bounds checks.
             *
             * The returned matrix is temporal and shouldn't be stored.
             */
            getBoundsTransformMatrix(): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Adds the given Game Object, or array of Game Objects, to this Container.
             *
             * Each Game Object must be unique within the Container.
             * @param child The Game Object, or array of Game Objects, to add to the Container.
             */
            add(child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): Phaser.GameObjects.Container;

            /**
             * Adds the given Game Object, or array of Game Objects, to this Container at the specified position.
             *
             * Existing Game Objects in the Container are shifted up.
             *
             * Each Game Object must be unique within the Container.
             * @param child The Game Object, or array of Game Objects, to add to the Container.
             * @param index The position to insert the Game Object/s at. Default 0.
             */
            addAt(child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[], index?: integer): Phaser.GameObjects.Container;

            /**
             * Returns the Game Object at the given position in this Container.
             * @param index The position to get the Game Object from.
             */
            getAt(index: integer): Phaser.GameObjects.GameObject;

            /**
             * Returns the index of the given Game Object in this Container.
             * @param child The Game Object to search for in this Container.
             */
            getIndex(child: Phaser.GameObjects.GameObject): integer;

            /**
             * Sort the contents of this Container so the items are in order based on the given property.
             * For example: `sort('alpha')` would sort the elements based on the value of their `alpha` property.
             * @param property The property to lexically sort by.
             */
            sort(property: string): Phaser.GameObjects.Container;

            /**
             * Searches for the first instance of a child with its `name` property matching the given argument.
             * Should more than one child have the same name only the first is returned.
             * @param name The name to search for.
             */
            getByName(name: string): Phaser.GameObjects.GameObject;

            /**
             * Returns a random Game Object from this Container.
             * @param startIndex An optional start index. Default 0.
             * @param length An optional length, the total number of elements (from the startIndex) to choose from.
             */
            getRandom(startIndex?: integer, length?: integer): Phaser.GameObjects.GameObject;

            /**
             * Gets the first Game Object in this Container.
             *
             * You can also specify a property and value to search for, in which case it will return the first
             * Game Object in this Container with a matching property and / or value.
             *
             * For example: `getFirst('visible', true)` would return the first Game Object that had its `visible` property set.
             *
             * You can limit the search to the `startIndex` - `endIndex` range.
             * @param property The property to test on each Game Object in the Container.
             * @param value The value to test the property against. Must pass a strict (`===`) comparison check.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            getFirst(property?: string, value?: any, startIndex?: integer, endIndex?: integer): Phaser.GameObjects.GameObject;

            /**
             * Returns all Game Objects in this Container.
             *
             * You can optionally specify a matching criteria using the `property` and `value` arguments.
             *
             * For example: `getAll('body')` would return only Game Objects that have a body property.
             *
             * You can also specify a value to compare the property to:
             *
             * `getAll('visible', true)` would return only Game Objects that have their visible property set to `true`.
             *
             * Optionally you can specify a start and end index. For example if this Container had 100 Game Objects,
             * and you set `startIndex` to 0 and `endIndex` to 50, it would return matches from only
             * the first 50 Game Objects.
             * @param property The property to test on each Game Object in the Container.
             * @param value If property is set then the `property` must strictly equal this value to be included in the results.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            getAll(property?: string, value?: any, startIndex?: integer, endIndex?: integer): Phaser.GameObjects.GameObject[];

            /**
             * Returns the total number of Game Objects in this Container that have a property
             * matching the given value.
             *
             * For example: `count('visible', true)` would count all the elements that have their visible property set.
             *
             * You can optionally limit the operation to the `startIndex` - `endIndex` range.
             * @param property The property to check.
             * @param value The value to check.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            count(property: string, value: any, startIndex?: integer, endIndex?: integer): integer;

            /**
             * Swaps the position of two Game Objects in this Container.
             * Both Game Objects must belong to this Container.
             * @param child1 The first Game Object to swap.
             * @param child2 The second Game Object to swap.
             */
            swap(child1: Phaser.GameObjects.GameObject, child2: Phaser.GameObjects.GameObject): Phaser.GameObjects.Container;

            /**
             * Moves a Game Object to a new position within this Container.
             *
             * The Game Object must already be a child of this Container.
             *
             * The Game Object is removed from its old position and inserted into the new one.
             * Therefore the Container size does not change. Other children will change position accordingly.
             * @param child The Game Object to move.
             * @param index The new position of the Game Object in this Container.
             */
            moveTo(child: Phaser.GameObjects.GameObject, index: integer): Phaser.GameObjects.Container;

            /**
             * Removes the given Game Object, or array of Game Objects, from this Container.
             *
             * The Game Objects must already be children of this Container.
             *
             * You can also optionally call `destroy` on each Game Object that is removed from the Container.
             * @param child The Game Object, or array of Game Objects, to be removed from the Container.
             * @param destroyChild Optionally call `destroy` on each child successfully removed from this Container. Default false.
             */
            remove(child: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[], destroyChild?: boolean): Phaser.GameObjects.Container;

            /**
             * Removes the Game Object at the given position in this Container.
             *
             * You can also optionally call `destroy` on the Game Object, if one is found.
             * @param index The index of the Game Object to be removed.
             * @param destroyChild Optionally call `destroy` on the Game Object if successfully removed from this Container. Default false.
             */
            removeAt(index: integer, destroyChild?: boolean): Phaser.GameObjects.Container;

            /**
             * Removes the Game Objects between the given positions in this Container.
             *
             * You can also optionally call `destroy` on each Game Object that is removed from the Container.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             * @param destroyChild Optionally call `destroy` on each Game Object successfully removed from this Container. Default false.
             */
            removeBetween(startIndex?: integer, endIndex?: integer, destroyChild?: boolean): Phaser.GameObjects.Container;

            /**
             * Removes all Game Objects from this Container.
             *
             * You can also optionally call `destroy` on each Game Object that is removed from the Container.
             * @param destroyChild Optionally call `destroy` on each Game Object successfully removed from this Container. Default false.
             */
            removeAll(destroyChild?: boolean): Phaser.GameObjects.Container;

            /**
             * Brings the given Game Object to the top of this Container.
             * This will cause it to render on-top of any other objects in the Container.
             * @param child The Game Object to bring to the top of the Container.
             */
            bringToTop(child: Phaser.GameObjects.GameObject): Phaser.GameObjects.Container;

            /**
             * Sends the given Game Object to the bottom of this Container.
             * This will cause it to render below any other objects in the Container.
             * @param child The Game Object to send to the bottom of the Container.
             */
            sendToBack(child: Phaser.GameObjects.GameObject): Phaser.GameObjects.Container;

            /**
             * Moves the given Game Object up one place in this Container, unless it's already at the top.
             * @param child The Game Object to be moved in the Container.
             */
            moveUp(child: Phaser.GameObjects.GameObject): Phaser.GameObjects.Container;

            /**
             * Moves the given Game Object down one place in this Container, unless it's already at the bottom.
             * @param child The Game Object to be moved in the Container.
             */
            moveDown(child: Phaser.GameObjects.GameObject): Phaser.GameObjects.Container;

            /**
             * Reverses the order of all Game Objects in this Container.
             */
            reverse(): Phaser.GameObjects.Container;

            /**
             * Shuffles the all Game Objects in this Container using the Fisher-Yates implementation.
             */
            shuffle(): Phaser.GameObjects.Container;

            /**
             * Replaces a Game Object in this Container with the new Game Object.
             * The new Game Object cannot already be a child of this Container.
             * @param oldChild The Game Object in this Container that will be replaced.
             * @param newChild The Game Object to be added to this Container.
             * @param destroyChild Optionally call `destroy` on the Game Object if successfully removed from this Container. Default false.
             */
            replace(oldChild: Phaser.GameObjects.GameObject, newChild: Phaser.GameObjects.GameObject, destroyChild?: boolean): Phaser.GameObjects.Container;

            /**
             * Returns `true` if the given Game Object is a direct child of this Container.
             *
             * This check does not scan nested Containers.
             * @param child The Game Object to check for within this Container.
             */
            exists(child: Phaser.GameObjects.GameObject): boolean;

            /**
             * Sets the property to the given value on all Game Objects in this Container.
             *
             * Optionally you can specify a start and end index. For example if this Container had 100 Game Objects,
             * and you set `startIndex` to 0 and `endIndex` to 50, it would return matches from only
             * the first 50 Game Objects.
             * @param property The property that must exist on the Game Object.
             * @param value The value to get the property to.
             * @param startIndex An optional start index to search from. Default 0.
             * @param endIndex An optional end index to search up to (but not included) Default Container.length.
             */
            setAll(property: string, value: any, startIndex?: integer, endIndex?: integer): Phaser.GameObjects.Container;

            /**
             * Passes all Game Objects in this Container to the given callback.
             *
             * A copy of the Container is made before passing each entry to your callback.
             * This protects against the callback itself modifying the Container.
             *
             * If you know for sure that the callback will not change the size of this Container
             * then you can use the more performant `Container.iterate` method instead.
             * @param callback The function to call.
             * @param context Value to use as `this` when executing callback.
             * @param args Additional arguments that will be passed to the callback, after the child.
             */
            each(callback: Function, context?: object, ...args: any[]): Phaser.GameObjects.Container;

            /**
             * Passes all Game Objects in this Container to the given callback.
             *
             * Only use this method when you absolutely know that the Container will not be modified during
             * the iteration, i.e. by removing or adding to its contents.
             * @param callback The function to call.
             * @param context Value to use as `this` when executing callback.
             * @param args Additional arguments that will be passed to the callback, after the child.
             */
            iterate(callback: Function, context?: object, ...args: any[]): Phaser.GameObjects.Container;

            /**
             * The number of Game Objects inside this Container.
             */
            readonly length: integer;

            /**
             * Returns the first Game Object within the Container, or `null` if it is empty.
             *
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly first: Phaser.GameObjects.GameObject;

            /**
             * Returns the last Game Object within the Container, or `null` if it is empty.
             *
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly last: Phaser.GameObjects.GameObject;

            /**
             * Returns the next Game Object within the Container, or `null` if it is empty.
             *
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly next: Phaser.GameObjects.GameObject;

            /**
             * Returns the previous Game Object within the Container, or `null` if it is empty.
             *
             * You can move the cursor by calling `Container.next` and `Container.previous`.
             */
            readonly previous: Phaser.GameObjects.GameObject;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The native (un-scaled) width of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             *
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * The Display List plugin.
         *
         * Display Lists belong to a Scene and maintain the list of Game Objects to render every frame.
         *
         * Some of these Game Objects may also be part of the Scene's [Update List]{@link Phaser.GameObjects.UpdateList}, for updating.
         */
        class DisplayList extends Phaser.Structs.List<Phaser.GameObjects.GameObject> {
            /**
             *
             * @param scene The Scene that this Display List belongs to.
             */
            constructor(scene: Phaser.Scene);

            /**
             * The flag the determines whether Game Objects should be sorted when `depthSort()` is called.
             */
            sortChildrenFlag: boolean;

            /**
             * The Scene that this Display List belongs to.
             */
            scene: Phaser.Scene;

            /**
             * The Scene's Systems.
             */
            systems: Phaser.Scenes.Systems;

            /**
             * Force a sort of the display list on the next call to depthSort.
             */
            queueDepthSort(): void;

            /**
             * Immediately sorts the display list if the flag is set.
             */
            depthSort(): void;

            /**
             * Compare the depth of two Game Objects.
             * @param childA The first Game Object.
             * @param childB The second Game Object.
             */
            sortByDepth(childA: Phaser.GameObjects.GameObject, childB: Phaser.GameObjects.GameObject): integer;

            /**
             * Given an array of Game Objects, sort the array and return it, so that
             * the objects are in index order with the lowest at the bottom.
             * @param gameObjects The array of Game Objects to sort.
             */
            sortGameObjects(gameObjects: Phaser.GameObjects.GameObject[]): any[];

            /**
             * Get the top-most Game Object in the given array of Game Objects, after sorting it.
             *
             * Note that the given array is sorted in place, even though it isn't returned directly it will still be updated.
             * @param gameObjects The array of Game Objects.
             */
            getTopGameObject(gameObjects: Phaser.GameObjects.GameObject[]): Phaser.GameObjects.GameObject;

            /**
             * All members of the group.
             */
            getChildren(): Phaser.GameObjects.GameObject[];

        }

        /**
         * [description]
         */
        class DOMElement extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.ComputedSize, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param element The DOM Element to use.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, element?: string | HTMLElement);

            /**
             * Compares the renderMask with the renderFlags to see if this Game Object will render or not.
             *
             * DOMElements always return `true` as they need to still set values during the render pass, even if not visible.
             */
            willRender(): boolean;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The native (un-scaled) width of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             *
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             *
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * The base class that all Game Objects extend.
         * You don't create GameObjects directly and they cannot be added to the display list.
         * Instead, use them as the base for your own custom classes.
         */
        class GameObject extends Phaser.Events.EventEmitter {
            /**
             *
             * @param scene The Scene to which this Game Object belongs.
             * @param type A textual representation of the type of Game Object, i.e. `sprite`.
             */
            constructor(scene: Phaser.Scene, type: string);

            /**
             * The Scene to which this Game Object belongs.
             * Game Objects can only belong to one Scene.
             */
            protected scene: Phaser.Scene;

            /**
             * A textual representation of this Game Object, i.e. `sprite`.
             * Used internally by Phaser but is available for your own custom classes to populate.
             */
            type: string;

            /**
             * The parent Container of this Game Object, if it has one.
             */
            parentContainer: Phaser.GameObjects.Container;

            /**
             * The name of this Game Object.
             * Empty by default and never populated by Phaser, this is left for developers to use.
             */
            name: string;

            /**
             * The active state of this Game Object.
             * A Game Object with an active state of `true` is processed by the Scenes UpdateList, if added to it.
             * An active object is one which is having its logic and internal systems updated.
             */
            active: boolean;

            /**
             * The Tab Index of the Game Object.
             * Reserved for future use by plugins and the Input Manager.
             */
            tabIndex: integer;

            /**
             * A Data Manager.
             * It allows you to store, query and get key/value paired information specific to this Game Object.
             * `null` by default. Automatically created if you use `getData` or `setData` or `setDataEnabled`.
             */
            data: Phaser.Data.DataManager;

            /**
             * The flags that are compared against `RENDER_MASK` to determine if this Game Object will render or not.
             * The bits are 0001 | 0010 | 0100 | 1000 set by the components Visible, Alpha, Transform and Texture respectively.
             * If those components are not used by your custom class then you can use this bitmask as you wish.
             */
            renderFlags: integer;

            /**
             * A bitmask that controls if this Game Object is drawn by a Camera or not.
             * Not usually set directly, instead call `Camera.ignore`, however you can
             * set this property directly using the Camera.id property:
             */
            cameraFilter: number;

            /**
             * If this Game Object is enabled for input then this property will contain an InteractiveObject instance.
             * Not usually set directly. Instead call `GameObject.setInteractive()`.
             */
            input: Phaser.Input.InteractiveObject;

            /**
             * If this Game Object is enabled for physics then this property will contain a reference to a Physics Body.
             */
            body: object | Phaser.Physics.Arcade.Body | Phaser.Physics.Impact.Body;

            /**
             * This Game Object will ignore all calls made to its destroy method if this flag is set to `true`.
             * This includes calls that may come from a Group, Container or the Scene itself.
             * While it allows you to persist a Game Object across Scenes, please understand you are entirely
             * responsible for managing references to and from this Game Object.
             */
            ignoreDestroy: boolean;

            /**
             * Sets the `active` property of this Game Object and returns this Game Object for further chaining.
             * A Game Object with its `active` property set to `true` will be updated by the Scenes UpdateList.
             * @param value True if this Game Object should be set as active, false if not.
             */
            setActive(value: boolean): this;

            /**
             * Sets the `name` property of this Game Object and returns this Game Object for further chaining.
             * The `name` property is not populated by Phaser and is presented for your own use.
             * @param value The name to be given to this Game Object.
             */
            setName(value: string): this;

            /**
             * Adds a Data Manager component to this Game Object.
             */
            setDataEnabled(): this;

            /**
             * Allows you to store a key value pair within this Game Objects Data Manager.
             *
             * If the Game Object has not been enabled for data (via `setDataEnabled`) then it will be enabled
             * before setting the value.
             *
             * If the key doesn't already exist in the Data Manager then it is created.
             *
             * ```javascript
             * sprite.setData('name', 'Red Gem Stone');
             * ```
             *
             * You can also pass in an object of key value pairs as the first argument:
             *
             * ```javascript
             * sprite.setData({ name: 'Red Gem Stone', level: 2, owner: 'Link', gold: 50 });
             * ```
             *
             * To get a value back again you can call `getData`:
             *
             * ```javascript
             * sprite.getData('gold');
             * ```
             *
             * Or you can access the value directly via the `values` property, where it works like any other variable:
             *
             * ```javascript
             * sprite.data.values.gold += 50;
             * ```
             *
             * When the value is first set, a `setdata` event is emitted from this Game Object.
             *
             * If the key already exists, a `changedata` event is emitted instead, along an event named after the key.
             * For example, if you updated an existing key called `PlayerLives` then it would emit the event `changedata_PlayerLives`.
             * These events will be emitted regardless if you use this method to set the value, or the direct `values` setter.
             *
             * Please note that the data keys are case-sensitive and must be valid JavaScript Object property strings.
             * This means the keys `gold` and `Gold` are treated as two unique values within the Data Manager.
             * @param key The key to set the value for. Or an object or key value pairs. If an object the `data` argument is ignored.
             * @param data The value to set for the given key. If an object is provided as the key this argument is ignored.
             */
            setData(key: string | object, data: any): this;

            /**
             * Retrieves the value for the given key in this Game Objects Data Manager, or undefined if it doesn't exist.
             *
             * You can also access values via the `values` object. For example, if you had a key called `gold` you can do either:
             *
             * ```javascript
             * sprite.getData('gold');
             * ```
             *
             * Or access the value directly:
             *
             * ```javascript
             * sprite.data.values.gold;
             * ```
             *
             * You can also pass in an array of keys, in which case an array of values will be returned:
             *
             * ```javascript
             * sprite.getData([ 'gold', 'armor', 'health' ]);
             * ```
             *
             * This approach is useful for destructuring arrays in ES6.
             * @param key The key of the value to retrieve, or an array of keys.
             */
            getData(key: string | string[]): any;

            /**
             * Pass this Game Object to the Input Manager to enable it for Input.
             *
             * Input works by using hit areas, these are nearly always geometric shapes, such as rectangles or circles, that act as the hit area
             * for the Game Object. However, you can provide your own hit area shape and callback, should you wish to handle some more advanced
             * input detection.
             *
             * If no arguments are provided it will try and create a rectangle hit area based on the texture frame the Game Object is using. If
             * this isn't a texture-bound object, such as a Graphics or BitmapText object, this will fail, and you'll need to provide a specific
             * shape for it to use.
             *
             * You can also provide an Input Configuration Object as the only argument to this method.
             * @param shape Either an input configuration object, or a geometric shape that defines the hit area for the Game Object. If not specified a Rectangle will be used.
             * @param callback A callback to be invoked when the Game Object is interacted with. If you provide a shape you must also provide a callback.
             * @param dropZone Should this Game Object be treated as a drop zone target? Default false.
             */
            setInteractive(shape?: Phaser.Input.InputConfiguration | any, callback?: HitAreaCallback, dropZone?: boolean): this;

            /**
             * If this Game Object has previously been enabled for input, this will disable it.
             *
             * An object that is disabled for input stops processing or being considered for
             * input events, but can be turned back on again at any time by simply calling
             * `setInteractive()` with no arguments provided.
             *
             * If want to completely remove interaction from this Game Object then use `removeInteractive` instead.
             */
            disableInteractive(): this;

            /**
             * If this Game Object has previously been enabled for input, this will queue it
             * for removal, causing it to no longer be interactive. The removal happens on
             * the next game step, it is not immediate.
             *
             * The Interactive Object that was assigned to this Game Object will be destroyed,
             * removed from the Input Manager and cleared from this Game Object.
             *
             * If you wish to re-enable this Game Object at a later date you will need to
             * re-create its InteractiveObject by calling `setInteractive` again.
             *
             * If you wish to only temporarily stop an object from receiving input then use
             * `disableInteractive` instead, as that toggles the interactive state, where-as
             * this erases it completely.
             *
             * If you wish to resize a hit area, don't remove and then set it as being
             * interactive. Instead, access the hitarea object directly and resize the shape
             * being used. I.e.: `sprite.input.hitArea.setSize(width, height)` (assuming the
             * shape is a Rectangle, which it is by default.)
             */
            removeInteractive(): this;

            /**
             * To be overridden by custom GameObjects. Allows base objects to be used in a Pool.
             * @param args args
             */
            update(...args: any[]): void;

            /**
             * Returns a JSON representation of the Game Object.
             */
            toJSON(): JSONGameObject;

            /**
             * Compares the renderMask with the renderFlags to see if this Game Object will render or not.
             * Also checks the Game Object against the given Cameras exclusion list.
             * @param camera The Camera to check against this Game Object.
             */
            willRender(camera: Phaser.Cameras.Scene2D.Camera): boolean;

            /**
             * Returns an array containing the display list index of either this Game Object, or if it has one,
             * its parent Container. It then iterates up through all of the parent containers until it hits the
             * root of the display list (which is index 0 in the returned array).
             *
             * Used internally by the InputPlugin but also useful if you wish to find out the display depth of
             * this Game Object and all of its ancestors.
             */
            getIndexList(): integer[];

            /**
             * Destroys this Game Object removing it from the Display List and Update List and
             * severing all ties to parent resources.
             *
             * Also removes itself from the Input Manager and Physics Manager if previously enabled.
             *
             * Use this to remove a Game Object from your game if you don't ever plan to use it again.
             * As long as no reference to it exists within your own code it should become free for
             * garbage collection by the browser.
             *
             * If you just want to temporarily disable an object then look at using the
             * Game Object Pool instead of destroying it, as destroyed objects cannot be resurrected.
             * @param fromScene Is this Game Object being destroyed as the result of a Scene shutdown? Default false.
             */
            destroy(fromScene?: boolean): void;

            /**
             * The bitmask that `GameObject.renderFlags` is compared against to determine if the Game Object will render or not.
             */
            static readonly RENDER_MASK: integer;

        }

        /**
         * The Game Object Creator is a Scene plugin that allows you to quickly create many common
         * types of Game Objects and return them. Unlike the Game Object Factory, they are not automatically
         * added to the Scene.
         *
         * Game Objects directly register themselves with the Creator and inject their own creation
         * methods into the class.
         */
        class GameObjectCreator {
            /**
             *
             * @param scene The Scene to which this Game Object Factory belongs.
             */
            constructor(scene: Phaser.Scene);

            /**
             * Creates a new Dynamic Bitmap Text Game Object and returns it.
             *
             * Note: This method will only be available if the Dynamic Bitmap Text Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            dynamicBitmapText(config: BitmapTextConfig, addToScene?: boolean): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Creates a new Bitmap Text Game Object and returns it.
             *
             * Note: This method will only be available if the Bitmap Text Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            bitmapText(config: BitmapTextConfig, addToScene?: boolean): Phaser.GameObjects.BitmapText;

            /**
             * Creates a new Blitter Game Object and returns it.
             *
             * Note: This method will only be available if the Blitter Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            blitter(config: object, addToScene?: boolean): Phaser.GameObjects.Blitter;

            /**
             * Creates a new Container Game Object and returns it.
             *
             * Note: This method will only be available if the Container Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            container(config: object, addToScene?: boolean): Phaser.GameObjects.Container;

            /**
             * The Scene to which this Game Object Creator belongs.
             */
            protected scene: Phaser.Scene;

            /**
             * A reference to the Scene.Systems.
             */
            protected systems: Phaser.Scenes.Systems;

            /**
             * A reference to the Scene Display List.
             */
            protected displayList: Phaser.GameObjects.DisplayList;

            /**
             * A reference to the Scene Update List.
             */
            protected "updateList;": Phaser.GameObjects.UpdateList;

            /**
             * Creates a new Graphics Game Object and returns it.
             *
             * Note: This method will only be available if the Graphics Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            graphics(config: object, addToScene?: boolean): Phaser.GameObjects.Graphics;

            /**
             * Creates a new Group Game Object and returns it.
             *
             * Note: This method will only be available if the Group Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             */
            group(config: GroupConfig): Phaser.GameObjects.Group;

            /**
             * Creates a new Image Game Object and returns it.
             *
             * Note: This method will only be available if the Image Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            image(config: object, addToScene?: boolean): Phaser.GameObjects.Image;

            /**
             * Creates a new Mesh Game Object and returns it.
             *
             * Note: This method will only be available if the Mesh Game Object and WebGL support have been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            mesh(config: object, addToScene?: boolean): Phaser.GameObjects.Mesh;

            /**
             * Creates a new Particle Emitter Manager Game Object and returns it.
             *
             * Note: This method will only be available if the Particles Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            particles(config: object, addToScene?: boolean): Phaser.GameObjects.Particles.ParticleEmitterManager;

            /**
             * Creates a new Quad Game Object and returns it.
             *
             * Note: This method will only be available if the Quad Game Object and WebGL support have been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            quad(config: object, addToScene?: boolean): Phaser.GameObjects.Quad;

            /**
             * Creates a new Render Texture Game Object and returns it.
             *
             * Note: This method will only be available if the Render Texture Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            renderTexture(config: RenderTextureConfig, addToScene?: boolean): Phaser.GameObjects.RenderTexture;

            /**
             * Creates a new Sprite Game Object and returns it.
             *
             * Note: This method will only be available if the Sprite Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            sprite(config: SpriteConfig, addToScene?: boolean): Phaser.GameObjects.Sprite;

            /**
             * Creates a new Text Game Object and returns it.
             *
             * Note: This method will only be available if the Text Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            text(config: object, addToScene?: boolean): Phaser.GameObjects.Text;

            /**
             * Creates a new TileSprite Game Object and returns it.
             *
             * Note: This method will only be available if the TileSprite Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             * @param addToScene Add this Game Object to the Scene after creating it? If set this argument overrides the `add` property in the config object.
             */
            tileSprite(config: TileSprite, addToScene?: boolean): Phaser.GameObjects.TileSprite;

            /**
             * Creates a new Zone Game Object and returns it.
             *
             * Note: This method will only be available if the Zone Game Object has been built into Phaser.
             * @param config The configuration object this Game Object will use to create itself.
             */
            zone(config: object): Phaser.GameObjects.Zone;

            /**
             * Creates a Tilemap from the given key or data, or creates a blank Tilemap if no key/data provided.
             * When loading from CSV or a 2D array, you should specify the tileWidth & tileHeight. When parsing
             * from a map from Tiled, the tileWidth, tileHeight, width & height will be pulled from the map
             * data. For an empty map, you should specify tileWidth, tileHeight, width & height.
             * @param config The config options for the Tilemap.
             */
            tilemap(config?: TilemapConfig): Phaser.Tilemaps.Tilemap;

            /**
             * Creates a new Tween object and returns it.
             *
             * Note: This method will only be available if Tweens have been built into Phaser.
             * @param config The Tween configuration.
             */
            tween(config: object): Phaser.Tweens.Tween;

        }

        /**
         * The Game Object Factory is a Scene plugin that allows you to quickly create many common
         * types of Game Objects and have them automatically registered with the Scene.
         *
         * Game Objects directly register themselves with the Factory and inject their own creation
         * methods into the class.
         */
        class GameObjectFactory {
            /**
             *
             * @param scene The Scene to which this Game Object Factory belongs.
             */
            constructor(scene: Phaser.Scene);

            /**
             * Creates a new Path Object.
             * @param x The horizontal position of this Path.
             * @param y The vertical position of this Path.
             */
            path(x: number, y: number): Phaser.Curves.Path;

            /**
             * Creates a new Dynamic Bitmap Text Game Object and adds it to the Scene.
             *
             * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
             *
             * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
             * match the font structure.
             *
             * Dynamic Bitmap Text objects are different from Static Bitmap Text in that they invoke a callback for each
             * letter being rendered during the render pass. This callback allows you to manipulate the properties of
             * each letter being rendered, such as its position, scale or tint, allowing you to create interesting effects
             * like jiggling text, which can't be done with Static text. This means that Dynamic Text takes more processing
             * time, so only use them if you require the callback ability they have.
             *
             * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
             * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
             * processing the font texture in an image editor, applying fills and any other effects required.
             *
             * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
             *
             * To create a BitmapText data files you need a 3rd party app such as:
             *
             * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
             * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
             * Littera (Web-based, free): http://kvazars.com/littera/
             *
             * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
             * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
             *
             * Note: This method will only be available if the Dynamic Bitmap Text Game Object has been built into Phaser.
             * @param x The x position of the Game Object.
             * @param y The y position of the Game Object.
             * @param font The key of the font to use from the BitmapFont cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size to set.
             */
            dynamicBitmapText(x: number, y: number, font: string, text?: string | string[], size?: number): Phaser.GameObjects.DynamicBitmapText;

            /**
             * Creates a new Bitmap Text Game Object and adds it to the Scene.
             *
             * BitmapText objects work by taking a texture file and an XML or JSON file that describes the font structure.
             *
             * During rendering for each letter of the text is rendered to the display, proportionally spaced out and aligned to
             * match the font structure.
             *
             * BitmapText objects are less flexible than Text objects, in that they have less features such as shadows, fills and the ability
             * to use Web Fonts, however you trade this flexibility for rendering speed. You can also create visually compelling BitmapTexts by
             * processing the font texture in an image editor, applying fills and any other effects required.
             *
             * To create multi-line text insert \r, \n or \r\n escape codes into the text string.
             *
             * To create a BitmapText data files you need a 3rd party app such as:
             *
             * BMFont (Windows, free): http://www.angelcode.com/products/bmfont/
             * Glyph Designer (OS X, commercial): http://www.71squared.com/en/glyphdesigner
             * Littera (Web-based, free): http://kvazars.com/littera/
             *
             * For most use cases it is recommended to use XML. If you wish to use JSON, the formatting should be equal to the result of
             * converting a valid XML file through the popular X2JS library. An online tool for conversion can be found here: http://codebeautify.org/xmltojson
             *
             * Note: This method will only be available if the Bitmap Text Game Object has been built into Phaser.
             * @param x The x position of the Game Object.
             * @param y The y position of the Game Object.
             * @param font The key of the font to use from the BitmapFont cache.
             * @param text The string, or array of strings, to be set as the content of this Bitmap Text.
             * @param size The font size to set.
             * @param align The alignment of the text in a multi-line BitmapText object. Default 0.
             */
            bitmapText(x: number, y: number, font: string, text?: string | string[], size?: number, align?: integer): Phaser.GameObjects.BitmapText;

            /**
             * Creates a new Blitter Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Blitter Game Object has been built into Phaser.
             * @param x The x position of the Game Object.
             * @param y The y position of the Game Object.
             * @param key The key of the Texture the Blitter object will use.
             * @param frame The default Frame children of the Blitter will use.
             */
            blitter(x: number, y: number, key: string, frame?: string | integer): Phaser.GameObjects.Blitter;

            /**
             * Creates a new Container Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Container Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param children An optional array of Game Objects to add to this Container.
             */
            container(x: number, y: number, children?: Phaser.GameObjects.GameObject | Phaser.GameObjects.GameObject[]): Phaser.GameObjects.Container;

            /**
             * Creates a new Image Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Image Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param element The DOM element.
             */
            dom(x: number, y: number, element: string): Phaser.GameObjects.DOMElement;

            /**
             * The Scene to which this Game Object Factory belongs.
             */
            protected scene: Phaser.Scene;

            /**
             * A reference to the Scene.Systems.
             */
            protected systems: Phaser.Scenes.Systems;

            /**
             * A reference to the Scene Display List.
             */
            protected displayList: Phaser.GameObjects.DisplayList;

            /**
             * A reference to the Scene Update List.
             */
            protected "updateList;": Phaser.GameObjects.UpdateList;

            /**
             * Adds an existing Game Object to this Scene.
             *
             * If the Game Object renders, it will be added to the Display List.
             * If it has a `preUpdate` method, it will be added to the Update List.
             * @param child The child to be added to this Scene.
             */
            existing(child: Phaser.GameObjects.GameObject): Phaser.GameObjects.GameObject;

            /**
             * Creates a new Graphics Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Graphics Game Object has been built into Phaser.
             * @param config The Graphics configuration.
             */
            graphics(config?: GraphicsOptions): Phaser.GameObjects.Graphics;

            /**
             * Creates a new Group Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Group Game Object has been built into Phaser.
             * @param children Game Objects to add to this Group; or the `config` argument.
             * @param config A Group Configuration object.
             */
            group(children?: Phaser.GameObjects.GameObject[] | GroupConfig | GroupConfig[], config?: GroupConfig): Phaser.GameObjects.Group;

            /**
             * Creates a new Image Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Image Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            image(x: number, y: number, texture: string, frame?: string | integer): Phaser.GameObjects.Image;

            /**
             * Creates a new Mesh Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Mesh Game Object and WebGL support have been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param vertices An array containing the vertices data for this Mesh.
             * @param uv An array containing the uv data for this Mesh.
             * @param colors An array containing the color data for this Mesh.
             * @param alphas An array containing the alpha data for this Mesh.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            mesh(x: number, y: number, vertices: number[], uv: number[], colors: number[], alphas: number[], texture: string, frame?: string | integer): Phaser.GameObjects.Mesh;

            /**
             * Creates a new Particle Emitter Manager Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Particles Game Object has been built into Phaser.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             * @param emitters Configuration settings for one or more emitters to create.
             */
            particles(texture: string, frame?: string | integer | object, emitters?: ParticleEmitterConfig | ParticleEmitterConfig[]): Phaser.GameObjects.Particles.ParticleEmitterManager;

            /**
             * Creates a new PathFollower Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the PathFollower Game Object has been built into Phaser.
             * @param path The Path this PathFollower is connected to.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            follower(path: Phaser.Curves.Path, x: number, y: number, texture: string, frame?: string | integer): Phaser.GameObjects.PathFollower;

            /**
             * Creates a new Quad Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Quad Game Object and WebGL support have been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            quad(x: number, y: number, texture: string, frame?: string | integer): Phaser.GameObjects.Quad;

            /**
             * Creates a new Render Texture Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Render Texture Game Object has been built into Phaser.
             *
             * A Render Texture is a special texture that allows any number of Game Objects to be drawn to it. You can take many complex objects and
             * draw them all to this one texture, which can they be used as the texture for other Game Object's. It's a way to generate dynamic
             * textures at run-time that are WebGL friendly and don't invoke expensive GPU uploads.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param width The width of the Render Texture. Default 32.
             * @param height The height of the Render Texture. Default 32.
             */
            renderTexture(x: number, y: number, width?: integer, height?: integer): Phaser.GameObjects.RenderTexture;

            /**
             * Creates a new Arc Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Arc Game Object has been built into Phaser.
             *
             * The Arc Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * When it renders it displays an arc shape. You can control the start and end angles of the arc,
             * as well as if the angles are winding clockwise or anti-clockwise. With the default settings
             * it renders as a complete circle. By changing the angles you can create other arc shapes,
             * such as half-circles.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param radius The radius of the arc. Default 128.
             * @param startAngle The start angle of the arc, in degrees. Default 0.
             * @param endAngle The end angle of the arc, in degrees. Default 360.
             * @param anticlockwise The winding order of the start and end angles. Default false.
             * @param fillColor The color the arc will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the arc will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            arc(x?: number, y?: number, radius?: number, startAngle?: integer, endAngle?: integer, anticlockwise?: boolean, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Arc;

            /**
             * Creates a new Circle Shape Game Object and adds it to the Scene.
             *
             * A Circle is an Arc with no defined start and end angle, making it render as a complete circle.
             *
             * Note: This method will only be available if the Arc Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param radius The radius of the circle. Default 128.
             * @param fillColor The color the circle will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the circle will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            circle(x?: number, y?: number, radius?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Arc;

            /**
             * Creates a new Curve Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Curve Game Object has been built into Phaser.
             *
             * The Curve Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * To render a Curve Shape you must first create a `Phaser.Curves.Curve` object, then pass it to
             * the Curve Shape in the constructor.
             *
             * The Curve shape also has a `smoothness` property and corresponding `setSmoothness` method.
             * This allows you to control how smooth the shape renders in WebGL, by controlling the number of iterations
             * that take place during construction. Increase and decrease the default value for smoother, or more
             * jagged, shapes.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param curve The Curve object to use to create the Shape.
             * @param fillColor The color the curve will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the curve will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            curve(x?: number, y?: number, curve?: Phaser.Curves.Curve, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Curve;

            /**
             * Creates a new Ellipse Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Ellipse Game Object has been built into Phaser.
             *
             * The Ellipse Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * When it renders it displays an ellipse shape. You can control the width and height of the ellipse.
             * If the width and height match it will render as a circle. If the width is less than the height,
             * it will look more like an egg shape.
             *
             * The Ellipse shape also has a `smoothness` property and corresponding `setSmoothness` method.
             * This allows you to control how smooth the shape renders in WebGL, by controlling the number of iterations
             * that take place during construction. Increase and decrease the default value for smoother, or more
             * jagged, shapes.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the ellipse. An ellipse with equal width and height renders as a circle. Default 128.
             * @param height The height of the ellipse. An ellipse with equal width and height renders as a circle. Default 128.
             * @param fillColor The color the ellipse will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the ellipse will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            ellipse(x?: number, y?: number, width?: number, height?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Ellipse;

            /**
             * Creates a new Grid Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Grid Game Object has been built into Phaser.
             *
             * The Grid Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports only fill colors and cannot be stroked.
             *
             * A Grid Shape allows you to display a grid in your game, where you can control the size of the
             * grid as well as the width and height of the grid cells. You can set a fill color for each grid
             * cell as well as an alternate fill color. When the alternate fill color is set then the grid
             * cells will alternate the fill colors as they render, creating a chess-board effect. You can
             * also optionally have an outline fill color. If set, this draws lines between the grid cells
             * in the given color. If you specify an outline color with an alpha of zero, then it will draw
             * the cells spaced out, but without the lines between them.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the grid. Default 128.
             * @param height The height of the grid. Default 128.
             * @param cellWidth The width of one cell in the grid. Default 32.
             * @param cellHeight The height of one cell in the grid. Default 32.
             * @param fillColor The color the grid cells will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the grid cells will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             * @param outlineFillColor The color of the lines between the grid cells.
             * @param outlineFillAlpha The alpha of the lines between the grid cells.
             */
            grid(x?: number, y?: number, width?: number, height?: number, cellWidth?: number, cellHeight?: number, fillColor?: number, fillAlpha?: number, outlineFillColor?: number, outlineFillAlpha?: number): Phaser.GameObjects.Grid;

            /**
             * Creates a new IsoBox Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the IsoBox Game Object has been built into Phaser.
             *
             * The IsoBox Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports only fill colors and cannot be stroked.
             *
             * An IsoBox is an 'isometric' rectangle. Each face of it has a different fill color. You can set
             * the color of the top, left and right faces of the rectangle respectively. You can also choose
             * which of the faces are rendered via the `showTop`, `showLeft` and `showRight` properties.
             *
             * You cannot view an IsoBox from under-neath, however you can change the 'angle' by setting
             * the `projection` property.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param size The width of the iso box in pixels. The left and right faces will be exactly half this value. Default 48.
             * @param height The height of the iso box. The left and right faces will be this tall. The overall height of the isobox will be this value plus half the `size` value. Default 32.
             * @param fillTop The fill color of the top face of the iso box. Default 0xeeeeee.
             * @param fillLeft The fill color of the left face of the iso box. Default 0x999999.
             * @param fillRight The fill color of the right face of the iso box. Default 0xcccccc.
             */
            isobox(x?: number, y?: number, size?: number, height?: number, fillTop?: number, fillLeft?: number, fillRight?: number): Phaser.GameObjects.IsoBox;

            /**
             * Creates a new IsoTriangle Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the IsoTriangle Game Object has been built into Phaser.
             *
             * The IsoTriangle Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports only fill colors and cannot be stroked.
             *
             * An IsoTriangle is an 'isometric' triangle. Think of it like a pyramid. Each face has a different
             * fill color. You can set the color of the top, left and right faces of the triangle respectively
             * You can also choose which of the faces are rendered via the `showTop`, `showLeft` and `showRight` properties.
             *
             * You cannot view an IsoTriangle from under-neath, however you can change the 'angle' by setting
             * the `projection` property. The `reversed` property controls if the IsoTriangle is rendered upside
             * down or not.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param size The width of the iso triangle in pixels. The left and right faces will be exactly half this value. Default 48.
             * @param height The height of the iso triangle. The left and right faces will be this tall. The overall height of the iso triangle will be this value plus half the `size` value. Default 32.
             * @param reversed Is the iso triangle upside down? Default false.
             * @param fillTop The fill color of the top face of the iso triangle. Default 0xeeeeee.
             * @param fillLeft The fill color of the left face of the iso triangle. Default 0x999999.
             * @param fillRight The fill color of the right face of the iso triangle. Default 0xcccccc.
             */
            isotriangle(x?: number, y?: number, size?: number, height?: number, reversed?: boolean, fillTop?: number, fillLeft?: number, fillRight?: number): Phaser.GameObjects.IsoTriangle;

            /**
             * Creates a new Line Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Line Game Object has been built into Phaser.
             *
             * The Line Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports only stroke colors and cannot be filled.
             *
             * A Line Shape allows you to draw a line between two points in your game. You can control the
             * stroke color and thickness of the line. In WebGL only you can also specify a different
             * thickness for the start and end of the line, allowing you to render lines that taper-off.
             *
             * If you need to draw multiple lines in a sequence you may wish to use the Polygon Shape instead.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param x1 The horizontal position of the start of the line. Default 0.
             * @param y1 The vertical position of the start of the line. Default 0.
             * @param x2 The horizontal position of the end of the line. Default 128.
             * @param y2 The vertical position of the end of the line. Default 0.
             * @param strokeColor The color the line will be drawn in, i.e. 0xff0000 for red.
             * @param strokeAlpha The alpha the line will be drawn in. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            line(x?: number, y?: number, x1?: number, y1?: number, x2?: number, y2?: number, strokeColor?: number, strokeAlpha?: number): Phaser.GameObjects.Line;

            /**
             * Creates a new Polygon Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Polygon Game Object has been built into Phaser.
             *
             * The Polygon Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * The Polygon Shape is created by providing a list of points, which are then used to create an
             * internal Polygon geometry object. The points can be set from a variety of formats:
             *
             * - An array of Point or Vector2 objects: `[new Phaser.Math.Vec2(x1, y1), ...]`
             * - An array of objects with public x/y properties: `[obj1, obj2, ...]`
             * - An array of paired numbers that represent point coordinates: `[x1,y1, x2,y2, ...]`
             * - An array of arrays with two elements representing x/y coordinates: `[[x1, y1], [x2, y2], ...]`
             *
             * By default the `x` and `y` coordinates of this Shape refer to the center of it. However, depending
             * on the coordinates of the points provided, the final shape may be rendered offset from its origin.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param points The points that make up the polygon.
             * @param fillColor The color the polygon will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the polygon will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            polygon(x?: number, y?: number, points?: any, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Polygon;

            /**
             * Creates a new Rectangle Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Rectangle Game Object has been built into Phaser.
             *
             * The Rectangle Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * You can change the size of the rectangle by changing the `width` and `height` properties.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param width The width of the rectangle. Default 128.
             * @param height The height of the rectangle. Default 128.
             * @param fillColor The color the rectangle will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the rectangle will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            rectangle(x?: number, y?: number, width?: number, height?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Rectangle;

            /**
             * Creates a new Star Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Star Game Object has been built into Phaser.
             *
             * The Star Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * As the name implies, the Star shape will display a star in your game. You can control several
             * aspects of it including the number of points that constitute the star. The default is 5. If
             * you change it to 4 it will render as a diamond. If you increase them, you'll get a more spiky
             * star shape.
             *
             * You can also control the inner and outer radius, which is how 'long' each point of the star is.
             * Modify these values to create more interesting shapes.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param points The number of points on the star. Default 5.
             * @param innerRadius The inner radius of the star. Default 32.
             * @param outerRadius The outer radius of the star. Default 64.
             * @param fillColor The color the star will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the star will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            star(x?: number, y?: number, points?: number, innerRadius?: number, outerRadius?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Star;

            /**
             * Creates a new Triangle Shape Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Triangle Game Object has been built into Phaser.
             *
             * The Triangle Shape is a Game Object that can be added to a Scene, Group or Container. You can
             * treat it like any other Game Object in your game, such as tweening it, scaling it, or enabling
             * it for input or physics. It provides a quick and easy way for you to render this shape in your
             * game without using a texture, while still taking advantage of being fully batched in WebGL.
             *
             * This shape supports both fill and stroke colors.
             *
             * The Triangle consists of 3 lines, joining up to form a triangular shape. You can control the
             * position of each point of these lines. The triangle is always closed and cannot have an open
             * face. If you require that, consider using a Polygon instead.
             * @param x The horizontal position of this Game Object in the world. Default 0.
             * @param y The vertical position of this Game Object in the world. Default 0.
             * @param x1 The horizontal position of the first point in the triangle. Default 0.
             * @param y1 The vertical position of the first point in the triangle. Default 128.
             * @param x2 The horizontal position of the second point in the triangle. Default 64.
             * @param y2 The vertical position of the second point in the triangle. Default 0.
             * @param x3 The horizontal position of the third point in the triangle. Default 128.
             * @param y3 The vertical position of the third point in the triangle. Default 128.
             * @param fillColor The color the triangle will be filled with, i.e. 0xff0000 for red.
             * @param fillAlpha The alpha the triangle will be filled with. You can also set the alpha of the overall Shape using its `alpha` property.
             */
            triangle(x?: number, y?: number, x1?: number, y1?: number, x2?: number, y2?: number, x3?: number, y3?: number, fillColor?: number, fillAlpha?: number): Phaser.GameObjects.Triangle;

            /**
             * Creates a new Sprite Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Sprite Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            sprite(x: number, y: number, texture: string, frame?: string | integer): Phaser.GameObjects.Sprite;

            /**
             * Creates a new Text Game Object and adds it to the Scene.
             *
             * A Text Game Object.
             *
             * Text objects work by creating their own internal hidden Canvas and then renders text to it using
             * the standard Canvas `fillText` API. It then creates a texture from this canvas which is rendered
             * to your game during the render pass.
             *
             * Because it uses the Canvas API you can take advantage of all the features this offers, such as
             * applying gradient fills to the text, or strokes, shadows and more. You can also use custom fonts
             * loaded externally, such as Google or TypeKit Web fonts.
             *
             * You can only display fonts that are currently loaded and available to the browser: therefore fonts must
             * be pre-loaded. Phaser does not do ths for you, so you will require the use of a 3rd party font loader,
             * or have the fonts ready available in the CSS on the page in which your Phaser game resides.
             *
             * See {@link http://www.jordanm.co.uk/tinytype this compatibility table} for the available default fonts
             * across mobile browsers.
             *
             * A note on performance: Every time the contents of a Text object changes, i.e. changing the text being
             * displayed, or the style of the text, it needs to remake the Text canvas, and if on WebGL, re-upload the
             * new texture to the GPU. This can be an expensive operation if used often, or with large quantities of
             * Text objects in your game. If you run into performance issues you would be better off using Bitmap Text
             * instead, as it benefits from batching and avoids expensive Canvas API calls.
             *
             * Note: This method will only be available if the Text Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param text The text this Text object will display.
             * @param style The Text style configuration object.
             */
            text(x: number, y: number, text: string | string[], style?: object): Phaser.GameObjects.Text;

            /**
             * Creates a new TileSprite Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the TileSprite Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param width The width of the Game Object.
             * @param height The height of the Game Object.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            tileSprite(x: number, y: number, width: number, height: number, texture: string, frame?: string | integer): Phaser.GameObjects.TileSprite;

            /**
             * Creates a new Zone Game Object and adds it to the Scene.
             *
             * Note: This method will only be available if the Zone Game Object has been built into Phaser.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param width The width of the Game Object.
             * @param height The height of the Game Object.
             */
            zone(x: number, y: number, width: number, height: number): Phaser.GameObjects.Zone;

            /**
             * Creates a Tilemap from the given key or data, or creates a blank Tilemap if no key/data provided.
             * When loading from CSV or a 2D array, you should specify the tileWidth & tileHeight. When parsing
             * from a map from Tiled, the tileWidth, tileHeight, width & height will be pulled from the map
             * data. For an empty map, you should specify tileWidth, tileHeight, width & height.
             * @param key The key in the Phaser cache that corresponds to the loaded tilemap data.
             * @param tileWidth The width of a tile in pixels. Pass in `null` to leave as the
             * default. Default 32.
             * @param tileHeight The height of a tile in pixels. Pass in `null` to leave as the
             * default. Default 32.
             * @param width The width of the map in tiles. Pass in `null` to leave as the
             * default. Default 10.
             * @param height The height of the map in tiles. Pass in `null` to leave as the
             * default. Default 10.
             * @param data Instead of loading from the cache, you can also load directly from
             * a 2D array of tile indexes. Pass in `null` for no data.
             * @param insertNull Controls how empty tiles, tiles with an index of -1, in the
             * map data are handled. If `true`, empty locations will get a value of `null`. If `false`, empty
             * location will get a Tile object with an index of -1. If you've a large sparsely populated map and
             * the tile data doesn't need to change then setting this value to `true` will help with memory
             * consumption. However if your map is small or you need to update the tiles dynamically, then leave
             * the default value set. Default false.
             */
            tilemap(key?: string, tileWidth?: integer, tileHeight?: integer, width?: integer, height?: integer, data?: integer[][], insertNull?: boolean): Phaser.Tilemaps.Tilemap;

            /**
             * Creates a new Tween object.
             *
             * Note: This method will only be available Tweens have been built into Phaser.
             * @param config The Tween configuration.
             */
            tween(config: object): Phaser.Tweens.Tween;

        }

        /**
         * A Graphics object is a way to draw primitive shapes to your game. Primitives include forms of geometry, such as
         * Rectangles, Circles, and Polygons. They also include lines, arcs and curves. When you initially create a Graphics
         * object it will be empty.
         *
         * To draw to it you must first specify a line style or fill style (or both), draw shapes using paths, and finally
         * fill or stroke them. For example:
         *
         * ```javascript
         * graphics.lineStyle(5, 0xFF00FF, 1.0);
         * graphics.beginPath();
         * graphics.moveTo(100, 100);
         * graphics.lineTo(200, 200);
         * graphics.closePath();
         * graphics.strokePath();
         * ```
         *
         * There are also many helpful methods that draw and fill/stroke common shapes for you.
         *
         * ```javascript
         * graphics.lineStyle(5, 0xFF00FF, 1.0);
         * graphics.fillStyle(0xFFFFFF, 1.0);
         * graphics.fillRect(50, 50, 400, 200);
         * graphics.strokeRect(50, 50, 400, 200);
         * ```
         *
         * When a Graphics object is rendered it will render differently based on if the game is running under Canvas or WebGL.
         * Under Canvas it will use the HTML Canvas context drawing operations to draw the path.
         * Under WebGL the graphics data is decomposed into polygons. Both of these are expensive processes, especially with
         * complex shapes.
         *
         * If your Graphics object doesn't change much (or at all) once you've drawn your shape to it, then you will help
         * performance by calling {@link Phaser.GameObjects.Graphics#generateTexture}. This will 'bake' the Graphics object into
         * a Texture, and return it. You can then use this Texture for Sprites or other display objects. If your Graphics object
         * updates frequently then you should avoid doing this, as it will constantly generate new textures, which will consume
         * memory.
         *
         * As you can tell, Graphics objects are a bit of a trade-off. While they are extremely useful, you need to be careful
         * in their complexity and quantity of them in your game.
         */
        class Graphics extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible, Phaser.GameObjects.Components.ScrollFactor {
            /**
             *
             * @param scene The Scene to which this Graphics object belongs.
             * @param options Options that set the position and default style of this Graphics object.
             */
            constructor(scene: Phaser.Scene, options?: GraphicsOptions);

            /**
             * The horizontal display origin of the Graphics.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of the Graphics.
             */
            displayOriginY: number;

            /**
             * The array of commands used to render the Graphics.
             */
            commandBuffer: any[];

            /**
             * The default fill color for shapes rendered by this Graphics object.
             */
            defaultFillColor: number;

            /**
             * The default fill alpha for shapes rendered by this Graphics object.
             */
            defaultFillAlpha: number;

            /**
             * The default stroke width for shapes rendered by this Graphics object.
             */
            defaultStrokeWidth: number;

            /**
             * The default stroke color for shapes rendered by this Graphics object.
             */
            defaultStrokeColor: number;

            /**
             * The default stroke alpha for shapes rendered by this Graphics object.
             */
            defaultStrokeAlpha: number;

            /**
             * Set the default style settings for this Graphics object.
             * @param options The styles to set as defaults.
             */
            setDefaultStyles(options: GraphicsStyles): Phaser.GameObjects.Graphics;

            /**
             * Set the current line style.
             * @param lineWidth The stroke width.
             * @param color The stroke color.
             * @param alpha The stroke alpha. Default 1.
             */
            lineStyle(lineWidth: number, color: number, alpha?: number): Phaser.GameObjects.Graphics;

            /**
             * Set the current fill style.
             * @param color The fill color.
             * @param alpha The fill alpha. Default 1.
             */
            fillStyle(color: number, alpha?: number): Phaser.GameObjects.Graphics;

            /**
             * Sets a gradient fill style. This is a WebGL only feature.
             *
             * The gradient color values represent the 4 corners of an untransformed rectangle.
             * The gradient is used to color all filled shapes and paths drawn after calling this method.
             * If you wish to turn a gradient off, call `fillStyle` and provide a new single fill color.
             *
             * When filling a triangle only the first 3 color values provided are used for the 3 points of a triangle.
             *
             * This feature is best used only on rectangles and triangles. All other shapes will give strange results.
             *
             * Note that for objects such as arcs or ellipses, or anything which is made out of triangles, each triangle used
             * will be filled with a gradient on its own. There is no ability to gradient fill a shape or path as a single
             * entity at this time.
             * @param topLeft The tint being applied to the top-left of the Game Object.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             * @param alpha The fill alpha. Default 1.
             */
            fillGradientStyle(topLeft: integer, topRight: integer, bottomLeft: integer, bottomRight: integer, alpha?: number): Phaser.GameObjects.Graphics;

            /**
             * Sets a gradient line style. This is a WebGL only feature.
             *
             * The gradient color values represent the 4 corners of an untransformed rectangle.
             * The gradient is used to color all stroked shapes and paths drawn after calling this method.
             * If you wish to turn a gradient off, call `lineStyle` and provide a new single line color.
             *
             * This feature is best used only on single lines. All other shapes will give strange results.
             *
             * Note that for objects such as arcs or ellipses, or anything which is made out of triangles, each triangle used
             * will be filled with a gradient on its own. There is no ability to gradient stroke a shape or path as a single
             * entity at this time.
             * @param lineWidth The stroke width.
             * @param topLeft The tint being applied to the top-left of the Game Object.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             * @param alpha The fill alpha. Default 1.
             */
            lineGradientStyle(lineWidth: number, topLeft: integer, topRight: integer, bottomLeft: integer, bottomRight: integer, alpha?: number): Phaser.GameObjects.Graphics;

            /**
             * Sets the texture frame this Graphics Object will use when drawing all shapes defined after calling this.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             *
             * Once set, all shapes will use this texture. Call this method with no arguments to clear it.
             *
             * The textures are not tiled. They are stretched to the dimensions of the shapes being rendered. For this reason,
             * it works best with seamless / tileable textures.
             *
             * The mode argument controls how the textures are combined with the fill colors. The default value (0) will
             * multiply the texture by the fill color. A value of 1 will use just the fill color, but the alpha data from the texture,
             * and a value of 2 will use just the texture and no fill color at all.
             * @param key The key of the texture to be used, as stored in the Texture Manager. Leave blank to clear a previously set texture.
             * @param frame The name or index of the frame within the Texture.
             * @param mode The texture tint mode. 0 is multiply, 1 is alpha only and 2 is texture only. Default 0.
             */
            setTexture(key?: string, frame?: string | integer, mode?: number): this;

            /**
             * Start a new shape path.
             */
            beginPath(): Phaser.GameObjects.Graphics;

            /**
             * Close the current path.
             */
            closePath(): Phaser.GameObjects.Graphics;

            /**
             * Fill the current path.
             */
            fillPath(): Phaser.GameObjects.Graphics;

            /**
             * Stroke the current path.
             */
            strokePath(): Phaser.GameObjects.Graphics;

            /**
             * Fill the given circle.
             * @param circle The circle to fill.
             */
            fillCircleShape(circle: Phaser.Geom.Circle): Phaser.GameObjects.Graphics;

            /**
             * Stroke the given circle.
             * @param circle The circle to stroke.
             */
            strokeCircleShape(circle: Phaser.Geom.Circle): Phaser.GameObjects.Graphics;

            /**
             * Fill a circle with the given position and radius.
             * @param x The x coordinate of the center of the circle.
             * @param y The y coordinate of the center of the circle.
             * @param radius The radius of the circle.
             */
            fillCircle(x: number, y: number, radius: number): Phaser.GameObjects.Graphics;

            /**
             * Stroke a circle with the given position and radius.
             * @param x The x coordinate of the center of the circle.
             * @param y The y coordinate of the center of the circle.
             * @param radius The radius of the circle.
             */
            strokeCircle(x: number, y: number, radius: number): Phaser.GameObjects.Graphics;

            /**
             * Fill the given rectangle.
             * @param rect The rectangle to fill.
             */
            fillRectShape(rect: Phaser.Geom.Rectangle): Phaser.GameObjects.Graphics;

            /**
             * Stroke the given rectangle.
             * @param rect The rectangle to stroke.
             */
            strokeRectShape(rect: Phaser.Geom.Rectangle): Phaser.GameObjects.Graphics;

            /**
             * Fill a rectangle with the given position and size.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             */
            fillRect(x: number, y: number, width: number, height: number): Phaser.GameObjects.Graphics;

            /**
             * Stroke a rectangle with the given position and size.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             */
            strokeRect(x: number, y: number, width: number, height: number): Phaser.GameObjects.Graphics;

            /**
             * Fill a rounded rectangle with the given position, size and radius.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             * @param radius The corner radius; It can also be an object to specify different radii for corners Default 20.
             * @param radius.tl Top left Default 20.
             * @param radius.tr Top right Default 20.
             * @param radius.br Bottom right Default 20.
             * @param radius.bl Bottom left Default 20.
             */
            fillRoundedRect(x: number, y: number, width: number, height: number, radius?: number): Phaser.GameObjects.Graphics;

            /**
             * Stroke a rounded rectangle with the given position, size and radius.
             * @param x The x coordinate of the top-left of the rectangle.
             * @param y The y coordinate of the top-left of the rectangle.
             * @param width The width of the rectangle.
             * @param height The height of the rectangle.
             * @param radius The corner radius; It can also be an object to specify different radii for corners Default 20.
             * @param radius.tl Top left Default 20.
             * @param radius.tr Top right Default 20.
             * @param radius.br Bottom right Default 20.
             * @param radius.bl Bottom left Default 20.
             */
            strokeRoundedRect(x: number, y: number, width: number, height: number, radius?: number): Phaser.GameObjects.Graphics;

            /**
             * Fill the given point.
             *
             * Draws a square at the given position, 1 pixel in size by default.
             * @param point The point to fill.
             * @param size The size of the square to draw. Default 1.
             */
            fillPointShape(point: Phaser.Geom.Point | Phaser.Math.Vector2 | object, size?: number): Phaser.GameObjects.Graphics;

            /**
             * Fill a point at the given position.
             *
             * Draws a square at the given position, 1 pixel in size by default.
             * @param x The x coordinate of the point.
             * @param y The y coordinate of the point.
             * @param size The size of the square to draw. Default 1.
             */
            fillPoint(x: number, y: number, size?: number): Phaser.GameObjects.Graphics;

            /**
             * Fill the given triangle.
             * @param triangle The triangle to fill.
             */
            fillTriangleShape(triangle: Phaser.Geom.Triangle): Phaser.GameObjects.Graphics;

            /**
             * Stroke the given triangle.
             * @param triangle The triangle to stroke.
             */
            strokeTriangleShape(triangle: Phaser.Geom.Triangle): Phaser.GameObjects.Graphics;

            /**
             * Fill a triangle with the given points.
             * @param x0 The x coordinate of the first point.
             * @param y0 The y coordinate of the first point.
             * @param x1 The x coordinate of the second point.
             * @param y1 The y coordinate of the second point.
             * @param x2 The x coordinate of the third point.
             * @param y2 The y coordinate of the third point.
             */
            fillTriangle(x0: number, y0: number, x1: number, y1: number, x2: number, y2: number): Phaser.GameObjects.Graphics;

            /**
             * Stroke a triangle with the given points.
             * @param x0 The x coordinate of the first point.
             * @param y0 The y coordinate of the first point.
             * @param x1 The x coordinate of the second point.
             * @param y1 The y coordinate of the second point.
             * @param x2 The x coordinate of the third point.
             * @param y2 The y coordinate of the third point.
             */
            strokeTriangle(x0: number, y0: number, x1: number, y1: number, x2: number, y2: number): Phaser.GameObjects.Graphics;

            /**
             * Draw the given line.
             * @param line The line to stroke.
             */
            strokeLineShape(line: Phaser.Geom.Line): Phaser.GameObjects.Graphics;

            /**
             * Draw a line between the given points.
             * @param x1 The x coordinate of the start point of the line.
             * @param y1 The y coordinate of the start point of the line.
             * @param x2 The x coordinate of the end point of the line.
             * @param y2 The y coordinate of the end point of the line.
             */
            lineBetween(x1: number, y1: number, x2: number, y2: number): Phaser.GameObjects.Graphics;

            /**
             * Draw a line from the current drawing position to the given position.
             *
             * Moves the current drawing position to the given position.
             * @param x The x coordinate to draw the line to.
             * @param y The y coordinate to draw the line to.
             */
            lineTo(x: number, y: number): Phaser.GameObjects.Graphics;

            /**
             * Move the current drawing position to the given position.
             * @param x The x coordinate to move to.
             * @param y The y coordinate to move to.
             */
            moveTo(x: number, y: number): Phaser.GameObjects.Graphics;

            /**
             * [description]
             * @param x [description]
             * @param y [description]
             * @param width [description]
             * @param rgb [description]
             */
            lineFxTo(x: number, y: number, width: number, rgb: number): Phaser.GameObjects.Graphics;

            /**
             * [description]
             * @param x [description]
             * @param y [description]
             * @param width [description]
             * @param rgb [description]
             */
            moveFxTo(x: number, y: number, width: number, rgb: number): Phaser.GameObjects.Graphics;

            /**
             * Stroke the shape represented by the given array of points.
             *
             * Pass `true` to `autoClose` to close the shape automatically.
             * @param points The points to stroke.
             * @param autoClose When `true`, the shape is closed by joining the last point to the first point. Default false.
             * @param endIndex The index of `points` to stop drawing at. Defaults to `points.length`.
             */
            strokePoints(points: any[] | Phaser.Geom.Point[], autoClose?: boolean, endIndex?: integer): Phaser.GameObjects.Graphics;

            /**
             * Fill the shape represented by the given array of points.
             *
             * Pass `true` to `autoClose` to close the shape automatically.
             * @param points The points to fill.
             * @param autoClose Whether to automatically close the polygon. Default false.
             * @param endIndex The index of `points` to stop at. Defaults to `points.length`.
             */
            fillPoints(points: any[] | Phaser.Geom.Point[], autoClose?: boolean, endIndex?: integer): Phaser.GameObjects.Graphics;

            /**
             * Stroke the given ellipse.
             * @param ellipse The ellipse to stroke.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            strokeEllipseShape(ellipse: Phaser.Geom.Ellipse, smoothness?: integer): Phaser.GameObjects.Graphics;

            /**
             * Stroke an ellipse with the given position and size.
             * @param x The x coordinate of the center of the ellipse.
             * @param y The y coordinate of the center of the ellipse.
             * @param width The width of the ellipse.
             * @param height The height of the ellipse.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            strokeEllipse(x: number, y: number, width: number, height: number, smoothness?: integer): Phaser.GameObjects.Graphics;

            /**
             * Fill the given ellipse.
             * @param ellipse The ellipse to fill.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            fillEllipseShape(ellipse: Phaser.Geom.Ellipse, smoothness?: integer): Phaser.GameObjects.Graphics;

            /**
             * Fill an ellipse with the given position and size.
             * @param x The x coordinate of the center of the ellipse.
             * @param y The y coordinate of the center of the ellipse.
             * @param width The width of the ellipse.
             * @param height The height of the ellipse.
             * @param smoothness The number of points to draw the ellipse with. Default 32.
             */
            fillEllipse(x: number, y: number, width: number, height: number, smoothness?: integer): Phaser.GameObjects.Graphics;

            /**
             * Draw an arc.
             *
             * This method can be used to create circles, or parts of circles.
             *
             * Make sure you call `beginPath` before starting the arc unless you wish for the arc to automatically
             * close when filled or stroked.
             *
             * Use the optional `overshoot` argument increase the number of iterations that take place when
             * the arc is rendered in WebGL. This is useful if you're drawing an arc with an especially thick line,
             * as it will allow the arc to fully join-up. Try small values at first, i.e. 0.01.
             *
             * Call {@link Phaser.GameObjects.Graphics#fillPath} or {@link Phaser.GameObjects.Graphics#strokePath} after calling
             * this method to draw the arc.
             * @param x The x coordinate of the center of the circle.
             * @param y The y coordinate of the center of the circle.
             * @param radius The radius of the circle.
             * @param startAngle The starting angle, in radians.
             * @param endAngle The ending angle, in radians.
             * @param anticlockwise Whether the drawing should be anticlockwise or clockwise. Default false.
             * @param overshoot This value allows you to increase the segment iterations in WebGL rendering. Useful if the arc has a thick stroke and needs to overshoot to join-up cleanly. Use small numbers such as 0.01 to start with and increase as needed. Default 0.
             */
            arc(x: number, y: number, radius: number, startAngle: number, endAngle: number, anticlockwise?: boolean, overshoot?: number): Phaser.GameObjects.Graphics;

            /**
             * Creates a pie-chart slice shape centered at `x`, `y` with the given radius.
             * You must define the start and end angle of the slice.
             *
             * Setting the `anticlockwise` argument to `true` creates a shape similar to Pacman.
             * Setting it to `false` creates a shape like a slice of pie.
             *
             * This method will begin a new path and close the path at the end of it.
             * To display the actual slice you need to call either `strokePath` or `fillPath` after it.
             * @param x The horizontal center of the slice.
             * @param y The vertical center of the slice.
             * @param radius The radius of the slice.
             * @param startAngle The start angle of the slice, given in radians.
             * @param endAngle The end angle of the slice, given in radians.
             * @param anticlockwise Whether the drawing should be anticlockwise or clockwise. Default false.
             * @param overshoot This value allows you to overshoot the endAngle by this amount. Useful if the arc has a thick stroke and needs to overshoot to join-up cleanly. Default 0.
             */
            slice(x: number, y: number, radius: number, startAngle: number, endAngle: number, anticlockwise?: boolean, overshoot?: number): Phaser.GameObjects.Graphics;

            /**
             * Saves the state of the Graphics by pushing the current state onto a stack.
             *
             * The most recently saved state can then be restored with {@link Phaser.GameObjects.Graphics#restore}.
             */
            save(): Phaser.GameObjects.Graphics;

            /**
             * Restores the most recently saved state of the Graphics by popping from the state stack.
             *
             * Use {@link Phaser.GameObjects.Graphics#save} to save the current state, and call this afterwards to restore that state.
             *
             * If there is no saved state, this command does nothing.
             */
            restore(): Phaser.GameObjects.Graphics;

            /**
             * Translate the graphics.
             * @param x The horizontal translation to apply.
             * @param y The vertical translation to apply.
             */
            translate(x: number, y: number): Phaser.GameObjects.Graphics;

            /**
             * Scale the graphics.
             * @param x The horizontal scale to apply.
             * @param y The vertical scale to apply.
             */
            scale(x: number, y: number): Phaser.GameObjects.Graphics;

            /**
             * Rotate the graphics.
             * @param radians The rotation angle, in radians.
             */
            rotate(radians: number): Phaser.GameObjects.Graphics;

            /**
             * Clear the command buffer and reset the fill style and line style to their defaults.
             */
            clear(): Phaser.GameObjects.Graphics;

            /**
             * Generate a texture from this Graphics object.
             *
             * If `key` is a string it'll generate a new texture using it and add it into the
             * Texture Manager (assuming no key conflict happens).
             *
             * If `key` is a Canvas it will draw the texture to that canvas context. Note that it will NOT
             * automatically upload it to the GPU in WebGL mode.
             * @param key The key to store the texture with in the Texture Manager, or a Canvas to draw to.
             * @param width The width of the graphics to generate.
             * @param height The height of the graphics to generate.
             */
            generateTexture(key: string | HTMLCanvasElement, width?: integer, height?: integer): Phaser.GameObjects.Graphics;

            /**
             * Internal destroy handler, called as part of the destroy process.
             */
            protected preDestroy(): void;

            /**
             * A Camera used specifically by the Graphics system for rendering to textures.
             */
            static TargetCamera: Phaser.Cameras.Scene2D.Camera;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

        }

        /**
         * A Group is a way for you to create, manipulate, or recycle similar Game Objects.
         *
         * Group membership is non-exclusive. A Game Object can belong to several groups, one group, or none.
         *
         * Groups themselves aren't displayable, and can't be positioned, rotated, scaled, or hidden.
         */
        class Group {
            /**
             *
             * @param scene The scene this group belongs to.
             * @param children Game Objects to add to this group; or the `config` argument.
             * @param config Settings for this group. If `key` is set, Phaser.GameObjects.Group#createMultiple is also called with these settings.
             */
            constructor(scene: Phaser.Scene, children?: Phaser.GameObjects.GameObject[] | GroupConfig | GroupCreateConfig, config?: GroupConfig | GroupCreateConfig);

            /**
             * This scene this group belongs to.
             */
            scene: Phaser.Scene;

            /**
             * Members of this group.
             */
            children: Phaser.Structs.Set<Phaser.GameObjects.GameObject>;

            /**
             * A flag identifying this object as a group.
             */
            isParent: boolean;

            /**
             * The class to create new group members from.
             */
            classType: object;

            /**
             * Whether this group runs its {@link Phaser.GameObjects.Group#preUpdate} method
             * (which may update any members).
             */
            active: boolean;

            /**
             * The maximum size of this group, if used as a pool. -1 is no limit.
             */
            maxSize: integer;

            /**
             * A default texture key to use when creating new group members.
             *
             * This is used in {@link Phaser.GameObjects.Group#create}
             * but not in {@link Phaser.GameObjects.Group#createMultiple}.
             */
            defaultKey: string;

            /**
             * A default texture frame to use when creating new group members.
             */
            defaultFrame: string | integer;

            /**
             * Whether to call the update method of any members.
             */
            runChildUpdate: boolean;

            /**
             * A function to be called when adding or creating group members.
             */
            createCallback: GroupCallback;

            /**
             * A function to be called when removing group members.
             */
            removeCallback: GroupCallback;

            /**
             * A function to be called when creating several group members at once.
             */
            createMultipleCallback: GroupMultipleCreateCallback;

            /**
             * Creates a new Game Object and adds it to this group, unless the group {@link Phaser.GameObjects.Group#isFull is full}.
             *
             * Calls {@link Phaser.GameObjects.Group#createCallback}.
             * @param x The horizontal position of the new Game Object in the world. Default 0.
             * @param y The vertical position of the new Game Object in the world. Default 0.
             * @param key The texture key of the new Game Object. Default defaultKey.
             * @param frame The texture frame of the new Game Object. Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of the new Game Object. Default true.
             * @param active The {@link Phaser.GameObjects.GameObject#active} state of the new Game Object. Default true.
             */
            create(x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean, active?: boolean): any;

            /**
             * Creates several Game Objects and adds them to this group.
             *
             * If the group becomes {@link Phaser.GameObjects.Group#isFull}, no further Game Objects are created.
             *
             * Calls {@link Phaser.GameObjects.Group#createMultipleCallback} and {@link Phaser.GameObjects.Group#createCallback}.
             * @param config Creation settings. This can be a single configuration object or an array of such objects, which will be applied in turn.
             */
            createMultiple(config: GroupCreateConfig | GroupCreateConfig[]): any[];

            /**
             * A helper for {@link Phaser.GameObjects.Group#createMultiple}.
             * @param options Creation settings.
             */
            createFromConfig(options: GroupCreateConfig): any[];

            /**
             * Updates any group members, if {@link Phaser.GameObjects.Group#runChildUpdate} is enabled.
             * @param time The current timestamp.
             * @param delta The delta time elapsed since the last frame.
             */
            preUpdate(time: number, delta: number): void;

            /**
             * Adds a Game Object to this group.
             *
             * Calls {@link Phaser.GameObjects.Group#createCallback}.
             * @param child The Game Object to add.
             * @param addToScene Also add the Game Object to the scene. Default false.
             */
            add(child: Phaser.GameObjects.GameObject, addToScene?: boolean): Phaser.GameObjects.Group;

            /**
             * Adds several Game Objects to this group.
             *
             * Calls {@link Phaser.GameObjects.Group#createCallback}.
             * @param children The Game Objects to add.
             * @param addToScene Also add the Game Objects to the scene. Default false.
             */
            addMultiple(children: Phaser.GameObjects.GameObject[], addToScene?: boolean): Phaser.GameObjects.Group;

            /**
             * Removes a member of this Group and optionally removes it from the Scene and / or destroys it.
             *
             * Calls {@link Phaser.GameObjects.Group#removeCallback}.
             * @param child The Game Object to remove.
             * @param removeFromScene Optionally remove the Group member from the Scene it belongs to. Default false.
             * @param destroyChild Optionally call destroy on the removed Group member. Default false.
             */
            remove(child: Phaser.GameObjects.GameObject, removeFromScene?: boolean, destroyChild?: boolean): Phaser.GameObjects.Group;

            /**
             * Removes all members of this Group and optionally removes them from the Scene and / or destroys them.
             *
             * Does not call {@link Phaser.GameObjects.Group#removeCallback}.
             * @param removeFromScene Optionally remove each Group member from the Scene. Default false.
             * @param destroyChild Optionally call destroy on the removed Group members. Default false.
             */
            clear(removeFromScene?: boolean, destroyChild?: boolean): Phaser.GameObjects.Group;

            /**
             * Tests if a Game Object is a member of this group.
             * @param child A Game Object.
             */
            contains(child: Phaser.GameObjects.GameObject): boolean;

            /**
             * All members of the group.
             */
            getChildren(): Phaser.GameObjects.GameObject[];

            /**
             * The number of members of the group.
             */
            getLength(): integer;

            /**
             * Scans the Group, from top to bottom, for the first member that has an {@link Phaser.GameObjects.GameObject#active} state matching the argument,
             * assigns `x` and `y`, and returns the member.
             *
             * If no matching member is found and `createIfNull` is true and the group isn't full then it will create a new Game Object using `x`, `y`, `key`, `frame`, and `visible`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param state The {@link Phaser.GameObjects.GameObject#active} value to match. Default false.
             * @param createIfNull Create a new Game Object if no matching members are found, using the following arguments. Default false.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            getFirst(state?: boolean, createIfNull?: boolean, x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * Scans the Group, from top to bottom, for the nth member that has an {@link Phaser.GameObjects.GameObject#active} state matching the argument,
             * assigns `x` and `y`, and returns the member.
             *
             * If no matching member is found and `createIfNull` is true and the group isn't full then it will create a new Game Object using `x`, `y`, `key`, `frame`, and `visible`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param nth The nth matching Group member to search for.
             * @param state The {@link Phaser.GameObjects.GameObject#active} value to match. Default false.
             * @param createIfNull Create a new Game Object if no matching members are found, using the following arguments. Default false.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            getFirstNth(nth: integer, state?: boolean, createIfNull?: boolean, x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * Scans the Group for the last member that has an {@link Phaser.GameObjects.GameObject#active} state matching the argument,
             * assigns `x` and `y`, and returns the member.
             *
             * If no matching member is found and `createIfNull` is true and the group isn't full then it will create a new Game Object using `x`, `y`, `key`, `frame`, and `visible`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param state The {@link Phaser.GameObjects.GameObject#active} value to match. Default false.
             * @param createIfNull Create a new Game Object if no matching members are found, using the following arguments. Default false.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            getLast(state?: boolean, createIfNull?: boolean, x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * Scans the Group for the last nth member that has an {@link Phaser.GameObjects.GameObject#active} state matching the argument,
             * assigns `x` and `y`, and returns the member.
             *
             * If no matching member is found and `createIfNull` is true and the group isn't full then it will create a new Game Object using `x`, `y`, `key`, `frame`, and `visible`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param nth The nth matching Group member to search for.
             * @param state The {@link Phaser.GameObjects.GameObject#active} value to match. Default false.
             * @param createIfNull Create a new Game Object if no matching members are found, using the following arguments. Default false.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            getLastNth(nth: integer, state?: boolean, createIfNull?: boolean, x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * Scans the group for the first member that has an {@link Phaser.GameObjects.GameObject#active} state set to `false`,
             * assigns `x` and `y`, and returns the member.
             *
             * If no inactive member is found and the group isn't full then it will create a new Game Object using `x`, `y`, `key`, `frame`, and `visible`.
             * The new Game Object will have its active state set to `true`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            get(x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * Scans the group for the first member that has an {@link Phaser.GameObjects.GameObject#active} state set to `true`,
             * assigns `x` and `y`, and returns the member.
             *
             * If no active member is found and `createIfNull` is `true` and the group isn't full then it will create a new one using `x`, `y`, `key`, `frame`, and `visible`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param createIfNull Create a new Game Object if no matching members are found, using the following arguments. Default false.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            getFirstAlive(createIfNull?: boolean, x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * Scans the group for the first member that has an {@link Phaser.GameObjects.GameObject#active} state set to `false`,
             * assigns `x` and `y`, and returns the member.
             *
             * If no inactive member is found and `createIfNull` is `true` and the group isn't full then it will create a new one using `x`, `y`, `key`, `frame`, and `visible`.
             * The new Game Object will have an active state set to `true`.
             * Unless a new member is created, `key`, `frame`, and `visible` are ignored.
             * @param createIfNull Create a new Game Object if no matching members are found, using the following arguments. Default false.
             * @param x The horizontal position of the Game Object in the world.
             * @param y The vertical position of the Game Object in the world.
             * @param key The texture key assigned to a new Game Object (if one is created). Default defaultKey.
             * @param frame A texture frame assigned to a new Game Object (if one is created). Default defaultFrame.
             * @param visible The {@link Phaser.GameObjects.Components.Visible#visible} state of a new Game Object (if one is created). Default true.
             */
            getFirstDead(createIfNull?: boolean, x?: number, y?: number, key?: string, frame?: string | integer, visible?: boolean): any;

            /**
             * {@link Phaser.GameObjects.Components.Animation#play Plays} an animation for all members of this group.
             * @param key The string-based key of the animation to play.
             * @param startFrame Optionally start the animation playing from this frame index. Default 0.
             */
            playAnimation(key: string, startFrame?: string): Phaser.GameObjects.Group;

            /**
             * Whether this group's size at its {@link Phaser.GameObjects.Group#maxSize maximum}.
             */
            isFull(): boolean;

            /**
             * Counts the number of active (or inactive) group members.
             * @param value Count active (true) or inactive (false) group members. Default true.
             */
            countActive(value?: boolean): integer;

            /**
             * Counts the number of in-use (active) group members.
             */
            getTotalUsed(): integer;

            /**
             * The difference of {@link Phaser.GameObjects.Group#maxSize} and the number of active group members.
             *
             * This represents the number of group members that could be created or reactivated before reaching the size limit.
             */
            getTotalFree(): integer;

            /**
             * Sets the depth of each group member.
             * @param value The amount to set the property to.
             * @param step This is added to the `value` amount, multiplied by the iteration counter.
             */
            setDepth(value: number, step: number): Phaser.GameObjects.Group;

            /**
             * Deactivates a member of this group.
             * @param gameObject A member of this group.
             */
            kill(gameObject: Phaser.GameObjects.GameObject): void;

            /**
             * Deactivates and hides a member of this group.
             * @param gameObject A member of this group.
             */
            killAndHide(gameObject: Phaser.GameObjects.GameObject): void;

            /**
             * Toggles (flips) the visible state of each member of this group.
             */
            toggleVisible(): Phaser.GameObjects.Group;

            /**
             * Empties this group and removes it from the Scene.
             *
             * Does not call {@link Phaser.GameObjects.Group#removeCallback}.
             * @param destroyChildren Also {@link Phaser.GameObjects.GameObject#destroy} each group member. Default false.
             */
            destroy(destroyChildren?: boolean): void;

        }

        /**
         * An Image Game Object.
         *
         * An Image is a light-weight Game Object useful for the display of static images in your game,
         * such as logos, backgrounds, scenery or other non-animated elements. Images can have input
         * events and physics bodies, or be tweened, tinted or scrolled. The main difference between an
         * Image and a Sprite is that you cannot animate an Image as they do not have the Animation component.
         */
        class Image extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Flip, Phaser.GameObjects.Components.GetBounds, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.ScaleMode, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Size, Phaser.GameObjects.Components.TextureCrop, Phaser.GameObjects.Components.Tint, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, texture: string, frame?: string | integer);

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The horizontally flipped state of the Game Object.
             * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             */
            flipX: boolean;

            /**
             * The vertically flipped state of the Game Object.
             * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             */
            flipY: boolean;

            /**
             * Toggles the horizontal flipped state of this Game Object.
             */
            toggleFlipX(): this;

            /**
             * Toggles the vertical flipped state of this Game Object.
             */
            toggleFlipY(): this;

            /**
             * Sets the horizontal flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipX(value: boolean): this;

            /**
             * Sets the vertical flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipY(value: boolean): this;

            /**
             * Sets the horizontal and vertical flipped state of this Game Object.
             * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlip(x: boolean, y: boolean): this;

            /**
             * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
             */
            resetFlip(): this;

            /**
             * Gets the center coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             */
            getCenter<O extends Phaser.Math.Vector2>(output?: O): O;

            /**
             * Gets the top-left corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getTopLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the top-right corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getTopRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bottom-left corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getBottomLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bottom-right corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getBottomRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bounds of this Game Object, regardless of origin.
             * The values are stored and returned in a Rectangle, or Rectangle-like, object.
             * @param output An object to store the values in. If not provided a new Rectangle will be created.
             */
            getBounds<O extends Phaser.Geom.Rectangle>(output?: O): O;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             *
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             */
            scaleMode: Phaser.ScaleModes;

            /**
             * Sets the Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             * @param value The Scale Mode to be used by this Game Object.
             */
            setScaleMode(value: Phaser.ScaleModes): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The native (un-scaled) width of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the size of this Game Object to be that of the given Frame.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param frame The frame to base the size of this Game Object on.
             */
            setSizeToFrame(frame: Phaser.Textures.Frame): this;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             *
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * A boolean flag indicating if this Game Object is being cropped or not.
             * You can toggle this at any time after `setCrop` has been called, to turn cropping on or off.
             * Equally, calling `setCrop` with no arguments will reset the crop and disable it.
             */
            isCropped: boolean;

            /**
             * Applies a crop to a texture based Game Object, such as a Sprite or Image.
             *
             * The crop is a rectangle that limits the area of the texture frame that is visible during rendering.
             *
             * Cropping a Game Object does not change its size, dimensions, physics body or hit area, it just
             * changes what is shown when rendered.
             *
             * The crop coordinates are relative to the texture frame, not the Game Object, meaning 0 x 0 is the top-left.
             *
             * Therefore, if you had a Game Object that had an 800x600 sized texture, and you wanted to show only the left
             * half of it, you could call `setCrop(0, 0, 400, 600)`.
             *
             * It is also scaled to match the Game Object scale automatically. Therefore a crop rect of 100x50 would crop
             * an area of 200x100 when applied to a Game Object that had a scale factor of 2.
             *
             * You can either pass in numeric values directly, or you can provide a single Rectangle object as the first argument.
             *
             * Call this method with no arguments at all to reset the crop, or toggle the property `isCropped` to `false`.
             *
             * You should do this if the crop rectangle becomes the same size as the frame itself, as it will allow
             * the renderer to skip several internal calculations.
             * @param x The x coordinate to start the crop from. Or a Phaser.Geom.Rectangle object, in which case the rest of the arguments are ignored.
             * @param y The y coordinate to start the crop from.
             * @param width The width of the crop rectangle in pixels.
             * @param height The height of the crop rectangle in pixels.
             */
            setCrop(x?: number | Phaser.Geom.Rectangle, y?: number, width?: number, height?: number): this;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string, frame?: string | integer): this;

            /**
             * Sets the frame this Game Object will use to render with.
             *
             * The Frame has to belong to the current Texture being used.
             *
             * It can be either a string or an index.
             *
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * Fill or additive?
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             *
             * Immediately sets the color values back to 0xffffff and the tint type to 'additive',
             * which results in no visible change to the texture.
             */
            clearTint(): this;

            /**
             * Sets an additive tint on this Game Object.
             *
             * The tint works by taking the pixel color values from the Game Objects texture, and then
             * multiplying it by the color value of the tint. You can provide either one color value,
             * in which case the whole Game Object will be tinted in that color. Or you can provide a color
             * per corner. The colors are blended together across the extent of the Game Object.
             *
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             *
             * To remove a tint call `clearTint`.
             *
             * To swap this from being an additive tint to a fill based tint set the property `tintFill` to `true`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If no other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTint(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

            /**
             * Sets a fill-based tint on this Game Object.
             *
             * Unlike an additive tint, a fill-tint literally replaces the pixel colors from the texture
             * with those in the tint. You can use this for effects such as making a player flash 'white'
             * if hit by something. You can provide either one color value, in which case the whole
             * Game Object will be rendered in that color. Or you can provide a color per corner. The colors
             * are blended together across the extent of the Game Object.
             *
             * To modify the tint color once set, either call this method again with new values or use the
             * `tint` property to set all colors at once. Or, use the properties `tintTopLeft`, `tintTopRight,
             * `tintBottomLeft` and `tintBottomRight` to set the corner color values independently.
             *
             * To remove a tint call `clearTint`.
             *
             * To swap this from being a fill-tint to an additive tint set the property `tintFill` to `false`.
             * @param topLeft The tint being applied to the top-left of the Game Object. If not other values are given this value is applied evenly, tinting the whole Game Object. Default 0xffffff.
             * @param topRight The tint being applied to the top-right of the Game Object.
             * @param bottomLeft The tint being applied to the bottom-left of the Game Object.
             * @param bottomRight The tint being applied to the bottom-right of the Game Object.
             */
            setTintFill(topLeft?: integer, topRight?: integer, bottomLeft?: integer, bottomRight?: integer): this;

            /**
             * The tint value being applied to the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintTopLeft: integer;

            /**
             * The tint value being applied to the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintTopRight: integer;

            /**
             * The tint value being applied to the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintBottomLeft: integer;

            /**
             * The tint value being applied to the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            tintBottomRight: integer;

            /**
             * The tint value being applied to the whole of the Game Object.
             */
            tint: integer;

            /**
             * Does this Game Object have a tint applied to it or not?
             */
            readonly isTinted: boolean;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

        }

        /**
         * A 2D point light.
         *
         * These are typically created by a {@link Phaser.GameObjects.LightsManager}, available from within a scene via `this.lights`.
         *
         * Any Game Objects using the Light2D pipeline will then be affected by these Lights.
         *
         * They can also simply be used to represent a point light for your own purposes.
         */
        class Light {
            /**
             *
             * @param x The horizontal position of the light.
             * @param y The vertical position of the light.
             * @param radius The radius of the light.
             * @param r The red color of the light. A value between 0 and 1.
             * @param g The green color of the light. A value between 0 and 1.
             * @param b The blue color of the light. A value between 0 and 1.
             * @param intensity The intensity of the light.
             */
            constructor(x: number, y: number, radius: number, r: number, g: number, b: number, intensity: number);

            /**
             * The horizontal position of the light.
             */
            x: number;

            /**
             * The vertical position of the light.
             */
            y: number;

            /**
             * The radius of the light.
             */
            radius: number;

            /**
             * The red color of the light. A value between 0 and 1.
             */
            r: number;

            /**
             * The green color of the light. A value between 0 and 1.
             */
            g: number;

            /**
             * The blue color of the light. A value between 0 and 1.
             */
            b: number;

            /**
             * The intensity of the light.
             */
            intensity: number;

            /**
             * The horizontal scroll factor of the light.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of the light.
             */
            scrollFactorY: number;

            /**
             * Set the properties of the light.
             *
             * Sets both horizontal and vertical scroll factor to 1. Use {@link Phaser.GameObjects.Light#setScrollFactor} to set
             * the scroll factor.
             * @param x The horizontal position of the light.
             * @param y The vertical position of the light.
             * @param radius The radius of the light.
             * @param r The red color. A value between 0 and 1.
             * @param g The green color. A value between 0 and 1.
             * @param b The blue color. A value between 0 and 1.
             * @param intensity The intensity of the light.
             */
            set(x: number, y: number, radius: number, r: number, g: number, b: number, intensity: number): Phaser.GameObjects.Light;

            /**
             * Set the scroll factor of the light.
             * @param x The horizontal scroll factor of the light.
             * @param y The vertical scroll factor of the light.
             */
            setScrollFactor(x: number, y: number): Phaser.GameObjects.Light;

            /**
             * Set the color of the light from a single integer RGB value.
             * @param rgb The integer RGB color of the light.
             */
            setColor(rgb: number): Phaser.GameObjects.Light;

            /**
             * Set the intensity of the light.
             * @param intensity The intensity of the light.
             */
            setIntensity(intensity: number): Phaser.GameObjects.Light;

            /**
             * Set the position of the light.
             * @param x The horizontal position of the light.
             * @param y The vertical position of the light.
             */
            setPosition(x: number, y: number): Phaser.GameObjects.Light;

            /**
             * Set the radius of the light.
             * @param radius The radius of the light.
             */
            setRadius(radius: number): Phaser.GameObjects.Light;

        }

        /**
         * Manages Lights for a Scene.
         *
         * Affects the rendering of Game Objects using the `Light2D` pipeline.
         */
        class LightsManager {
            /**
             * The pool of Lights.
             *
             * Used to recycle removed Lights for a more efficient use of memory.
             */
            lightPool: Phaser.GameObjects.Light[];

            /**
             * The Lights in the Scene.
             */
            lights: Phaser.GameObjects.Light[];

            /**
             * Lights that have been culled from a Camera's viewport.
             *
             * Lights in this list will not be rendered.
             */
            culledLights: Phaser.GameObjects.Light[];

            /**
             * The ambient color.
             */
            ambientColor: Object;

            /**
             * Whether the Lights Manager is enabled.
             */
            active: boolean;

            /**
             * Enable the Lights Manager.
             */
            enable(): Phaser.GameObjects.LightsManager;

            /**
             * Disable the Lights Manager.
             */
            disable(): Phaser.GameObjects.LightsManager;

            /**
             * Cull any Lights that aren't visible to the given Camera.
             *
             * Culling Lights improves performance by ensuring that only Lights within a Camera's viewport are rendered.
             * @param camera The Camera to cull Lights for.
             */
            cull(camera: Phaser.Cameras.Scene2D.Camera): Phaser.GameObjects.Light[];

            /**
             * Iterate over each Light with a callback.
             * @param callback The callback that is called with each Light.
             */
            forEachLight(callback: LightForEach): Phaser.GameObjects.LightsManager;

            /**
             * Set the ambient light color.
             * @param rgb The integer RGB color of the ambient light.
             */
            setAmbientColor(rgb: number): Phaser.GameObjects.LightsManager;

            /**
             * Returns the maximum number of Lights allowed to appear at once.
             */
            getMaxVisibleLights(): integer;

            /**
             * Get the number of Lights managed by this Lights Manager.
             */
            getLightCount(): integer;

            /**
             * Add a Light.
             * @param x The horizontal position of the Light. Default 0.
             * @param y The vertical position of the Light. Default 0.
             * @param radius The radius of the Light. Default 100.
             * @param rgb The integer RGB color of the light. Default 0xffffff.
             * @param intensity The intensity of the Light. Default 1.
             */
            addLight(x?: number, y?: number, radius?: number, rgb?: number, intensity?: number): Phaser.GameObjects.Light;

            /**
             * Remove a Light.
             * @param light The Light to remove.
             */
            removeLight(light: Phaser.GameObjects.Light): Phaser.GameObjects.LightsManager;

            /**
             * Shut down the Lights Manager.
             *
             * Recycles all active Lights into the Light pool, resets ambient light color and clears the lists of Lights and
             * culled Lights.
             */
            shutdown(): void;

            /**
             * Destroy the Lights Manager.
             *
             * Cleans up all references by calling {@link Phaser.GameObjects.LightsManager#shutdown}.
             */
            destroy(): void;

        }

        /**
         * A Scene plugin that provides a {@link Phaser.GameObjects.LightsManager} for the Light2D pipeline.
         *
         * Available from within a Scene via `this.lights`.
         *
         * Add Lights using the {@link Phaser.GameObjects.LightsManager#addLight} method:
         *
         * ```javascript
         * // Enable the Lights Manager because it is disabled by default
         * this.lights.enable();
         *
         * // Create a Light at [400, 300] with a radius of 200
         * this.lights.addLight(400, 300, 200);
         * ```
         *
         * For Game Objects to be affected by the Lights when rendered, you will need to set them to use the `Light2D` pipeline like so:
         *
         * ```javascript
         * sprite.setPipeline('Light2D');
         * ```
         */
        class LightsPlugin extends Phaser.GameObjects.LightsManager {
            /**
             *
             * @param scene The Scene that this Lights Plugin belongs to.
             */
            constructor(scene: Phaser.Scene);

            /**
             * A reference to the Scene that this Lights Plugin belongs to.
             */
            scene: Phaser.Scene;

            /**
             * A reference to the Scene's systems.
             */
            systems: Phaser.Scenes.Systems;

            /**
             * Boot the Lights Plugin.
             */
            boot(): void;

            /**
             * Destroy the Lights Plugin.
             *
             * Cleans up all references.
             */
            destroy(): void;

        }

        /**
         * A Mesh Game Object.
         */
        class Mesh extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Alpha, Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Flip, Phaser.GameObjects.Components.GetBounds, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.Origin, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.ScaleMode, Phaser.GameObjects.Components.Size, Phaser.GameObjects.Components.Texture, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible, Phaser.GameObjects.Components.ScrollFactor {
            /**
             *
             * @param scene The Scene to which this Game Object belongs. A Game Object can only belong to one Scene at a time.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param vertices An array containing the vertices data for this Mesh.
             * @param uv An array containing the uv data for this Mesh.
             * @param colors An array containing the color data for this Mesh.
             * @param alphas An array containing the alpha data for this Mesh.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            constructor(scene: Phaser.Scene, x: number, y: number, vertices: number[], uv: number[], colors: number[], alphas: number[], texture: string, frame?: string | integer);

            /**
             * An array containing the vertices data for this Mesh.
             */
            vertices: Float32Array;

            /**
             * An array containing the uv data for this Mesh.
             */
            uv: Float32Array;

            /**
             * An array containing the color data for this Mesh.
             */
            colors: Uint32Array;

            /**
             * An array containing the alpha data for this Mesh.
             */
            alphas: Float32Array;

            /**
             * Fill or additive mode used when blending the color values?
             */
            tintFill: boolean;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The horizontally flipped state of the Game Object.
             * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             */
            flipX: boolean;

            /**
             * The vertically flipped state of the Game Object.
             * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             */
            flipY: boolean;

            /**
             * Toggles the horizontal flipped state of this Game Object.
             */
            toggleFlipX(): this;

            /**
             * Toggles the vertical flipped state of this Game Object.
             */
            toggleFlipY(): this;

            /**
             * Sets the horizontal flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipX(value: boolean): this;

            /**
             * Sets the vertical flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipY(value: boolean): this;

            /**
             * Sets the horizontal and vertical flipped state of this Game Object.
             * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlip(x: boolean, y: boolean): this;

            /**
             * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
             */
            resetFlip(): this;

            /**
             * Gets the center coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             */
            getCenter<O extends Phaser.Math.Vector2>(output?: O): O;

            /**
             * Gets the top-left corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getTopLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the top-right corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getTopRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bottom-left corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getBottomLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bottom-right corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getBottomRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bounds of this Game Object, regardless of origin.
             * The values are stored and returned in a Rectangle, or Rectangle-like, object.
             * @param output An object to store the values in. If not provided a new Rectangle will be created.
             */
            getBounds<O extends Phaser.Geom.Rectangle>(output?: O): O;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             *
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             */
            scaleMode: Phaser.ScaleModes;

            /**
             * Sets the Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             * @param value The Scale Mode to be used by this Game Object.
             */
            setScaleMode(value: Phaser.ScaleModes): this;

            /**
             * The native (un-scaled) width of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the size of this Game Object to be that of the given Frame.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param frame The frame to base the size of this Game Object on.
             */
            setSizeToFrame(frame: Phaser.Textures.Frame): this;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             *
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string, frame?: string | integer): this;

            /**
             * Sets the frame this Game Object will use to render with.
             *
             * The Frame has to belong to the current Texture being used.
             *
             * It can be either a string or an index.
             *
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * The x position of this Game Object.
             */
            x: number;

            /**
             * The y position of this Game Object.
             */
            y: number;

            /**
             * The z position of this Game Object.
             * Note: Do not use this value to set the z-index, instead see the `depth` property.
             */
            z: number;

            /**
             * The w position of this Game Object.
             */
            w: number;

            /**
             * The horizontal scale of this Game Object.
             */
            scaleX: number;

            /**
             * The vertical scale of this Game Object.
             */
            scaleY: number;

            /**
             * The angle of this Game Object as expressed in degrees.
             *
             * Where 0 is to the right, 90 is down, 180 is left.
             *
             * If you prefer to work in radians, see the `rotation` property instead.
             */
            angle: integer;

            /**
             * The angle of this Game Object in radians.
             *
             * If you prefer to work in degrees, see the `angle` property instead.
             */
            rotation: number;

            /**
             * Sets the position of this Game Object.
             * @param x The x position of this Game Object. Default 0.
             * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
             * @param z The z position of this Game Object. Default 0.
             * @param w The w position of this Game Object. Default 0.
             */
            setPosition(x?: number, y?: number, z?: number, w?: number): this;

            /**
             * Sets the position of this Game Object to be a random position within the confines of
             * the given area.
             *
             * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
             *
             * The position does not factor in the size of this Game Object, meaning that only the origin is
             * guaranteed to be within the area.
             * @param x The x position of the top-left of the random area. Default 0.
             * @param y The y position of the top-left of the random area. Default 0.
             * @param width The width of the random area.
             * @param height The height of the random area.
             */
            setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

            /**
             * Sets the rotation of this Game Object.
             * @param radians The rotation of this Game Object, in radians. Default 0.
             */
            setRotation(radians?: number): this;

            /**
             * Sets the angle of this Game Object.
             * @param degrees The rotation of this Game Object, in degrees. Default 0.
             */
            setAngle(degrees?: number): this;

            /**
             * Sets the scale of this Game Object.
             * @param x The horizontal scale of this Game Object.
             * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScale(x: number, y?: number): this;

            /**
             * Sets the x position of this Game Object.
             * @param value The x position of this Game Object. Default 0.
             */
            setX(value?: number): this;

            /**
             * Sets the y position of this Game Object.
             * @param value The y position of this Game Object. Default 0.
             */
            setY(value?: number): this;

            /**
             * Sets the z position of this Game Object.
             * @param value The z position of this Game Object. Default 0.
             */
            setZ(value?: number): this;

            /**
             * Sets the w position of this Game Object.
             * @param value The w position of this Game Object. Default 0.
             */
            setW(value?: number): this;

            /**
             * Gets the local transform matrix for this Game Object.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             */
            getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
             * @param tempMatrix The matrix to populate with the values from this Game Object.
             * @param parentMatrix A temporary matrix to hold parent values during the calculations.
             */
            getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

            /**
             * The visible state of the Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             */
            visible: boolean;

            /**
             * Sets the visibility of this Game Object.
             *
             * An invisible Game Object will skip rendering, but will still process update logic.
             * @param value The visible state of the Game Object.
             */
            setVisible(value: boolean): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

        }

        namespace Particles {
            /**
             * A Particle Emitter property.
             *
             * Facilitates changing Particle properties as they are emitted and throughout their lifetime.
             */
            class EmitterOp {
                /**
                 *
                 * @param config Settings for the Particle Emitter that owns this property.
                 * @param key The name of the property.
                 * @param defaultValue The default value of the property.
                 * @param emitOnly Whether the property can only be modified when a Particle is emitted. Default false.
                 */
                constructor(config: ParticleEmitterConfig, key: string, defaultValue: number, emitOnly?: boolean);

                /**
                 * The name of this property.
                 */
                propertyKey: string;

                /**
                 * The value of this property.
                 */
                propertyValue: number;

                /**
                 * The default value of this property.
                 */
                defaultValue: number;

                /**
                 * The number of steps for stepped easing between {@link Phaser.GameObjects.Particles.EmitterOp#start} and
                 * {@link Phaser.GameObjects.Particles.EmitterOp#end} values, per emit.
                 */
                steps: number;

                /**
                 * The step counter for stepped easing, per emit.
                 */
                counter: number;

                /**
                 * The start value for this property to ease between.
                 */
                start: number;

                /**
                 * The end value for this property to ease between.
                 */
                end: number;

                /**
                 * The easing function to use for updating this property.
                 */
                ease: Function;

                /**
                 * Whether this property can only be modified when a Particle is emitted.
                 *
                 * Set to `true` to allow only {@link Phaser.GameObjects.Particles.EmitterOp#onEmit} callbacks to be set and
                 * affect this property.
                 *
                 * Set to `false` to allow both {@link Phaser.GameObjects.Particles.EmitterOp#onEmit} and
                 * {@link Phaser.GameObjects.Particles.EmitterOp#onUpdate} callbacks to be set and affect this property.
                 */
                emitOnly: boolean;

                /**
                 * The callback to run for Particles when they are emitted from the Particle Emitter.
                 */
                onEmit: EmitterOpOnEmitCallback;

                /**
                 * The callback to run for Particles when they are updated.
                 */
                onUpdate: EmitterOpOnUpdateCallback;

                /**
                 * Load the property from a Particle Emitter configuration object.
                 *
                 * Optionally accepts a new property key to use, replacing the current one.
                 * @param config Settings for the Particle Emitter that owns this property.
                 * @param newKey The new key to use for this property, if any.
                 */
                loadConfig(config?: ParticleEmitterConfig, newKey?: string): void;

                /**
                 * Build a JSON representation of this Particle Emitter property.
                 */
                toJSON(): object;

                /**
                 * Change the current value of the property and update its callback methods.
                 * @param value The value of the property.
                 */
                onChange(value: number): Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * Update the {@link Phaser.GameObjects.Particles.EmitterOp#onEmit} and
                 * {@link Phaser.GameObjects.Particles.EmitterOp#onUpdate} callbacks based on the type of the current
                 * {@link Phaser.GameObjects.Particles.EmitterOp#propertyValue}.
                 */
                setMethods(): Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * Check whether an object has the given property.
                 * @param object The object to check.
                 * @param key The key of the property to look for in the object.
                 */
                has(object: object, key: string): boolean;

                /**
                 * Check whether an object has both of the given properties.
                 * @param object The object to check.
                 * @param key1 The key of the first property to check the object for.
                 * @param key2 The key of the second property to check the object for.
                 */
                hasBoth(object: object, key1: string, key2: string): boolean;

                /**
                 * Check whether an object has at least one of the given properties.
                 * @param object The object to check.
                 * @param key1 The key of the first property to check the object for.
                 * @param key2 The key of the second property to check the object for.
                 */
                hasEither(object: object, key1: string, key2: string): boolean;

                /**
                 * The returned value sets what the property will be at the START of the particles life, on emit.
                 * @param particle The particle.
                 * @param key The name of the property.
                 * @param value The current value of the property.
                 */
                defaultEmit(particle: Phaser.GameObjects.Particles.Particle, key: string, value?: number): number;

                /**
                 * The returned value updates the property for the duration of the particles life.
                 * @param particle The particle.
                 * @param key The name of the property.
                 * @param t The T value (between 0 and 1)
                 * @param value The current value of the property.
                 */
                defaultUpdate(particle: Phaser.GameObjects.Particles.Particle, key: string, t: number, value: number): number;

                /**
                 * An `onEmit` callback that returns the current value of the property.
                 */
                staticValueEmit(): number;

                /**
                 * An `onUpdate` callback that returns the current value of the property.
                 */
                staticValueUpdate(): number;

                /**
                 * An `onEmit` callback that returns a random value from the current value array.
                 */
                randomStaticValueEmit(): number;

                /**
                 * An `onEmit` callback that returns a value between the {@link Phaser.GameObjects.Particles.EmitterOp#start} and
                 * {@link Phaser.GameObjects.Particles.EmitterOp#end} range.
                 * @param particle The particle.
                 * @param key The key of the property.
                 */
                randomRangedValueEmit(particle: Phaser.GameObjects.Particles.Particle, key: string): number;

                /**
                 * An `onEmit` callback that returns a stepped value between the
                 * {@link Phaser.GameObjects.Particles.EmitterOp#start} and {@link Phaser.GameObjects.Particles.EmitterOp#end}
                 * range.
                 */
                steppedEmit(): number;

                /**
                 * An `onEmit` callback that returns an eased value between the
                 * {@link Phaser.GameObjects.Particles.EmitterOp#start} and {@link Phaser.GameObjects.Particles.EmitterOp#end}
                 * range.
                 * @param particle The particle.
                 * @param key The name of the property.
                 */
                easedValueEmit(particle: Phaser.GameObjects.Particles.Particle, key: string): number;

                /**
                 * An `onUpdate` callback that returns an eased value between the
                 * {@link Phaser.GameObjects.Particles.EmitterOp#start} and {@link Phaser.GameObjects.Particles.EmitterOp#end}
                 * range.
                 * @param particle The particle.
                 * @param key The name of the property.
                 * @param t The T value (between 0 and 1)
                 */
                easeValueUpdate(particle: Phaser.GameObjects.Particles.Particle, key: string, t: number): number;

            }

            /**
             * [description]
             */
            class GravityWell {
                /**
                 *
                 * @param x The x coordinate of the Gravity Well, in world space. Default 0.
                 * @param y The y coordinate of the Gravity Well, in world space. Default 0.
                 * @param power The power of the Gravity Well. Default 0.
                 * @param epsilon [description] Default 100.
                 * @param gravity The gravitational force of this Gravity Well. Default 50.
                 */
                constructor(x?: number | GravityWellConfig, y?: number, power?: number, epsilon?: number, gravity?: number);

                /**
                 * The x coordinate of the Gravity Well, in world space.
                 */
                x: number;

                /**
                 * The y coordinate of the Gravity Well, in world space.
                 */
                y: number;

                /**
                 * The active state of the Gravity Well. An inactive Gravity Well will not influence any particles.
                 */
                active: boolean;

                /**
                 * The power of the Gravity Well.
                 */
                power: number;

                /**
                 * [description]
                 */
                epsilon: number;

                /**
                 * Takes a Particle and updates it based on the properties of this Gravity Well.
                 * @param particle The Particle to update.
                 * @param delta The delta time in ms.
                 * @param step The delta value divided by 1000.
                 */
                update(particle: Phaser.GameObjects.Particles.Particle, delta: number, step: number): void;

            }

            /**
             * A Particle is a simple Game Object controlled by a Particle Emitter and Manager, and rendered by the Manager.
             * It uses its own lightweight physics system, and can interact only with its Emitter's bounds and zones.
             */
            class Particle {
                /**
                 *
                 * @param emitter The Emitter to which this Particle belongs.
                 */
                constructor(emitter: Phaser.GameObjects.Particles.ParticleEmitter);

                /**
                 * The Emitter to which this Particle belongs.
                 *
                 * A Particle can only belong to a single Emitter and is created, updated and destroyed via it.
                 */
                emitter: Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * The texture frame used to render this Particle.
                 */
                frame: Phaser.Textures.Frame;

                /**
                 * The position of this Particle within its Emitter's particle pool.
                 */
                index: number;

                /**
                 * The x coordinate of this Particle.
                 */
                x: number;

                /**
                 * The y coordinate of this Particle.
                 */
                y: number;

                /**
                 * The x velocity of this Particle.
                 */
                velocityX: number;

                /**
                 * The y velocity of this Particle.
                 */
                velocityY: number;

                /**
                 * The x acceleration of this Particle.
                 */
                accelerationX: number;

                /**
                 * The y acceleration of this Particle.
                 */
                accelerationY: number;

                /**
                 * The maximum horizontal velocity this Particle can travel at.
                 */
                maxVelocityX: number;

                /**
                 * The maximum vertical velocity this Particle can travel at.
                 */
                maxVelocityY: number;

                /**
                 * The bounciness, or restitution, of this Particle.
                 */
                bounce: number;

                /**
                 * The horizontal scale of this Particle.
                 */
                scaleX: number;

                /**
                 * The vertical scale of this Particle.
                 */
                scaleY: number;

                /**
                 * The alpha value of this Particle.
                 */
                alpha: number;

                /**
                 * The angle of this Particle in degrees.
                 */
                angle: number;

                /**
                 * The angle of this Particle in radians.
                 */
                rotation: number;

                /**
                 * The tint applied to this Particle.
                 */
                tint: integer;

                /**
                 * The lifespan of this Particle in ms.
                 */
                life: number;

                /**
                 * The current life of this Particle in ms.
                 */
                lifeCurrent: number;

                /**
                 * The delay applied to this Particle upon emission, in ms.
                 */
                delayCurrent: number;

                /**
                 * The normalized lifespan T value, where 0 is the start and 1 is the end.
                 */
                lifeT: number;

                /**
                 * The data used by the ease equation.
                 */
                data: object;

                /**
                 * Checks to see if this Particle is alive and updating.
                 */
                isAlive(): boolean;

                /**
                 * Starts this Particle from the given coordinates.
                 * @param x The x coordinate to launch this Particle from.
                 * @param y The y coordinate to launch this Particle from.
                 */
                fire(x: number, y: number): void;

                /**
                 * An internal method that calculates the velocity of the Particle.
                 * @param emitter The Emitter that is updating this Particle.
                 * @param delta The delta time in ms.
                 * @param step The delta value divided by 1000.
                 * @param processors Particle processors (gravity wells).
                 */
                computeVelocity(emitter: Phaser.GameObjects.Particles.ParticleEmitter, delta: number, step: number, processors: any[]): void;

                /**
                 * Checks if this Particle is still within the bounds defined by the given Emitter.
                 *
                 * If not, and depending on the Emitter collision flags, the Particle may either stop or rebound.
                 * @param emitter The Emitter to check the bounds against.
                 */
                checkBounds(emitter: Phaser.GameObjects.Particles.ParticleEmitter): void;

                /**
                 * The main update method for this Particle.
                 *
                 * Updates its life values, computes the velocity and repositions the Particle.
                 * @param delta The delta time in ms.
                 * @param step The delta value divided by 1000.
                 * @param processors An optional array of update processors.
                 */
                update(delta: number, step: number, processors: any[]): boolean;

            }

            /**
             * A particle emitter represents a single particle stream.
             * It controls a pool of {@link Phaser.GameObjects.Particles.Particle Particles} and is controlled by a {@link Phaser.GameObjects.Particles.ParticleEmitterManager Particle Emitter Manager}.
             */
            class ParticleEmitter implements Phaser.GameObjects.Components.BlendMode, Phaser.GameObjects.Components.Mask, Phaser.GameObjects.Components.ScrollFactor, Phaser.GameObjects.Components.Visible {
                /**
                 *
                 * @param manager The Emitter Manager this Emitter belongs to.
                 * @param config Settings for this emitter.
                 */
                constructor(manager: Phaser.GameObjects.Particles.ParticleEmitterManager, config: ParticleEmitterConfig);

                /**
                 * The Emitter Manager this Emitter belongs to.
                 */
                manager: Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * The texture assigned to particles.
                 */
                texture: Phaser.Textures.Texture;

                /**
                 * The texture frames assigned to particles.
                 */
                frames: Phaser.Textures.Frame[];

                /**
                 * The default texture frame assigned to particles.
                 */
                defaultFrame: Phaser.Textures.Frame;

                /**
                 * Names of simple configuration properties.
                 */
                configFastMap: object;

                /**
                 * Names of complex configuration properties.
                 */
                configOpMap: object;

                /**
                 * The name of this Particle Emitter.
                 *
                 * Empty by default and never populated by Phaser, this is left for developers to use.
                 */
                name: string;

                /**
                 * The Particle Class which will be emitted by this Emitter.
                 */
                particleClass: Phaser.GameObjects.Particles.Particle;

                /**
                 * The x-coordinate of the particle origin (where particles will be emitted).
                 */
                x: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The y-coordinate of the particle origin (where particles will be emitted).
                 */
                y: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * A radial emitter will emit particles in all directions between angle min and max,
                 * using {@link Phaser.GameObjects.Particles.ParticleEmitter#speed} as the value. If set to false then this acts as a point Emitter.
                 * A point emitter will emit particles only in the direction derived from the speedX and speedY values.
                 */
                radial: boolean;

                /**
                 * Horizontal acceleration applied to emitted particles, in pixels per second squared.
                 */
                gravityX: number;

                /**
                 * Vertical acceleration applied to emitted particles, in pixels per second squared.
                 */
                gravityY: number;

                /**
                 * Whether accelerationX and accelerationY are non-zero. Set automatically during configuration.
                 */
                acceleration: boolean;

                /**
                 * Horizontal acceleration applied to emitted particles, in pixels per second squared.
                 */
                accelerationX: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * Vertical acceleration applied to emitted particles, in pixels per second squared.
                 */
                accelerationY: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The maximum horizontal velocity of emitted particles, in pixels per second squared.
                 */
                maxVelocityX: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The maximum vertical velocity of emitted particles, in pixels per second squared.
                 */
                maxVelocityY: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The initial horizontal speed of emitted particles, in pixels per second.
                 */
                speedX: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The initial vertical speed of emitted particles, in pixels per second.
                 */
                speedY: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * Whether moveToX and moveToY are nonzero. Set automatically during configuration.
                 */
                moveTo: boolean;

                /**
                 * The x-coordinate emitted particles move toward, when {@link Phaser.GameObjects.Particles.ParticleEmitter#moveTo} is true.
                 */
                moveToX: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The y-coordinate emitted particles move toward, when {@link Phaser.GameObjects.Particles.ParticleEmitter#moveTo} is true.
                 */
                moveToY: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * Whether particles will rebound when they meet the emitter bounds.
                 */
                bounce: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The horizontal scale of emitted particles.
                 */
                scaleX: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The vertical scale of emitted particles.
                 */
                scaleY: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * Color tint applied to emitted particles. Any alpha component (0xAA000000) is ignored.
                 */
                tint: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The alpha (transparency) of emitted particles.
                 */
                alpha: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The lifespan of emitted particles, in ms.
                 */
                lifespan: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The angle of the initial velocity of emitted particles, in degrees.
                 */
                angle: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * The rotation of emitted particles, in degrees.
                 */
                rotate: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * A function to call when a particle is emitted.
                 */
                emitCallback: ParticleEmitterCallback;

                /**
                 * The calling context for {@link Phaser.GameObjects.Particles.ParticleEmitter#emitCallback}.
                 */
                emitCallbackScope: any;

                /**
                 * A function to call when a particle dies.
                 */
                deathCallback: ParticleDeathCallback;

                /**
                 * The calling context for {@link Phaser.GameObjects.Particles.ParticleEmitter#deathCallback}.
                 */
                deathCallbackScope: any;

                /**
                 * Set to hard limit the amount of particle objects this emitter is allowed to create.
                 * 0 means unlimited.
                 */
                maxParticles: integer;

                /**
                 * How many particles are emitted each time particles are emitted (one explosion or one flow cycle).
                 */
                quantity: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * How many ms to wait after emission before the particles start updating.
                 */
                delay: Phaser.GameObjects.Particles.EmitterOp;

                /**
                 * For a flow emitter, the time interval (>= 0) between particle flow cycles in ms.
                 * A value of 0 means there is one particle flow cycle for each logic update (the maximum flow frequency). This is the default setting.
                 * For an exploding emitter, this value will be -1.
                 * Calling {@link Phaser.GameObjects.Particles.ParticleEmitter#flow} also puts the emitter in flow mode (frequency >= 0).
                 * Calling {@link Phaser.GameObjects.Particles.ParticleEmitter#explode} also puts the emitter in explode mode (frequency = -1).
                 */
                frequency: number;

                /**
                 * Controls if the emitter is currently emitting a particle flow (when frequency >= 0).
                 * Already alive particles will continue to update until they expire.
                 * Controlled by {@link Phaser.GameObjects.Particles.ParticleEmitter#start} and {@link Phaser.GameObjects.Particles.ParticleEmitter#stop}.
                 */
                on: boolean;

                /**
                 * Newly emitted particles are added to the top of the particle list, i.e. rendered above those already alive.
                 * Set to false to send them to the back.
                 */
                particleBringToTop: boolean;

                /**
                 * The time rate applied to active particles, affecting lifespan, movement, and tweens. Values larger than 1 are faster than normal.
                 */
                timeScale: number;

                /**
                 * An object describing a shape to emit particles from.
                 */
                emitZone: Phaser.GameObjects.Particles.Zones.EdgeZone | Phaser.GameObjects.Particles.Zones.RandomZone;

                /**
                 * An object describing a shape that deactivates particles when they interact with it.
                 */
                deathZone: Phaser.GameObjects.Particles.Zones.DeathZone;

                /**
                 * A rectangular boundary constraining particle movement.
                 */
                bounds: Phaser.Geom.Rectangle;

                /**
                 * Whether particles interact with the left edge of the emitter {@link Phaser.GameObjects.Particles.ParticleEmitter#bounds}.
                 */
                collideLeft: boolean;

                /**
                 * Whether particles interact with the right edge of the emitter {@link Phaser.GameObjects.Particles.ParticleEmitter#bounds}.
                 */
                collideRight: boolean;

                /**
                 * Whether particles interact with the top edge of the emitter {@link Phaser.GameObjects.Particles.ParticleEmitter#bounds}.
                 */
                collideTop: boolean;

                /**
                 * Whether particles interact with the bottom edge of the emitter {@link Phaser.GameObjects.Particles.ParticleEmitter#bounds}.
                 */
                collideBottom: boolean;

                /**
                 * Whether this emitter updates itself and its particles.
                 *
                 * Controlled by {@link Phaser.GameObjects.Particles.ParticleEmitter#pause}
                 * and {@link Phaser.GameObjects.Particles.ParticleEmitter#resume}.
                 */
                active: boolean;

                /**
                 * Set this to false to hide any active particles.
                 */
                visible: boolean;

                /**
                 * The blend mode of this emitter's particles.
                 */
                blendMode: integer;

                /**
                 * A Game Object whose position is used as the particle origin.
                 */
                follow: Phaser.GameObjects.GameObject;

                /**
                 * The offset of the particle origin from the {@link Phaser.GameObjects.Particles.ParticleEmitter#follow} target.
                 */
                followOffset: Phaser.Math.Vector2;

                /**
                 * Whether the emitter's {@link Phaser.GameObjects.Particles.ParticleEmitter#visible} state will track
                 * the {@link Phaser.GameObjects.Particles.ParticleEmitter#follow} target's visibility state.
                 */
                trackVisible: boolean;

                /**
                 * The current texture frame, as an index of {@link Phaser.GameObjects.Particles.ParticleEmitter#frames}.
                 */
                currentFrame: integer;

                /**
                 * Whether texture {@link Phaser.GameObjects.Particles.ParticleEmitter#frames} are selected at random.
                 */
                randomFrame: boolean;

                /**
                 * The number of consecutive particles that receive a single texture frame (per frame cycle).
                 */
                frameQuantity: integer;

                /**
                 * Merges configuration settings into the emitter's current settings.
                 * @param config Settings for this emitter.
                 */
                fromJSON(config: ParticleEmitterConfig): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Creates a description of this emitter suitable for JSON serialization.
                 * @param output An object to copy output into.
                 */
                toJSON(output?: object): object;

                /**
                 * Continuously moves the particle origin to follow a Game Object's position.
                 * @param target The Game Object to follow.
                 * @param offsetX Horizontal offset of the particle origin from the Game Object. Default 0.
                 * @param offsetY Vertical offset of the particle origin from the Game Object. Default 0.
                 * @param trackVisible Whether the emitter's visible state will track the target's visible state. Default false.
                 */
                startFollow(target: Phaser.GameObjects.GameObject, offsetX?: number, offsetY?: number, trackVisible?: boolean): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Stops following a Game Object.
                 */
                stopFollow(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Chooses a texture frame from {@link Phaser.GameObjects.Particles.ParticleEmitter#frames}.
                 */
                getFrame(): Phaser.Textures.Frame;

                /**
                 * Sets a pattern for assigning texture frames to emitted particles.
                 * @param frames One or more texture frames, or a configuration object.
                 * @param pickRandom Whether frames should be assigned at random from `frames`. Default true.
                 * @param quantity The number of consecutive particles that will receive each frame. Default 1.
                 */
                setFrame(frames: any[] | string | integer | ParticleEmitterFrameConfig, pickRandom?: boolean, quantity?: integer): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Turns {@link Phaser.GameObjects.Particles.ParticleEmitter#radial} particle movement on or off.
                 * @param value Radial mode (true) or point mode (true). Default true.
                 */
                setRadial(value?: boolean): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the position of the emitter's particle origin.
                 * New particles will be emitted here.
                 * @param x The x-coordinate of the particle origin.
                 * @param y The y-coordinate of the particle origin.
                 */
                setPosition(x: number | number[] | EmitterOpOnEmitCallback | object, y: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets or modifies a rectangular boundary constraining the particles.
                 *
                 * To remove the boundary, set {@link Phaser.GameObjects.Particles.ParticleEmitter#bounds} to null.
                 * @param x The x-coordinate of the left edge of the boundary, or an object representing a rectangle.
                 * @param y The y-coordinate of the top edge of the boundary.
                 * @param width The width of the boundary.
                 * @param height The height of the boundary.
                 */
                setBounds(x: number | ParticleEmitterBounds | ParticleEmitterBoundsAlt, y: number, width: number, height: number): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the initial horizontal speed of emitted particles.
                 * Changes the emitter to point mode.
                 * @param value The speed, in pixels per second.
                 */
                setSpeedX(value: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the initial vertical speed of emitted particles.
                 * Changes the emitter to point mode.
                 * @param value The speed, in pixels per second.
                 */
                setSpeedY(value: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the initial radial speed of emitted particles.
                 * Changes the emitter to radial mode.
                 * @param value The speed, in pixels per second.
                 */
                setSpeed(value: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the horizontal scale of emitted particles.
                 * @param value The scale, relative to 1.
                 */
                setScaleX(value: number | number[] | EmitterOpOnUpdateCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the vertical scale of emitted particles.
                 * @param value The scale, relative to 1.
                 */
                setScaleY(value: number | number[] | EmitterOpOnUpdateCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the scale of emitted particles.
                 * @param value The scale, relative to 1.
                 */
                setScale(value: number | number[] | EmitterOpOnUpdateCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the horizontal gravity applied to emitted particles.
                 * @param value Acceleration due to gravity, in pixels per second squared.
                 */
                setGravityX(value: number): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the vertical gravity applied to emitted particles.
                 * @param value Acceleration due to gravity, in pixels per second squared.
                 */
                setGravityY(value: number): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the gravity applied to emitted particles.
                 * @param x Horizontal acceleration due to gravity, in pixels per second squared.
                 * @param y Vertical acceleration due to gravity, in pixels per second squared.
                 */
                setGravity(x: number, y: number): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the opacity of emitted particles.
                 * @param value A value between 0 (transparent) and 1 (opaque).
                 */
                setAlpha(value: number | number[] | EmitterOpOnUpdateCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the angle of a {@link Phaser.GameObjects.Particles.ParticleEmitter#radial} particle stream.
                 * @param value The angle of the initial velocity of emitted particles.
                 */
                setEmitterAngle(value: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the angle of a {@link Phaser.GameObjects.Particles.ParticleEmitter#radial} particle stream.
                 * @param value The angle of the initial velocity of emitted particles.
                 */
                setAngle(value: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the lifespan of newly emitted particles.
                 * @param value The particle lifespan, in ms.
                 */
                setLifespan(value: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the number of particles released at each flow cycle or explosion.
                 * @param quantity The number of particles to release at each flow cycle or explosion.
                 */
                setQuantity(quantity: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets the emitter's {@link Phaser.GameObjects.Particles.ParticleEmitter#frequency}
                 * and {@link Phaser.GameObjects.Particles.ParticleEmitter#quantity}.
                 * @param frequency The time interval (>= 0) of each flow cycle, in ms; or -1 to put the emitter in explosion mode.
                 * @param quantity The number of particles to release at each flow cycle or explosion.
                 */
                setFrequency(frequency: number, quantity?: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets or removes the {@link Phaser.GameObjects.Particles.ParticleEmitter#emitZone}.
                 *
                 * An {@link ParticleEmitterEdgeZoneConfig EdgeZone} places particles on its edges. Its {@link EdgeZoneSource source} can be a Curve, Path, Circle, Ellipse, Line, Polygon, Rectangle, or Triangle; or any object with a suitable {@link EdgeZoneSourceCallback getPoints} method.
                 *
                 * A {@link ParticleEmitterRandomZoneConfig RandomZone} places randomly within its interior. Its {@link RandomZoneSource source} can be a Circle, Ellipse, Line, Polygon, Rectangle, or Triangle; or any object with a suitable {@link RandomZoneSourceCallback getRandomPoint} method.
                 * @param zoneConfig An object describing the zone, or `undefined` to remove any current emit zone.
                 */
                setEmitZone(zoneConfig?: ParticleEmitterEdgeZoneConfig | ParticleEmitterRandomZoneConfig): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets or removes the {@link Phaser.GameObjects.Particles.ParticleEmitter#deathZone}.
                 * @param zoneConfig An object describing the zone, or `undefined` to remove any current death zone.
                 */
                setDeathZone(zoneConfig?: ParticleEmitterDeathZoneConfig): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Creates inactive particles and adds them to this emitter's pool.
                 * @param particleCount The number of particles to create.
                 */
                reserve(particleCount: integer): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Gets the number of active (in-use) particles in this emitter.
                 */
                getAliveParticleCount(): integer;

                /**
                 * Gets the number of inactive (available) particles in this emitter.
                 */
                getDeadParticleCount(): integer;

                /**
                 * Gets the total number of particles in this emitter.
                 */
                getParticleCount(): integer;

                /**
                 * Whether this emitter is at its limit (if set).
                 */
                atLimit(): boolean;

                /**
                 * Sets a function to call for each newly emitted particle.
                 * @param callback The function.
                 * @param context The calling context.
                 */
                onParticleEmit(callback: ParticleEmitterCallback, context?: any): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sets a function to call for each particle death.
                 * @param callback The function.
                 * @param context The function's calling context.
                 */
                onParticleDeath(callback: ParticleDeathCallback, context?: any): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Deactivates every particle in this emitter.
                 */
                killAll(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Calls a function for each active particle in this emitter.
                 * @param callback The function.
                 * @param context The function's calling context.
                 */
                forEachAlive(callback: ParticleEmitterCallback, context: any): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Calls a function for each inactive particle in this emitter.
                 * @param callback The function.
                 * @param context The function's calling context.
                 */
                forEachDead(callback: ParticleEmitterCallback, context: any): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Turns {@link Phaser.GameObjects.Particles.ParticleEmitter#on} the emitter and resets the flow counter.
                 *
                 * If this emitter is in flow mode (frequency >= 0; the default), the particle flow will start (or restart).
                 *
                 * If this emitter is in explode mode (frequency = -1), nothing will happen.
                 * Use {@link Phaser.GameObjects.Particles.ParticleEmitter#explode} or {@link Phaser.GameObjects.Particles.ParticleEmitter#flow} instead.
                 */
                start(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Turns {@link Phaser.GameObjects.Particles.ParticleEmitter#on off} the emitter.
                 */
                stop(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * {@link Phaser.GameObjects.Particles.ParticleEmitter#active Deactivates} the emitter.
                 */
                pause(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * {@link Phaser.GameObjects.Particles.ParticleEmitter#active Activates} the emitter.
                 */
                resume(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Sorts active particles with {@link Phaser.GameObjects.Particles.ParticleEmitter#depthSortCallback}.
                 */
                depthSort(): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Puts the emitter in flow mode (frequency >= 0) and starts (or restarts) a particle flow.
                 *
                 * To resume a flow at the current frequency and quantity, use {@link Phaser.GameObjects.Particles.ParticleEmitter#start} instead.
                 * @param frequency The time interval (>= 0) of each flow cycle, in ms.
                 * @param count The number of particles to emit at each flow cycle. Default 1.
                 */
                flow(frequency: number, count?: number | number[] | EmitterOpOnEmitCallback | object): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Puts the emitter in explode mode (frequency = -1), stopping any current particle flow, and emits several particles all at once.
                 * @param count The amount of Particles to emit.
                 * @param x The x coordinate to emit the Particles from.
                 * @param y The y coordinate to emit the Particles from.
                 */
                explode(count: integer, x: number, y: number): Phaser.GameObjects.Particles.Particle;

                /**
                 * Emits particles at a given position (or the emitter's current position).
                 * @param x The x coordinate to emit the Particles from. Default this.x.
                 * @param y The y coordinate to emit the Particles from. Default this.x.
                 * @param count The number of Particles to emit. Default this.quantity.
                 */
                emitParticleAt(x?: number, y?: number, count?: integer): Phaser.GameObjects.Particles.Particle;

                /**
                 * Emits particles at a given position (or the emitter's current position).
                 * @param count The number of Particles to emit. Default this.quantity.
                 * @param x The x coordinate to emit the Particles from. Default this.x.
                 * @param y The y coordinate to emit the Particles from. Default this.x.
                 */
                emitParticle(count?: integer, x?: number, y?: number): Phaser.GameObjects.Particles.Particle;

                /**
                 * Updates this emitter and its particles.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                preUpdate(time: integer, delta: number): void;

                /**
                 * Calculates the difference of two particles, for sorting them by depth.
                 * @param a The first particle.
                 * @param b The second particle.
                 */
                depthSortCallback(a: object, b: object): integer;

                /**
                 * Calculates the difference of two particles, for sorting them by index.
                 * @param a The first particle.
                 * @param b The second particle.
                 */
                indexSortCallback(a: object, b: object): integer;

                /**
                 * Sets the Blend Mode being used by this Game Object.
                 *
                 * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
                 *
                 * Under WebGL only the following Blend Modes are available:
                 *
                 * * ADD
                 * * MULTIPLY
                 * * SCREEN
                 *
                 * Canvas has more available depending on browser support.
                 *
                 * You can also create your own custom Blend Modes in WebGL.
                 *
                 * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
                 * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
                 * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
                 * are used.
                 * @param value The BlendMode value. Either a string or a CONST.
                 */
                setBlendMode(value: string | Phaser.BlendModes): this;

                /**
                 * The Mask this Game Object is using during render.
                 */
                mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

                /**
                 * Sets the mask that this Game Object will use to render with.
                 *
                 * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
                 * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
                 *
                 * If a mask is already set on this Game Object it will be immediately replaced.
                 *
                 * Masks are positioned in global space and are not relative to the Game Object to which they
                 * are applied. The reason for this is that multiple Game Objects can all share the same mask.
                 *
                 * Masks have no impact on physics or input detection. They are purely a rendering component
                 * that allows you to limit what is visible during the render pass.
                 * @param mask The mask this Game Object will use when rendering.
                 */
                setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

                /**
                 * Clears the mask that this Game Object was using.
                 * @param destroyMask Destroy the mask before clearing it? Default false.
                 */
                clearMask(destroyMask?: boolean): this;

                /**
                 * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
                 * including this one.
                 *
                 * To create the mask you need to pass in a reference to a renderable Game Object.
                 * A renderable Game Object is one that uses a texture to render with, such as an
                 * Image, Sprite, Render Texture or BitmapText.
                 *
                 * If you do not provide a renderable object, and this Game Object has a texture,
                 * it will use itself as the object. This means you can call this method to create
                 * a Bitmap Mask from any renderable Game Object.
                 * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
                 */
                createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

                /**
                 * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
                 * including this one.
                 *
                 * To create the mask you need to pass in a reference to a Graphics Game Object.
                 *
                 * If you do not provide a graphics object, and this Game Object is an instance
                 * of a Graphics object, then it will use itself to create the mask.
                 *
                 * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
                 * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
                 */
                createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

                /**
                 * The horizontal scroll factor of this Game Object.
                 *
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 *
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 *
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 *
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 */
                scrollFactorX: number;

                /**
                 * The vertical scroll factor of this Game Object.
                 *
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 *
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 *
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 *
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 */
                scrollFactorY: number;

                /**
                 * Sets the scroll factor of this Game Object.
                 *
                 * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
                 *
                 * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
                 * It does not change the Game Objects actual position values.
                 *
                 * A value of 1 means it will move exactly in sync with a camera.
                 * A value of 0 means it will not move at all, even if the camera moves.
                 * Other values control the degree to which the camera movement is mapped to this Game Object.
                 *
                 * Please be aware that scroll factor values other than 1 are not taken in to consideration when
                 * calculating physics collisions. Bodies always collide based on their world position, but changing
                 * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
                 * them from physics bodies if not accounted for in your code.
                 * @param x The horizontal scroll factor of this Game Object.
                 * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
                 */
                setScrollFactor(x: number, y?: number): this;

                /**
                 * Sets the visibility of this Game Object.
                 *
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 * @param value The visible state of the Game Object.
                 */
                setVisible(value: boolean): this;

            }

            /**
             * A Particle Emitter Manager creates and controls {@link Phaser.GameObjects.Particles.ParticleEmitter Particle Emitters} and {@link Phaser.GameObjects.Particles.GravityWell Gravity Wells}.
             */
            class ParticleEmitterManager extends Phaser.GameObjects.GameObject implements Phaser.GameObjects.Components.Depth, Phaser.GameObjects.Components.Pipeline, Phaser.GameObjects.Components.Transform, Phaser.GameObjects.Components.Visible {
                /**
                 *
                 * @param scene The Scene to which this Emitter Manager belongs.
                 * @param texture The key of the Texture this Emitter Manager will use to render particles, as stored in the Texture Manager.
                 * @param frame An optional frame from the Texture this Emitter Manager will use to render particles.
                 * @param emitters Configuration settings for one or more emitters to create.
                 */
                constructor(scene: Phaser.Scene, texture: string, frame?: string | integer, emitters?: ParticleEmitterConfig | ParticleEmitterConfig[]);

                /**
                 * The time scale applied to all emitters and particles, affecting flow rate, lifespan, and movement.
                 * Values larger than 1 are faster than normal.
                 * This is multiplied with any timeScale set on each individual emitter.
                 */
                timeScale: number;

                /**
                 * The texture used to render this Emitter Manager's particles.
                 */
                texture: Phaser.Textures.Texture;

                /**
                 * The texture frame used to render this Emitter Manager's particles.
                 */
                frame: Phaser.Textures.Frame;

                /**
                 * Names of this Emitter Manager's texture frames.
                 */
                frameNames: string[];

                /**
                 * A list of Emitters being managed by this Emitter Manager.
                 */
                emitters: Phaser.Structs.List<Phaser.GameObjects.Particles.ParticleEmitter>;

                /**
                 * A list of Gravity Wells being managed by this Emitter Manager.
                 */
                wells: Phaser.Structs.List<Phaser.GameObjects.Particles.GravityWell>;

                /**
                 * Sets the texture and frame this Emitter Manager will use to render with.
                 *
                 * Textures are referenced by their string-based keys, as stored in the Texture Manager.
                 * @param key The key of the texture to be used, as stored in the Texture Manager.
                 * @param frame The name or index of the frame within the Texture.
                 */
                setTexture(key: string, frame?: string | integer): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Sets the frame this Emitter Manager will use to render with.
                 *
                 * The Frame has to belong to the current Texture being used.
                 *
                 * It can be either a string or an index.
                 * @param frame The name or index of the frame within the Texture.
                 */
                setFrame(frame?: string | integer): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Assigns texture frames to an emitter.
                 * @param frames The texture frames.
                 * @param emitter The particle emitter to modify.
                 */
                setEmitterFrames(frames: Phaser.Textures.Frame | Phaser.Textures.Frame[], emitter: Phaser.GameObjects.Particles.ParticleEmitter): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Adds an existing Particle Emitter to this Emitter Manager.
                 * @param emitter The Particle Emitter to add to this Emitter Manager.
                 */
                addEmitter(emitter: Phaser.GameObjects.Particles.ParticleEmitter): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Creates a new Particle Emitter object, adds it to this Emitter Manager and returns a reference to it.
                 * @param config Configuration settings for the Particle Emitter to create.
                 */
                createEmitter(config: ParticleEmitterConfig): Phaser.GameObjects.Particles.ParticleEmitter;

                /**
                 * Adds an existing Gravity Well object to this Emitter Manager.
                 * @param well The Gravity Well to add to this Emitter Manager.
                 */
                addGravityWell(well: Phaser.GameObjects.Particles.GravityWell): Phaser.GameObjects.Particles.GravityWell;

                /**
                 * Creates a new Gravity Well, adds it to this Emitter Manager and returns a reference to it.
                 * @param config Configuration settings for the Gravity Well to create.
                 */
                createGravityWell(config: GravityWellConfig): Phaser.GameObjects.Particles.GravityWell;

                /**
                 * Emits particles from each active emitter.
                 * @param count The number of particles to release from each emitter. The default is the emitter's own {@link Phaser.GameObjects.Particles.ParticleEmitter#quantity}.
                 * @param x The x-coordinate to to emit particles from. The default is the x-coordinate of the emitter's current location.
                 * @param y The y-coordinate to to emit particles from. The default is the y-coordinate of the emitter's current location.
                 */
                emitParticle(count?: integer, x?: number, y?: number): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Emits particles from each active emitter.
                 * @param x The x-coordinate to to emit particles from. The default is the x-coordinate of the emitter's current location.
                 * @param y The y-coordinate to to emit particles from. The default is the y-coordinate of the emitter's current location.
                 * @param count The number of particles to release from each emitter. The default is the emitter's own {@link Phaser.GameObjects.Particles.ParticleEmitter#quantity}.
                 */
                emitParticleAt(x?: number, y?: number, count?: integer): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Pauses this Emitter Manager.
                 *
                 * This has the effect of pausing all emitters, and all particles of those emitters, currently under its control.
                 *
                 * The particles will still render, but they will not have any of their logic updated.
                 */
                pause(): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Resumes this Emitter Manager, should it have been previously paused.
                 */
                resume(): Phaser.GameObjects.Particles.ParticleEmitterManager;

                /**
                 * Gets all active particle processors (gravity wells).
                 */
                getProcessors(): Phaser.GameObjects.Particles.GravityWell[];

                /**
                 * Updates all active emitters.
                 * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
                 * @param delta The delta time, in ms, elapsed since the last frame.
                 */
                preUpdate(time: integer, delta: number): void;

                /**
                 * The depth of this Game Object within the Scene.
                 *
                 * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
                 * of Game Objects, without actually moving their position in the display list.
                 *
                 * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
                 * value will always render in front of one with a lower value.
                 *
                 * Setting the depth will queue a depth sort event within the Scene.
                 */
                depth: number;

                /**
                 * The depth of this Game Object within the Scene.
                 *
                 * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
                 * of Game Objects, without actually moving their position in the display list.
                 *
                 * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
                 * value will always render in front of one with a lower value.
                 *
                 * Setting the depth will queue a depth sort event within the Scene.
                 * @param value The depth of this Game Object.
                 */
                setDepth(value: integer): this;

                /**
                 * The initial WebGL pipeline of this Game Object.
                 */
                defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

                /**
                 * The current WebGL pipeline of this Game Object.
                 */
                pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

                /**
                 * Sets the initial WebGL Pipeline of this Game Object.
                 * This should only be called during the instantiation of the Game Object.
                 * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
                 */
                initPipeline(pipelineName?: string): boolean;

                /**
                 * Sets the active WebGL Pipeline of this Game Object.
                 * @param pipelineName The name of the pipeline to set on this Game Object.
                 */
                setPipeline(pipelineName: string): this;

                /**
                 * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
                 */
                resetPipeline(): boolean;

                /**
                 * Gets the name of the WebGL Pipeline this Game Object is currently using.
                 */
                getPipelineName(): string;

                /**
                 * The x position of this Game Object.
                 */
                x: number;

                /**
                 * The y position of this Game Object.
                 */
                y: number;

                /**
                 * The z position of this Game Object.
                 * Note: Do not use this value to set the z-index, instead see the `depth` property.
                 */
                z: number;

                /**
                 * The w position of this Game Object.
                 */
                w: number;

                /**
                 * The horizontal scale of this Game Object.
                 */
                scaleX: number;

                /**
                 * The vertical scale of this Game Object.
                 */
                scaleY: number;

                /**
                 * The angle of this Game Object as expressed in degrees.
                 *
                 * Where 0 is to the right, 90 is down, 180 is left.
                 *
                 * If you prefer to work in radians, see the `rotation` property instead.
                 */
                angle: integer;

                /**
                 * The angle of this Game Object in radians.
                 *
                 * If you prefer to work in degrees, see the `angle` property instead.
                 */
                rotation: number;

                /**
                 * Sets the position of this Game Object.
                 * @param x The x position of this Game Object. Default 0.
                 * @param y The y position of this Game Object. If not set it will use the `x` value. Default x.
                 * @param z The z position of this Game Object. Default 0.
                 * @param w The w position of this Game Object. Default 0.
                 */
                setPosition(x?: number, y?: number, z?: number, w?: number): this;

                /**
                 * Sets the position of this Game Object to be a random position within the confines of
                 * the given area.
                 *
                 * If no area is specified a random position between 0 x 0 and the game width x height is used instead.
                 *
                 * The position does not factor in the size of this Game Object, meaning that only the origin is
                 * guaranteed to be within the area.
                 * @param x The x position of the top-left of the random area. Default 0.
                 * @param y The y position of the top-left of the random area. Default 0.
                 * @param width The width of the random area.
                 * @param height The height of the random area.
                 */
                setRandomPosition(x?: number, y?: number, width?: number, height?: number): this;

                /**
                 * Sets the rotation of this Game Object.
                 * @param radians The rotation of this Game Object, in radians. Default 0.
                 */
                setRotation(radians?: number): this;

                /**
                 * Sets the angle of this Game Object.
                 * @param degrees The rotation of this Game Object, in degrees. Default 0.
                 */
                setAngle(degrees?: number): this;

                /**
                 * Sets the scale of this Game Object.
                 * @param x The horizontal scale of this Game Object.
                 * @param y The vertical scale of this Game Object. If not set it will use the `x` value. Default x.
                 */
                setScale(x: number, y?: number): this;

                /**
                 * Sets the x position of this Game Object.
                 * @param value The x position of this Game Object. Default 0.
                 */
                setX(value?: number): this;

                /**
                 * Sets the y position of this Game Object.
                 * @param value The y position of this Game Object. Default 0.
                 */
                setY(value?: number): this;

                /**
                 * Sets the z position of this Game Object.
                 * @param value The z position of this Game Object. Default 0.
                 */
                setZ(value?: number): this;

                /**
                 * Sets the w position of this Game Object.
                 * @param value The w position of this Game Object. Default 0.
                 */
                setW(value?: number): this;

                /**
                 * Gets the local transform matrix for this Game Object.
                 * @param tempMatrix The matrix to populate with the values from this Game Object.
                 */
                getLocalTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

                /**
                 * Gets the world transform matrix for this Game Object, factoring in any parent Containers.
                 * @param tempMatrix The matrix to populate with the values from this Game Object.
                 * @param parentMatrix A temporary matrix to hold parent values during the calculations.
                 */
                getWorldTransformMatrix(tempMatrix?: Phaser.GameObjects.Components.TransformMatrix, parentMatrix?: Phaser.GameObjects.Components.TransformMatrix): Phaser.GameObjects.Components.TransformMatrix;

                /**
                 * The visible state of the Game Object.
                 *
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 */
                visible: boolean;

                /**
                 * Sets the visibility of this Game Object.
                 *
                 * An invisible Game Object will skip rendering, but will still process update logic.
                 * @param value The visible state of the Game Object.
                 */
                setVisible(value: boolean): this;

            }

            namespace Zones {
                /**
                 * A Death Zone.
                 *
                 * A Death Zone is a special type of zone that will kill a Particle as soon as it either enters, or leaves, the zone.
                 *
                 * The zone consists of a `source` which could be a Geometric shape, such as a Rectangle or Ellipse, or your own
                 * object as long as it includes a `contains` method for which the Particles can be tested against.
                 */
                class DeathZone {
                    /**
                     *
                     * @param source An object instance that has a `contains` method that returns a boolean when given `x` and `y` arguments.
                     * @param killOnEnter Should the Particle be killed when it enters the zone? `true` or leaves it? `false`
                     */
                    constructor(source: DeathZoneSource, killOnEnter: boolean);

                    /**
                     * An object instance that has a `contains` method that returns a boolean when given `x` and `y` arguments.
                     * This could be a Geometry shape, such as `Phaser.Geom.Circle`, or your own custom object.
                     */
                    source: DeathZoneSource;

                    /**
                     * Set to `true` if the Particle should be killed if it enters this zone.
                     * Set to `false` to kill the Particle if it leaves this zone.
                     */
                    killOnEnter: boolean;

                    /**
                     * Checks if the given Particle will be killed or not by this zone.
                     * @param particle The Particle to be checked against this zone.
                     */
                    willKill(particle: Phaser.GameObjects.Particles.Particle): boolean;

                }

                /**
                 * A zone that places particles on a shape's edges.
                 */
                class EdgeZone {
                    /**
                     *
                     * @param source An object instance with a `getPoints(quantity, stepRate)` method returning an array of points.
                     * @param quantity The number of particles to place on the source edge. Set to 0 to use `stepRate` instead.
                     * @param stepRate The distance between each particle. When set, `quantity` is implied and should be set to 0.
                     * @param yoyo Whether particles are placed from start to end and then end to start. Default false.
                     * @param seamless Whether one endpoint will be removed if it's identical to the other. Default true.
                     */
                    constructor(source: EdgeZoneSource, quantity: integer, stepRate: number, yoyo?: boolean, seamless?: boolean);

                    /**
                     * An object instance with a `getPoints(quantity, stepRate)` method returning an array of points.
                     */
                    source: EdgeZoneSource | RandomZoneSource;

                    /**
                     * The points placed on the source edge.
                     */
                    points: Phaser.Geom.Point[];

                    /**
                     * The number of particles to place on the source edge. Set to 0 to use `stepRate` instead.
                     */
                    quantity: integer;

                    /**
                     * The distance between each particle. When set, `quantity` is implied and should be set to 0.
                     */
                    stepRate: number;

                    /**
                     * Whether particles are placed from start to end and then end to start.
                     */
                    yoyo: boolean;

                    /**
                     * The counter used for iterating the EdgeZone's points.
                     */
                    counter: number;

                    /**
                     * Whether one endpoint will be removed if it's identical to the other.
                     */
                    seamless: boolean;

                    /**
                     * Update the {@link Phaser.GameObjects.Particles.Zones.EdgeZone#points} from the EdgeZone's
                     * {@link Phaser.GameObjects.Particles.Zones.EdgeZone#source}.
                     *
                     * Also updates internal properties.
                     */
                    updateSource(): Phaser.GameObjects.Particles.Zones.EdgeZone;

                    /**
                     * Change the EdgeZone's source.
                     * @param source An object instance with a `getPoints(quantity, stepRate)` method returning an array of points.
                     */
                    changeSource(source: EdgeZoneSource): Phaser.GameObjects.Particles.Zones.EdgeZone;

                    /**
                     * Get the next point in the Zone and set its coordinates on the given Particle.
                     * @param particle The Particle.
                     */
                    getPoint(particle: Phaser.GameObjects.Particles.Particle): void;

                }

                /**
                 * A zone that places particles randomly within a shape's area.
                 */
                class RandomZone {
                    /**
                     *
                     * @param source An object instance with a `getRandomPoint(point)` method.
                     */
                    constructor(source: RandomZoneSource);

                    /**
                     * An object instance with a `getRandomPoint(point)` method.
                     */
                    source: RandomZoneSource;

                    /**
                     * Get the next point in the Zone and set its coordinates on the given Particle.
                     * @param particle The Particle.
                     */
                    getPoint(particle: Phaser.GameObjects.Particles.Particle): void;

                }

            }

        }

        /**
         * A PathFollower Game Object.
         *
         * A PathFollower is a Sprite Game Object with some extra helpers to allow it to follow a Path automatically.
         *
         * Anything you can do with a standard Sprite can be done with this PathFollower, such as animate it, tint it,
         * scale it and so on.
         *
         * PathFollowers are bound to a single Path at any one time and can traverse the length of the Path, from start
         * to finish, forwards or backwards, or from any given point on the Path to its end. They can optionally rotate
         * to face the direction of the path, be offset from the path coordinates or rotate independently of the Path.
         */
        class PathFollower extends Phaser.GameObjects.Sprite {
            /**
             *
             * @param scene The Scene to which this PathFollower belongs.
             * @param path The Path this PathFollower is following. It can only follow one Path at a time.
             * @param x The horizontal position of this Game Object in the world.
             * @param y The vertical position of this Game Object in the world.
             * @param texture The key of the Texture this Game Object will use to render with, as stored in the Texture Manager.
             * @param frame An optional frame from the Texture this Game Object is rendering with.
             */
            constructor(scene: Phaser.Scene, path: Phaser.Curves.Path, x: number, y: number, texture: string, frame?: string | integer);

            /**
             * The Path this PathFollower is following. It can only follow one Path at a time.
             */
            path: Phaser.Curves.Path;

            /**
             * Should the PathFollower automatically rotate to point in the direction of the Path?
             */
            rotateToPath: boolean;

            /**
             * [description]
             */
            pathRotationVerticalAdjust: boolean;

            /**
             * If the PathFollower is rotating to match the Path (@see Phaser.GameObjects.PathFollower#rotateToPath)
             * this value is added to the rotation value. This allows you to rotate objects to a path but control
             * the angle of the rotation as well.
             */
            pathRotationOffset: number;

            /**
             * An additional vector to add to the PathFollowers position, allowing you to offset it from the
             * Path coordinates.
             */
            pathOffset: Phaser.Math.Vector2;

            /**
             * [description]
             */
            pathVector: Phaser.Math.Vector2;

            /**
             * The Tween used for following the Path.
             */
            pathTween: Phaser.Tweens.Tween;

            /**
             * Settings for the PathFollower.
             */
            pathConfig: PathConfig;

            /**
             * Set the Path that this PathFollower should follow.
             *
             * Optionally accepts {@link PathConfig} settings.
             * @param path The Path this PathFollower is following. It can only follow one Path at a time.
             * @param config Settings for the PathFollower.
             */
            setPath(path: Phaser.Curves.Path, config?: PathConfig): Phaser.GameObjects.PathFollower;

            /**
             * Set whether the PathFollower should automatically rotate to point in the direction of the Path.
             * @param value Whether the PathFollower should automatically rotate to point in the direction of the Path.
             * @param offset Rotation offset in degrees. Default 0.
             * @param verticalAdjust [description] Default false.
             */
            setRotateToPath(value: boolean, offset?: number, verticalAdjust?: boolean): Phaser.GameObjects.PathFollower;

            /**
             * Is this PathFollower actively following a Path or not?
             *
             * To be considered as `isFollowing` it must be currently moving on a Path, and not paused.
             */
            isFollowing(): boolean;

            /**
             * Starts this PathFollower following its given Path.
             * @param config The duration of the follow, or a PathFollower config object. Default {}.
             * @param startAt Optional start position of the follow, between 0 and 1. Default 0.
             */
            startFollow(config?: number | PathConfig, startAt?: number): Phaser.GameObjects.PathFollower;

            /**
             * Pauses this PathFollower. It will still continue to render, but it will remain motionless at the
             * point on the Path at which you paused it.
             */
            pauseFollow(): Phaser.GameObjects.PathFollower;

            /**
             * Resumes a previously paused PathFollower.
             *
             * If the PathFollower was not paused this has no effect.
             */
            resumeFollow(): Phaser.GameObjects.PathFollower;

            /**
             * Stops this PathFollower from following the path any longer.
             *
             * This will invoke any 'stop' conditions that may exist on the Path, or for the follower.
             */
            stopFollow(): Phaser.GameObjects.PathFollower;

            /**
             * Internal update handler that advances this PathFollower along the path.
             *
             * Called automatically by the Scene step, should not typically be called directly.
             * @param time The current timestamp as generated by the Request Animation Frame or SetTimeout.
             * @param delta The delta time, in ms, elapsed since the last frame.
             */
            protected preUpdate(time: integer, delta: number): void;

            /**
             * Clears all alpha values associated with this Game Object.
             *
             * Immediately sets the alpha levels back to 1 (fully opaque).
             */
            clearAlpha(): this;

            /**
             * Set the Alpha level of this Game Object. The alpha controls the opacity of the Game Object as it renders.
             * Alpha values are provided as a float between 0, fully transparent, and 1, fully opaque.
             *
             * If your game is running under WebGL you can optionally specify four different alpha values, each of which
             * correspond to the four corners of the Game Object. Under Canvas only the `topLeft` value given is used.
             * @param topLeft The alpha value used for the top-left of the Game Object. If this is the only value given it's applied across the whole Game Object. Default 1.
             * @param topRight The alpha value used for the top-right of the Game Object. WebGL only.
             * @param bottomLeft The alpha value used for the bottom-left of the Game Object. WebGL only.
             * @param bottomRight The alpha value used for the bottom-right of the Game Object. WebGL only.
             */
            setAlpha(topLeft?: number, topRight?: number, bottomLeft?: number, bottomRight?: number): this;

            /**
             * The alpha value of the Game Object.
             *
             * This is a global value, impacting the entire Game Object, not just a region of it.
             */
            alpha: number;

            /**
             * The alpha value starting from the top-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopLeft: number;

            /**
             * The alpha value starting from the top-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaTopRight: number;

            /**
             * The alpha value starting from the bottom-left of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomLeft: number;

            /**
             * The alpha value starting from the bottom-right of the Game Object.
             * This value is interpolated from the corner to the center of the Game Object.
             */
            alphaBottomRight: number;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             */
            blendMode: Phaser.BlendModes | string;

            /**
             * Sets the Blend Mode being used by this Game Object.
             *
             * This can be a const, such as `Phaser.BlendModes.SCREEN`, or an integer, such as 4 (for Overlay)
             *
             * Under WebGL only the following Blend Modes are available:
             *
             * * ADD
             * * MULTIPLY
             * * SCREEN
             *
             * Canvas has more available depending on browser support.
             *
             * You can also create your own custom Blend Modes in WebGL.
             *
             * Blend modes have different effects under Canvas and WebGL, and from browser to browser, depending
             * on support. Blend Modes also cause a WebGL batch flush should it encounter a new blend mode. For these
             * reasons try to be careful about the construction of your Scene and the frequency of which blend modes
             * are used.
             * @param value The BlendMode value. Either a string or a CONST.
             */
            setBlendMode(value: string | Phaser.BlendModes): this;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             */
            depth: number;

            /**
             * The depth of this Game Object within the Scene.
             *
             * The depth is also known as the 'z-index' in some environments, and allows you to change the rendering order
             * of Game Objects, without actually moving their position in the display list.
             *
             * The depth starts from zero (the default value) and increases from that point. A Game Object with a higher depth
             * value will always render in front of one with a lower value.
             *
             * Setting the depth will queue a depth sort event within the Scene.
             * @param value The depth of this Game Object.
             */
            setDepth(value: integer): this;

            /**
             * The horizontally flipped state of the Game Object.
             * A Game Object that is flipped horizontally will render inversed on the horizontal axis.
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             */
            flipX: boolean;

            /**
             * The vertically flipped state of the Game Object.
             * A Game Object that is flipped vertically will render inversed on the vertical axis (i.e. upside down)
             * Flipping always takes place from the middle of the texture and does not impact the scale value.
             */
            flipY: boolean;

            /**
             * Toggles the horizontal flipped state of this Game Object.
             */
            toggleFlipX(): this;

            /**
             * Toggles the vertical flipped state of this Game Object.
             */
            toggleFlipY(): this;

            /**
             * Sets the horizontal flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipX(value: boolean): this;

            /**
             * Sets the vertical flipped state of this Game Object.
             * @param value The flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlipY(value: boolean): this;

            /**
             * Sets the horizontal and vertical flipped state of this Game Object.
             * @param x The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             * @param y The horizontal flipped state. `false` for no flip, or `true` to be flipped.
             */
            setFlip(x: boolean, y: boolean): this;

            /**
             * Resets the horizontal and vertical flipped state of this Game Object back to their default un-flipped state.
             */
            resetFlip(): this;

            /**
             * Gets the center coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             */
            getCenter<O extends Phaser.Math.Vector2>(output?: O): O;

            /**
             * Gets the top-left corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getTopLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the top-right corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getTopRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bottom-left corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getBottomLeft<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bottom-right corner coordinate of this Game Object, regardless of origin.
             * The returned point is calculated in local space and does not factor in any parent containers
             * @param output An object to store the values in. If not provided a new Vector2 will be created.
             * @param includeParent If this Game Object has a parent Container, include it (and all other ancestors) in the resulting vector? Default false.
             */
            getBottomRight<O extends Phaser.Math.Vector2>(output?: O, includeParent?: boolean): O;

            /**
             * Gets the bounds of this Game Object, regardless of origin.
             * The values are stored and returned in a Rectangle, or Rectangle-like, object.
             * @param output An object to store the values in. If not provided a new Rectangle will be created.
             */
            getBounds<O extends Phaser.Geom.Rectangle>(output?: O): O;

            /**
             * The Mask this Game Object is using during render.
             */
            mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask;

            /**
             * Sets the mask that this Game Object will use to render with.
             *
             * The mask must have been previously created and can be either a GeometryMask or a BitmapMask.
             * Note: Bitmap Masks only work on WebGL. Geometry Masks work on both WebGL and Canvas.
             *
             * If a mask is already set on this Game Object it will be immediately replaced.
             *
             * Masks are positioned in global space and are not relative to the Game Object to which they
             * are applied. The reason for this is that multiple Game Objects can all share the same mask.
             *
             * Masks have no impact on physics or input detection. They are purely a rendering component
             * that allows you to limit what is visible during the render pass.
             * @param mask The mask this Game Object will use when rendering.
             */
            setMask(mask: Phaser.Display.Masks.BitmapMask | Phaser.Display.Masks.GeometryMask): this;

            /**
             * Clears the mask that this Game Object was using.
             * @param destroyMask Destroy the mask before clearing it? Default false.
             */
            clearMask(destroyMask?: boolean): this;

            /**
             * Creates and returns a Bitmap Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a renderable Game Object.
             * A renderable Game Object is one that uses a texture to render with, such as an
             * Image, Sprite, Render Texture or BitmapText.
             *
             * If you do not provide a renderable object, and this Game Object has a texture,
             * it will use itself as the object. This means you can call this method to create
             * a Bitmap Mask from any renderable Game Object.
             * @param renderable A renderable Game Object that uses a texture, such as a Sprite.
             */
            createBitmapMask(renderable?: Phaser.GameObjects.GameObject): Phaser.Display.Masks.BitmapMask;

            /**
             * Creates and returns a Geometry Mask. This mask can be used by any Game Object,
             * including this one.
             *
             * To create the mask you need to pass in a reference to a Graphics Game Object.
             *
             * If you do not provide a graphics object, and this Game Object is an instance
             * of a Graphics object, then it will use itself to create the mask.
             *
             * This means you can call this method to create a Geometry Mask from any Graphics Game Object.
             * @param graphics A Graphics Game Object. The geometry within it will be used as the mask.
             */
            createGeometryMask(graphics?: Phaser.GameObjects.Graphics): Phaser.Display.Masks.GeometryMask;

            /**
             * The horizontal origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the left of the Game Object.
             */
            originX: number;

            /**
             * The vertical origin of this Game Object.
             * The origin maps the relationship between the size and position of the Game Object.
             * The default value is 0.5, meaning all Game Objects are positioned based on their center.
             * Setting the value to 0 means the position now relates to the top of the Game Object.
             */
            originY: number;

            /**
             * The horizontal display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginX: number;

            /**
             * The vertical display origin of this Game Object.
             * The origin is a normalized value between 0 and 1.
             * The displayOrigin is a pixel value, based on the size of the Game Object combined with the origin.
             */
            displayOriginY: number;

            /**
             * Sets the origin of this Game Object.
             *
             * The values are given in the range 0 to 1.
             * @param x The horizontal origin value. Default 0.5.
             * @param y The vertical origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setOrigin(x?: number, y?: number): this;

            /**
             * Sets the origin of this Game Object based on the Pivot values in its Frame.
             */
            setOriginFromFrame(): this;

            /**
             * Sets the display origin of this Game Object.
             * The difference between this and setting the origin is that you can use pixel values for setting the display origin.
             * @param x The horizontal display origin value. Default 0.
             * @param y The vertical display origin value. If not defined it will be set to the value of `x`. Default x.
             */
            setDisplayOrigin(x?: number, y?: number): this;

            /**
             * Updates the Display Origin cached values internally stored on this Game Object.
             * You don't usually call this directly, but it is exposed for edge-cases where you may.
             */
            updateDisplayOrigin(): this;

            /**
             * The initial WebGL pipeline of this Game Object.
             */
            defaultPipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * The current WebGL pipeline of this Game Object.
             */
            pipeline: Phaser.Renderer.WebGL.WebGLPipeline;

            /**
             * Sets the initial WebGL Pipeline of this Game Object.
             * This should only be called during the instantiation of the Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object. Defaults to the Texture Tint Pipeline. Default TextureTintPipeline.
             */
            initPipeline(pipelineName?: string): boolean;

            /**
             * Sets the active WebGL Pipeline of this Game Object.
             * @param pipelineName The name of the pipeline to set on this Game Object.
             */
            setPipeline(pipelineName: string): this;

            /**
             * Resets the WebGL Pipeline of this Game Object back to the default it was created with.
             */
            resetPipeline(): boolean;

            /**
             * Gets the name of the WebGL Pipeline this Game Object is currently using.
             */
            getPipelineName(): string;

            /**
             * The Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             */
            scaleMode: Phaser.ScaleModes;

            /**
             * Sets the Scale Mode being used by this Game Object.
             * Can be either `ScaleModes.LINEAR` or `ScaleModes.NEAREST`.
             * @param value The Scale Mode to be used by this Game Object.
             */
            setScaleMode(value: Phaser.ScaleModes): this;

            /**
             * The horizontal scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorX: number;

            /**
             * The vertical scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             */
            scrollFactorY: number;

            /**
             * Sets the scroll factor of this Game Object.
             *
             * The scroll factor controls the influence of the movement of a Camera upon this Game Object.
             *
             * When a camera scrolls it will change the location at which this Game Object is rendered on-screen.
             * It does not change the Game Objects actual position values.
             *
             * A value of 1 means it will move exactly in sync with a camera.
             * A value of 0 means it will not move at all, even if the camera moves.
             * Other values control the degree to which the camera movement is mapped to this Game Object.
             *
             * Please be aware that scroll factor values other than 1 are not taken in to consideration when
             * calculating physics collisions. Bodies always collide based on their world position, but changing
             * the scroll factor is a visual adjustment to where the textures are rendered, which can offset
             * them from physics bodies if not accounted for in your code.
             * @param x The horizontal scroll factor of this Game Object.
             * @param y The vertical scroll factor of this Game Object. If not set it will use the `x` value. Default x.
             */
            setScrollFactor(x: number, y?: number): this;

            /**
             * The native (un-scaled) width of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayWidth` property.
             */
            width: number;

            /**
             * The native (un-scaled) height of this Game Object.
             *
             * Changing this value will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or use
             * the `displayHeight` property.
             */
            height: number;

            /**
             * The displayed width of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayWidth: number;

            /**
             * The displayed height of this Game Object.
             *
             * This value takes into account the scale factor.
             *
             * Setting this value will adjust the Game Object's scale property.
             */
            displayHeight: number;

            /**
             * Sets the size of this Game Object to be that of the given Frame.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param frame The frame to base the size of this Game Object on.
             */
            setSizeToFrame(frame: Phaser.Textures.Frame): this;

            /**
             * Sets the internal size of this Game Object, as used for frame or physics body creation.
             *
             * This will not change the size that the Game Object is rendered in-game.
             * For that you need to either set the scale of the Game Object (`setScale`) or call the
             * `setDisplaySize` method, which is the same thing as changing the scale but allows you
             * to do so by giving pixel values.
             *
             * If you have enabled this Game Object for input, changing the size will _not_ change the
             * size of the hit area. To do this you should adjust the `input.hitArea` object directly.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setSize(width: number, height: number): this;

            /**
             * Sets the display size of this Game Object.
             *
             * Calling this will adjust the scale.
             * @param width The width of this Game Object.
             * @param height The height of this Game Object.
             */
            setDisplaySize(width: number, height: number): this;

            /**
             * The Texture this Game Object is using to render with.
             */
            texture: Phaser.Textures.Texture | Phaser.Textures.CanvasTexture;

            /**
             * The Texture Frame this Game Object is using to render with.
             */
            frame: Phaser.Textures.Frame;

            /**
             * A boolean flag indicating if this Game Object is being cropped or not.
             * You can toggle this at any time after `setCrop` has been called, to turn cropping on or off.
             * Equally, calling `setCrop` with no arguments will reset the crop and disable it.
             */
            isCropped: boolean;

            /**
             * Applies a crop to a texture based Game Object, such as a Sprite or Image.
             *
             * The crop is a rectangle that limits the area of the texture frame that is visible during rendering.
             *
             * Cropping a Game Object does not change its size, dimensions, physics body or hit area, it just
             * changes what is shown when rendered.
             *
             * The crop coordinates are relative to the texture frame, not the Game Object, meaning 0 x 0 is the top-left.
             *
             * Therefore, if you had a Game Object that had an 800x600 sized texture, and you wanted to show only the left
             * half of it, you could call `setCrop(0, 0, 400, 600)`.
             *
             * It is also scaled to match the Game Object scale automatically. Therefore a crop rect of 100x50 would crop
             * an area of 200x100 when applied to a Game Object that had a scale factor of 2.
             *
             * You can either pass in numeric values directly, or you can provide a single Rectangle object as the first argument.
             *
             * Call this method with no arguments at all to reset the crop, or toggle the property `isCropped` to `false`.
             *
             * You should do this if the crop rectangle becomes the same size as the frame itself, as it will allow
             * the renderer to skip several internal calculations.
             * @param x The x coordinate to start the crop from. Or a Phaser.Geom.Rectangle object, in which case the rest of the arguments are ignored.
             * @param y The y coordinate to start the crop from.
             * @param width The width of the crop rectangle in pixels.
             * @param height The height of the crop rectangle in pixels.
             */
            setCrop(x?: number | Phaser.Geom.Rectangle, y?: number, width?: number, height?: number): this;

            /**
             * Sets the texture and frame this Game Object will use to render with.
             *
             * Textures are referenced by their string-based keys, as stored in the Texture Manager.
             * @param key The key of the texture to be used, as stored in the Texture Manager.
             * @param frame The name or index of the frame within the Texture.
             */
            setTexture(key: string, frame?: string | integer): this;

            /**
             * Sets the frame this Game Object will use to render with.
             *
             * The Frame has to belong to the current Texture being used.
             *
             * It can be either a string or an index.
             *
             * Calling `setFrame` will modify the `width` and `height` properties of your Game Object.
             * It will also change the `origin` if the Frame has a custom pivot point, as exported from packages like Texture Packer.
             * @param frame The name or index of the frame within the Texture.
             * @param updateSize Should this call adjust the size of the Game Object? Default true.
             * @param updateOrigin Should this call adjust the origin of the Game Object? Default true.
             */
            setFrame(frame: string | integer, updateSize?: boolean, updateOrigin?: boolean): this;

            /**
             * Fill or additive?
             */
            tintFill: boolean;

            /**
             * Clears all tint values associated with this Game Object.
             *
             * Immediately sets the color values back to 0x